/* --------------------------------------------------------------------------
 * module_test.c
 * code for test module
 *
 * Copyright 2002 Matthias Grimm
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 * -------------------------------------------------------------------------*/
#ifdef DEBUG

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <sys/time.h>

#include <pbb.h>

#include "gettext_macros.h"
#include "input_manager.h"
#include "module_peep.h"
#include "module_pmac.h"
#include "support.h"
#include "debug.h"

static struct tagtable tagtab[] = {
  { "TAG_SKIP",              TAG_SKIP},
  { "TAG_SAVECONFIG",        TAG_SAVECONFIG},
  { "TAG_USERALLOWED",       TAG_USERALLOWED},
  { "TAG_AUTORESCAN",        TAG_AUTORESCAN},
  { "TAG_TIMEFORCMD",        TAG_TIMEFORCMD},
  { "TAG_VERSION",           TAG_VERSION},
  { "TAG_REINIT",            TAG_REINIT}, 
  { "TAG_KEYCODE",           TAG_KEYCODE},
  { "TAG_KEYREPEAT",         TAG_KEYREPEAT},
  { "TAG_MODIFIER",          TAG_MODIFIER},
  { "TAG_MOUSEBUTTONS",      TAG_MOUSEBUTTONS},
  { "TAG_MOUSERELX",         TAG_MOUSERELX},
  { "TAG_MOUSERELY",         TAG_MOUSERELY},
  { "TAG_MOUSEWHEEL",        TAG_MOUSEWHEEL},
  { "TAG_SYSINFO",           TAG_SYSINFO},
  { "TAG_BACKLIGHTMAX",      TAG_BACKLIGHTMAX},
  { "TAG_BACKLIGHTLEVEL",    TAG_BACKLIGHTLEVEL},
  { "TAG_COVERSTATUS",       TAG_COVERSTATUS},
  { "TAG_BATLOG",            TAG_BATLOG},
  { "TAG_BATCYCLE",          TAG_BATCYCLE},
#if PEEP_NOCYCLIC
  { "TAG_TIMEREMAINING",     TAG_TIMEREMAINING | FLG_NOPEEP},
  { "TAG_TIMECHANGED",       TAG_TIMECHANGED | FLG_NOPEEP},
  { "TAG_POWERSOURCE",       TAG_POWERSOURCE | FLG_NOPEEP},
  { "TAG_AMBIENTLIGHT",      TAG_AMBIENTLIGHT | FLG_NOPEEP},
  { "TAG_BATTERYPRESENT",    TAG_BATTERYPRESENT | FLG_NOPEEP},
#else
  { "TAG_TIMEREMAINING",     TAG_TIMEREMAINING},
  { "TAG_TIMECHANGED",       TAG_TIMECHANGED},
  { "TAG_POWERSOURCE",       TAG_POWERSOURCE},
  { "TAG_AMBIENTLIGHT",      TAG_AMBIENTLIGHT},
  { "TAG_BATTERYPRESENT",    TAG_BATTERYPRESENT},
#endif
  { "TAG_POWERCHANGED",      TAG_POWERCHANGED},
  { "TAG_REQUESTSLEEP",      TAG_REQUESTSLEEP},
  { "TAG_SLEEPSUPPORTED",    TAG_SLEEPSUPPORTED},
  { "TAG_PREPAREFORSLEEP",   TAG_PREPAREFORSLEEP},
  { "TAG_WAKEUPFROMSLEEP",   TAG_WAKEUPFROMSLEEP},
  { "TAG_PMUDEVICE",         TAG_PMUDEVICE},
  { "TAG_ADBDEVICE",         TAG_ADBDEVICE},
  { "TAG_TPMODEUPKEY",       TAG_TPMODEUPKEY},
  { "TAG_TPMODEUPMOD",       TAG_TPMODEUPMOD},
  { "TAG_TPMODEDOWNKEY",     TAG_TPMODEDOWNKEY},
  { "TAG_TPMODEDOWNMOD",     TAG_TPMODEDOWNMOD},
  { "TAG_TPMODE",            TAG_TPMODE},
  { "TAG_KBDMODE",           TAG_KBDMODE},
  { "TAG_REPLACEPMUD",       TAG_REPLACEPMUD},
  { "TAG_KEYBLIGHTLEVEL",    TAG_KEYBLIGHTLEVEL },
  { "TAG_KEYBLIGHTMAX",      TAG_KEYBLIGHTMAX },
  { "TAG_LCDILLUMUPKEY",     TAG_LCDILLUMUPKEY},
  { "TAG_LCDILLUMUPMOD",     TAG_LCDILLUMUPMOD},
  { "TAG_LCDILLUMDOWNKEY",   TAG_LCDILLUMDOWNKEY},
  { "TAG_LCDILLUMDOWNMOD",   TAG_LCDILLUMDOWNMOD},
  { "TAG_KBDILLUMUPKEY",     TAG_KBDILLUMUPKEY},
  { "TAG_KBDILLUMUPMOD",     TAG_KBDILLUMUPMOD},
  { "TAG_KBDILLUMDOWNKEY",   TAG_KBDILLUMDOWNKEY},
  { "TAG_KBDILLUMDOWNMOD",   TAG_KBDILLUMDOWNMOD},
  { "TAG_KBDILLUMONKEY",     TAG_KBDILLUMONKEY},
  { "TAG_KBDILLUMONMOD",     TAG_KBDILLUMONMOD},
  { "TAG_LCDAUTOADJUST",     TAG_LCDAUTOADJUST },
  { "TAG_LCDBRIGHTNESS",     TAG_LCDBRIGHTNESS},
  { "TAG_LCDBRIGHTNESSMAX",  TAG_LCDBRIGHTNESSMAX},
  { "TAG_LCDFADINGSPEED",    TAG_LCDFADINGSPEED},
  { "TAG_LCDTHRESHOLD",      TAG_LCDTHRESHOLD },
  { "TAG_LCDAUTOADJMINBAT",  TAG_LCDAUTOADJMINBAT },
  { "TAG_LCDAUTOADJMAXBAT",  TAG_LCDAUTOADJMAXBAT },
  { "TAG_LCDAUTOADJMINAC",   TAG_LCDAUTOADJMINAC },
  { "TAG_LCDAUTOADJMAXAC",   TAG_LCDAUTOADJMAXAC },
  { "TAG_KBDAUTOADJUST",     TAG_KBDAUTOADJUST },
  { "TAG_KBDBRIGHTNESS",     TAG_KBDBRIGHTNESS},
  { "TAG_KBDONBRIGHTNESS",   TAG_KBDONBRIGHTNESS},
  { "TAG_KBDBRIGHTNESSMAX",  TAG_KBDBRIGHTNESSMAX},
  { "TAG_KBDFADINGSPEED",    TAG_KBDFADINGSPEED},
  { "TAG_KBDTHRESHOLD",      TAG_KBDTHRESHOLD },
  { "TAG_BRIGHTNESSOP",      TAG_BRIGHTNESSOP},
  { "TAG_FRAMEBUFFERDEVICE", TAG_FRAMEBUFFERDEVICE},
  { "TAG_BLANKFRAMEBUFFER",  TAG_BLANKFRAMEBUFFER},
  { "TAG_DIMFULLYDARK",      TAG_DIMFULLYDARK},
  { "TAG_MIXERDEVICE",       TAG_MIXERDEVICE},
  { "TAG_VOLUMEUPKEY",       TAG_VOLUMEUPKEY},
  { "TAG_VOLUMEUPMOD",       TAG_VOLUMEUPMOD},
  { "TAG_VOLUMEDOWNKEY",     TAG_VOLUMEDOWNKEY},
  { "TAG_VOLUMEDOWNMOD",     TAG_VOLUMEDOWNMOD},
  { "TAG_MUTEKEY",           TAG_MUTEKEY},
  { "TAG_MUTEMOD",           TAG_MUTEMOD},
  { "TAG_MIXERINITDELAY",    TAG_MIXERINITDELAY},
  { "TAG_VOLUME",            TAG_VOLUME },
  { "TAG_MUTE",              TAG_MUTE },
  { "TAG_MIXERCHANNELS",     TAG_MIXERCHANNELS },
  { "TAG_CDROMDEVICE",       TAG_CDROMDEVICE },
  { "TAG_EJECTCDKEY",        TAG_EJECTCDKEY },
  { "TAG_EJECTCDMOD",        TAG_EJECTCDMOD },
  { "TAG_EJECTCDKEYDELAY",   TAG_EJECTCDKEYDELAY },
  { "TAG_EJECTCD",           TAG_EJECTCD },
  { "TAG_SCRIPTPMCS",        TAG_SCRIPTPMCS },
  { "TAG_HEARTBEATBEEP",     TAG_HEARTBEATBEEP },
  { "TAG_SLEEPKEY",          TAG_SLEEPKEY },
  { "TAG_SLEEPMOD",          TAG_SLEEPMOD },
  { "TAG_SLEEPKEYDELAY",     TAG_SLEEPKEYDELAY },
  { "TAG_SLEEPKEYSLEEP",     TAG_SLEEPKEYSLEEP },
  { "TAG_GOTOSLEEP",         TAG_GOTOSLEEP },
  { "TAG_BWLFIRST",          TAG_BWLFIRST },
  { "TAG_BWLSECOND",         TAG_BWLSECOND },
  { "TAG_BWLLAST",           TAG_BWLLAST },
  { "TAG_CURRENTBWL",        TAG_CURRENTBWL },
  { "TAG_SLEEPINSECONDS",    TAG_SLEEPINSECONDS },
  { "TAG_CPULOADSLEEPLOCK",  TAG_CPULOADSLEEPLOCK },
  { "TAG_CPULOADMIN",        TAG_CPULOADMIN },
  { "TAG_CPULOADPERIOD",     TAG_CPULOADPERIOD },
  { "TAG_NETLOADSLEEPLOCK",  TAG_NETLOADSLEEPLOCK },
  { "TAG_NETLOADMIN",        TAG_NETLOADMIN },
  { "TAG_NETLOADPERIOD",     TAG_NETLOADPERIOD },
  { "TAG_NETLOADDEV",        TAG_NETLOADDEV },
  { "TAG_POLICY",            TAG_POLICY },
  { "TAG_ONAC_POLICY",       TAG_ONAC_POLICY },
  { "TAG_ONAC_TIMERACTION",  TAG_ONAC_TIMERACTION },
  { "TAG_ONAC_COVERACTION",  TAG_ONAC_COVERACTION },
  { "TAG_ONAC_KEYACTION",    TAG_ONAC_KEYACTION },
  { "TAG_ONAC_TIMESUSPEND",  TAG_ONAC_TIMESUSPEND },
  { "TAG_ONAC_TIMEDIM",      TAG_ONAC_TIMEDIM },
  { "TAG_ONBATT_POLICY",     TAG_ONBATT_POLICY },
  { "TAG_ONBATT_TIMERACTION",TAG_ONBATT_TIMERACTION },
  { "TAG_ONBATT_COVERACTION",TAG_ONBATT_COVERACTION },
  { "TAG_ONBATT_KEYACTION",  TAG_ONBATT_KEYACTION },
  { "TAG_ONBATT_TIMESUSPEND",TAG_ONBATT_TIMESUSPEND },
  { "TAG_ONBATT_TIMEDIM",    TAG_ONBATT_TIMEDIM },
  { "TAG_EMERGENCYACTION",   TAG_EMERGENCYACTION },
  { "TAG_IDENTITY",          TAG_IDENTITY },

  { "TAG_ONACPOLICY",        TAG_ONACPOLICY         | FLG_DEPRECIATED },
  { "TAG_ONACSLEEP",         TAG_ONACSLEEP          | FLG_DEPRECIATED },
  { "TAG_ONACCOVERSLEEP",    TAG_ONACCOVERSLEEP     | FLG_DEPRECIATED },
  { "TAG_ONACTIMESLEEP",     TAG_ONACTIMESLEEP      | FLG_DEPRECIATED },
  { "TAG_ONACTIMEDIM",       TAG_ONACTIMEDIM        | FLG_DEPRECIATED },
  { "TAG_ONBATTERYPOLICY",   TAG_ONBATTERYPOLICY    | FLG_DEPRECIATED },
  { "TAG_ONBATTERYSLEEP",    TAG_ONBATTERYSLEEP     | FLG_DEPRECIATED },
  { "TAG_ONBATTERYCOVERSLEEP",TAG_ONBATTERYCOVERSLEEP | FLG_DEPRECIATED },
  { "TAG_ONBATTERYTIMESLEEP",TAG_ONBATTERYTIMESLEEP | FLG_DEPRECIATED },
  { "TAG_ONBATTERYTIMEDIM",  TAG_ONBATTERYTIMEDIM   | FLG_DEPRECIATED },
  { "TAG_ONBATTERYTIMEHDOFF",TAG_ONBATTERYTIMEHDOFF | FLG_DEPRECIATED },
  { "TAG_ONBATTERYDIM",      TAG_ONBATTERYDIM       | FLG_DEPRECIATED },
  { "TAG_ONBATTERYBLANK",    TAG_ONBATTERYBLANK     | FLG_DEPRECIATED },
  { "TAG_ONACTIMEHDOFF",     TAG_ONACTIMEHDOFF      | FLG_DEPRECIATED },
  { "TAG_ONACDIM",           TAG_ONACDIM            | FLG_DEPRECIATED },
  { "TAG_ONACBLANK",         TAG_ONACBLANK          | FLG_DEPRECIATED },
  { "TAG_SCRIPTBATCRITICAL", TAG_SCRIPTBATCRITICAL  | FLG_DEPRECIATED },
  { "TAG_SCRIPTPROFCHANGED", TAG_SCRIPTPROFCHANGED  | FLG_DEPRECIATED }, 
  { "TAG_SCRIPTHDSETUP",     TAG_SCRIPTHDSETUP      | FLG_DEPRECIATED },
  { NULL, 0 }};

  static struct tagtable errortab[] = {
  { N_("Permission denied"),           E_PERM},
  { N_("Private Tag"),                 E_PRIVATE},
  { N_("File doesn't exist"),          E_NOEXIST},
  { N_("File not a charakter device"), E_NOCHAR},
  { N_("File not a block device"),     E_NOBLK},
  { N_("File not a file"),             E_NOFILE},
  { N_("Buffer overflow"),             E_BUFOVL},
  { N_("open error"),                  E_OPEN},
  { N_("format error"),                E_FORMAT},
  { N_("Messageport not available"),    E_MSGPORT},
  { N_("Server already running"),         E_TWICE},
  { N_("Help or version info"),             E_INFO},
  { N_("Unsecure script owner"),             E_USER},
  { N_("Script must be write-only by owner"), E_RIGHTS},
  { N_("read-only value"),			E_NOWRITE},
  { N_("write-only value"), 			E_NOREAD},
  { N_("argument invalid"),	  		 E_INVALID},
  { N_("function not supported"),		  E_NOSUPPORT},
  { NULL, 0 }};

struct moddata_peep {
  int timer10;
  int timer100;
  int timer1000;
  int secure;
} modbase_peep;

int
peep_init (struct tagitem *taglist)
{
	struct moddata_peep *base = &modbase_peep;

	base->timer10    = 0;
	base->timer100   = 0;
	base->timer1000  = 0;
	base->secure     = 0;

#if PEEP_KEYBOARD
	register_function (KBDQUEUE, peep_keyboard);
#endif
#if PEEP_MOUSE
	register_function (MOUSEQUEUE, peep_mouse);
#endif
#if PEEP_TIMER
	register_function (T10QUEUE, peep_t10);
	register_function (T100QUEUE, peep_t100);
	register_function (T1000QUEUE, peep_t1000);
#endif
	register_function (CONFIGQUEUE, peep_config);
	register_function (QUERYQUEUE, peep_query);
	register_function (SECUREQUEUE, peep_secure);
	return 0;
}

int
peep_exit ()
{
  return 0;
}

#if PEEP_KEYBOARD
void
peep_keyboard (struct tagitem *taglist)
{
	int code = (int) tagfind (taglist, TAG_KEYCODE, 0);
	int value = (int) tagfind (taglist, TAG_KEYREPEAT, 0);
	int mod = (int) tagfind (taglist, TAG_MODIFIER, 0);
	printf("\tKeyboard: code %3d, value %d, modifier %d\r", code, value, mod);
	fflush(stdout);
}
#endif

#if PEEP_MOUSE
void
peep_mouse (struct tagitem *taglist)
{
	int buttons = (int) tagfind (taglist, TAG_MOUSEBUTTONS, 0);
	int rel_x = (int) tagfind (taglist, TAG_MOUSERELX, 0);
	int rel_y = (int) tagfind (taglist, TAG_MOUSERELY, 0);
	int wheel = (int) tagfind (taglist, TAG_MOUSEWHEEL, -1);
	if (wheel == -1)
		printf("Mouse: buttons %x, rel-X %d, rel-Y %d\n", buttons, rel_x, rel_y);
	else
		printf("Mouse: buttons %x, rel-X %d, rel-Y %d, wheel %d\n", buttons, rel_x, rel_y, wheel);
}
#endif

#if PEEP_TIMER
void
peep_t10 (struct tagitem *taglist)
{
	struct moddata_peep *base = &modbase_peep;
	base->timer10++;
}

void
peep_t100 (struct tagitem *taglist)
{
	struct moddata_peep *base = &modbase_peep;
	base->timer100++;
}

void
peep_t1000 (struct tagitem *taglist)
{
	struct moddata_peep *base = &modbase_peep;
	base->timer1000++;
	printf ("timer10: %d   timer100: %d   timer1000: %d   secure: %d\n", base->timer10, base->timer100, base->timer1000, base->secure);
}
#endif

void
peep_ipc (struct tagitem *taglist)
{
#if PEEP_IPC
	peep_taglist ("IPC", taglist);
#endif
}

void
peep_config (struct tagitem *taglist)
{
#if PEEP_CONFIG
	peep_taglist ("Config", taglist);
#endif
}

void
peep_query (struct tagitem *taglist)
{
#if PEEP_QUERY
	peep_taglist ("Query", taglist);
#endif
}

void
peep_secure (struct tagitem *taglist)
{
#if PEEP_SECURE
	peep_taglist ("Secure", taglist);
#endif
}

void
peep_taglist (char *source, struct tagitem *taglist)
{
	struct tagtable *tt, *tt2;
	char mark;

	if (taglist != NULL)
		while (taglist->tag != TAG_END) {
			if ((tt = findttentry (tagtab, taglist->tag & TAGMASK))) {
				if (tt->tag & FLG_NOPEEP) {
					taglist++;
					continue;         /* skip tag */
				}

				if (taglist->tag & FLG_ERROR)
					mark = 'E';
				else if (tt->tag & FLG_DEPRECIATED)
					mark = 'D';
				else if (tt->tag & FLG_PRIVATE)
					mark = 'P';
				else
					mark = ' ';

				printf ("%-10s: %c %-25s: ", source, mark, tt->tagname);
				if (tt->tag == TAG_SKIP) {
					tt2 = findttentry (tagtab,  taglist->data);
					if (tt2 != NULL)
						printf ("%s", tt2->tagname);
					else
						printf ("???");
				} else if (taglist->tag & FLG_ERROR) {
					tt2 = findttentry (errortab,  taglist->data);
					if (tt2 != NULL)
						printf ("%s", tt2->tagname);
					else
						printf ("???");
				} else if (tt->tag & FLG_STRING) {
						printf ("\"%s\"", (char *) taglist->data);
				} else {
						printf ("%ld", taglist->data);
				}
				printf ("                    \n");
			} else
				printf ("%-10s:   %-25s: %lx\n", source, _("unknown tag"), taglist->tag & TAGMASK);
			taglist++;
		}
}

struct tagtable *
findttentry (struct tagtable *ttab, long tag)
{
	while (ttab->tag != 0) {
		if ((ttab->tag & TAGMASK) == tag)
			return ttab;
		ttab++;
	}
	return NULL;
}

#endif /* Debug */
