///
// Copyright (C) 2002 - 2004, Fredrik Arnerup & Rasmus Kaj, See COPYING
///
#include "textstream.h"
#include "getxsltparams.h"
#include <algorithm>
#include <fstream>
#include <sigc++/slot.h>

#include "util/processman.h"
#include "util/warning.h"
#include "util/stringutil.h"
#include "util/filesys.h"
#include "util/os.h"
#include "util/barrier.h"
#include "xml2ps/typesetter.hh"
#include "xml2ps/pdfcanvas.hh"

#include "textframe.h"
#include "page.h"

#include <sys/types.h>
#include <signal.h>

namespace {			// local
  class SortingPredicate {
  public:
    bool operator () (const TextFrame* a, const TextFrame* b) {
      try {
	int a_num = a->get_page_num(); 
	int b_num = b->get_page_num();
	if(a_num < b_num)
	  return true;
	if(a_num > b_num)
	  return false;
      } catch (Error::InvalidPageNum e) {
	// shouldn't happen, but it has happened
	return true;		// something broken; any order!
      }
      
      // Same page, look at boxes.
      const Vector a_vec = a->get_box()->getCorner(Corner::UL);
      const Vector b_vec = b->get_box()->getCorner(Corner::UL);
      static const float k = 0.1;  // 10% slope
      const float m = a_vec.y - k * a_vec.x;
      return b_vec.y < k * b_vec.x + m;
    }
  };

  bool gtk_main_running = false;

  /**
   * Start an external process running a XLST transform.
   * \return a Process handle
   */
  Process runXsltProc(const std::string& transform,
		      const std::string& association,
		      const TextStream::ParamIter& begin,
		      const TextStream::ParamIter& end) {
    debug << "Applying XSL Transform" << std::endl;
    // Assume xsltproc is in the path
    std::vector<std::string> command;
    command.push_back("xsltproc");
    command.push_back("--nonet");
    command.push_back("--catalogs");
    
    for(TextStream::ParamIter i = begin; i != end; ++i)
      if(!i->second.empty()) {
	command.push_back("--stringparam");
	command.push_back(i->first);
	command.push_back(i->second);
      }
    command.push_back(transform);
    command.push_back(association);
    
    Process xformproc = ProcessManager::instance().run(command);
    if(debug) {
      debug << "Started " << xformproc->get_pid() << ':';
      for(std::vector<std::string>::const_iterator i = command.begin();
	  i != command.end(); ++i)
	debug << ' ' << *i;
      debug << "" << std::endl;
    }
    return xformproc;
  }
}


class TextStream::WorkerThread : public SigC::Object {
public:
  static Glib::RefPtr<WorkerThread> create(TextStream& stream,
                                           const Frames& frames) {
    Glib::RefPtr<WorkerThread> tmp(new WorkerThread(stream, frames));
    tmp->reference();
    tmp->self = tmp;
    return tmp;
  }
  
  /// replaces SigC::Object::reference
  void reference() const { ++count_; }
  /// replaces SigC::Object::unreference
  void unreference() const { if(--count_ == 0) delete this; }

  bool is_done();
  
private:
  WorkerThread(TextStream& stream, const Frames& frames);
  ~WorkerThread();

  void run();
  void run_xml2ps(istream& source);

  void on_done();
  void on_error();
  
  void process_stopped(pid_t  pid, bool exited_normally, int exit_code);
  
  /// reference count
  mutable unsigned int count_;

  /// Keeps a reference to itself, so that the object is not deleted
  /// until self is set to null explicitly, which should happen in
  /// on_done() or on_error(). The idea is that the object should be
  /// kept from being deleted until we are sure the Dispatcher pipe is
  /// empty.
  Glib::RefPtr<WorkerThread> self;

  Barrier started, done;
  TextStream& stream;
  Frames sorted_frames;
  bool truncated;
  Glib::Dispatcher signal_done, signal_error;
  SigC::Connection proc_stop_connection;
  Glib::Thread* thread;
  Process xformproc;
  Glib::RefPtr<xml2ps::PsCanvas> new_canvas;
};

TextStream::WorkerThread::WorkerThread(TextStream& s, const Frames& frames)
  : count_(0), stream(s), sorted_frames(frames), truncated(false),
    proc_stop_connection(ProcessManager::instance().process_stopped.connect
			 (SigC::slot(*this, 
				     &WorkerThread::process_stopped)))
{
  sorted_frames.sort(SortingPredicate());
  
  signal_done.connect(SigC::slot(*this, &WorkerThread::on_done));
  signal_error.connect(SigC::slot(*this,  &WorkerThread::on_error));
  
  thread = Glib::Thread::create(SigC::slot(*this, &WorkerThread::run), false);
  debug << "Started thread " << thread << " from " << Glib::Thread::self()
	<< std::endl;
  
  started.wait();
  debug << "Thread " << thread << " initialized. "
	<< Glib::Thread::self() << " may continue." << std::endl;
}

TextStream::WorkerThread::~WorkerThread() {
  proc_stop_connection.disconnect();
  if(xformproc)
    kill(xformproc->get_pid(), SIGTERM);
}

bool TextStream::WorkerThread::is_done() {
  return done.get_open();
}


TextStream::TextStream(const std::string& _name, 
		       const std::string& _association,
		       const std::string& _transform)
  : name(_name), association(_association), transform(_transform),
    failed(false),
    association_watcher(association), transform_watcher(transform),
    typesetter_thread(0)
{
  typedef std::vector<std::string> Vec;
  Vec params = getXsltParams(transform);
  for(Vec::const_iterator i = params.begin(); i != params.end(); ++i)
    parameters[*i] = "";
  
  association_watcher.modified_signal.connect
    (SigC::slot(*this, &TextStream::on_file_modified));
  transform_watcher.modified_signal.connect
    (SigC::slot(*this, &TextStream::on_file_modified));

  // make sure Gtk::main() is running 
  Glib::signal_idle().connect(SigC::slot(*this, &TextStream::on_idle));
}

TextStream::TextStream(const ElementWrap& xml)
  : name(xml.get_attribute<string>("name")),
    association(xml.get_filename("file")),
    transform(xml.get_filename("transform")),
    failed(false),
    association_watcher(association), transform_watcher(transform),
    typesetter_thread(0)
{
  // read param defs
  typedef std::vector<std::string> Vec;
  Vec params = getXsltParams(transform);
  for(Vec::const_iterator i = params.begin(); i != params.end(); ++i)
    parameters[*i] = "";

  //read param values
  xmlpp::Element::NodeList children = xml.element().get_children();
  for(xmlpp::Element::NodeList::iterator i = children.begin();
      i != children.end();
      ++i) 
    {
      if(xmlpp::Element *pnode = dynamic_cast<xmlpp::Element*>(*i))
	if(pnode->get_name() == "parameter") {
	  set_parameter
	    (pnode->get_attribute("name")->get_value(),
	     pnode->get_attribute("value")->get_value());
	}
    }

  association_watcher.modified_signal.connect
    (SigC::slot(*this, &TextStream::on_file_modified));
  transform_watcher.modified_signal.connect
    (SigC::slot(*this, &TextStream::on_file_modified));

  // make sure Gtk::main() is running 
  Glib::signal_idle().connect(SigC::slot(*this, &TextStream::on_idle));
}

TextStream::~TextStream() {
  /// \todo  Wait for thread to finish or make sure thread can finish without
  /// this data.
  
  /// \todo  Do this on a copy of frames, after clearing frames.

  // remove_frame has the property that removing an unexisting frame is not an
  // error, just a no-op.
  for(Frames::iterator i = frames.begin(); i != frames.end(); i++)
    (*i)->set_stream(0, false);
  // do not let the frame remove itself from this stream
}

xmlpp::Element *TextStream::save(xmlpp::Element& parent_node, 
				 const FileContext &context) const 
{
  xmlpp::Element *tmp = parent_node.add_child("text_stream");
  tmp->set_attribute("name", name);
  tmp->set_attribute("file", context.to(association));
  tmp->set_attribute("transform", context.to(transform));
      
  for(TextStream::ParamIter
	j = param_begin(); j != param_end(); ++j)
    if(!j->second.empty()) {
      xmlpp::Element *par = tmp->add_child("parameter");
      par->set_attribute("name", j->first);
      par->set_attribute("value", j->second);
    }

  return tmp;
}

void TextStream::on_file_modified() {
  /// \todo  This method should do something that triggers a
  /// generate_ps_request from any associated TextFrame that actually is
  /// visible.  But for now, just call run_typesetter() directly.
  run_typesetter();
}

void TextStream::add_frame(TextFrame *text_frame) {
  // Don't insert a duplicate entry
  if(find(frames.begin(), frames.end(), text_frame) == frames.end())
    frames.push_back(text_frame);
}

void TextStream::remove_frame(TextFrame *text_frame) {
  Frames::iterator i = find(frames.begin(), frames.end(),
			    text_frame);
  if(i != frames.end()) 
    frames.erase(i);

  run_typesetter();
}

void TextStream::generate_ps_request(TextFrame *frame) {
  if(frame)
    debug << "Request by " << frame->get_name() << std::endl;

  if(failed) { // the last attempt failed
    failed = false; // so we can try again
    throw BasicFrame::GenPicError(BasicFrame::GENERATION, 
				  typesetter_error);
  }

  run_typesetter();
}

void TextStream::run_typesetter() {
  if(association.empty())
    throw BasicFrame::GenPicError(BasicFrame::ASSOCIATION, 
				  "No associated file");
  if(!access(association))
    throw BasicFrame::GenPicError(BasicFrame::ASSOCIATION, 
  				     "Could not open \"" + association + "\"");

  if(frames.empty())
    return;

  if(!gtk_main_running) {
    debug << "Not starting typesetter thread because Gtk::Main isn't running."
	  << std::endl;
    return;
  }
  
  /// \todo {Tell the old thread to stop and start a new one, this method is
  /// (should be) called when there is a _new_ change in the source or frame
  /// formats only.  This way we lag behind.
  /// ... or possibly just accept a little lag, but in that case, make sure a
  /// new typesetter_thread is started after the last one that was ignored.}
  
  // start typesetter in separate thread
  if(!typesetter_thread || typesetter_thread->is_done())
    typesetter_thread = WorkerThread::create(*this, frames);
  else
    debug << "Typesetter already working, not starting new." << std::endl;
}

void TextStream::WorkerThread::run() {
  // this method is supposed to run in a separate thread
  debug << "inside typesetter thread" << std::endl;
  try {
    if(stream.transform.empty()) {
      // Run xml2ps directly on the source xml
      std::ifstream source(stream.association.c_str());
      if(!source) {
	stream.typesetter_error = "Could not read from " + stream.association;
	throw std::runtime_error(stream.typesetter_error);
      }
      run_xml2ps(source);
      
    } else {
      xformproc = runXsltProc(stream.transform, stream.association,
			      stream.parameters.begin(),
			      stream.parameters.end());
      
      run_xml2ps(xformproc->get_cout());
    }
    debug << "Signalling done from " << Glib::Thread::self() << std::endl;
    done.open();
    signal_done();
    debug << " signalled done from " << Glib::Thread::self() << std::endl;
  } catch (...) {
    started.open(); // make sure the barrier actually opens even on errors.
    // have to let main thread resume
    // even if there is an error
    if(stream.typesetter_error.empty()) {
      try { throw; }
      catch (const std::exception& err) {
	stream.typesetter_error = err.what();
      } catch(...) {
	stream.typesetter_error = "unknown error";
      }
    }
    debug << "typesetter error: " << stream.typesetter_error << std::endl;
    done.open();
    signal_error();
  }
}

void TextStream::WorkerThread::run_xml2ps(std::istream& source) {
  xml2ps::Canvas::PageVec pages;
  for(Frames::const_iterator i = sorted_frames.begin(); 
      i != sorted_frames.end(); 
      i++) {
    (*i)->begin_write_ps();
    const Vector framesize = (*i)->get_inherent_size();
    pages.push_back(xml2ps::PageBoundary((*i)->get_page_num(),
					 framesize.x, framesize.y,
					 (*i)->get_num_columns(),
					 (*i)->get_gutter()));
    
    // A boundary for the actual text frame, to make sure it doesn't try to
    // flow around itself.
    const Boundary boundary = (*i)->get_obstacle_boundary();
    
    // Transform from page coords to local textframe coords.
    const Matrix xform((*i)->get_matrix().inv());
    
    // List of boundaries the text should flow around
    const BoundaryVect obstacle_list = Page::containing(**i).obstacle_list();
    for(BoundaryVect::const_iterator j = obstacle_list.begin();
	j != obstacle_list.end();
	j++) {
      if(boundary != *j) {
	pages.back().addObstacle(*j * xform);
      }
    }
  }
  
  // ok for caller to proceed:
  started.open();
  
  debug << "running internal xml2ps ..." << std::endl;
  new_canvas = (new xml2ps::PsCanvas(pages, false /* no extra pages */));
  new_canvas->setSubstituteFontAliases(true);
  xml2ps::PsConverter parser(*new_canvas.operator->());
  try {
    parser.parse_stream(source);
  } catch(const xml2ps::OutOfPages&) {
    truncated = true;
  }
}

/**
 * This method should run in the main GUI thread when the worker is done.
 */
void TextStream::WorkerThread::on_done() {
  // make sure the the WorkerThread won't keep any references to
  // itself after this method has returned.
  Glib::RefPtr<WorkerThread> tmp = self;
  self.clear();

  debug << "Acting on done signal in " << Glib::Thread::self()
	<< std::endl;
  stream.used_fonts = new_canvas->getUsedFonts();
  
  // The new canvas is now ready to get data from
  stream.pageno_map.clear();
  swap(new_canvas, stream.canvas);
  int num = 1;
  for(Frames::const_iterator i = sorted_frames.begin(); 
      i != sorted_frames.end(); 
      ++i, ++num) {
    stream.pageno_map[*i] = num;
    Frames::const_iterator j = i;
    (*i)->end_write_ps(true, truncated && (++j == sorted_frames.end()));
  }
  debug << "internal xml2ps done" << std::endl;
}

/**
 * This method should run in the main GUI thread if the worker fails.
 */
void TextStream::WorkerThread::on_error() {
  // make sure the the WorkerThread won't keep any references to
  // itself after this method has returned.
  Glib::RefPtr<WorkerThread> tmp = self;
  self.clear();

  for(Frames::const_iterator i = sorted_frames.begin(); 
      i != sorted_frames.end();  ++i)
    (*i)->end_write_ps(false, false, stream.typesetter_error);

  stream.failed = true; // report the last attempt to be a failure  

  /// \todo Report this in a better way!
  warning << "TextStream " << stream.name << ": "
	  << stream.typesetter_error << std::endl;
}

void TextStream::WorkerThread::process_stopped(pid_t pid, 
					       bool exited_normally, 
					       int exit_code)
{
  if(!xformproc || pid != xformproc->get_pid())
    return;
  
  if(!exited_normally) {
    warning << "Process " << pid << " exited abnormally" << std::endl;
    return; /// \todo do something clever
  }

  if(exit_code != 0) {
    /// \todo check that the return code 127 is not unique to Bash.
    if(exit_code == 127) // the shell could not find xsltproc
      warning << "xsltproc is not in $PATH" << std::endl;
    else
      warning << "xsltproc process with pid " << pid
	      << " failed with exit code " << exit_code << std::endl;
    return;
  }
  
  debug << "Stylesheet done" << std::endl;
}

void TextStream::set_association(const std::string &s) {
  association = s;
  // perhaps it has been applied, but not for this file
  try { on_file_modified(); } 
  catch(const BasicFrame::GenPicError& e) {
    warning << e.what() << std::endl;
  }
  association_watcher.set_file(association);
}

const std::string &TextStream::get_association() const { return association; } 

void TextStream::set_name(const std::string &s) { name = s; }

const std::string &TextStream::get_name() const { return name; } 

void TextStream::set_transform(const std::string& s) {
  if(s == transform)
    return;
  
  transform = s;
  typedef std::vector<std::string> Vec;
  Vec params = getXsltParams(transform);
  for(Vec::const_iterator i = params.begin(); i != params.end(); ++i)
    parameters[*i] = "";
  on_file_modified();
  transform_watcher.set_file(transform);
}

const std::string &TextStream::get_transform() const { return transform; }

void TextStream::set_parameter(const std::string& name, 
			       const std::string& value) {
  parameters[name] = value;
  on_file_modified();
}

void TextStream::outputPageRaw(std::ostream& out, const TextFrame* frame) {
  if(canvas) {
    canvas->closePage();
    canvas->appendPage(pageno_map[frame], out);
  } else
    throw std::runtime_error("Missing canvas while running outputPageRaw");
}
  
void TextStream::outputPageEps(std::ostream& out, const TextFrame* frame) {
  if(canvas) {
    canvas->closePage();
    canvas->pageEps(pageno_map[frame], out);
  } else
    throw std::runtime_error("Missing canvas while running outputPageEps");
}


void TextStream::print_pdf(PDF::Document::Ptr pdfdoc) {
  // don't try to print streams with no pages
  if(frames.empty())
    return;

  debug << "TextStream:p-pdf: a\n";
  Process xformproc;
  // run xsltproc, but only if there is a stylesheet
  if(!transform.empty()) {
    xformproc = runXsltProc(transform, association,
			    parameters.begin(), parameters.end());
  }

  debug << "TextStream:p-pdf: c\n";
  
  xml2ps::Canvas::PageVec pages;
  Frames sorted_frames(frames);
  sorted_frames.sort(SortingPredicate());
  for(Frames::const_iterator i = sorted_frames.begin(); 
      i != sorted_frames.end(); 
      i++) {
    const Vector framesize = (*i)->get_inherent_size();
    pages.push_back(xml2ps::PageBoundary((*i)->get_page_num(),
					 framesize.x, framesize.y,
					 (*i)->get_num_columns(),
					 (*i)->get_gutter()));
    
    // A boundary for the actual text frame, to make sure it doesn't try to
    // flow around itself.
    const Boundary boundary = (*i)->get_obstacle_boundary();
    
    // Transform from page coords to local textframe coords.
    const Matrix xform((*i)->get_matrix().inv());
    
    // List of boundaries the text should flow around
    const BoundaryVect obstacle_list = Page::containing(**i).obstacle_list();
    for(BoundaryVect::const_iterator j = obstacle_list.begin();
	j != obstacle_list.end();
	j++) {
      if(boundary != *j) {
	pages.back().addObstacle(*j * xform);
      }
    }
  }
  
  debug << "TextStream:p-pdf: running internal xml2ps ..." << std::endl;
  
  pdfcanvas.reset(new xml2ps::PDFCanvas(pdfdoc, xml2ps::PDFCanvas::XObjs,
					pages,
					false /* no extra pages */));
  //canvas.setSubstituteFontAliases(true);
  debug << "TextStream:p-pdf: parser" << std::endl;
  xml2ps::PsConverter parser(*pdfcanvas);
  try {
    debug << "TextStream:p-pdf: parsing" << std::endl;
    if(xformproc) {
      parser.parse_stream(xformproc->get_cout());
    } else {
      std::ifstream file(association.c_str());
      parser.parse_stream(file);
    }
    pdfcanvas->closePage();
  } catch(const xml2ps::OutOfPages&) {
    warning << "content truncated\n";
  }
  debug << "TextStream:p-pdf: done" << std::endl;
}

PDF::Object::Ptr TextStream::getPagePDF(const TextFrame* frame) {
  if(pdfcanvas.get()) {
    return pdfcanvas->getPageObject(pageno_map[frame]);
  } else
    throw std::runtime_error("Missing canvas while running getPagePDF");
}


// a hack to make sure the gtkmm main loop is running before we
// start any threads
bool TextStream::on_idle() {
  if(!gtk_main_running) {
    debug << "Gtk::Main is running" << std::endl;
    // if there has been an idle signal, then everything is cool
    gtk_main_running = true;
  }
  if(!association.empty())
    run_typesetter();
  // disconnect from handler
  return false;
}
