# ui-mbtools.tcl --
#
#       FIXME: This file needs a description here.
#
# Copyright (c) 1997-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
# @(#) $Header: /usr/mash/src/repository/mash/mash-1/tcl/mb/ui-mbtools.tcl,v 1.19 2002/02/03 04:27:17 lim Exp $


import MBWidget tips mb_utils

Class MBTool

#
# Import classes here after we defined MBTool
#
import MBImportTool MBFHTool MBTextTool MBShapeTool MBSelectTool MBEraseTool

#
# Contains routines that manipulates the screen
#

#
# Starts up netscape
#
proc StartNetscape {url} {

	set tmp [eval {exec netscape -remote openURL($url) >& /dev/null} ]
	if { $tmp == 0} {
		set err [catch { eval {exec netscape $url &} }]
		if { $err == 0} {
			puts "Could not start netscape."
		}
	}
}

MBTool instproc init {toolbar mgr sender} {
	$self set toolbar_ $toolbar
	$self set sender_ $sender
	$self set mgr_ $mgr
}

#
# returns the current property of a tool
#
MBTool instproc property {tool} {
	$self instvar toolbar_
	return [[$toolbar_ set mbui_] property $tool]
}

# virtual function called AFTER we switch modes or switch page
# NOTE when it is called, the modes/page must already have changed!
# FIXME: not there yet
MBTool instproc deactivate {} {
}

#
# Selection Tool: For Copy and Move
#
Class MBSelectTool -superclass MBTool

MBSelectTool instproc init {toolbar mgr sender} {
	$self next $toolbar $mgr $sender
	$self set selectedObj_ {}
}

#
# starts the <i>move</i> operation
#
MBSelectTool instproc activate_move { page_id } {
	$self instvar canv_ page_id_ mgr_
	set page_id_ $page_id
	set canv_ [[$mgr_ page_manager] page2canv $page_id]
	$canv_ resetBindings
	$canv_ setHilit
	$canv_ config -cursor arrow

	set w [$canv_ get_win]
	bind $w <Button-1> "$self start_move \[$canv_ canvasxy %x %y\]"
	bind $w <B1-Motion> "$self move \[$canv_ canvasxy %x %y\]"
	bind $w <ButtonRelease-1> "$self end_move \[$canv_ canvasxy %x %y\]"
}

#
# Duplicate an item, almost the same as moving, except the first step
# entails a <i>copy</i> operation
#
MBSelectTool instproc activate_copy { page_id } {
	$self instvar canv_ page_id_ mgr_
	set page_id_ $page_id
	set canv_ [[$mgr_ page_manager] page2canv $page_id]
	$canv_ resetBindings
	$canv_ setHilit
	$canv_ config -cursor arrow

	set w [$canv_ get_win]
	bind $w <Button-1> "$self start_copy \[$canv_ canvasxy %x %y\]"
	bind $w <B1-Motion> "$self move \[$canv_ canvasxy %x %y\]"
	bind $w <ButtonRelease-1> "$self end_move \[$canv_ canvasxy %x %y\]"
}

#
# copy the nearest object, if it is close enough
#
MBSelectTool instproc start_copy {pt} {
	$self instvar selectedObj_ lastpt_ startpt_ sender_ canv_ mgr_ toolbar_
	set nearbyObjs [eval $sender_ nearest $pt 2]
    	set pgId [[$mgr_ page_manager] current_page]

	if { [llength $nearbyObjs] > 0 } {
		set obj [lindex $nearbyObjs 0]
		set selectedObj_ [$sender_ dup_item $obj]

		$toolbar_ add_item $pgId $selectedObj_
		set lastpt_ $pt
		set startpt_ $pt
		$canv_ config -cursor fleur
		$sender_ interactive 1
	}
}

#
# selects the nearest object
#
MBSelectTool instproc start_move {pt} {
	$self instvar selectedObj_ lastpt_ startpt_ sender_ canv_
	set nearbyObjs [eval $sender_ nearest $pt 2]

	if { [llength $nearbyObjs] > 0 } {
		set selectedObj_ [lindex $nearbyObjs 0]
		set lastpt_ $pt
		set startpt_ $pt
		$canv_ config -cursor fleur
		$sender_ interactive 1
	}
}


#
# Moves selected object, if any
#
MBSelectTool instproc move {pt} {
	$self instvar selectedObj_ lastpt_ sender_ canv_

	if {$selectedObj_ != {}} {
		set dx [expr {[lindex $pt 0] - [lindex $lastpt_ 0]}]
		set dy [expr {[lindex $pt 1] - [lindex $lastpt_ 1]}]
		$sender_ move $selectedObj_ $dx $dy
		set lastpt_ $pt
		$canv_ hilit
	}
}

#
# End of one move operation
#
# - put current object on top of the display list
#
MBSelectTool instproc end_move {pt} {
    	$self instvar selectedObj_ lastpt_ startpt_ sender_ canv_ page_id_ \
			toolbar_
	if {$selectedObj_ == {}} {
		return
	} else {
		$canv_ config -cursor arrow
		# move object back
		set dx [expr {[lindex $startpt_ 0] - [lindex $lastpt_ 0]}]
		set dy [expr {[lindex $startpt_ 1] - [lindex $lastpt_ 1]}]
		$sender_ move $selectedObj_ $dx $dy

		$sender_ interactive 0
		# do the real move
		set newobj [$sender_ -page $page_id_ move $selectedObj_ \
			    [expr {[lindex $pt 0] - [lindex $startpt_ 0]}] \
			    [expr {[lindex $pt 1] - [lindex $startpt_ 1]}]]

		# DbgOut "New obj=$newobj"
		$toolbar_ replace_item $page_id_ $selectedObj_ $newobj
		set selectedObj_ {}
	}
	$canv_ resetMarker
}

# use last pt tracked from motion, this could be a bit different
# that the current point, but is the best we can do
# FIXME: should use pointerx stuff to get more accurate positioning of last pt
MBSelectTool instproc deactivate {} {
	$self instvar selectedObj_ lastpt_
	if { [info exists selectedObj_] && [info exists lastpt_] } {
		$self end_move $lastpt_
	}
}

#
# MBTextTool
#    - maintains one text tool on one group of canvas
#
Class MBTextTool -superclass MBTool

MBTextTool instproc init {toolbar mgr sender} {
	$self next $toolbar $mgr $sender
	$self set textLast_ {}
	$self set currText_ {}
}

MBTextTool instproc activate { page_id } {
	$self instvar page_id_ canv_ mgr_

	set page_id_ $page_id
	set canv_ [[$mgr_ page_manager] page2canv $page_id]
	$canv_ resetBindings

	$canv_ config -cursor xterm
    	$self set textLast_ {}
	$self set currText_ {}

	set w [$canv_ get_win]
	bind $w <Button-1> "$self new_group $canv_ \[$canv_ canvasxy %x %y\]"
}

#
# select position to type a new group of text
#
MBTextTool instproc new_group {canv pt} {
	$self instvar canv_ currText_ sender_
	# end the previous group, if any
	if [info exists canv_] {
		$self end_group
	}
	set canv_ $canv

	$self set textLast_ {}

	# item created will have tag currtext
	set currText_ [eval $sender_ create_item text $pt \
		    [$self property text]]

	$canv focus currtext
	$canv icursor currtext 1

	set w [$canv get_win]

	$canv bind currtext <Any-Key>   "$self insert_char %A; break"
	$canv bind currtext <Return>    "$self insert_char \\n"
	$canv bind currtext <Control-h> [list $self insert_char \b]
	$canv bind currtext <Tab>       "$self insert_char \\t; break"
	$canv bind currtext <BackSpace> [$canv bind currtext <Control-h>]
	$canv bind currtext <Delete> [$canv bind currtext <Control-h>]
#	bind $w <<Paste>> [list $self paste $canv]
	# don't insert control characters
	$canv bind currtext <Control-v> {}
	$canv bind currtext <Control-y> {}
#	$canv bind currtext <Shift-Insert> [$canv bind currtext <Control-v>]
}

#
# Returns whether we are in the middle of defining a new group
# in the given canvas widget
#
MBTextTool instproc pending {canvWgt} {
	$self instvar canv_ currText_
	if {![info exist canv_]} {
		return 0
	}
	return [expr { $canvWgt==[$canv_ get_win] &&  {} != $currText_ } ]
}

#
# insert one string of characters
#
MBTextTool instproc insert_string {str} {
	#mtrace trcMB "inserting string {$str}"
	# REVIEW: we should make MBSender take multiple characters
	for {set i 0} {$i < [string length $str]} {incr i} {
		#mtrace trcMB [string index $str $i]
		$self insert_char [string index $str $i]
	}
}

#
# Pastes the current selection into the pending text group, if there is one,
#   otherwise creates a new group with the selection.
#
MBTextTool instproc paste {canv {pt {}}} {
	# interrupt the current tool
	$self instvar toolbar_
	set currTool [$toolbar_ current_tool]
	if {$currTool!="text"} {
		$toolbar_ deactivate_tool $currTool
	}

	set canvWgt [$canv get_win]
	set oldcursor [$canv show_busy 1]
	# get selection from either PRIMARY or CLIPBOARD,
	# ask using default type STRING
	if [catch {selection get} sel] {
		if [catch {selection get -selection CLIPBOARD} sel] {
			$canv show_busy 0 $oldcursor
			return
		}
	}
	if {$sel == {}} {
		$canv show_busy 0 $oldcursor
		return
	}

	# if we are not in text mode or not working on a pending group,
	# create the text item as a separate text item, otherwise insert
	# the selection into the current text item
	set newgroup 0
	$self instvar toolbar_
	if {[$toolbar_ current_tool] != "text" || \
		    ![$self pending $canvWgt]} {
		set newgroup 1
	}

	if {$newgroup} {
		if {$pt == {}} {
			set pt [$canv canvasxy 0.0 0.0]
		}
		$self new_group $canv $pt
	}
	$self insert_string $sel
	if {$newgroup} {
		$self end_group
	}
	if {[$toolbar_ current_tool] != "text"} {
		$canvWgt focus {}
	}
	$canv show_busy 0 $oldcursor
}

#
# Insert one character
#
MBTextTool instproc insert_char {char} {
	if {$char=={}} { return	}
	$self instvar canv_ currText_ textLast_ sender_
	# move cursor to the end (play safe)
	$canv_ icursor currtext end
	set i [$canv_ index currtext insert]
	set textLast_ [$sender_ insert $currText_ $i $char]
	# move cursor again
	$canv_ icursor currtext end
}

#
# define the pending group
#
MBTextTool instproc end_group {} {
	$self instvar canv_ textLast_ currText_ sender_ mgr_ toolbar_
	if {![info exists canv_]} { return }
	set tags [$canv_ gettags currtext]

	# create the new text group if we are working on one
	if {$tags!={}} {
		set page_id [[$mgr_ page_manager] canv2page $canv_]
		if {$textLast_!={} && $currText_!={} \
				&& $currText_ < $textLast_} {
			set newobj [$sender_ -page $page_id \
				    create_item group text \
				    $currText_ $textLast_]
			$toolbar_ add_item $page_id $newobj
		}
		set currText_ {}
		set textLast_ {}
	}
	#    set tags [$canv gettags currtext]
	#    DbgOut "tags for currtext after grouping: $tags"
	$canv_ dtag currtext currtext
}

MBTextTool instproc deactivate {} {
	$self end_group
}

#
# Class MBEraseTool --
#     -- the Eraser
#
Class MBEraseTool -superclass MBTool

MBEraseTool instproc init {toolbar mgr sender} {
	$self next $toolbar $mgr $sender
}

#
# Erase mode activated
#
MBEraseTool instproc activate { page_id } {
	$self instvar mgr_
	set canv [[$mgr_ page_manager] page2canv $page_id]
	$canv resetBindings
	$canv setHilit

	$canv config -cursor pirate
	set w [$canv get_win]

	bind $w <Button-3>  "$self erase_last $page_id"
	bind $w <Button-1>  "$self erase_here \[$canv canvasxy %x %y\]"
	bind $w <B1-Motion> "$self erase_here \[$canv canvasxy %x %y\]"

}

#
# Erase last drawn item on the current page
#
MBEraseTool instproc erase_last {pageid} {
	# interrupt the current tool
        $self instvar toolbar_
	$toolbar_ deactivate_tool [$toolbar_ current_tool]

	$self instvar trashList_ sender_ mgr_ toolbar_

	set lastItem [$toolbar_ get_item $pageid end]
	if {$lastItem != {}} {
		if ![info exists trashList_($pageid)] {
			set trashList_($pageid) {}
		}
		set trashList_($pageid) \
				[linsert $trashList_($pageid) 0 $lastItem]

		$sender_ delete_item $lastItem
		$toolbar_ replace_item $pageid $lastItem {}
	}
	[[$mgr_ page_manager] page2canv $pageid] unhilit
}

#
# Unerase last erased item
#
MBEraseTool instproc unerase { pageid } {
	$self instvar trashList_ sender_ toolbar_

	if {![info exists trashList_($pageid)] \
			|| $trashList_($pageid) == {}} {
		return
	}

	set obj [lindex $trashList_($pageid) 0]
	set trashList_($pageid) [lrange $trashList_($pageid) 1 end]

	if [catch {$sender_ dup_item $obj} id] {
		puts stderr "unerase failed!"
	} else {
		$toolbar_ add_item $pageid $id
	}
}

#
# Erase the objects close to $pt
#
#
MBEraseTool instproc erase_here {pt} {
	$self instvar trashList_ sender_ mgr_ toolbar_
	set killObjs [eval $sender_ nearest $pt 2]

	# DbgOut "KillObjs= ($killObjs)"
	set pageid [[$mgr_ page_manager] current_page]
	foreach killObj $killObjs {
		if ![info exists trashList_($pageid)] {
			set trashList_($pageid) {}
		}
		set trashList_($pageid) \
				[linsert $trashList_($pageid) 0 $killObj]
		$sender_ delete_item $killObj
		$toolbar_ replace_item $pageid $killObj {}
		[[$mgr_ page_manager] page2canv $pageid] unhilit
	}
}

#---------------------------------------------------
#  Class MBShapeTool --
#
#   Shapes = (rect, oval, straight line)
#
#  This group of items are specified by pressing on
#  the starting point, dragging to the end-point,
#  and release the button
#
#---------------------------------------------------

Class MBShapeTool -superclass MBTool

MBShapeTool instproc init {toolbar mgr sender} {
	$self next $toolbar $mgr $sender
	$self set fakeObj_ {}
}

#
# Change current shape to "type"
#
MBShapeTool instproc activate_shape { type page_id } {
	$self instvar type_ canv_ page_id_ mgr_
	set page_id_ $page_id
	set canv_ [[$mgr_ page_manager] page2canv $page_id]
	$canv_ resetBindings
	$canv_ config -cursor tcross
	set type_ $type

	set w [$canv_ get_win]
	bind $w <Button-1> "$self begin %x %y"
	bind $w <B1-Motion> \
		    "$self motion \[$w canvasx %x\] \[$w canvasy %y\]"
	bind $w <ButtonRelease-1> "$self end %x %y"
}

#
# Start of shape drawing mode
#   create the shape
#
MBShapeTool instproc begin {rootx rooty} {
	$self instvar fakestartpt_ startpt_ fakeObj_ canv_ type_
	set startpt_ [$canv_ canvasxy $rootx $rooty]
	set x [$canv_ canvasx $rootx]
	set y [$canv_ canvasy $rooty]
	set fakestartpt_ [list $x $y]
	# Note: the created obj is an tkCanvas object and not recognized
	#       by MBCanvas!

	set fakeObj_ [eval $canv_ create $type_ $x $y $x $y \
		    [$self property $type_]]
}

#
# Mouse is moving while in shape drawing mode
#
MBShapeTool instproc motion {x y} {
	$self instvar fakestartpt_ fakeObj_ canv_
	if {$fakeObj_ == {}} { return }

	$canv_ config -cursor cross
	eval $canv_ coords $fakeObj_ $fakestartpt_ $x $y
}

#
# Mouse released in shape drawing mode
#
MBShapeTool instproc end {rootX rootY} {
	$self instvar startpt_ fakeObj_ sender_ canv_ type_ page_id_ toolbar_
	if {$fakeObj_ == {}} { return }
	$canv_ delete $fakeObj_
	set endpt [$canv_ canvasxy $rootX $rootY]
	set newobj [eval $sender_ -page $page_id_ \
		    create_item $type_ $startpt_ \
		    $endpt [$self property $type_]]
	$toolbar_ add_item $page_id_ $newobj
	set startpt_ {}
	set fakeObj_ {}
	$canv_ config -cursor tcross
}

MBShapeTool instproc deactivate {} {
	mtrace trcExcessive "$class deactivate"
	$self instvar canv_
	set w [$canv_ get_win]
	set rx [winfo rootx $w]
	set wx [winfo pointerx $w]
	set ry [winfo rooty $w]
	set wy [winfo pointery $w]

	$self end [expr {$wx - $rx}] [expr {$wy - $ry}]
}


#---------------------------------------------------
#
# Freehand sketching
#
#    Press and hold down to sketch, release to stop
#
#---------------------------------------------------

#
# Free hand drawing Tool
#
Class MBFHTool -superclass MBTool

MBFHTool instproc init {toolbar mgr sender} {
	$self next $toolbar $mgr $sender
}

#
# Change the mode to freehand
#
MBFHTool instproc activate { page_id } {
	$self instvar canv_ page_id_ mgr_
	set page_id_ $page_id
	set canv_ [[$mgr_ page_manager] page2canv $page_id]
	$canv_ config -cursor pencil
	$canv_ resetBindings
	set w [$canv_ get_win]
	$self set firstpt_ {}
	$self set start_ {}

	bind $w <Button-1>  "$self begin \[$canv_ canvasxy %x %y\]"
	bind $w <B1-Motion> "$self motion %x %y"
	bind $w <ButtonRelease-1> "$self end"
}

#
# create the line segment and start interactive mode
#
MBFHTool instproc begin {pt} {
	$self instvar start_ last_ prop_ firstpt_ sender_

	set firstpt_ $pt
	set prop_ [$self property plainline]
	set start_ [eval $sender_ create_item line $pt $pt $prop_]
	set last_ $start_
}

#
# Mouse is moving while in freehand mode
#
MBFHTool instproc motion {x y} {
	$self instvar start_ last_ prop_ firstpt_ sender_ canv_
	if {$firstpt_ == {}} return
	set pt [$canv_ canvasxy $x $y]
	set last_ [eval $sender_ create_item line $firstpt_ $pt $prop_]
	set firstpt_ $pt
}

#
# Bundle all the individual lines into one multi-line
#       note that we assumed that from start_ to last_ they are all line
#       segments, if this is not true, there will be an error.
#
MBFHTool instproc end {} {
	$self instvar start_ last_ firstpt_ sender_ page_id_ toolbar_
	if {$start_ == {}} return
	set newObj [eval $sender_ -page $page_id_ \
		    create_item group mline $start_ $last_]
	$toolbar_ add_item $page_id_ $newObj
	set start_ {}
	set last_ {}
	set firstpt_ {}
}

MBFHTool instproc deactivate {} {
	$self end
}

# ---------------------------
# Class MBScanTool --
#    Tool for scan in canvas
#   FIXME: this is not ready yet!
Class MBScanTool -superclass MBTool

MBScanTool instproc init {toolbar mgr sender} {
	$self next $toolbar $mgr $sender
}

MBScanTool instproc activate { page_id } {
	$self instvar mgr_
	set canv [[$mgr_ page_manager] page2canv $page_id]
	$canv resetBindings
	$canv setHilit
	set w [$canv get_win]
	bind $w <Button-1> "$self start_scan $w %x %y"
	bind $w <B1-Motion> "$self scan_to $w %x %y"
	bind $w <ButtonRelease-1> "$self end_scan"
}

MBScanTool instproc start_scan { canvwgt x y } {
	$canvwgt scan mark $x $y
}

MBScanTool instproc scan_to { canvwgt x y } {
	$canvwgt scan dragto $x $y
	$canvwgt scan mark $x $y
}

MBScanTool instproc end_scan { } {
}

Class MBWhoTool -superclass MBTool

# tool that shows the owner of each item
MBWhoTool instproc init {toolbar mgr sender} {
	$self next $toolbar $mgr $sender
	$self set canv_ {}
	$self set font_ [$self get_option smallfont]
}

MBWhoTool instproc activate { page_id } {
	$self instvar canv_ mgr_
	set canv_ [[$mgr_ page_manager] page2canv $page_id]
	set w [$canv_ get_win]
	bind $w <B1-Motion> "$self show_owner %x %y"
}

MBWhoTool instproc show_owner { x y } {
	$self instvar canv_ label_ font_ labelw_
	set px [$canv_ canvasx $x]
	set py [$canv_ canvasy $y]
	set tag [$canv_ find closest $px $py]
	set source [$canv_ owner $tag]
	if {$source == {}} {
		return
	}
	set cname [$source cname]
	$canv_ hilit $tag
	if {![info exists label_]} {
		set label_ [label .l -text $cname -font $font_ -bg beige]
		set labelw_ [$canv_ create window $px $py -window $label_]
	} else {
		$label_ configure -text $cname
		$canv_ coord $labelw_ $px $py
	}
	$canv_ raise $labelw_
}

# ---------------------------
# Class MBToolbar --
#    Displays all the tools
#
Class MBToolbar -superclass MBWidget

#
# Displays the toolbar panel with all the buttons
#
MBToolbar instproc init {ui fr mgr sender ops} {
	$self instvar mbui_ mgr_
	set mbui_ $ui
	set mgr_ $mgr

	$self set currTool_ {}
	$self set lastTool_ {}
	set p [eval frame $fr.toolbar $ops]
	$self set path_ $p

	# REVIEW:
	# these tools really does more than just implement the buttons
	# may be they should be part of the ui?
	$self instvar tools_
	set tools_(freehand) [new MBFHTool $self $mgr $sender]
	set tools_(import)   [new MBImportTool $self $mgr $sender]
	set tools_(text)     [new MBTextTool $self $mgr $sender]
	set tools_(line)     [new MBShapeTool $self $mgr $sender]
	set tools_(rectangle) $tools_(line)
	set tools_(oval)     $tools_(line)
	set tools_(copy)     [new MBSelectTool $self $mgr $sender]
	set tools_(move)     $tools_(copy)
	set tools_(erase)    [new MBEraseTool $self $mgr $sender]
	set tools_(unerase)  $tools_(erase)

#	set scan_     [new MBScanTool $self $sender]

	$self tkvar currTool_
	set currToolVarName [$self tkvarname currTool_]

	set smallfont_ [$self get_option smallfont]

	button $p.sep -text "" -bitmap "sep" -height 4 -relief flat -state disabled

# 	radiobutton $p.who -text who -command \
# 			"$tools_(who) activate  \[$pageMgr current_page\]" \
# 			-justify center
# 	$self add_tip $p.who "Show Owner under cursor"

	set pageMgr [$mgr page_manager]
	button $p.unerase -bitmap unerase -command \
			"$tools_(unerase) unerase \[$pageMgr current_page\]" \
			-justify center
	$self add_tip $p.unerase "Unerase Object"

	radiobutton $p.erase -bitmap erase -command \
			"$tools_(erase) activate \[$pageMgr current_page\]" \
			-variable $currToolVarName \
			-value erase -indicatoron false
	$self add_tip $p.erase "Erase"

	radiobutton $p.copy -bitmap copy -command \
			"$tools_(copy) activate_copy \[$pageMgr current_page\]" \
			-variable $currToolVarName -value copy \
			-indicatoron false
	$self add_tip $p.copy "Copy"

	radiobutton $p.move -bitmap move -command \
			"$tools_(move) activate_move \[$pageMgr current_page\]" \
			-variable $currToolVarName -value move \
			-indicatoron false
	$self add_tip $p.move "Move"

	radiobutton $p.freehand -bitmap freehand -command \
			"$tools_(freehand) activate \[$pageMgr current_page\]" \
			-variable $currToolVarName -value freehand \
			-indicatoron false
	$self add_tip $p.freehand "Freehand Sketch"

	radiobutton $p.line -bitmap line -command \
			"$tools_(line) activate_shape line \[$pageMgr current_page\]" \
			-variable $currToolVarName -value line \
			-indicatoron false
	$self add_tip $p.line "Straight Line"

	radiobutton $p.oval -bitmap oval -command  \
			"$tools_(oval) activate_shape oval \[$pageMgr current_page\]" \
			-variable $currToolVarName -value oval \
			-indicatoron false
	$self add_tip $p.oval "Oval"

	radiobutton $p.rectangle -bitmap rectangle -command  \
			"$tools_(rectangle) activate_shape rectangle \
			 \[$pageMgr current_page\]" \
			-variable $currToolVarName -value rectangle \
			-indicatoron false
	$self add_tip $p.rectangle "Rectangle"

	radiobutton $p.text -bitmap text -command  \
			"$tools_(text) activate  \[$pageMgr current_page\]" \
			-variable $currToolVarName -value text \
			-indicatoron false
	$self add_tip $p.text "Text"

# 	radiobutton $p.scan -text scan -variable $currToolVarName -value scan \
# 			-indicatoron false \
# 			-command "$scan_ activate \[$pageMgr current_page\]"
# 	$self add_tip $p.scan "Move View point"

	set buttonList {unerase erase copy move sep \
			oval rectangle line text freehand}

	set omitTools [string trim [$self get_option omitTools]]
	foreach elt $buttonList {
		# display all buttons not in this list
		if {-1==[lsearch -exact $omitTools $elt]} {
			pack $p.$elt -side bottom -pady 1 -padx 1 -ipadx 0 -ipady 0
		}
	}
}

MBToolbar instproc pack {args} {
	eval $self next $args
	$self select_tool [$self current_tool]
}

MBToolbar instproc unpack {} {
	$self deactivate_tool [$self current_tool]
	$self instvar mgr_
	[[$mgr_ page_manager] current_canvas] resetBindings
	$self next
}

MBToolbar instproc add_tip {args} {
	return [eval add_tip $args]
}

MBToolbar instproc current_tool {} {
	$self tkvar currTool_
	return $currTool_
}

MBToolbar instproc select_tool {toolname} {
	$self instvar path_
	if {$toolname != {}} {
		$path_.$toolname invoke
	}
}

#
# Add a callback to be triggered when a different tool has been selected
#       the format of the callback will be:
#		callback_cmd new_tool_name
#
MBToolbar instproc trace_tool {cmd} {
	$self instvar tool_change_callbacks_ traced_
	$self tkvar currTool_
	if ![info exists traced_] {
		trace variable currTool_ w \
				"$self tool_changed"
		set traced_ 1
	}
	lappend tool_change_callbacks_ $cmd
}

# allow the tool (defaults to last tool) to clean up
MBToolbar instproc deactivate_tool {{toolname {}}} {
	$self instvar path_ lastTool_ tools_
	#	puts "in deactivate, lastTool_ is $lastTool_"
	if {$toolname == {}} {
		set toolname $lastTool_
	}
	if {$toolname != {}} {
		$tools_($toolname) deactivate
	}
}

MBToolbar instproc tool {toolname} {
	$self instvar tools_
	if [info exists tools_($toolname)] {
		return $tools_($toolname)
	}
}

#
# Our own callback who a tool changes, it calls all registered callbacks
#
MBToolbar instproc tool_changed {args} {
	$self tkvar currTool_
	$self instvar lastTool_
	# allow the last tool to cleanup
	$self deactivate_tool
	set lastTool_ $currTool_
	$self instvar tool_change_callbacks_
	foreach cmd $tool_change_callbacks_ {
		eval $cmd $currTool_
	}
}

MBToolbar instproc destroy {} {
	trace vdelete [$self tkvarname currTool_] w "$self tool_changed"
	$self instvar tools_
	delete $tools_(freehand)
	delete $tools_(import)
	delete $tools_(text)
	delete $tools_(line)
	delete $tools_(copy)
	delete $tools_(erase)
	delete $tools_(none)
}

MBToolbar instproc add_item {pageId id} {
	$self instvar arItems_
	lappend arItems_($pageId) $id
	mtrace trcVerbose "adding $id to arItems_($pageId), new: $arItems_($pageId)"
}

MBToolbar instproc replace_item {pageId oldId newId} {
	$self instvar arItems_
	set arItems_($pageId) [lsubst $arItems_($pageId) $oldId $newId]
}

MBToolbar instproc get_item {pageId index} {
	$self instvar arItems_
	if [info exists arItems_($pageId)] {
		mtrace trcVerbose "get_item $pageId $index: returning  [lindex $arItems_($pageId) $index]"
		return [lindex $arItems_($pageId) $index]
	} else {
		mtrace trcVerbose "cannot find arItems($pageId), returning null"
		return {}
	}
}
