/*
 * tkCanvPS.c --
 *
 *      This file implements postscript items for canvas widgets.
 *
 * Copyright (c) 1996-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * @(#) $Header: /usr/mash/src/repository/mash/mash-1/mb/tkCanvPS.c,v 1.7 2002/02/03 03:16:30 lim Exp $
 */

#include <stdio.h>
#include <stdlib.h>
#include "tcl.h"
#include "tk.h"
#include <math.h>
#include "tkCanvPS.h"

/*
 * The structure below defines the record for each PScript item.
 */

typedef struct PScriptItem  {
    Tk_Item header;		    /* Generic stuff that's the same for all
                             * types.  MUST BE FIRST IN STRUCTURE. */
    Tk_Canvas canvas;		/* Canvas containing the PScript. */
    double x1, y1, x2, y2;	/* Coordinates of positioning point for
                             * PScript. */
    char *szFile;           /* string describing -file option (malloc-ed).
                             * NULL means no PScript right now. */
    GC gc_;
    Drawable pixmap_;
} PScriptItem;

/*
 * Information used for parsing configuration specs:
 */

static Tk_CustomOption tagsOption = {Tk_CanvasTagsParseProc,
    Tk_CanvasTagsPrintProc, (ClientData) NULL
};

static Tk_ConfigSpec configSpecs[] = {
    {TK_CONFIG_STRING, "-file", (char *) NULL, (char *) NULL,
     (char *) NULL, Tk_Offset(PScriptItem, szFile), TK_CONFIG_NULL_OK},
    {TK_CONFIG_CUSTOM, "-tags", (char *) NULL, (char *) NULL,
	(char *) NULL, 0, TK_CONFIG_NULL_OK, &tagsOption},
    {TK_CONFIG_END, (char *) NULL, (char *) NULL, (char *) NULL,
	(char *) NULL, 0, 0}
};

/*
 * Prototypes for procedures defined in this file:
 */
#if 0
static void		PScriptChangedProc _ANSI_ARGS_((ClientData clientData,
			    int x, int y, int width, int height, int psWidth,
			    int psHeight));
#endif
static int		PScriptCoords _ANSI_ARGS_((Tcl_Interp *interp,
			    Tk_Canvas canvas, Tk_Item *itemPtr, int argc,
			    Tcl_Obj *CONST argv[]));
static int		PScriptToArea _ANSI_ARGS_((Tk_Canvas canvas,
			    Tk_Item *itemPtr, double *rectPtr));
static double		PScriptToPoint _ANSI_ARGS_((Tk_Canvas canvas,
			    Tk_Item *itemPtr, double *coordPtr));
static void		ComputePScriptBbox _ANSI_ARGS_((Tk_Canvas canvas,
			    PScriptItem *psPtr));
static int		ConfigurePScript _ANSI_ARGS_((Tcl_Interp *interp,
			    Tk_Canvas canvas, Tk_Item *itemPtr, int argc,
			    Tcl_Obj *CONST argv[], int flags));
static int		CreatePScript _ANSI_ARGS_((Tcl_Interp *interp,
			    Tk_Canvas canvas, Tk_Item *itemPtr, int argc,
			    Tcl_Obj *CONST argv[]));
static void		DeletePScript _ANSI_ARGS_((Tk_Canvas canvas,
			    Tk_Item *itemPtr, Display *display));
static void		DisplayPScript _ANSI_ARGS_((Tk_Canvas canvas,
			    Tk_Item *itemPtr, Display *display, Drawable dst,
			    int x, int y, int width, int height));
static void		ScalePScript _ANSI_ARGS_((Tk_Canvas canvas,
			    Tk_Item *itemPtr, double originX, double originY,
			    double scaleX, double scaleY));
static void		TranslatePScript _ANSI_ARGS_((Tk_Canvas canvas,
			    Tk_Item *itemPtr, double deltaX, double deltaY));

/*
 * The structures below defines the PScript item type in terms of
 * procedures that can be invoked by generic item code.
 */

Tk_ItemType tkPScriptType = {
    "pscript",				/* name */
    sizeof(PScriptItem),			/* itemSize */
    CreatePScript,			/* createProc */
    configSpecs,			/* configSpecs */
    ConfigurePScript,			/* configureProc */
    PScriptCoords,			/* coordProc */
    DeletePScript,			/* deleteProc */
    DisplayPScript,			/* displayProc */
    0,					/* alwaysRedraw */
    PScriptToPoint,			/* pointProc */
    PScriptToArea,			/* areaProc */
    (Tk_ItemPostscriptProc *) NULL,	/* postscriptProc */
    ScalePScript,				/* scaleProc */
    TranslatePScript,			/* translateProc */
    (Tk_ItemIndexProc *) NULL,		/* indexProc */
    (Tk_ItemCursorProc *) NULL,		/* icursorProc */
    (Tk_ItemSelectionProc *) NULL,	/* selectionProc */
    (Tk_ItemInsertProc *) NULL,		/* insertProc */
    (Tk_ItemDCharsProc *) NULL,		/* dTextProc */
    (Tk_ItemType *) NULL		/* nextPtr */
};

/*
 *--------------------------------------------------------------
 *
 * CreatePScript --
 *
 *	This procedure is invoked to create a new PScript
 *	item in a canvas.
 *
 * Results:
 *	A standard Tcl return value.  If an error occurred in
 *	creating the item, then an error message is left in
 *	interp->result;  in this case itemPtr is left uninitialized,
 *	so it can be safely freed by the caller.
 *
 * Side effects:
 *	A new PScript item is created.
 *
 *--------------------------------------------------------------
 */

static int
CreatePScript(Tcl_Interp *interp, Tk_Canvas canvas, Tk_Item *itemPtr,
              int argc, Tcl_Obj *CONST argv[])
#if 0
    Tcl_Interp *interp;			/* For error reporting. */
    Tk_Canvas canvas;			/* Canvas to hold new item. */
    Tk_Item *itemPtr;			/* Record to hold new item;  header
					 * has been initialized by caller. */
    int argc;				/* Number of arguments in argv. */
    Tcl_Obj *CONST argv[];		/* Arguments describing rectangle. */
#endif
{
    PScriptItem *psPtr = (PScriptItem *) itemPtr;

    fprintf(stderr,"ps create called\n");

    if (argc < 2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"",
		Tk_PathName(Tk_CanvasTkwin(canvas)), " create ",
		itemPtr->typePtr->name, " x y ?options?\"",
		(char *) NULL);
	return TCL_ERROR;
    }

    /*
     * Initialize item's record.
     */

    psPtr->canvas = canvas;
    psPtr->szFile = NULL;
    psPtr->gc_ = None;
    psPtr->pixmap_ = None;

    /*
     * Process the arguments to fill in the item record.
     */

    if ((Tk_CanvasGetCoordFromObj(interp, canvas, argv[0], &psPtr->x1) != TCL_OK)
	    || (Tk_CanvasGetCoordFromObj(interp, canvas, argv[1], &psPtr->y1)!= TCL_OK)
        || (Tk_CanvasGetCoordFromObj(interp, canvas, argv[2], &psPtr->x2)!= TCL_OK)
        || (Tk_CanvasGetCoordFromObj(interp, canvas, argv[3], &psPtr->y2)!= TCL_OK)
        ) {
        return TCL_ERROR;
    }

    if (ConfigurePScript(interp, canvas, itemPtr, argc-4, argv+4, 0)
        != TCL_OK) {
	DeletePScript(canvas, itemPtr, Tk_Display(Tk_CanvasTkwin(canvas)));
	return TCL_ERROR;
    }
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * PScriptCoords --
 *
 *	This procedure is invoked to process the "coords" widget
 *	command on PScript items.  See the user documentation for
 *	details on what it does.
 *
 * Results:
 *	Returns TCL_OK or TCL_ERROR, and sets interp->result.
 *
 * Side effects:
 *	The coordinates for the given item may be changed.
 *
 *--------------------------------------------------------------
 */

static int
PScriptCoords(Tcl_Interp *interp, Tk_Canvas canvas, Tk_Item *itemPtr,
              int argc, Tcl_Obj *CONST argv[])
#if 0
    Tcl_Interp *interp;			/* Used for error reporting. */
    Tk_Canvas canvas;			/* Canvas containing item. */
    Tk_Item *itemPtr;			/* Item whose coordinates are to be
					 * read or modified. */
    int argc;				/* Number of coordinates supplied in
					 * argv. */
    Tcl_Obj *CONST argv[];		/* Array of coordinates: x1, y1,
					 * x2, y2, ... */
#endif
{
    PScriptItem *psPtr = (PScriptItem *) itemPtr;
    char x1[TCL_DOUBLE_SPACE], y1[TCL_DOUBLE_SPACE],
        x2[TCL_DOUBLE_SPACE], y2[TCL_DOUBLE_SPACE];

    fprintf(stderr,"ps coords called\n");
    if (argc == 0) {
        Tcl_PrintDouble(interp, psPtr->x1, x1);
        Tcl_PrintDouble(interp, psPtr->y1, y1);
        Tcl_PrintDouble(interp, psPtr->x2, x2);
        Tcl_PrintDouble(interp, psPtr->y2, y2);
        Tcl_AppendResult(interp, x1, " ", y1,
                         " ", x2, " ", y2, (char *) NULL);
    } else if (argc == 4) {
        if ((Tk_CanvasGetCoordFromObj(interp, canvas, argv[0], &psPtr->x1)!=TCL_OK)
            || (Tk_CanvasGetCoordFromObj(interp, canvas, argv[1], &psPtr->y1)!=TCL_OK)
            || (Tk_CanvasGetCoordFromObj(interp, canvas, argv[2], &psPtr->x2)!=TCL_OK)
            || (Tk_CanvasGetCoordFromObj(interp, canvas, argv[3], &psPtr->y2)!=TCL_OK)
            ) {
            return TCL_ERROR;
        }
        ComputePScriptBbox(canvas, psPtr);
    } else {
        sprintf(interp->result,
                "wrong # coordinates: expected 0 or 4, got %d", argc);
        return TCL_ERROR;
    }
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * ConfigurePScript --
 *
 *	This procedure is invoked to configure various aspects
 *	of an PScript item, such as its anchor position.
 *
 * Results:
 *	A standard Tcl result code.  If an error occurs, then
 *	an error message is left in interp->result.
 *
 * Side effects:
 *	Configuration information may be set for itemPtr.
 *
 *--------------------------------------------------------------
 */

static int
ConfigurePScript(Tcl_Interp *interp, Tk_Canvas canvas, Tk_Item *itemPtr,
                 int argc, Tcl_Obj *CONST argv[], int flags)
#if 0
    Tcl_Interp *interp;		/* Used for error reporting. */
    Tk_Canvas canvas;		/* Canvas containing itemPtr. */
    Tk_Item *itemPtr;		/* Rectangle item to reconfigure. */
    int argc;			/* Number of elements in argv.  */
    Tcl_Obj *CONST argv[];	/* Arguments describing things to configure. */
    int flags;			/* Flags to pass to Tk_ConfigureWidget. */
#endif
{
    PScriptItem *psPtr = (PScriptItem *) itemPtr;
    Tk_Window tkwin;
    Tcl_DString dstr;
    char szPixmap[TCL_DOUBLE_SPACE]; /* should be more than enough for long */
    XGCValues v;
    unsigned long mask;

#ifdef DEBUG
    fprintf(stderr,"ps config called\n");
#endif

    tkwin = Tk_CanvasTkwin(canvas);
    if (Tk_ConfigureWidget(interp, tkwin, configSpecs, argc,
	    (char **) argv, (char *) psPtr, flags) != TCL_OK) {
        return TCL_ERROR;
    }

    /*
     * Create the PScript.  Save the old PScript around and don't free it
     * until after the new one is allocated.  This keeps the reference
     * count from going to zero so the PScript doesn't have to be recreated
     * if it hasn't changed.
     */
    ComputePScriptBbox(canvas, psPtr);
    if (psPtr->szFile != NULL) {
        int width  = psPtr->header.x2 - psPtr->header.x1;
        int height = psPtr->header.y2 - psPtr->header.y1;

        psPtr->pixmap_ = Tk_GetPixmap(Tk_Display(tkwin), Tk_WindowId(tkwin),
                                      width, height, Tk_Depth(tkwin));
        Tcl_DStringInit(&dstr);
        Tcl_DStringAppend(&dstr,"$mb(gsinterp) get_pixmap ", -1);

        sprintf(szPixmap,"%ld ", psPtr->pixmap_);
        Tcl_DStringAppend(&dstr,szPixmap, -1);
#if 0
        sprintf(szPixmap,"%ld ", Tk_WindowId(tkwin));
        Tcl_DStringAppend(&dstr,szPixmap, -1);
#endif
        sprintf(szPixmap,"%d ", width);
        Tcl_DStringAppend(&dstr,szPixmap, -1);
        sprintf(szPixmap,"%d ", height);
        Tcl_DStringAppend(&dstr,szPixmap, -1);
        Tcl_DStringAppend(&dstr,psPtr->szFile, -1);

#ifdef CANV_DBG
        fprintf(stderr,"initialized pixmap_ to %ld\n", psPtr->pixmap_);
#endif
        if (psPtr->gc_ == None) {
            XColor* c = Tk_GetColor(interp, tkwin, Tk_GetUid("grey"));
            if (c == 0) abort();
            mask = GCForeground|GCBackground|GCGraphicsExposures;
            v.background = c->pixel;
            v.foreground = c->pixel;
            v.graphics_exposures = 0;
            psPtr->gc_ = Tk_GetGC(tkwin, mask, &v);
        }
        /* clear the pixmap */
        XFillRectangle(Tk_Display(tkwin),
                       psPtr->pixmap_, psPtr->gc_, 0, 0, width, height);

#ifdef CANV_DEBUG
        fprintf(stderr, "executing %s\n", Tcl_DStringValue(&dstr));
#endif
        if (TCL_OK!=Tcl_GlobalEval(interp,Tcl_DStringValue(&dstr))) {
            Tcl_DStringFree(&dstr);
            return TCL_ERROR;
        }
        sscanf(interp->result, "%ld", &(psPtr->pixmap_));
        Tcl_DStringFree(&dstr);
    }
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * DeletePScript --
 *
 *	This procedure is called to clean up the data structure
 *	associated with a PScript item.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Resources associated with itemPtr are released.
 *
 *--------------------------------------------------------------
 */

static void
DeletePScript(Tk_Canvas canvas, Tk_Item *itemPtr, Display *display)
#if 0
    Tk_Canvas canvas;			/* Info about overall canvas widget. */
    Tk_Item *itemPtr;			/* Item that is being deleted. */
    Display *display;			/* Display containing window for
					 * canvas. */
#endif
{
    PScriptItem *psPtr = (PScriptItem *) itemPtr;

#ifdef CANV_DEBUG
    fprintf(stderr,"ps del called\n");
#endif

    if (psPtr->szFile != NULL) {
	Tcl_Free(psPtr->szFile);
    }
    if (psPtr->pixmap_ != None) {
        Tk_FreePixmap(display, psPtr->pixmap_);
    }
    if (psPtr->gc_ != None) {
        Tk_FreeGC(display, psPtr->gc_);
    }
}

/*
 *--------------------------------------------------------------
 *
 * ComputePScriptBbox --
 *
 *	This procedure is invoked to compute the bounding box of
 *	all the pixels that may be drawn as part of a PScript item.
 *	This procedure is where the child PScript's placement is
 *	computed.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The fields x1, y1, x2, and y2 are updated in the header
 *	for itemPtr.
 *
 *--------------------------------------------------------------
 */

	/* ARGSUSED */
static void
ComputePScriptBbox( Tk_Canvas canvas, PScriptItem *psPtr)
#if 0
    Tk_Canvas canvas;			/* Canvas that contains item. */
    PScriptItem *psPtr;			/* Item whose bbox is to be
					 * recomputed. */
#endif
{
    int x1, y1, x2, y2;

#ifdef CANV_DEBUG
    fprintf(stderr,"ps computeBBox called\n");
#endif // CANV_DEBUG

    x1 = (int) ( psPtr->x1 + ((psPtr->x1 >= 0) ? 0.5 : - 0.5) );
    y1 = (int) ( psPtr->y1 + ((psPtr->y2 >= 0) ? 0.5 : - 0.5) );

    if (psPtr->szFile == NULL) {
        psPtr->header.x1 = psPtr->header.x2 = x1;
        psPtr->header.y1 = psPtr->header.y2 = y1;
        return;
    }

    x2 = (int) ( psPtr->x2 + ((psPtr->x2 >= 0) ? 0.5 : - 0.5) );
    y2 = (int) ( psPtr->y2 + ((psPtr->y2 >= 0) ? 0.5 : - 0.5) );

    psPtr->header.x1 = x1;
    psPtr->header.y1 = y1;
    psPtr->header.x2 = x2;
    psPtr->header.y2 = y2;
}

/*
 *--------------------------------------------------------------
 *
 * DisplayPScript --
 *
 *	This procedure is invoked to draw a PScript item in a given
 *	drawable.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	ItemPtr is drawn in drawable using the transformation
 *	information in canvas.
 *
 *--------------------------------------------------------------
 */
static void DisplayPScript _ANSI_ARGS_(
    (Tk_Canvas canvas,
     Tk_Item *itemPtr, Display *display, Drawable drawable,
     int x, int y, int width, int height))
#if 0
    Tk_Canvas canvas;			/* Canvas that contains item. */
    Tk_Item *itemPtr;			/* Item to be displayed. */
    Display *display;			/* Display on which to draw item. */
    Drawable drawable;			/* Pixmap or window in which to draw
					 * item. */
    int x, y, width, height;		/* Describes region of canvas that
					 * must be redisplayed (not used). */
#endif
{
    PScriptItem *psPtr = (PScriptItem *) itemPtr;
    short drawableX, drawableY;

#ifdef CANV_DEBUG
    fprintf(stderr,"in DispPS()\n");
#endif

    if (psPtr->szFile == NULL) {
	return;
    }

    /*
     * Translate the coordinates to those of the PScript, then redisplay it.
     */

    Tk_CanvasDrawableCoords(canvas, (double) x, (double) y,
	    &drawableX, &drawableY);
#ifdef CANV_DEBUG
    fprintf(stderr, "Executing XCopyArea[ src=%ld,dest=%ld, from (%d,%d)"
            "to (%d,%d) for wd=%d ht=%d ]\n",psPtr->pixmap_,drawable,
            x - psPtr->header.x1,y - psPtr->header.y1,drawableX,
            drawableY,width,height);
#endif
#if 0
    if (width > psPtr->header.x2 - (psPtr->header.x1 - x))
        width = psPtr->header.x2 - (psPtr->header.x1 - x);
    if (height > psPtr->header.y2 - psPtr->header.y1 - y)
        height = psPtr->header.y2 - psPtr->header.y1 - y;
#endif
    XCopyArea(display, psPtr->pixmap_, drawable, psPtr->gc_,
              x - psPtr->header.x1, y - psPtr->header.y1,
              width, height, drawableX, drawableY);
}

/*
 *--------------------------------------------------------------
 *
 * PScriptToPoint --
 *
 *	Computes the distance from a given point to a given
 *	rectangle, in canvas units.
 *
 * Results:
 *	The return value is 0 if the point whose x and y coordinates
 *	are coordPtr[0] and coordPtr[1] is inside the PScript.  If the
 *	point isn't inside the PScript then the return value is the
 *	distance from the point to the PScript.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static double
PScriptToPoint(Tk_Canvas canvas, Tk_Item *itemPtr, double *coordPtr)
#if 0
    Tk_Canvas canvas;		/* Canvas containing item. */
    Tk_Item *itemPtr;		/* Item to check against point. */
    double *coordPtr;		/* Pointer to x and y coordinates. */
#endif
{
    PScriptItem *psPtr = (PScriptItem *) itemPtr;
    double x1, x2, y1, y2, xDiff, yDiff;

#if 0
    fprintf(stderr,"ToPt()");
#endif

    x1 = psPtr->header.x1;
    y1 = psPtr->header.y1;
    x2 = psPtr->header.x2;
    y2 = psPtr->header.y2;

    /*
     * Point is outside rectangle.
     */

    if (coordPtr[0] < x1) {
	xDiff = x1 - coordPtr[0];
    } else if (coordPtr[0] > x2)  {
	xDiff = coordPtr[0] - x2;
    } else {
	xDiff = 0;
    }

    if (coordPtr[1] < y1) {
	yDiff = y1 - coordPtr[1];
    } else if (coordPtr[1] > y2)  {
	yDiff = coordPtr[1] - y2;
    } else {
	yDiff = 0;
    }

    return sqrt(xDiff*xDiff + yDiff*yDiff);

}

/*
 *--------------------------------------------------------------
 *
 * PScriptToArea --
 *
 *	This procedure is called to determine whether an item
 *	lies entirely inside, entirely outside, or overlapping
 *	a given rectangle.
 *
 * Results:
 *	-1 is returned if the item is entirely outside the area
 *	given by rectPtr, 0 if it overlaps, and 1 if it is entirely
 *	inside the given area.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static int
PScriptToArea(Tk_Canvas canvas, Tk_Item *itemPtr, double *rectPtr)
#if 0
    Tk_Canvas canvas;		/* Canvas containing item. */
    Tk_Item *itemPtr;		/* Item to check against rectangle. */
    double *rectPtr;		/* Pointer to array of four coordinates
				 * (x1, y1, x2, y2) describing rectangular
				 * area.  */
#endif
{
    PScriptItem *psPtr = (PScriptItem *) itemPtr;

#ifdef CANV_DEBUG
    fprintf(stderr,"ps toArea called\n");
#endif

    if ((rectPtr[2] <= psPtr->header.x1)
	    || (rectPtr[0] >= psPtr->header.x2)
	    || (rectPtr[3] <= psPtr->header.y1)
	    || (rectPtr[1] >= psPtr->header.y2)) {
	return -1;
    }
    if ((rectPtr[0] <= psPtr->header.x1)
	    && (rectPtr[1] <= psPtr->header.y1)
	    && (rectPtr[2] >= psPtr->header.x2)
	    && (rectPtr[3] >= psPtr->header.y2)) {
	return 1;
    }
    return 0;
}

/*
 *--------------------------------------------------------------
 *
 * ScalePScript --
 *
 *	This procedure is invoked to rescale an item.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The item referred to by itemPtr is rescaled so that the
 *	following transformation is applied to all point coordinates:
 *		x' = originX + scaleX*(x-originX)
 *		y' = originY + scaleY*(y-originY)
 *
 *--------------------------------------------------------------
 */

static void
ScalePScript(Tk_Canvas canvas, Tk_Item *itemPtr,
             double originX, double originY, double  scaleX, double scaleY)
#if 0
    Tk_Canvas canvas;			/* Canvas containing rectangle. */
    Tk_Item *itemPtr;			/* Rectangle to be scaled. */
    double originX, originY;		/* Origin about which to scale rect. */
    double scaleX;			/* Amount to scale in X direction. */
    double scaleY;			/* Amount to scale in Y direction. */
#endif
{
    PScriptItem *psPtr = (PScriptItem *) itemPtr;

#ifdef CANV_DEBUG
    fprintf(stderr,"ps scale called\n");
#endif

    psPtr->x1 = originX + scaleX*(psPtr->x1 - originX);
    psPtr->y1 = originY + scaleY*(psPtr->y1 - originY);
    psPtr->x2 = originX + scaleX*(psPtr->x2 - originX);
    psPtr->y2 = originY + scaleY*(psPtr->y2 - originY);
    ComputePScriptBbox(canvas, psPtr);
}

/*
 *--------------------------------------------------------------
 *
 * TranslatePScript --
 *
 *	This procedure is called to move an item by a given amount.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The position of the item is offset by (xDelta, yDelta), and
 *	the bounding box is updated in the generic part of the item
 *	structure.
 *
 *--------------------------------------------------------------
 */

static void
TranslatePScript(Tk_Canvas canvas, Tk_Item *itemPtr,
                 double deltaX, double deltaY)
#if 0
    Tk_Canvas canvas;			/* Canvas containing item. */
    Tk_Item *itemPtr;			/* Item that is being moved. */
    double deltaX, deltaY;		/* Amount by which item is to be
					 * moved. */
#endif
{
    PScriptItem *psPtr = (PScriptItem *) itemPtr;

#ifdef CANV_DEBUG
    fprintf(stderr,"ps translate called\n");
#endif

    psPtr->x1 += deltaX;
    psPtr->y1 += deltaY;
    psPtr->x2 += deltaX;
    psPtr->y2 += deltaY;
    ComputePScriptBbox(canvas, psPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * PScriptChangedProc --
 *
 *	This procedure is invoked by the PScript code whenever the manager
 *	for an PScript does something that affects the PScript's size or
 *	how it is displayed.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Arranges for the canvas to get redisplayed.
 *
 *----------------------------------------------------------------------
 */
#if 0
static void
PScriptChangedProc(ClientData clientData, int x, int y,
                   int width, int height, int psWidth, int psHeight)
#if 0
    ClientData clientData;		/* Pointer to canvas item for PScript. */
    int x, y;				/* Upper left pixel (within PScript)
					 * that must be redisplayed. */
    int width, height;			/* Dimensions of area to redisplay
					 * (may be <= 0). */
    int psWidth, psHeight;		/* New dimensions of PScript. */
#endif
{
    PScriptItem *psPtr = (PScriptItem *) clientData;

#ifdef CANV_DEBUG
    fprintf(stderr,"pschange called\n");
#endif
    /*
     * If the PScript's size changed and it's not anchored at its
     * northwest corner then just redisplay the entire area of the
     * PScript.  This is a bit over-conservative, but we need to do
     * something because a size change also means a position change.
     */

    if (((psPtr->header.x2 - psPtr->header.x1) != psWidth)
	    || ((psPtr->header.y2 - psPtr->header.y1) != psHeight)) {
	x = y = 0;
	width = psWidth;
	height = psHeight;
	Tk_CanvasEventuallyRedraw(psPtr->canvas, psPtr->header.x1,
		psPtr->header.y1, psPtr->header.x2, psPtr->header.y2);
    }
    ComputePScriptBbox(psPtr->canvas, psPtr);
    Tk_CanvasEventuallyRedraw(psPtr->canvas, psPtr->header.x1 + x,
	    psPtr->header.y1 + y, (int) (psPtr->header.x1 + x + width),
	    (int) (psPtr->header.y1 + y + height));
}
#endif
