/*
 * mb-sndr.cc --
 *
 *      MediaBoard Sender (MBSender)
 *      The main object from which the UI manipulate mb drawing objects.
 *
 * Copyright (c) 1996-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * $Header: /usr/mash/src/repository/mash/mash-1/mb/mb-sndr.cc,v 1.17 2002/02/03 03:16:30 lim Exp $
 */

#ifndef MB_SNDR_CC
#define MB_SNDR_CC

#include "mb.h"
#include "mb-string.h"

#ifndef lint
static const char rcsid[] =
"@(#) $Header: /usr/mash/src/repository/mash/mash-1/mb/mb-sndr.cc,v 1.17 2002/02/03 03:16:30 lim Exp $";
#endif

#include <tclcl.h>
#include "mb-sndr.h"
#include "mb-page.h"
#include "mb-cmd.h"
#include "mb-rcvr.h"
#include "testpkt.h"

#ifdef MB_DEBUG
#define TEST                    // we turn this on by default
#endif

static class MBSenderClass : public TclClass {
public:
	MBSenderClass() : TclClass("MB_Sender") {}

	TclObject* create(int /*argc*/, const char*const* /*argv*/) {
        return (new MBSender());
    }
} mbsc;

MBSender::MBSender() :
     pCurrPage_(NULL), pLocalRcvr_(NULL), pMgr_(NULL), interactiveLvl_(0)
{
}

#define RET_ERR(str) \
    do { \
        Tcl_AddErrorInfo(MB_Interp,str); \
        return TCL_ERROR; \
    } while (FALSE)

int MBSender::command(int argc, const char*const* origArgv)
{
    MB_DefTcl(tcl);
    Tcl_ResetResult(tcl.interp());

    const char** argv = CONST_CAST(const char**) (origArgv);
    if (argc<=1)
        RET_ERR("too few arguments");

    Page* pPage=pCurrPage_;
    if (!strcmp(argv[1], "-page") && argc >= 3) {
            PageId pgId;
            if (!Str2PgId(argv[2], pgId)) RET_ERR("Invalid page id!");
            pPage = (Page*) pLocalRcvr_->DefinePage(pgId);
            argc -= 2;
            argv += 2;
    }

#ifdef MB_DEBUG
    if (!strcmp(argv[1], "dump")) {
        int dumptype=0;
        if (!strcmp(argv[2],"timestamps"))
            dumptype=1;
        else if(!strcmp(argv[2],"canvas"))
            dumptype=2;
	Tcl_Obj* pObj = Tcl_NewObj();
        if (dumptype)
            pMgr_->Dump(dumptype, pObj);
	tcl.result(pObj);
        return TCL_OK;
    }
#else
    if (!strcmp(argv[1], "dump")) {
            return TCL_OK;
    }
#endif

    // commands with variable number of parameters
    PageItem *pItem=NULL;
    if (!strcmp(argv[1],"create_item"))
    {
        /* pCurrPage_ could be null if the page is not visible yet */
	if (pPage == NULL)
	    RET_ERR("page not defined yet");

        if (argc<=2)
            RET_ERR("too few arguments");

        if (!strcmp(argv[2],"group")) {
            if (!strcmp(argv[3],"mline")) {
                u_long itemId =
                    CreateGroup(pPage, PgItemMLine, strtoul(argv[4],NULL,10),
                                strtoul(argv[5],NULL,10), pItem);

                tcl.resultf("%lu", itemId);
                return itemId ? TCL_OK : TCL_ERROR;
            } else if (!strcmp(argv[3],"text")) {
                u_long itemId =
                    CreateGroup(pPage, PgItemText, strtol(argv[4],NULL,10),
                                strtol(argv[5],NULL,10),pItem);
                if (!itemId) RET_ERR("unable to create item");

                tcl.resultf("%lu", itemId);
                return TCL_OK;
            }
        } else {                // !create_item group
            PageItemType pit = PageItem::strName2Type(argv[2]);
            if (pit==PgItemInvalid) RET_ERR("invalid item");

            pItem = PageItem::createItem(pit);
            if (!pItem) RET_ERR("invalid item");

            int numPoints=PageItem::type2NumPts(pit);
            const int cStartPts=3;
            int startConfig=cStartPts+numPoints*2;
            if ( argc>startConfig &&
                 (!pItem->configure(argc-startConfig, argv+startConfig)) )
                RET_ERR("invalid configuration");

            if (argc<startConfig)
                RET_ERR("too few parameters!");

            Point* aPoints=new Point[numPoints];
            for (int i=0; i<numPoints; i++) {
                if ( !( Str2Coord(argv[cStartPts+(2*i)],aPoints[i].x)
                      && Str2Coord(argv[cStartPts+(2*i)+1],aPoints[i].y) ))
                {
                    delete[] aPoints;
                    RET_ERR("Invalid Point");
                }
            }
            pItem->setPoints(aPoints,numPoints);
            u_long itemId = 0;
            if (!pItem->mustFrag())
                itemId = CreateItem(pPage, pItem);
            else
                itemId = CreateAsFrag(pPage, pItem, argc-startConfig,
                                      argv+startConfig);

            if (!itemId) RET_ERR("unable to create item");
            tcl.resultf("%lu", itemId);
            return TCL_OK;
        } // end !create_item group
    }

    // fixed parameters
    switch (argc) {

    case 3:
        if (!strcmp(argv[1],"attach")) {
            Attach( (MBManager*) TclObject::lookup(argv[2]) );
        }
        else if (!strcmp(argv[1],"create_page")) {
		PageId newPageId;
		int retcode = CreatePage(argv[2], newPageId);
		char* szNewPageId = PgId2Str(newPageId);
		tcl.result(Tcl_NewStringObj(szNewPageId, -1));
		delete[] szNewPageId;
		return retcode;
        } else if (!strcmp(argv[1],"switch_page")) {
            PageId pgid;
            if (!Str2PgId(argv[2],pgid)) RET_ERR("Invalid page id!");
            if (!SwitchPage(pgid)) RET_ERR("page id not found");
        }
        else if (!strcmp(argv[1],"delete_item")) {
            if (pPage == NULL)
		RET_ERR("page not defined yet");

            u_long itemId = DeleteItem(pPage, strtoul(argv[2],NULL,10));
            tcl.resultf("%lu", itemId);
        }
        else if (!strcmp(argv[1],"dup_item")) {
            if (pPage == NULL)
		RET_ERR("page not defined yet");
            u_long itemId = DupItem(pPage, strtoul(argv[2],NULL,10));
            tcl.resultf("%lu", itemId);
        }
        else if (!strcmp(argv[1],"interactive")) {
            return SetInteractive(atoi(argv[2])) ? TCL_OK : TCL_ERROR;
        }
        else return TclObject::command(argc, argv);
        break;

    case 5:
        if (!strcmp(argv[1],"move")) {
            if (pPage == NULL)

		RET_ERR("page not defined yet");

            Coord dx, dy;
            if (!Str2Coord(argv[3],dx)) return TCL_ERROR;
            if (!Str2Coord(argv[4],dy)) return TCL_ERROR;
            u_long itemId = MoveItem(pPage, strtoul(argv[2],NULL,10),
                                     dx,dy);
            tcl.resultf("%lu",itemId);
        } else if (!strcmp(argv[1],"insert")) {
            if (pPage == NULL)
		RET_ERR("page not defined yet");

            u_long itemId = Insert(pPage, strtoul(argv[2],NULL,10),
                                   atoi(argv[3]), argv[4]);
            if (!itemId) RET_ERR("cannot insert item");
            tcl.resultf("%lu", itemId);
        } else if (!strcmp(argv[1],"nearest")) {
            if (pPage == NULL)
		RET_ERR("page not defined yet");
            Coord x, y, dist;
            if (!( Str2Coord(argv[2],x) && Str2Coord(argv[3],y) &&
                   Str2Coord(argv[4],dist) ))
                RET_ERR("Invalid Coords!");
            u_long itemId=NearestItem(pPage, x, y, dist);
            MTrace(trcMB|trcExcessive,("nearest item is: %lu", itemId));
            if (itemId) tcl.resultf("%lu",itemId);
            else tcl.result("");
        }
        else return TclObject::command(argc, argv);
        break;

#ifdef OLD
    case 7:
        if (strcmp(argv[1],"coords") == 0) {
            Coord x1,y1,x2,y2;
            if (!Str2Coord(argv[3],x1)) return TCL_ERROR;
            if (!Str2Coord(argv[4],y1)) return TCL_ERROR;
            if (!Str2Coord(argv[5],x2)) return TCL_ERROR;
            if (!Str2Coord(argv[6],y2)) return TCL_ERROR;
            u_long seqno =
                ChangeItemCoord(pPage, strtoul(argv[2], NULL, 10), x1, y1, x2,
                                y2);
            tcl.resultf("%lu", seqno);
        }
        else return TclObject::command(argc, argv);
        break;
#endif

    default:
        Tcl_AddErrorInfo(MB_Interp,"Argument mismatch\n");
        return TclObject::command(argc, argv);
    }
    // defaults to return OK
    return TCL_OK;
}


u_long MBSender::NearestItem(Page* pPage, Coord x, Coord y, Coord dist) {
	MBBaseRcvr *pRcvr = DYN_CAST(MBBaseRcvr *)(pLocalRcvr_);

	// we know that in MBSender the canvas is of class MBCanvas
	MBCanvas* pCanvas = DYN_CAST(MBCanvas*) (pPage->getCanvas());

	CanvItemId startcid =
		pCanvas->nextNearest(x, y, dist, 0);
	if (!startcid) return 0;         // implies no items are near enough

	// the canvas will return the closest item even if it is not within
	// dist, so we will have to check that it overlaps
	MBCmd* pCmd;
	if (pCanvas->overlap(x-dist, y-dist, x+dist, y+dist, startcid)) {
		pCmd = pCanvas->canvId2cmd(startcid);
		if (pCmd && pCmd->rcvr()==pRcvr) {
			return pCanvas->canvId2itemId(startcid);
		}
	} else { // the closest item is out of range
		return 0;
	}

	// canvas will loop thru all items, until startcid is returned again
	// all subsequent items returned should be within dist
	ulong nearestCid = 0;
	ulong nextCid=pCanvas->nextNearest(x,y,dist,startcid);
	while (nextCid!=startcid) {
		pCmd = pCanvas->canvId2cmd(nextCid);
		if (pCmd && pCmd->rcvr()==pRcvr) {
			nearestCid = nextCid;
			break;
		}
		nextCid=pCanvas->nextNearest(x, y, dist, nextCid);
	}
	MTrace(trcMB|trcVerbose,
	       ("Nearest Item returns (canvid)%ld",nearestCid));
	if (!nearestCid) return 0;
	else return pCanvas->canvId2itemId(nearestCid);
}


//
// change to the given page
//
Bool MBSender::SwitchPage(const PageId& pgId)
{
    if (pMgr_->FindPage(pgId)) {
        Page* pPage = NULL;
        if (pCurrPage_ && pgId!=pCurrPage_->getId() ) {
            pMgr_->ChangeStatus(pCurrPage_->getId(),FALSE);
        }
        pPage = (Page*) pLocalRcvr_->DefinePage(pgId);
        assert(pPage && "DefinePage should not fail");
	if (!pPage) return FALSE;
        pLocalRcvr_->SetCurrPage(pPage);
        pCurrPage_ = pPage;
        pMgr_->ChangeStatus(pCurrPage_->getId(),TRUE);
        return TRUE;
    }
    return FALSE;
}

int MBSender::CreatePage(const char *szName, PageId &pageId)
{
	Page* pPage = (Page*)pLocalRcvr_->NewPageObject();
	if (!pPage) {
		Tcl::instance().add_error("create page failed in NewPageObject");
		return TCL_ERROR;
	}
	MBCmd* pCmd = new MBCmdPgName(pMgr_->CurrTime(), szName);
        if (!pCmd) {
		SignalError(("Out of memory"));
		Tcl::instance().add_error("cannot create new command, out of memory?");
		return TCL_ERROR;
	}

        if (pLocalRcvr_->Dispatch(pCmd, pPage)) {
#ifdef TEST
                assert(TestPacketize(pCmd));
#endif
		pageId = pPage->getId();
                return TCL_OK;
        }
        return TCL_ERROR;
}

//
// Creates new item
//
u_long MBSender::CreateItem(Page* pPage, PageItem *pItem)
{
    MBCmd* pCmd = new MBCmdCreate(pMgr_->CurrTime(), pItem);
    assert(pCmd);

    if (pLocalRcvr_->Dispatch(pCmd, pPage)) {
#ifdef TEST
        assert(TestPacketize(pCmd));
#endif
        return (pCmd->getSeqno());
    }
    return 0;
}

//
// Creates new item
//
u_long MBSender::CreateAsFrag(Page* pPage, PageItem *pItem, int argc,
                              const char*const* argv)
{
    if (argc<1) {
        Tcl_AddErrorInfo(MB_Interp,"Argument mismatch");
        return (u_long)NULL;
    }
    MBFile f;
    if (!f.Init(argc, argv)) return (u_long)NULL;
    MBCmdFrag* pCmdFrag=NULL;
    u_long firstSN=0;
    u_long lastSN=0;
    Bool done=FALSE;
    int maxLen = 0;
    do {
        if (!pCmdFrag) {
            pCmdFrag = new MBCmdFrag(pMgr_->CurrTime());
            if (!maxLen)     // calculate the max data size for a fragment
                maxLen = pMgr_->getMaxCmdSz() - pCmdFrag->getPacketLen();
        }
        done=!f.getNextFrag(maxLen, pCmdFrag);
#ifdef TEST
        assert(TestPacketize(pCmdFrag));
#endif
        // add the command to the page if it is non-empty
        if (!pCmdFrag->IsEmpty()) {
            pLocalRcvr_->Dispatch(pCmdFrag, pPage);
            if (firstSN==0) firstSN=pCmdFrag->getSeqno();
            lastSN=pCmdFrag->getSeqno();
            pCmdFrag = NULL;
        }
    } while (!done);

    MBCmd* pCmd = MBCmdGroup::Create(pMgr_->CurrTime(), pItem->getType(),
                                     firstSN, lastSN, pItem, argc, argv);
    assert(pCmd);
#ifdef TEST
    assert(TestPacketize(pCmd));
#endif

    if (pLocalRcvr_->Dispatch(pCmd, pPage)) {
#ifdef TEST
        assert(TestPacketize(pCmd));
#endif
        return (pCmd->getSeqno());
    }
    return 0;
}

//
// Creates new multi-line segment
//
u_long MBSender::CreateGroup(Page* pPage, PageItemType type,
                             u_long snStart, u_long snEnd,
                             PageItem *pItem)
{
    MBCmd* pCmd = MBCmdGroup::Create(pMgr_->CurrTime(), type,
                                     snStart, snEnd, pItem);
#ifdef TEST
    assert(TestPacketize(pCmd));
#endif

	if (!pCmd) {
		SignalError(("Out of memory"));
		return 0;
	}
    if (pLocalRcvr_->Dispatch(pCmd, pPage))
        return (pCmd->getSeqno());
    else return 0;
}

//
// Insert a character at index into item# itemid
//
Bool MBSender::Insert(Page *pPage, u_long itemid, u_short index,
                      const char* pChar)
{
    // ignore underrun deletes
    if ((*pChar == cchDel) && (index == 0)) {
	return TRUE;
    }
    // only accept control characters that we understand
    // REVIEW: ascill is assumed!
    if ((*pChar < ' ' &&
	 !(*pChar==cchDel || *pChar==cchCR || *pChar==cchTab))) {
	return TRUE;
    }
    MTrace(trcMB|trcVerbose, ("inserting %c (%d)", *pChar, *pChar));
    MBCmd* pCmd = new MBCmdChar(pMgr_->CurrTime(), itemid,
                                index, *pChar);

#ifdef TEST
    assert(TestPacketize(pCmd));
#endif
    if (!pCmd) {
		SignalError(("Out of memory"));
		return 0;
	}
    if (pLocalRcvr_->Dispatch(pCmd, pPage))
        return (pCmd->getSeqno());
    else return 0;
}

//
// Delete an item.
//
u_long MBSender::DeleteItem(Page* pPage, u_long itemId)
{
    MBCmd *pCmd = new MBCmdDel(pMgr_->CurrTime(),
                                itemId);
#ifdef TEST
    assert(TestPacketize(pCmd));
#endif
    if (!pCmd) {
		SignalError(("Out of memory"));
		return 0;
	}
    if (0 == interactiveLvl_) {
        return pLocalRcvr_->Dispatch(pCmd, pPage) ? pCmd->getSeqno():0;
    }
    else {
        return pLocalRcvr_->executeInteractive(pCmd, pPage);
	delete pCmd;
    }
    return 0;
}

//
// Delete an item.
//
u_long MBSender::DupItem(Page* pPage, u_long itemId)
{
    MBCmd *pCmd = new MBCmdDup(pMgr_->CurrTime(),itemId);

#ifdef TEST
    assert(TestPacketize(pCmd));
#endif

    if (!pCmd) {
	    SignalError(("Out of memory"));
	    return 0;
    }
    if (pLocalRcvr_->Dispatch(pCmd, pPage)) {
	    return pCmd->getSeqno();
    }
    return 0;
}


//
// xi's and yi's give the new coords
//      this command is only useful for rects ovals, and not for
//      lines and multi-lines
//
#ifdef OLD // this command is obsolete

u_long MBSender::ChangeItemCoord(u_long /*itemId*/,
                                 Coord /*x1*/, Coord /*y1*/, Coord /*x2*/,
                                 Coord /*y2*/)
{
	MBCmd *pCmd = new MBCmdCoord(pMgr_->CurrTime(),
                                 itemId, x1, y1, x2, y2);
	if (!pCmd) {
		SignalError(("Out of memory"));
		return 0;
	}

#ifdef TEST
	assert(TestPacketize(pCmd));    `
#endif

    if (0 == interactiveLvl_) {
        return pLocalRcvr_->Dispatch(pCmd, pCurrPage_) ? pCmd->getSeqno():0;
    }
    else {
        return pLocalRcvr_->executeInteractive(pCmd, pCurrPage_);
	delete pCmd;
    }
}

#endif // OLD


u_long MBSender::MoveItem(Page* pPage, u_long itemId, Coord dx, Coord dy)
{
    MBCmd *pCmd = new MBCmdMove(pMgr_->CurrTime(),
                                itemId, dx, dy);
#ifdef TEST
    assert(TestPacketize(pCmd));
#endif
    if (!pCmd) {
        SignalError(("Out of memory"));
        return 0;
    }
    if (0==interactiveLvl_)
        return pLocalRcvr_->Dispatch(pCmd, pPage) ? pCmd->getSeqno():0;
    else {
        pLocalRcvr_->executeInteractive(pCmd, pPage);
	delete pCmd;
        return 0;
    }
}

#ifdef OLD
//
// return a TCL list of overlapped items
//
Tcl_DString* MBSender::OverlappedItems(Page *pPage,
                                       Coord x1, Coord y1, Coord x2, Coord y2)
{
    int count;
    u_long *arItemIds=NULL;
    pPage->OverlappedItems(x1, y1, x2, y2, count, arItemIds);
    Trace(VERYVERBOSE,("found %d items:",count));
    if (!count) return NULL;
    char szWrk[20]; // 20 should be long enough for all ulongs
    Tcl_DString* pDStr = new Tcl_DString;
    if (!pDStr) return NULL;
    Tcl_DStringInit(pDStr);
    for (u_long *pItemId=arItemIds; pItemId<arItemIds+count; pItemId++) {
        sprintf(szWrk, "%lu", *pItemId);
        assert(strlen(szWrk)<20 && "might have over-written memory" );
        Tcl_DStringAppendElement(pDStr, szWrk);
    }
    return pDStr;
}
#endif

#endif // #define MB_SNDR_CC
