/*
 * vidreps.cc --
 *
 *      Video frame memory representation method definitions.
 *
 * Copyright (c) 1998-2001 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/* @(#) $Header: /usr/mash/src/repository/mash/mash-1/fx/vidreps.cc,v 1.1 2002/02/07 04:17:01 chema Exp $ */


#include "vidreps.h"
#include "dvmpnm.h"


//
// PUT_BUF_NUM used to construct Dali buffername strings from buffer ids.
//
#define PUT_BUF_NUM(str,prefix,buf) sprintf(str, "%s%d", prefix, buf)


//
// Constructor for VidRep class.
//
VidRep::VidRep() :
		TclObject(),
		w_(0),
		h_(0),
		css_h_(1),
		css_v_(1),
		true_w_(0),
		true_h_(0),
		xoff_(0),
		yoff_(0),
		resized_(0),
		ssrc_(0),
		ts_(0),
		source_id_(0)
{
#if 0
	Tcl& tcl = Tcl::instance();
	tcl.eval("VidRep/Uncompressed set w_ 0");
	tcl.eval("VidRep/Uncompressed set h_ 0");
	tcl.eval("VidRep/Uncompressed set css_h_ 1");
	tcl.eval("VidRep/Uncompressed set css_v_ 1");
	tcl.eval("VidRep/Uncompressed set true_w_ 0");
	tcl.eval("VidRep/Uncompressed set true_h_ 0");
	tcl.eval("VidRep/Uncompressed set xoff_ 0");
	tcl.eval("VidRep/Uncompressed set yoff_ 0");
	tcl.eval("VidRep/Uncompressed set ssrc_ 0");
	tcl.eval("VidRep/Uncompressed set ts_ 0");

	bind("w_", (int*)&w_);
	bind("h_", (int*)&h_);
	bind("css_h_", (int*)&css_h_);
	bind("css_v_", (int*)&css_v_);
	bind("true_w_", (int*)&true_w_);
	bind("true_h_", (int*)&true_h_);
	bind("xoff_", (int*)&xoff_);
	bind("yoff_", (int*)&yoff_);
	bind("ssrc_", (unsigned int*)&ssrc_);
	bind("ts_", (unsigned int*)&ts_);
#endif
};

/*
 * Destructor for VidRep class. Currently empty.
 */
VidRep::~VidRep() {
};

/*
 * Copies geometry for representation passed in as template.
 * Does no allocation or deallocation, simply changes geometry variables.
 */
void 
VidRep::copy_geometry(VidRep *temp)
{
	w_ = temp->w_;
	h_ = temp->h_;
	css_h_ = temp->css_h_;
	css_v_ = temp->css_v_;
	true_w_ = temp->true_w_;
	true_h_ = temp->true_h_;
	xoff_ = temp->xoff_;
	yoff_ = temp->yoff_;
}

/*
 * Tcl command interface for VidRep.
 * Commands supported:
 *     copy_geometry <VidRepObject>
 */
int
VidRep::command(int argc, const char*const* argv) 
{
	if (argc == 3) {
		if (strcmp(argv[1], "copy_geometry") == 0) {
			VidRep *temp;

			temp = (VidRep *) TclObject::lookup(argv[2]);
			if (temp != 0) {
				copy_geometry(temp);
			}
			return TCL_OK;
		}
		if (strcmp(argv[1], "set_source_id") == 0) {
			if (source_id_ != 0) 
					delete[] source_id_;
			source_id_ = new char[strlen(argv[2]) + 1];
			strcpy(source_id_, argv[2]);
			return TCL_OK;
		}
	}
	return (TclObject::command(argc, argv));
}

//
// OTcl class binding for Uncompressed.
// OTcl class is "VidRep/Uncompressed"
//
static class UncompressedClass : public TclClass {
public:
	UncompressedClass() : TclClass("VidRep/Uncompressed") {}
	TclObject *create(int argc, const char*const* argv) {
		return (new Uncompressed());
	}
} uncompressed_class_;


//
// Constructor for Uncompressed
//
Uncompressed::Uncompressed() :
		VidRep(), 
		memory_allocation_(0),
		lum_(0),
		cr_(0),
		cb_(0),
		lum_id_(-1),
		cr_id_(-1),
		cb_id_(-1),
		alt_css_h_(0),
		alt_css_v_(0),
		alt_cr_(0),
		alt_cb_(0),
		alt_cr_id_(-1),
		alt_cb_id_(-1),
		bp_(0)
{
	// create and initialize the luminance structure
	lum_ = new ByteImage;
	lum_->parentWidth = lum_->width = 0;
	lum_->height = 0;
	lum_->x = 0;
	lum_->y = 0;
	lum_->isVirtual = 0;

	// create and initialize the main Cr chrominance structure
	cr_ = new ByteImage;
	cr_->parentWidth = cr_->width = 0;
	cr_->height = 0;
	cr_->x = 0;
	cr_->y = 0;
	cr_->isVirtual = 0;

	// create and initialize the main Cb chrominance structure
	cb_ = new ByteImage;
	cb_->parentWidth = 0;
	cb_->height = 0;
	cb_->x = 0;
	cb_->y = 0;
	cb_->isVirtual = 0;

	// create and initialize the alternative Cr chrominance structure
	alt_cr_ = new ByteImage;
	alt_cr_->parentWidth = alt_cr_->width = 0;
	alt_cr_->height = 0;
	alt_cr_->x = 0;
	alt_cr_->y = 0;
	alt_cr_->isVirtual = 0;

	// create and initialize the alternative Cb chrominance structure
	alt_cb_ = new ByteImage;
	alt_cb_->parentWidth = alt_cb_->width = 0;
	alt_cb_->height = 0;
	alt_cb_->x = 0;
	alt_cb_->y = 0;
	alt_cb_->isVirtual = 0;

	return;
}



//
// Destructor for Uncompressed.
//
Uncompressed::~Uncompressed() {
	// deallocate all memory used and unregister
	dealloc();

	return;
};



//
//	Uncompressed::init -
//
//	Inputs:
//	  - w, h: frame dimensions (in pixels)
//	  - css_h, css_v: horizontal and vertical chroma subsamplings. 
//	    4:4:4 is (1,1)
//	    4:2:2 is (2,1)
//	    4:1:1 is (4,1)
//	    4:2:0 is (2,2)
//	  - true_w, true_h: dimensions of the frame (aka true frame) for which 
//	    this frame is just a subrectangle (in pixels)
//	  - xoff, yoff: offset of the frame inside the true frame
//	  - ldata, crdata, cbdata: pointers to the 3 components bulk-memory
//	  - type: set of flags that marks if allocation of the 3 Dali structures
//	    and of the bulk-memory allocation is wanted, and if registering is
//		  needed
//
//	Description:
//	Initializes an Uncompressed object: fills main Dali ByteImage buffers 
//	(y, cr, and cb), registers Dali structures, and allocates the 
//	bulk-memory buffer if requested
//
void Uncompressed::init(int w, int h, int css_h, int css_v,
		int alt_css_h, int alt_css_v,
		int true_w, int true_h, 
		int xoff, int yoff,
		u_char *ldata, u_char *crdata, u_char *cbdata, 
		int type)
{
	// copy the frame dimensions
	w_ = w; h_ = h;
	css_h_ = css_h; css_v_ = css_v;
	alt_css_h_ = alt_css_h; alt_css_v_ = alt_css_v;
	true_w_ = true_w; true_h_ = true_h;
	xoff_ = xoff; yoff_ = yoff;

	// fill the ByteImage structures
	lum_->parentWidth = lum_->width = w_;
	lum_->height = h_;
	lum_->x = 0;
	lum_->y = 0;
	lum_->isVirtual = 0;

	cr_->parentWidth = cr_->width = w_ / css_h_;
	cr_->height = h_ / css_v_;
	cr_->x = 0;
	cr_->y = 0;
	cr_->isVirtual = 0;

	cb_->parentWidth = cb_->width = w_ / css_h_;
	cb_->height = h_ / css_v_;
	cb_->x = 0;
	cb_->y = 0;
	cb_->isVirtual = 0;

	// deallocate potential old memory
	if (ISSET(memory_allocation_, VIDREP_ALLOC_MEMORY)) {
		delete[] lum_->firstByte;
		lum_->firstByte = 0;
	} else if (ISSET(type, VIDREP_ALLOC_CHROMAS)) {
		delete[] cr_->firstByte;
		cr_->firstByte = 0;
	}

	if (ISSET(type, VIDREP_ALLOC_MEMORY)) {
		// allocate bulk memory for both luma and chromas
		memory_allocation_ = VIDREP_ALLOC_MEMORY;
		int n = w_*h_ + 2 * ((w_/css_h_) * (h_/css_v_));
		bp_ = new u_char[n];
		memset(bp_, 0x80, n);

		// set the Y-, Cr-, Cb-component pointers
		lum_->firstByte = bp_;
		cr_->firstByte = bp_ + (w_*h_);
		cb_->firstByte = bp_ + (w_*h_) + ((w_/css_h_) * (h_/css_v_));
		alt_cr_->firstByte = 0;
		alt_cb_->firstByte = 0;

	} else if (ISSET(type, VIDREP_ALLOC_CHROMAS)) {
		// allocate chroma buffers only (luma memory comes from elsewhere)
		memory_allocation_ = VIDREP_ALLOC_CHROMAS;
		int n = 2 * ((w_/css_h_) * (h_/css_v_));
		cr_->firstByte = new u_char[n];
		// allocate the bulk memory and paint it in gray
		memset(cr_->firstByte, 0x80, n);

		cb_->firstByte = cr_->firstByte + ((w_/css_h_) * (h_/css_v_));
		bp_ = ldata;
		lum_->firstByte = ldata;
		alt_cr_->firstByte = crdata;
		alt_cb_->firstByte = cbdata;

		alt_cr_->parentWidth = alt_cr_->width = w_ / alt_css_h_;
		alt_cr_->height = h_ / alt_css_v_;
		alt_cr_->x = 0;
		alt_cr_->y = 0;
		alt_cr_->isVirtual = 0;

		alt_cb_->parentWidth = alt_cb_->width = w_ / alt_css_h_;
		alt_cb_->height = h_ / alt_css_v_;
		alt_cb_->x = 0;
		alt_cb_->y = 0;
		alt_cb_->isVirtual = 0;

	} else {
		// all memory comes from elsewhere or doesn't exist already
		memory_allocation_ = VIDREP_OVERLOAD_MEMORY;
		bp_ = ldata;
		lum_->firstByte = ldata;
		cr_->firstByte = crdata;
		cb_->firstByte = cbdata;
		alt_cr_->firstByte = 0;
		alt_cb_->firstByte = 0;
	}

	if (ISSET(type, VIDREP_REGISTER)) {
		memory_allocation_ |= VIDREP_REGISTER;
		Tcl& tcl = Tcl::instance();
		lum_id_ = PutByteImage(tcl.interp(), lum_);
		cr_id_ = PutByteImage(tcl.interp(), cr_);
		cb_id_ = PutByteImage(tcl.interp(), cb_);
		alt_cr_id_ = PutByteImage(tcl.interp(), alt_cr_);
		alt_cb_id_ = PutByteImage(tcl.interp(), alt_cb_);
	}

	return;
}



//
//	Uncompressed::init -
//
//	Inputs:
//	  - w, h: frame dimensions (in pixels)
//	  - csss: chroma subsampling scheme
//
//	Description:
//	A commonly-used wrapper for the other Uncompressed::init. It gets all 
//	the possible information from another frame.
//
void Uncompressed::init(Uncompressed *fr, int csss, int type)
{
	u_char *ldata  = 0;
	u_char *crdata = 0;
	u_char *cbdata = 0;
	int css_h = 0;
	int css_v = 0;

	// convert the color subsampling scheme into horizontal and vertical 
	//	subsampling ratios
	convert_csss (csss, css_h, css_v);

	if (ISSET(type, VIDREP_OVERLOAD_MEMORY)) {
		// reset the allocate memory flag because we'll use the same memory twice
		type &= ~VIDREP_ALLOC_MEMORY;
		// get the luma and chroma buffer addresses
		ldata  = fr->lum_->firstByte;
		crdata = fr->cr_->firstByte;
		cbdata = fr->cb_->firstByte;
		if (csss != fr->get_csss()) {
			// different color subsampling schemes: get alternative chroma buffers
			type &= ~VIDREP_OVERLOAD_MEMORY;
			type |= VIDREP_ALLOC_CHROMAS;
		}
	}

	init(fr->w_, fr->h_, css_h, css_v, fr->css_h_, fr->css_v_,
			fr->true_w_, fr->true_h_, fr->xoff_, fr->yoff_,
			ldata, crdata, cbdata, type);

	return;
}



//
//	Uncompressed::dealloc -
//
//	Inputs:
//	  none
//
//	Description:
//	Unregisters Dali structures and deallocates any bulk-memory buffer 
//	originally self-allocated. 
//
void Uncompressed::dealloc()
{
	// deallocate full-frame memory
	if (ISSET(memory_allocation_, VIDREP_ALLOC_MEMORY)) {
		if (lum_->firstByte != 0) {
			delete[] lum_->firstByte;
			lum_->firstByte = 0;
		}
	}

	// deallocate alternative chroma buffers
	if (ISSET(memory_allocation_, VIDREP_ALLOC_CHROMAS)) {
		if (cr_->firstByte != 0) {
			delete[] cr_->firstByte;
			cr_->firstByte = 0;
		}
	}

	// free the source id name
	if (source_id_ != 0) {
		delete[] source_id_;
	}

	// unregister Dali structures
	if (ISSET(memory_allocation_, VIDREP_REGISTER)) {

		// unregister the luminance ByteImage
		if (lum_id_ != -1) {
			RemoveBufByNum(lum_id_);
			lum_id_ = -1;
		}

		// unregister the main Cr chrominance ByteImage
		if (cr_id_ != -1) {
			RemoveBufByNum(cr_id_);
			cr_id_ = -1;
		}

		// unregister the main Cb chrominance ByteImage
		if (cb_id_ != -1)  {
			RemoveBufByNum(cb_id_);
			cb_id_ = -1;
		}

		// unregister the alternative Cr chrominance ByteImage
		if (alt_cr_id_ != -1)  {
			RemoveBufByNum(alt_cr_id_);
			alt_cr_id_ = -1;
		}

		// unregister the alternative Cb chrominance ByteImage
		if (alt_cb_id_ != -1)  {
			RemoveBufByNum(alt_cb_id_);
			alt_cb_id_ = -1;
		}
	}

	// deallocate the ByteImage structures
	delete lum_;
	delete cr_;
	delete cb_;
	delete alt_cr_;
	delete alt_cb_;

	return;
}



//
//	Uncompressed::byte_copy
//
//	Description:
//		Copies the contents of an Uncompressed object
//
//	Inputs:
//		frame: the source Uncompressed
//
//	Outputs:
//		none
//
void Uncompressed::byte_copy(Uncompressed *frame)
{
	ByteImage *in_l = frame->lum_;
	ByteImage *in_cr = frame->cr_;
	ByteImage *in_cb = frame->cb_;
	ByteImage *out_l = this->lum_;
	ByteImage *out_cr = this->cr_;
	ByteImage *out_cb = this->cb_;

	ByteCopy (in_l, out_l);
	ByteCopy (in_cr, out_cr);
	ByteCopy (in_cb, out_cb);

	return;
}



//
//	Uncompressed::add_chroma
//
//	Inputs:
//	  - csss: second chroma subsampling scheme
//
//	Description:
//	Adds a second chroma representation to the frame.
//
void Uncompressed::add_chroma(int csss)
{
	u_char *mem;

	if (csss == get_csss()) {
		// we're not going to add two copies with the same chroma subsampling scheme
		return;
	}


	// move the current main chroma buffers to the alternative ones
	alt_css_h_ = css_h_;
	alt_css_v_ = css_v_;

	alt_cr_->firstByte = cr_->firstByte;
	alt_cr_->parentWidth = cr_->parentWidth;
	alt_cr_->width = cr_->width;
	alt_cr_->x = cr_->x;
	alt_cr_->y = cr_->y;
	alt_cr_->isVirtual = cr_->isVirtual;

	alt_cb_->firstByte = cb_->firstByte;
	alt_cb_->parentWidth = cb_->parentWidth;
	alt_cb_->width = cb_->width;
	alt_cb_->x = cb_->x;
	alt_cb_->y = cb_->y;
	alt_cb_->isVirtual = cb_->isVirtual;

	// allocate new chroma buffers
	convert_csss (csss, css_h_, css_v_);
	int n = 2 * ((w_/css_h_) * (h_/css_v_));
	mem = new u_char[n];
	memset(mem, 0x80, n);
	cr_->firstByte = mem;
	cb_->firstByte = mem + ((w_/css_h_) * (h_/css_v_));

	cr_->parentWidth = cr_->width = w_ / css_h_;
	cr_->height = h_ / css_v_;
	cr_->x = 0;
	cr_->y = 0;
	cr_->isVirtual = 0;

	cb_->parentWidth = cb_->width = w_ / css_h_;
	cb_->height = h_ / css_v_;
	cb_->x = 0;
	cb_->y = 0;
	cb_->isVirtual = 0;

	// add the allocated chromas flag
	memory_allocation_ |= VIDREP_ALLOC_CHROMAS;

	return;
}



//
//	Uncompressed::check_chroma
//
//	Description:
//	If a frame has two chroma representations, copy the alternative one 
//	into the main buffer by carrying out chroma downsizing or upsizing
//
//	TBD: this should use the rvts_ vector to avoid copying as much as 
//	possible
//
//
void Uncompressed::check_chroma()
{
	int main;
	int alt;

	if (!ISSET(memory_allocation_, VIDREP_ALLOC_CHROMAS)) {
		// only one chroma representation. Don't do anything
		return;
	}

	// get the alternate chroma subsampling
	alt = convert_csss(alt_css_h_, alt_css_v_);

	// get the main chroma subsampling
	main = get_csss();

	// carry out the conversion
  u_char *crsrc = alt_cr_->firstByte;
  u_char *crdst = cr_->firstByte;
  u_char *cbsrc = alt_cb_->firstByte;
  u_char *cbdst = cb_->firstByte;

	if ((alt == 422) && (main == 420)) {
		// 4:2:2 -> 4:2:0. Copy only the even lines
		for (int i = 0; i < h_; i++) {
			if (i % 2 == 0) {
				memcpy (crdst, crsrc, w_/2);
				crdst += w_/2;
				memcpy (cbdst, cbsrc, w_/2);
				cbdst += w_/2;
			}
			crsrc += w_/2;
			cbsrc += w_/2;
		}
		return;
	}

	if ((alt == 420) && (main == 422)) {
		// 4:2:0 -> 4:2:2. Duplicate all the lines
		for (int i = 0; i < h_/2; i++) {
			memcpy (crdst, crsrc, w_/2);
			crdst += w_/2;
			memcpy (crdst, crsrc, w_/2);
			crdst += w_/2;
			crsrc += w_/2;
			memcpy (cbdst, cbsrc, w_/2);
			cbdst += w_/2;
			memcpy (cbdst, cbsrc, w_/2);
			cbdst += w_/2;
			cbsrc += w_/2;
		}
		return;
	}

	if ((alt == 422) && (main == 411)) {
		// 4:2:2 -> 4:1:1. Copy every other byte
		for (int i = 0; i < h_; i++) {
			for (int j = 0; j < w_/2; j++) {
				if (j % 2 == 0) {
					*crdst = *crsrc;
					crdst++;
					*cbdst = *cbsrc;
					cbdst++;
				}
				crsrc++;
				cbsrc++;
			}
		}
		return;
	}

	if ((alt == 411) && (main == 422)) {
		// 4:1:1 -> 4:2:2. Duplicate all the bytes
		for (int i = 0; i < h_; i++) {
			for (int j = 0; j < w_/4; j++) {
				*crdst = *crsrc;
				crdst++;
				*crdst = *crsrc;
				crdst++;
				crsrc++;
				*cbdst = *cbsrc;
				cbdst++;
				*cbdst = *cbsrc;
				cbdst++;
				cbsrc++;
			}
		}
		return;
	}

	// unimplemented conversion
	fprintf (stderr, "check_chroma: unimplemented chroma conversion (%d->%d)\n", 
			alt, main);
	abort();
}


//
//	Uncompressed::save_disk
//
//	Description:
//		Writes an Uncompressed object into a PPM file
//		(TBD: currently it just writes the luminance into a grey-scale PGM file)
//
//	Inputs:
//		prefix: a file name prefix (the suffix will be the frame timestamp)
//
//	Outputs:
//		none
//
void Uncompressed::save_disk(char* prefix)
{
	char filename[1024];

#define PGM
// PPM not yet
//#define PPM

#if defined(PGM)
	sprintf  (filename, "/tmp/%s-%d.pgm", prefix, ts_);
#elif defined(PPM)
	sprintf  (filename, "/tmp/%s-%d.ppm", prefix, ts_);
#endif

	FILE* fid = fopen (filename, "w");


#ifdef PPM
	// carry out chroma conversion to provide 4:4:4 chromas
	ByteImage full_lum;
	ByteImage full_cr;
	ByteImage full_cb;

	// allocate memory for a 4:4:4 frame
	int n = w_ * h_;
	full_lum.firstByte = new u_char[n];
	full_cr.firstByte = new u_char[n];
	full_cb.firstByte = new u_char[n];

	full_lum.parentWidth = full_lum.width = w_;
	full_lum.height = h_;
	full_lum.x = 0;
	full_lum.y = 0;
	full_lum.isVirtual = 0;

	full_cr.parentWidth = full_cr.width = w_;
	full_cr.height = h_;
	full_cr.x = 0;
	full_cr.y = 0;
	full_cr.isVirtual = 0;

	full_cb.parentWidth = full_cb.width = w_;
	full_cb.height = h_;
	full_cb.x = 0;
	full_cb.y = 0;
	full_cb.isVirtual = 0;

	// copy the luma
	memcpy (full_lum.firstByte, lum_->firstByte, w_*h_);

	// convert both chromas to 4:4:4
	int csss = get_csss();

	if (csss == 444) {
		full_cr.firstByte = cr_->firstByte;
		full_cb.firstByte = cb_->firstByte;

	} else if (csss == 422) {
		// 4:2:2 -> 4:4:4: duplicate all bytes
		u_char *crsrc = cr_->firstByte;
		u_char *crdst = full_cr.firstByte;
		u_char *cbsrc = cb_->firstByte;
		u_char *cbdst = full_cb.firstByte;
		for (int i = 0; i < h_; i++) {
			for (int j = 0; j < w_; j+=2) {
				// duplicate all the bytes
				*crdst = *crsrc;
				crdst++;
				*crdst = *crsrc;
				crdst++;
				crsrc++;
				*cbdst = *cbsrc;
				cbdst++;
				*cbdst = *cbsrc;
				cbdst++;
				cbsrc++;
			}
		}

	} else if (csss == 420) {
		// 4:2:0 -> 4:4:4: duplicate all bytes and all lines
		u_char *crsrc = cr_->firstByte;
		u_char *crdst = full_cr.firstByte;
		u_char *cbsrc = cb_->firstByte;
		u_char *cbdst = full_cb.firstByte;
		u_char *linecr = full_cr.firstByte;
		u_char *linecb = full_cb.firstByte;
		for (int i = 0; i < h_; i++) {
			if (i%2 == 0) {
				for (int j = 0; j < w_; j+=2) {
					// duplicate all the bytes on even lines
					*crdst = *crsrc;
					crdst++;
					*crdst = *crsrc;
					crdst++;
					crsrc++;
					*cbdst = *cbsrc;
					cbdst++;
					*cbdst = *cbsrc;
					cbdst++;
					cbsrc++;
				}
			} else {
				// copy previous even line on odd lines
				memcpy (crdst, linecr, w_);
				crdst += w_;
				linecr = crdst;
				memcpy (cbdst, linecb, w_);
				cbdst += w_;
				linecb = cbdst;
			}
		}

	} else if (csss == 411) {
		// 4:1:1 -> 4:4:4: quadruplicate all bytes in all lines
		u_char *crsrc = cr_->firstByte;
		u_char *crdst = full_cr.firstByte;
		u_char *cbsrc = cb_->firstByte;
		u_char *cbdst = full_cb.firstByte;
		for (int i = 0; i < h_; i++) {
			for (int j = 0; j < w_; j+=4) {
				// quadruplicate all the bytes
				*crdst = *crsrc;
				crdst++;
				*crdst = *crsrc;
				crdst++;
				*crdst = *crsrc;
				crdst++;
				*crdst = *crsrc;
				crdst++;
				crsrc++;
				*cbdst = *cbsrc;
				cbdst++;
				*cbdst = *cbsrc;
				cbdst++;
				*cbdst = *cbsrc;
				cbdst++;
				*cbdst = *cbsrc;
				cbdst++;
				cbsrc++;
			}
		}
	}

	// carry out yuv2rgb conversion
	ByteImage *full_red, *full_green, *full_blue;
	full_red = &full_lum;
	full_green = &full_cr;
	full_blue = &full_cb;
	float y, u, v;
	float r, g, b;
	for (int i = 0; i < h_; i++) {
		for (int j = 0; j < w_; j++) {
			y = (float)full_lum.firstByte[i*w_ + j];
			u = (float)full_cr.firstByte[i*w_ + j];
			v = (float)full_cb.firstByte[i*w_ + j];
			r = y                          +  1.402   * (v - 128.0);
			g = y + -0.34414 * (u - 128.0) + -0.71414 * (v - 128.0);
			b = y +  1.772   * (u - 128.0);
			full_red->firstByte[i*w_ + j] = (u_int)r;
			full_green->firstByte[i*w_ + j] = (u_int)g;
			full_blue->firstByte[i*w_ + j] = (u_int)b;
		}
	}
#endif

	// create the PNM header object
	PnmHdr *hdr = PnmHdrNew ();
	PnmHdrSetWidth (hdr, w_);
	PnmHdrSetHeight (hdr, h_);
#if defined(PGM)
	PnmHdrSetType (hdr, PGM_BIN);
#elif defined(PPM)
	PnmHdrSetType (hdr, PPM_BIN);
#endif
	PnmHdrSetMaxVal (hdr, 255);

	// create the BitParser and BitStream objects
	BitParser *outbp = BitParserNew ();
	BitStream *outbs = BitStreamNew (20 + (w_ * h_));

	// attach them
	BitParserWrap (outbp, outbs);
	// do the actual encoding
	(void)PnmHdrEncode (hdr, outbp);
#if defined(PGM)
	PgmEncode (lum_, outbp);
#elif defined(PPM)
	PpmEncode (full_red, full_green, full_blue, outbp);
#endif
	(void) BitStreamFileWrite (outbs, fid, 0);

#if defined(PPM)
	// free the frame buffers
	delete[] full_lum.firstByte;
	delete[] full_cr.firstByte;
	delete[] full_cb.firstByte;
#endif

	// close the file and empty the BitParser and BitStream objects
	(void)fclose(fid);
	BitParserFree (outbp);
	BitStreamFree (outbs);

#if defined(PGM)
#undef PGM
#elif defined(PPM)
#undef PPM
#endif

	return;
}



/*
 * Tcl command interface to Uncompressed class.
 * Commands supported:
 *    get_lum_name - returns Dali buffer name of lum. plane
 *    get_cr_name -    "       "    "      "   "  cr    "
 *    get_cb_name -    "       "    "      "   "  cb    "
 *    allocate - creates new structures based on current geometry and
 *               allocates buffer memory.
 */
int 
Uncompressed::command(int argc, const char*const* argv)
{
	Tcl& tcl=Tcl::instance();
	char *wrk = tcl.buffer();

	if (argc == 2) {
		if (strcmp(argv[1], "get_lum_name") == 0) {
			if (lum_ == 0) {
				*wrk = 0;
			} else {
				PUT_BUF_NUM(wrk, BYTE_PREFIX, lum_id_);
			}
			tcl.result(wrk);
			return (TCL_OK);
		}
		if (strcmp(argv[1], "get_cr_name") == 0) {
			if (cr_ == 0) {
				*wrk = 0;
			} else {
				PUT_BUF_NUM(wrk, BYTE_PREFIX, cr_id_);
			}
			tcl.result(wrk);
			return (TCL_OK);
		}
		if (strcmp(argv[1], "get_cb_name") == 0) {
			if (cb_ == 0) {
				*wrk = 0;
			} else {
				PUT_BUF_NUM(wrk, BYTE_PREFIX, cb_id_);
			}
			tcl.result(wrk);
			return (TCL_OK);
		}
		if (strcmp(argv[1], "allocate") == 0) {
			// don't alloc ByteImage structures but alloc bulk-memory and 
			//	register in Dali
			init(w_, h_, css_h_, css_v_, 0, 0, true_w_, true_h_, 
					xoff_, yoff_, 0, 0, 0, 
					VIDREP_ALLOC_MEMORY | VIDREP_REGISTER);
			return (TCL_OK);
		}
	} 
	return (VidRep::command(argc, argv));
}

/*
 * OTcl class binding for JPEGCompressed.
 * OTcl class is "VidRep/JPEGCompressed"
 */
static class JPEGCompressedClass : public TclClass {
public:
	JPEGCompressedClass() : TclClass("VidRep/JPEGCompressed") {}
	TclObject *create(int argc, const char*const* argv) {
		return (new JPEGCompressed());
	}
} jpeg_compressed_class_;

/*
 * Constructor for JPEGCompressed
 * Changes css_h_ and css_v_ to reflect default 422 geometry.
 */
JPEGCompressed::JPEGCompressed() :
		VidRep(),
		self_allocated_(0),
		q_(0),
		len_(0),
		bp_(0)
{
	css_h_ = 2;
	css_v_ = 1;
	bind("q_", (int*)&q_);
	bind("len_", (int*)&len_);
}

/*
 * Destructor for JPEGCompressed.
 */
JPEGCompressed::~JPEGCompressed() {
	dealloc();
}

/*
 * Deallocates buffer memory if originally self-allocated.
 */
void JPEGCompressed::dealloc() {
	if (self_allocated_) {
		if (bp_ != 0) {
			free(bp_);
			bp_ = 0;
			self_allocated_ = 0;
		}
	}
}

/*
 * Given geometry info and pointer to data, initializes structure 
 * appropriately.
 */
void JPEGCompressed::set(int w, int h, int decimation, int q, int len,
		u_int32_t ssrc, u_int32_t ts, u_char *bp)
{
	dealloc();
	true_w_ = w_ = w;
	true_h_ = h_ = h;
	q_ = q;
	len_ = len;

	if (decimation == 422) {
		css_h_ = 2;
		css_v_ = 1;
	} else {
		css_h_ = 2;
		css_v_ = 2;
	}    
	xoff_ = yoff_ = 0;

	ssrc_ = ssrc;
	ts_ = ts;

	bp_ = bp;
	self_allocated_ = 0;
}

/*
 * Tcl command interface to JPEGCompressed object.
 * The following commands are supported:
 *       copy <JPEGCompressed> - copies geometry and buffer of 
 *                               passed in source object.
 */
int
JPEGCompressed::command(int argc, const char*const* argv) 
{
//	Tcl& tcl = Tcl::instance();

	if (argc == 3) {
		if (strcmp(argv[1], "copy") == 0) {
			JPEGCompressed *src = (JPEGCompressed *) TclObject::lookup(argv[2]);
			if (src != 0) {
				dealloc();
				copy_geometry((VidRep *)src);
				q_ = src->q_;
				len_ = src->len_;
				ssrc_ = src->ssrc_;
				ts_ = src->ts_;
				self_allocated_ = 1;
				bp_ = (u_char *) malloc(len_);
				memcpy(bp_, src->bp_, len_);
			}
			return TCL_OK;
		}
	}
	return (VidRep::command(argc, argv));
}

/*
 * OTcl class binding for Semicompressed.
 * OTcl class is "VidRep/Semicompressed"
 */
static class SemicompressedClass : public TclClass {
public:
	SemicompressedClass() : TclClass("VidRep/Semicompressed") {}
	TclObject *create(int argc, const char*const* argv) {
		return (new Semicompressed());
	}
} semi_compressed_class_;

/*
 * Constructor for Semicompressed.
 */
Semicompressed::Semicompressed() :
		VidRep(),
		self_allocated_(0),
		lum_(0),
		cr_(0),
		cb_(0),
		lum_id_(-1),
		cr_id_(-1),
		cb_id_(-1)
{
};

/*
 * Destructor for Semicompressed.
 */
Semicompressed::~Semicompressed()
{
	dealloc();
}

/*
 * Deallocates Dali structures and buffer memory if self-allocated.
 */
void Semicompressed::dealloc()
{
	if (lum_ != 0) {
		RemoveBufByNum(lum_id_);
		if (self_allocated_) {
			ScFree(lum_);
		}
	}
	if (cr_ != 0) {
		RemoveBufByNum(cr_id_);
		if (self_allocated_) {
			ScFree(cr_);
		}
	}
	if (cb_ != 0) {
		RemoveBufByNum(cb_id_);
		if (self_allocated_) {
			ScFree(cb_);
		}
	}
	lum_ = 0;
	cr_ = 0;
	cb_ = 0;
	lum_id_ = cr_id_ = cb_id_ = -1;
}

/* 
 * Tcl interface to Semicompressed objects.
 * Current commands supported:
 *      print
 *      print <x> <y>
 *      get_lum_name
 *      get_cr_name
 *      get_cb_name
 *      allocate
 */
int 
Semicompressed::command(int argc, const char*const* argv)
{
	Tcl& tcl = Tcl::instance();
	char *wrk = tcl.buffer();

	if (argc == 2) {
		if (strcmp(argv[1], "print") == 0) {
			print();
			return TCL_OK;
		}
		if (strcmp(argv[1], "get_lum_name") == 0) {
			if (lum_ == 0) {
				*wrk = 0;
			} else {
				PUT_BUF_NUM(wrk, SC_PREFIX, lum_id_);
				tcl.result(wrk);
			}
			return (TCL_OK);
		}
		if (strcmp(argv[1], "get_cr_name") == 0) {
			if (cr_ == 0) {
				*wrk = 0;
			} else {
				PUT_BUF_NUM(wrk, SC_PREFIX, cr_id_);
				tcl.result(wrk);
			}
			return (TCL_OK);
		}
		if (strcmp(argv[1], "get_cb_name") == 0) {
			if (cb_ == 0) {
				*wrk = 0;
			} else {
				PUT_BUF_NUM(wrk, SC_PREFIX, cb_id_);
				tcl.result(wrk);
			}
			return (TCL_OK);
		}
		if (strcmp(argv[1], "allocate") == 0) {
			allocate();
			return (TCL_OK);
		}
	}
	if (argc == 4) {
		if (strcmp(argv[1], "print") == 0) {
			int x, y;

			x = atoi(argv[2]);
			y = atoi(argv[3]);
 
			print(x,y);
			return TCL_OK;
		}
	}
	return (VidRep::command(argc, argv));
}

/*
 * Allocates Dali structures for current geometry. Deallocates first.
 */
void
Semicompressed::allocate()
{
	int i;
	Tcl& tcl=Tcl::instance();

	dealloc();
	lum_ = ScNew(w_/8,h_/8);
	cr_ = ScNew(w_/(8*css_h_),h_/(8*css_v_));
	cb_ = ScNew(w_/(8*css_h_),h_/(8*css_v_));

	for(i=0; i<(w_/8)*(h_/8); i++) {
		ScBlock *scb = &(lum_->firstBlock[i]);
		scb->intracoded = 1;
		scb->skipMB = 0;
		scb->skipBlock = 0;
		scb->dc = 0;
		scb->numOfAC = 0;
	}

	for(i=0; i<(w_/(8*css_h_))*(h_/(8*css_v_)); i++) {
		ScBlock *scb = &(cr_->firstBlock[i]);
		scb->intracoded = 1;
		scb->skipMB = 0;
		scb->skipBlock = 0;
		scb->dc = 0;
		scb->numOfAC = 0;

		scb = &(cb_->firstBlock[i]);
		scb->intracoded = 1;
		scb->skipMB = 0;
		scb->skipBlock = 0;
		scb->dc = 0;
		scb->numOfAC = 0;
	}

	lum_id_ = PutScImage(tcl.interp(), lum_);
	cr_id_ = PutScImage(tcl.interp(), cr_);
	cb_id_ = PutScImage(tcl.interp(), cb_);
	self_allocated_ = 1;
}

/*
 * Given geometry, sets and allocates appropriately.
 */
void
Semicompressed::set(int w, int h, int h_subsample, int v_subsample,
		int tw, int th, int xoff, int yoff) 
{
	w_ = w;
	h_ = h;
	css_h_ = h_subsample;
	css_v_ = v_subsample;
	true_w_ = tw;
	true_h_ = th;
	xoff_ = xoff;
	yoff_ = yoff;
	allocate();
}

/*
 * Print values for all block x,y in all planes.
 * Mostly a debugging function.
 */
void
Semicompressed::print(int x, int y) 
{
	if (lum_ != 0) {
		fprintf(stderr, "Luminance Plane\n");
		fprintf(stderr, "%d blocks wide by %d blocks high\n", lum_->width,
				lum_->height);
		if ((x < lum_->width) && (y < lum_->height)) {
			ScBlock *scb = &(lum_->firstBlock[y*lum_->width+x]);
			fprintf(stderr, "\t%d,%d block\n", x, y);
			fprintf(stderr, "\tDC value: %d\n", scb->dc);
			fprintf(stderr, "\tAC Index\t AC Value\n");
			for(int idx=0; idx<scb->numOfAC; idx++) {
				fprintf(stderr, "\t%d\t%d\n", scb->index[idx], scb->value[idx]);
			}
		} else {
			fprintf(stderr, "%d,%d out of range\n", x, y);
		}
	}

	x = x / css_h_;
	y = y / css_v_;

	if (cr_ != 0) {
		fprintf(stderr, "Cr Plane\n");
		fprintf(stderr, "%d blocks wide by %d blocks high\n", cr_->width,
				cr_->height);

		if ((x < cr_->width) && (y < cr_->height)) {
			ScBlock *scb = &(cr_->firstBlock[y*cr_->width+x]);
			fprintf(stderr, "\t%d,%d block\n", x, y);
			fprintf(stderr, "\tDC value: %d\n", scb->dc);
			fprintf(stderr, "\tAC Index\t AC Value\n");
			for(int idx=0; idx<scb->numOfAC; idx++) {
				fprintf(stderr, "\t%d\t%d\n", scb->index[idx], scb->value[idx]);
			}
		} else {
			fprintf(stderr, "%d,%d out of range\n", x, y);
		}
	}

	if (cb_ != 0) {
		fprintf(stderr, "Cb Plane\n");
		fprintf(stderr, "%d blocks wide by %d blocks high\n", cb_->width,
				cb_->height);
		if ((x < cb_->width) && (y < cb_->height)) {
			ScBlock *scb = &(cb_->firstBlock[y*cb_->width+x]);
			fprintf(stderr, "\t%d,%d block\n", x, y);
			fprintf(stderr, "\tDC value: %d\n", scb->dc);
			fprintf(stderr, "\tAC Index\t AC Value\n");
			for(int idx=0; idx<scb->numOfAC; idx++) {
				fprintf(stderr, "\t%d\t%d\n", scb->index[idx], scb->value[idx]);
			}
		} else {
			fprintf(stderr, "%d,%d out of range\n", x, y);
		}
	}
}

/*
 * Print all blocks of all planes.
 */
void
Semicompressed::print() 
{
	if (lum_ != 0) {
		fprintf(stderr, "Luminance Plane\n");
		fprintf(stderr, "%d blocks wide by %d blocks high\n", lum_->width,
				lum_->height);
		for (int row=0; row < lum_->height; row++) {
			for (int col=0; col < lum_->width; col++) {
				ScBlock *scb = &(lum_->firstBlock[row*lum_->width+col]);
				fprintf(stderr, "\t%d,%d block\n", col, row);
				fprintf(stderr, "\tDC value: %d\n", scb->dc);
				fprintf(stderr, "\tAC Index\t AC Value\n");
				for(int idx=0; idx<scb->numOfAC; idx++) {
					fprintf(stderr, "\t%d\t%d\n", scb->index[idx], scb->value[idx]);
				}
			}
		}
	}

	if (cr_ != 0) {
		fprintf(stderr, "Cr Plane\n");
		fprintf(stderr, "%d blocks wide by %d blocks high\n", cr_->width,
				cr_->height);
		for (int row=0; row < cr_->height; row++) {
			for (int col=0; col < cr_->width; col++) {
				ScBlock *scb = &(cr_->firstBlock[row*cr_->width+col]);
				fprintf(stderr, "\t%d,%d block\n", col, row);
				fprintf(stderr, "\tDC value: %d\n", scb->dc);
				fprintf(stderr, "\tAC Index\t AC Value\n");
				for(int idx=0; idx<scb->numOfAC; idx++) {
					fprintf(stderr, "\t%d\t%d\n", scb->index[idx], scb->value[idx]);
				}
			}
		}
	}

	if (cb_ != 0) {
		fprintf(stderr, "Cb Plane\n");
		fprintf(stderr, "%d blocks wide by %d blocks high\n", cb_->width,
				cb_->height);
		for (int row=0; row < cb_->height; row++) {
			for (int col=0; col < cb_->width; col++) {
				ScBlock *scb = &(cb_->firstBlock[row*cb_->width+col]);
				fprintf(stderr, "\t%d,%d block\n", col, row);
				fprintf(stderr, "\tDC value: %d\n", scb->dc);
				fprintf(stderr, "\tAC Index\t AC Value\n");
				for(int idx=0; idx<scb->numOfAC; idx++) {
					fprintf(stderr, "\t%d\t%d\n", scb->index[idx], scb->value[idx]);
				}
			}
		}
	}
}

