#include <assert.h>

#include "defs.h"
#include "inet.h"
#include "net.h"
#include "rtp.h"
#include "dct.h"
#include "bsd-endian.h"

#include "tclcl.h"
#include "crdef.h"
#include "pktbuf-rtp.h"
#include "module.h"

#include "h263/bitOut.h"
#include "h263/h263.h"
#include "encoder-h263.h"

extern "C" {
#include "h263/h263encoder.h"
#include "h263/h263encoder.p"

	int split_h263stream(
		u_char *bs,			/* bitstream [in] */
		u_int bitcount,			/* bitstream length in bits [in] */
		u_int gobs_per_frame,		/* gobs per frame [in] */
		u_int mbs_per_gob,		/* mbs per gob [in] */
		int *mbind,			/* macroblock indices [in] */
		int *mbquant,			/* quantifiers [in] */
		MVField mvfield,		/* motion vectors [in] */
		u_int ts_,			/* timestamp [in] */
		int headersize,			/* headerlength [in] */
		h263_rtp_packet	**packets	/* rtp packets [out] */
		);
}

H263Encoder::H263Encoder() : EncoderModule()
{	/* first frame must be coded INTRA */
	nexttype_ = PICTURE_CODING_TYPE_INTRA;
	
	lastw_ = 0;lasth_ = 0;
	oldpic_.w = 0;
	oldpic_.h = 0;
	oldpic_.y = new Byte[3];
	oldpic_.u = oldpic_.y+1;
	oldpic_.v = oldpic_.u+1;
	oldorigpic_.w = 0;
	oldorigpic_.h = 0;
	oldorigpic_.y = new Byte[3];
	oldorigpic_.u = oldorigpic_.y+1;
	oldorigpic_.v = oldorigpic_.u+1;
	
	mbind_=(int*)malloc(1);
	mbquant_=(int*)malloc(1);
	
	mvfield_.mx = (short*)malloc(1);
	mvfield_.my = (short*)malloc(1);
	mvfield_.mode = (short*)malloc(1);
	mvfield_.w = 1;
	mvfield_.h = 1;
	
	decpict_.w = 0; decpict_.h = 0;
	decpict_.y = new Byte[3];
	decpict_.u = decpict_.y+1;
	decpict_.v = decpict_.u+1;
	
	/* global maximum rate for the encoder.
	* The encoder itself does not have a fixed bitrate delimiter, so
	* we just allocate a very big buffer and hope that the encoder does
	* not overflow it.
	*/
	maxrate_ = 100000;
	bitstr_.b = new Byte[maxrate_*10/8];
	bitstr_.size = maxrate_*10;
	bitstr_.ind = 0;
	bitstr_.actualSize = 0;
	bitstr_.fp = NULL;
	
	/* We use only codingtime 3, since it is the best time/quality/space
	* tradeoff currently. 
	*/
	codingtime_ = 5;
	
	maycheck_ = 0;

	/* Last I frames */
	last_iframe = 0;
	
	/* default Quantisierer is 10 */
	q_ = 10;
}

H263Encoder::~H263Encoder()
{

}

void
H263Encoder::size(int w, int h)
{
    fprintf(stderr,"send-h263: new frame size: %dx%d\n",w,h);
    FrameModule::size(w, h);
}

void
H263Encoder::setq(int q)
{
  q_ = q;
}

int
H263Encoder::command(int argc, const char*const* argv)
{
    fprintf(stderr,"send-h263: command: %s\n",argv[1]);

    if (argc == 3) {
      if (strcmp(argv[1], "q") == 0) {
	q_ = atoi(argv[2]);
	return (TCL_OK);
      }
    }
    return (EncoderModule::command(argc, argv));
}

void
H263Encoder::recv(Buffer* bp)
{
  const VideoFrame* vf = (VideoFrame*)bp;
  if (!samesize(vf))
    size(vf->width_, vf->height_);
  encode(vf);
}

int BitsToByteAlign(Bitstr *bs)
{
  int n;
  
  n = 8-(bs->ind)%8;
  if(n==8) n=0;
 
  return n;
}

void PutNumber(int num, int n, Bitstr *b)
{
  if (n > 0) {
    int  i;
    long numl = (long)num;
    int  rem = b->ind & 0x7;
    long mask = (0x1 << (8 - rem)) - 1;
    int  ind = b->ind + n - 1;
    int  byteInd = ind >> 3;
    int  bitPEnd = 7 - (ind & 0x7);


    numl <<= bitPEnd;

    for (i = rem + n - 8; i > 0; i -= 8) {
      b->b[byteInd--] = (Byte)numl;
      numl >>= 8;
    }

    b->b[byteInd] &= ~mask;
    b->b[byteInd] |= (numl & mask);
    b->ind += n;
  }
}


int ByteAlign(Bitstr *b)
{
  int n;

  n = BitsToByteAlign(b);
  PutNumber(0, n, b);

  return(n);
}

/* This function encodes a intra frame and returns number of bits*/
int H263Encoder::encodeframe(int rate, int codingTime, int t, int gfid,
			     Picture *pict, Picture *decPict, Bitstr *bs,
			     int *mbInd, int *mbQuant, MVField *mvField)
{
  static int q = 20;
  static int bits = 8000;
  int        partRate, tolerance;
  int        sourceFormat;
  int        j;
  int        mx, my, mn;
  int        nMbVert = pict->h / MACROBLOCK_SIZE;/* wrong for 4CIF and 16CIF */


  if (pict->w == SQCIF_WIDTH && pict->h == SQCIF_HEIGHT) {
    sourceFormat = SQCIF;
  } else if (pict->w==QCIF_WIDTH && pict->h==QCIF_HEIGHT){
    sourceFormat = QCIF;
  } else if (pict->w==CIF_WIDTH && pict->h==CIF_HEIGHT){
    sourceFormat = CIF;
  } else if (pict->w==CIF4_WIDTH && pict->h==CIF4_HEIGHT){
    sourceFormat = CIF4;
  } else if (pict->w==CIF16_WIDTH && pict->h==CIF16_HEIGHT){
    sourceFormat = CIF16;
  } else if (!(pict->w % MACROBLOCK_SIZE || pict->h % MACROBLOCK_SIZE)) {
    sourceFormat = CUSTOM_SOURCE_FORMAT;
  } else {
    fprintf(stderr,"Picture format %dx%d is not allowed in this H.263 coder\n",
	    pict->w, pict->h);
    return(0);
  }

  if (bits > rate)
    q++;
  else if (bits < rate)
    q--;

  bits = 0;
  for (j = 0, my = 0, mn = 0; j < nMbVert; ++j, my += MACROBLOCK_SIZE) {
    /* wrong for 4CIF and 16CIF */
    partRate = j * rate / nMbVert;
    tolerance = partRate / 5;
    if (bits > partRate)
      q++;
    else if (bits < partRate - tolerance)
      q--;
    q = CLIP(q, 1, 31);

    /* Init quant tables */
    if (q != quantInit)
      H263initQuantTables(q);

    if (mbInd != NULL)
      mbInd[mn] = bs->ind;
    /* Send Header */
    if (j == 0)
      bits += H263putPictureHeader(t, sourceFormat, PICTURE_CODING_TYPE_INTRA,
				   q, pict->w, pict->h, bs);
    else
      bits += H263putGOBHeader(j, gfid, q, bs);
#ifdef PROTOCOL
    if (fp_prot_g != NULL && protFormat_g >= 3)
      fprintf(fp_prot_g, "\n");
#endif
    /* Encode and send data */
    for (mx = 0; mx < pict->w; mx += MACROBLOCK_SIZE, mn++) {
#ifdef PROTOCOL
      if (fp_prot_g != NULL && protFormat_g >= 3)
	fprintf(fp_prot_g, "I");
#endif
      if (mx != 0 && mbInd != NULL)  /* wrong for 4CIF and 16CIF */
	mbInd[mn] = bs->ind;
      if (mbQuant != NULL)
	mbQuant[mn] = q;
      if (mvField != NULL) {
	mvField->mode[mn] = MODE_INTRA;
	mvField->mx[mn] = 0;
	mvField->my[mn] = 0;
      }

      bits += H263dctMbEncodeIFrame(mx, my, q, pict, decPict, bs);
    }

    /* PSTUF or GSTUF to byte align Syncs */
    bits += ByteAlign(bs);
  }

  return(bits);
}

/***********************************************************************
* H263Encoder::encode()
*
* Main interface function to VIC is called for every grabbed frame.
* Encodes the frames into h.263, packetizes them according to RFC 2190 and
* sends the RTP packets using the RTP network layer of VIC.
*
* Returns number of sent bytes.
*/
int H263Encoder::encode(const VideoFrame* vf)
{
	YuvFrame*	yuv = (YuvFrame*)vf;	/* real YUV Frame */
	Picture		thispic;		/* current image */
	u_int		xfps,kbps;		/* parameter from Tcl */
	int		i;
	u_int		sentbytes=0;		/* sent bytes in this consume */
	u_int		bps;			/* target bitrate */
	int			k, mbs_per_gob, gobs_per_frame;
	static int gfid = -1;
	u_int			w, h;
	Tcl&		tcl = Tcl::instance();
	pktbuf*		pb;
	rtphdr*		rh;
	
	w = yuv->width_;
	h = yuv->height_;
	
	
	k = 1;
	mbs_per_gob = w/16;
	gobs_per_frame = h/16;
	if (w == CIF4_WIDTH) {
	  k = 2;
	  mbs_per_gob = 18;
	  gobs_per_frame = 88;
	}
	if (w == CIF16_WIDTH) {
	  k = 4;
	  mbs_per_gob = 18;
	  gobs_per_frame = 352;
	}
	
	/* did we the size of the grabbed frame change? */
	if (w!=lastw_ || h!=lasth_) {
	  gfid++;
		/* readjust structzures, temp. arrays, et.al. */
		delete[] oldpic_.y;
		delete[] oldorigpic_.y;
		delete[] decpict_.y;
		
		free(mvfield_.mx);
		free(mvfield_.my);
		free(mvfield_.mode);
		
		oldpic_.w = w;oldpic_.h = h;
		oldpic_.y = new Byte[w*h*3/2];
		oldpic_.u = oldpic_.y+w*h;
		oldpic_.v = oldpic_.u+w*h/4;
		
		oldorigpic_.w = w;oldpic_.h = h;
		oldorigpic_.y = new Byte[w*h*3/2];
		oldorigpic_.u = oldorigpic_.y+w*h;
		oldorigpic_.v = oldorigpic_.u+w*h/4;
		
		decpict_.w = w;decpict_.h = h;
		decpict_.y = new Byte[w*h*3/2];
		decpict_.u = decpict_.y+w*h;
		decpict_.v = decpict_.u+w*h/4;
		
		mvfield_.mx = (short*)malloc(mbs_per_gob*gobs_per_frame*sizeof(short));
		mvfield_.my = (short*)malloc(mbs_per_gob*gobs_per_frame*sizeof(short));
		mvfield_.mode = (short*)malloc(mbs_per_gob*gobs_per_frame*sizeof(short));
		mvfield_.w = mbs_per_gob;
		mvfield_.h = gobs_per_frame;
		nexttype_ = PICTURE_CODING_TYPE_INTRA;
		lastw_ = w;
		lasth_ = h;
		free(mbind_);free(mbquant_);
		mbind_=(int*)malloc((mbs_per_gob*gobs_per_frame+1)*sizeof(int));
		mbquant_=(int*)malloc(mbs_per_gob*gobs_per_frame*sizeof(int));
		
		
	}
	
	/* HACK: We query the Tcl layers for the specified bps and fps values.
	* Hack due to the fact, that no other encoder uses this values in this
	* way and we could get feedback loops from the network code (who uses
	* them). Bummer.
	*/
	tcl.evalc("$fps_slider get");sscanf(tcl.result(),"%d",&xfps);
	tcl.evalc("$bps_slider get");sscanf(tcl.result(),"%d",&kbps);
	
	/* calculate bits/frame */
	bps = kbps*1024/xfps;
	//       	fprintf(stderr,"bps = %d\n",bps);

	
	/* temporary picture from passed YUV Frame */
	thispic.w = w;
	thispic.h = h;
	thispic.y = (Byte*)yuv->bp_;
	thispic.u = (Byte*)yuv->bp_+w*h;
	thispic.v = (Byte*)yuv->bp_+w*h+(w*h)/4;
	
	/* Bitstream starts at 0 */
	bitstr_.ind = 0;
	
	/* and GO */
	bitstr_.b[bitstr_.size/8-1] = 0x42;
	
	encodeframe(q_,
		    codingtime_,
		    (int)vf->ts_,
		    gfid,
		    &thispic,
		    &decpict_,
		    &bitstr_,
		    mbind_,
		    mbquant_,
		    &mvfield_);

	{
		Picture tmppict;
		tmppict = oldpic_; oldpic_ = decpict_; decpict_ = tmppict;
	}
	memcpy(oldorigpic_.y,thispic.y,w*h*3/2);
	
	assert(bitstr_.ind < bitstr_.size);
	assert(bitstr_.b[bitstr_.size/8-1]== 0x42);
	
	/* mark the end of the bitstream */
	mbind_[mbs_per_gob*gobs_per_frame]=bitstr_.ind;
	
	/* The actual splitfunction */
	h263_rtp_packet	*packets = NULL;
	int nrofpackets = split_h263stream(
		bitstr_.b,			/* [in] bitstream */
		bitstr_.ind,		/* [in] length in bits*/
		gobs_per_frame,
		mbs_per_gob,
		mbind_,
		mbquant_,
		mvfield_,
		yuv->ts_,			/* [in] timestamp */
		mtu_,			/* [in] current MTU */
		&packets			/* [out] packets... */
		);
		/* And now we send the packets.
		* We use the 2 standard iov buffers, buffer 1 just for the RTP header,
		* buffer 2 for the H263 payloadheader and the h.263 bits.
		* Reason for this is that the VIC decoder structure does not like 
		* like differently sized headers and acts differently between loopback
		* and network-in decoding.
	*/
	for (i=0;i<nrofpackets;i++) {
		pb = pool_->alloc(yuv->ts_,RTP_PT_H263);
		rh = (rtphdr*)pb->data;
		
		memcpy(&pb->data[sizeof(rtphdr)],packets[i].header,packets[i].headersize);
		memcpy(&pb->data[sizeof(rtphdr)]+packets[i].headersize,packets[i].data,packets[i].datasize);
		pb->len = sizeof(rtphdr) + packets[i].datasize+packets[i].headersize;
		
		/* mark the last packet */
		if (i==nrofpackets-1)
			rh->rh_flags |= htons(RTP_M);
		/* sending ...  */
		target_->recv(pb);
		sentbytes+=packets[i].headersize+packets[i].datasize+sizeof(rtphdr);
	}

	nexttype_ = PICTURE_CODING_TYPE_INTRA;
	free(packets);

	return sentbytes;
}
