/*
 * decoder-jpeg.cc --
 *
 *      Motion Jpeg decoder source file
 *
 * Copyright (c) 1993-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

static const char rcsid[] =
    "@(#) $Header: /usr/mash/src/repository/mash/mash-1/codec/decoder-jpeg.cc,v 1.39 2002/02/03 03:13:33 lim Exp $";

#include <fstream.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <iostream.h>
#include "inet.h"
#include "rtp.h"
#include "decoder.h"
#include "bsd-endian.h"
#include "tclcl.h"
#include "jpeg/jpeg.h"
#include "renderer.h"
#include "pktbuf.h"
#include "postdct.h"

#include "decoder-jpeg.h"


static class MotionJpegDecoderClass : public TclClass {
public:
	MotionJpegDecoderClass() : TclClass("Module/VideoDecoder/JPEG") {}
	TclObject* create(int /* argc */, const char*const* /* argv */) {
		return (new MotionJpegDecoder());
	}
} dm_mjpeg;


// end of image
#define EOI 0xd9

// bit in the jpeg header type field that
// indicates if the image has restart markers
#define JPEG_HDR_RESTART_BIT 0x40
struct restarthdr {
    u_int16_t interval;
    u_int16_t count;
};

// initial/max size of each reassembly buffer.
#define JPEG_BUFSIZE (16*1024)
#define JPEG_MAXSIZE (512*1024)

#define STAT_BADOFF 0
#define STAT_HUGEFRM 1
#define STAT_BADTYPE 2


//
// JpegReassembler --
//

JpegReassembler::JpegReassembler()
	: decimate_(0), ndec_(0), hugefrm_(0), badoff_(0)
{
	rbsize_ = JPEG_BUFSIZE;
	rb0_.bp = new u_char[2 * JPEG_BUFSIZE];
	rb0_.ts = ~0;
	rb0_.drop = 0;
	rb0_.current = rb0_.bp;
	rb0_.restartInterval = 0;
	rb0_.restartCount = 0;
	rb0_.len = 0;
	rb1_.bp = &rb0_.bp[JPEG_BUFSIZE];
	rb1_.ts = ~0;
	rb1_.drop = 0;
	rb1_.current = rb1_.bp;
	rb1_.restartInterval = 0;
	rb1_.restartCount = 0;
	rb1_.len = 0;
	memset(rb0_.bp, 0, 2*JPEG_BUFSIZE);

	nslots_ = 64;
	slotmask_ = nslots_ - 1;
	slots_ = new slot[nslots_];
	memset(slots_, 0, nslots_ * sizeof(slot));
	prev_ = NULL;
	current_ = NULL;
	doneodd = false;
}

JpegReassembler::~JpegReassembler()
{
	delete[] rb0_.bp;
}

/*
 * Reassemble an RTP/JPEG stream.  Return a pointer to a buffer
 * each time we encounter an entire frame.  Otherwise, return 0.
 * Set len to the length of the jpeg data in the buffer.
 */
u_char* JpegReassembler::reassemble(const rtphdr* rh,
				    const u_char* bp, int& len)
{
        jpeghdr* p = (jpeghdr*)(rh + 1);
	int off = int(ntohl(p->off) & 0xffffff);
	
	int cc = len;
	if (off < 0) {
		++badoff_;
		return (0);
	}
	
	if (off + cc > rbsize_) {
		/*
		 * Check for outrageous frame size.
		 */
		if (off + cc > JPEG_MAXSIZE) {
			++hugefrm_;
			return (0);
		}

		/*
		 * Grow reassembly buffers.
		 */
		int nsize = rbsize_;
		do {
			nsize <<= 1;
		} while (off + cc > nsize);
		u_char* p = new u_char[2 * nsize];
		memcpy(p, rb0_.bp, rbsize_);
		memcpy(p + nsize, rb1_.bp, rbsize_);
		delete[] rb0_.bp;
		rb0_.bp = p;
		rb1_.bp = p + nsize;
		rbsize_ = nsize;
	}

	/*
	 * Initialize the slot data structure.
	 */
	int seqno = ntohs(rh->rh_seqno);
	int s = seqno & slotmask_;
	u_int32_t ts = ntohl(rh->rh_ts);
	slots_[s].seqno = seqno;
	slots_[s].off = off;
	slots_[s].ts = ts;

	/*
	 * Figure out which reassembly-buffer to use.  If we're not
	 * already reassembling this frame, take over the older buffer.
	 */
	rbuf* rb;

	if (ts == rb0_.ts)
		rb = &rb0_;
	else if (ts == rb1_.ts)
		rb = &rb1_;
	else {
		rb = ((int)(rb0_.ts - rb1_.ts) < 0) ? &rb0_ : &rb1_;
		rb->ts = ts;
		rb->drop = 0;
		/*
		 * If we're decimating frames (to save cycles),
		 * remember that we might want to drop the rest
		 * of the packets from this frame.
		 */

		if (decimate_) {
			if (--ndec_ <= 0)
				ndec_ = decimate_;
			else
				rb->drop = 1;
		}
	}
	if (rb->drop)
		return (0);

	memcpy((char*)&rb->bp[off], (char*)bp, cc);

	/*
	 * Check if we're at end-of-frame.  If not, see if we're
	 * filling a hole.  If not, return.  Otherwise, drop out
	 * below and check for an entire frame.  We set cc to be
	 * the entire frame size in the if-else below.
	 */
	if ((ntohs(rh->rh_flags) & RTP_M) != 0) {
		slots_[s].eof = cc;
		cc += off;
	} else {
		slots_[s].eof = 0;
	startsearch:
		int ns = s;
		do {
			ns = (ns + 1) & slotmask_;
			if (slots_[ns].ts != ts)
				return (0);

			/*
			 * If we wrap all the way around, our
			 * slots_ data structure is too small
			 * for the frame we're receiving so
			 * resize it now.
			 */
			if (ns == s) {
				if (nslots_ > MAX_JPEG_SLOTS) {
					// ++hugefrm_;
					return (0);
				}
				slot* newslots = new slot[2*nslots_];
				memset(newslots, 0, 2*nslots_*sizeof(slot));

				int newmask = 2*nslots_ - 1;
				int i;
				for (i=0; i<nslots_; i++) {
					int seqno = slots_[i].seqno;
					if (seqno == 0)
						continue;
					memcpy(&newslots[seqno&newmask],
					       &slots_[seqno&slotmask_],
					       sizeof(slot));
				}
				nslots_ *= 2;
				slotmask_ = newmask;
				delete[] slots_;
				slots_ = newslots;
				goto startsearch;
			}
		} while (slots_[ns].eof == 0);
		cc = int(slots_[ns].eof + slots_[ns].off);
	}

	/*
	 * At this point, we know we have an end-of-frame, and
	 * all packets from slot 's' up until the end-of-frame.
	 * Scan backward from slot 's' making sure we have all
	 * packets from the start-of-frame (off == 0) to 's'.
	 */
	int ps = s;
	do {
		ps = (ps - 1) & slotmask_;
		if (slots_[ps].ts != ts || ps == s)
			return (0);
	} while (slots_[ps].off != 0);

	len = cc;
	return (rb->bp);
}

#ifdef notyet
/*
 * Part of the recovery code.  Took from Matt.  Modified to fit into Mash
 */
bool JpegReassembler::advancePrevPtr(rbuf * curr, rbuf * prev)
{
  // check to see if we don't need to (or can't) advance the pointer
  if(prev->restartCount>=curr->restartCount) {
    // return true if equal, false if prev is ahead of us
    if (prev->restartCount != curr->restartCount) {
#ifndef NDEBUG
      cout << "ERROR !!!" << endl;
#endif
      // Error
      return false;
    } else {
      return true;
    }

  }

  // compute how far we need to advance the pointer
  int advance=curr->restartCount-prev->restartCount;

  // scan ahead through the appropriate number of restart intervals
  unsigned char * currPtr=prev->current;
  for(; advance>0; --advance) {
    int lastFF=0;
    while(1) {
      if(lastFF) { // if last character was a 0xff
	if(*currPtr!=0) { // if not a 0xff in the byte stream
	  if(*currPtr==EOI && advance>1) { // if got eoi when need more data
#ifndef NDEBUG
	    cout << "ERROR: got EOI when when still need more to copy!!!" 
		 << endl;
#endif
	    advance=0;
	    return false;
	  }
	  ++currPtr; // advance past the 'opcode' for the 0xff
	  break;
	}
	lastFF=0; // we've processed the 0xff, so unmark
      }
      else if(*currPtr==0xff)
	lastFF=1; // mark that we saw a 0xff
      ++currPtr;
    }
  }

  // update the pointers and corresponding restart interval #
  prev->current=currPtr;
  //  cout << "In AdvPrePt, before prev->RestartCount is " << prev->restartCount << endl;
  prev->restartCount=curr->restartCount;
  //  cout << "In AdvPrePt, after prev->RestartCount is " << prev->restartCount << endl;
  return true;
}


/*
 * New Recover function.  Took from Matt's recovery code.  Modified to fit
 * into Mash code
 */

bool JpegReassembler::recoverFromPrev(rbuf * curr, rbuf * prev, int recoverTo)
{
  //  cout << "Doing recovering" << endl;
  bool result;
  result = advancePrevPtr(curr, prev);
  if (!result) {
#ifndef NDEBUG
    cout << "ERROR in AdvancePrevPtr" << endl;
#endif
    abort();
  }
  // get the starting/ending pointers for copying data
  unsigned char * startPtr=prev->current;
  unsigned char * endPtr=prev->current;

  if(recoverTo>=0) { // if we need to recover to a particular location

    int i=recoverTo - (curr->restartCount*curr->restartInterval); // count how many to copy

    if(!i)
      return true; // no need to advance
    else if(i<0)
      return false; // we've already got this recovered

    i/=prev->restartInterval; // convert MCUs to restart intervals
    
    // scan for restart intervals to discover how much data to copy
    for(; i>0; --i) { // see previous function for comments on code
      int lastFF=0;
      while(1) {
	if(lastFF) {
	  if(*endPtr!=0) {
	    if(*endPtr==EOI && i>1) {
#ifndef NDEBUG
	      cout << "ERROR: got EOI when when still need more to copy!!!" 
		   << endl;
#endif
	      return false;
	    }
	    ++endPtr;
	    break;
	  }
	  lastFF=0;
	}
	else if(*endPtr==0xff)
	  lastFF=1; 
	++endPtr;
      }
    }
    curr->restartCount=recoverTo / (curr->restartInterval);
  }
  else { // recover until EOI (the end of the jpeg image)
    int lastFF=0;
    while(1) {
      if(lastFF) {
	if(*endPtr==EOI) {
	  ++endPtr;
	  break;
	}
	lastFF=0;
      }
      else if(*endPtr==0xff)
	lastFF=1; 
      ++endPtr;
    }
#ifndef NDEBUG
    cout << " Recover to " << (unsigned int)(endPtr-startPtr) << endl;
#endif
    // what should nextIn be updated to?????
  }

  // copy data and update/advance the pointers (if something to copy)
  if(endPtr-startPtr) {
    memcpy(curr->current, startPtr, endPtr-startPtr);
    curr->current+=endPtr-startPtr;
    curr->len+=endPtr-startPtr;
    //    curr->recoveredFrame=1; // mark that recovery was performed
  }
  return true;
}




/*
 * New Reassemble function.  This function will attempt recover on an RTP/JPEG
 * Stream.  If a complete frame is done, it will return the entire frame,
 * Otherwise, return 0.
 */
u_char* JpegReassembler::recoverReassem(const rtphdr* rh, const u_char* bp, int& len, u_int32_t restartInterval, u_int32_t restartCount)
{

  jpeghdr* p = (jpeghdr*)(rh + 1);
  bool result = false;
  unsigned int firstWord=(unsigned int)(ntohl(p->off)); // first word of jpeg header
  unsigned int typeSpec=(firstWord&0xff000000)>>24;
  int off = firstWord&0x00ffffff;
  rbuf* returnFrame = NULL;

  int cc = len;
  if (off < 0) {
    ++badoff_;
    return (0);
  }

  /*
   * Check for outrageous frame size.
   */
  if (off + cc > 256*1024) {
    ++hugefrm_;
    return (0);
  }
	
  if (off + cc > rbsize_) {
    // This should be an error
#ifndef NDEBUG
    cout << " Error in rbsize_ " << endl;
#endif
    return (0);
  }
  //  int seqno = ntohs(rh->rh_seqno);
  // int s = seqno & JPEG_SLOTMASK;
  u_int32_t ts = ntohl(rh->rh_ts);

  // Check to see if we are in the middle of a frame

  if ((current_ != NULL) && (ts == current_->ts)) {
    if (restartCount == current_->restartCount) {
      // Don't need recovery
      memcpy((char*)(current_->current), (char*)bp, cc);
      current_->len = current_->len + cc;
      

      /*
    ** count number of intervals in data so we know what to expect next time
    */

      int lastFF=0;
      for( unsigned char * i= current_->current; i< current_->current+cc; ++i) {
	if(lastFF) {
	  if(*i!=0)
	    current_->restartCount = current_->restartCount + 1;
	  lastFF=0;
	}
	else if(*i==0xff)
	  lastFF=1;
      }
      current_->current = current_->current + cc;

    } else if (restartCount < current_->restartCount) {
      // toss the frame because we have recovered
      return (0);
    } else {
      // Need recovery
      if (!((typeSpec == 1) && (doneodd))) {
	// Don't have a previous frame
	current_ = NULL;
	return (0);
      } else {
	recoverFromPrev(current_, prev_, restartInterval*restartCount);

	// Copy in the rest
	memcpy((char*)(current_->current), (char*)bp, cc);
	current_->len = current_->len + cc;
	/*
	** count number of intervals in data so we know what to expect next time
	*/

      int lastFF=0;
      for( unsigned char * i= current_->current; i< current_->current+cc; ++i) {
	if(lastFF) {
	  if(*i!=0)
	    current_->restartCount = current_->restartCount + 1;
	  lastFF=0;
	}
	else if(*i==0xff)
	  lastFF=1;
      }
      current_->current = current_->current + cc;

      }
    }
    // Need to check if we are at the end of a frame
    if ((ntohs(rh->rh_flags) & RTP_M) != 0) {
      // We are indeed at the end of the frame
      // Need to setup for the next field
      if (typeSpec == 1) {
	doneodd = true;
      }
      returnFrame = current_;
      current_->current = current_->bp;
      current_->restartCount = 0;
      current_ = NULL;
      prev_->current = prev_->bp;
      prev_->restartCount = 0;
      len = returnFrame->len;
      return returnFrame->bp;
    } else {
      // Not at the end of the frame yet
      return (0);
    }
  } else {
    // New Frame
    // We need to recover if current frame exists
    if (current_ != NULL) {
      // Need to recover
      //      cout << "Recover with recoverto == -1" << endl;
      if (doneodd) {
	result = recoverFromPrev(current_, prev_, -1);
	current_->current = current_->bp;
	current_->restartCount = 0;
	prev_->current = prev_->bp;
	prev_->restartCount = 0;
	current_ = NULL;
	returnFrame = current_;
      }
    }
    if ((ts - rb0_.ts) <= (ts - rb1_.ts)) {
      // Picking rb1_ as the current odd field and
      // rb0_ as the previous
      //	cout << "Picking rb2" << endl;
      current_ = &rb1_;
      prev_ = &rb0_;
      prev_->current = prev_->bp;
      prev_->restartCount = 0;
    } else {
      // Picking rb0_ as the current odd field and
      // rb1_ as the old
      //	cout << "Picking r0" << endl;
      current_ = &rb0_;
      prev_ = &rb1_;
      prev_->current = prev_->bp;
      prev_->restartCount = 0;
    }
    // Copying
    current_->ts = ts;
    current_->len = 0;
    current_->restartInterval = restartInterval;
    current_->restartCount = 0;
    current_->current = current_->bp;
    if (restartCount != 0) {
      // Missing front of a frame
      // FIXME When Prev_ is not initialized, need to do something else
      if (doneodd)
	if ((ntohs(rh->rh_flags) & RTP_M) == 0) {
	  recoverFromPrev(current_, prev_, restartInterval*restartCount);
	} else {
	  current_ = NULL;
	  return (0);
	}
      else {
#ifndef NDEBUG
	cout << "can't handle without the start" << endl;
	cout << "The number for restartCount is " << restartCount << endl;
#endif
	current_=NULL;
	return(0);
      }
    }
    // Copying
    memcpy((char*)(current_->current), (char*)bp, cc);
    current_->len += cc;
    /*
    ** count number of intervals in data so we know what to expect next time
    */

    int lastFF=0;
    for( unsigned char * i= current_->current; i< current_->current+cc; ++i) {
      if(lastFF) {
	if(*i!=0)
	  current_->restartCount = current_->restartCount + 1;
	lastFF=0;
      }
      else if(*i==0xff)
	lastFF=1;
    }

    current_->current = current_->current + cc;
    if (returnFrame != NULL) {
      len = returnFrame->len;
      return returnFrame->bp;
    } else {
      return (0);
    }
  }
}
#endif  



//
// MotionJpegDecoder --
//

MotionJpegDecoder::MotionJpegDecoder()
	: Decoder(sizeof(jpeghdr)), codec_(0)
{
	JpegDecoder::defaults(config_);

	inw_ = 0;
	inh_ = 0;
	inq_ = 0;
	/* guess type 0 */
	type_ = 0;
	decimation_ = 422;

	stat_[STAT_BADOFF].name = "Bad-Offset";
	stat_[STAT_HUGEFRM].name = "Huge-Frame";
	stat_[STAT_BADTYPE].name = "Unsupported-Type";
	nstat_ = 3;

	drop_even_ = 1;
}

MotionJpegDecoder::~MotionJpegDecoder()
{
	delete codec_;
}

void MotionJpegDecoder::stats(char* wrk)
{
	/* pull stats out of reassembler */
	setstat(STAT_BADOFF, reasm_.badoff());
	setstat(STAT_HUGEFRM, reasm_.hugefrm());
	setstat(STAT_BADTYPE, badtype_);
	Decoder::stats(wrk);
}

int
MotionJpegDecoder::command(int argc, const char*const* argv)
{
    Tcl& tcl = Tcl::instance();
    if (strcmp(argv[1], "drop-even") == 0) {
	if (argc == 2) {
	    tcl.resultf("%d", drop_even_);
	    return (TCL_OK);
	}
	else if (argc == 3) {
	    int d;
	    if (Tcl_GetBoolean(tcl.interp(), (char*)argv[2], &d) != TCL_OK)
		return (TCL_ERROR);

	    drop_even_ = d;
	    return (TCL_OK);
	}
    }
    return (Decoder::command(argc, argv));
}


void MotionJpegDecoder::configure()
{
	config_.comp[0].hsf = 2;
	int old_decimation = decimation_;
	if (type_ == 1 || type_ == 65 ) { // MattD: spec now says 1, 65
		decimation_ = 411;
		config_.comp[0].vsf = 2;
	}
	else if (type_ == 0 || type_ == 64) {
		decimation_ = 422;
		config_.comp[0].vsf = 1;
	}
	else { // undefined type
	  // TODO: MattD: now what??; following old example
	  decimation_ = 422;
	  config_.comp[0].vsf = 1;
	}
	config_.comp[1].hsf = 1;
	config_.comp[1].vsf = 1;
	config_.comp[2].hsf = 1;
	config_.comp[2].vsf = 1;
	config_.width = inw_;
	config_.height = inh_;
	JpegDecoder::quantizer(config_, inq_);

	delete codec_;
	codec_ = JpegPixelDecoder::create(config_, inw_, inh_);
	Tcl& tcl = Tcl::instance();
	int q;
#ifdef notyet
	q = atoi(tcl.attr("softJPEGthresh"));
	if (q < 0)
#endif
		q = JpegDecoder::q_to_thresh(inq_);
	codec_->thresh(q);
#ifdef notyet
	int ct = atoi(tcl.attr("softJPEGcthresh"));
#else
	int ct = 6;
#endif
	codec_->cthresh(ct);

	if (old_decimation != decimation_)
		tcl.evalf("%s parameters_changed", name());
}

int MotionJpegDecoder::colorhist(u_int* hist) const
{
	const u_char* frm = codec_->frame();
	int off = inw_ * inh_;
	if (decimation_ == 411)
		colorhist_411_556(hist, frm, frm + off, frm + off + (off >> 2),
				  inw_, inh_);
	else
		colorhist_422_556(hist, frm, frm + off, frm + off + (off >> 1),
				  inw_, inh_);
	return (1);
}

void MotionJpegDecoder::recv(pktbuf* pb)
{
    rtphdr* rh = (rtphdr*)pb->dp;
    const jpeghdr* p = (const jpeghdr*)(rh + 1);
    unsigned int typeSpec = ntohl(p->off) >> 24;

    if (typeSpec != JPEG_FULL_FRAME
	&& typeSpec != JPEG_ODD_FIELD
	&& typeSpec != JPEG_EVEN_FIELD
	&& typeSpec != JPEG_SINGLE_FIELD) {

	++badtype_;
	pb->release();
	return;
    }

    int needConfig = 0;
    if (p->q != inq_ || p->type != type_) {
	type_ = p->type;
	inq_ = p->q;
	needConfig = 1;
    }

    int inw = p->width << 3;
    int inh = p->height << 3;
    if (typeSpec != JPEG_FULL_FRAME)
	inh *= 2;
	
    if (inw_ !=  inw || inh_ != inh) {
	resize(inw, inh);
	needConfig = 1;
    }

    if (needConfig) {
	configure();
    }

    u_int8_t* bp = (u_int8_t*)(p + 1);
    int cc = pb->len - (sizeof(*rh) + sizeof(*p));

    // skip over the restart header
    if (type_ & JPEG_HDR_RESTART_BIT) {
	restarthdr* rh = (restarthdr*) bp;
	bp += sizeof(restarthdr);
	cc -= sizeof(restarthdr);

	// FIXME Need to handle the case where F and L are not 1
	restart_interval_ = ntohs(rh->interval);
	restart_count_ = ntohs(rh->count) & 0x3fff;
    } else {
	restart_interval_ = 0;
    }

    /*
     * If we're not doing interlacing, drop the
     * even fields and line double the odd fields
     */
    if (drop_even_) {
	    if (typeSpec == JPEG_EVEN_FIELD) {
		    pb->release();
		    return;
	    }
	    else if (typeSpec == JPEG_ODD_FIELD) {
		    typeSpec = JPEG_SINGLE_FIELD;
	    }
    }
    
#ifdef ERROR_RECOVERY
    if (type_ & JPEG_HDR_RESTART_BIT) {
	bp = reasm_.recoverReassem(r, bp, cc,
				   restart_interval_, restart_count_);
    }
    else
#endif
	bp = reasm_.reassemble(rh, bp, cc);


    if (bp != 0) {
	// Spin through all the software renderers and only
	// decode the frame if at least one renderer wants it.
	// (e.g., in the case of a single thumbnail window,
	// we don't want to decode the stream at full rate).
	int doSoftwareDecode = 0;
/*FIXME need to make HW path work again */
#ifdef HW_JPEG
	JpegFrame jf(0, (u_int8_t*)bp, cc, inq_, type_, inw_, inh_);
#endif
	for (Renderer* r = engines_; r != 0; r = r->next_) {
	    if (r->update_interval() == 0 || r->need_update()) {
/*FIXME need to make HW path work again */
#ifdef HW_JPEG
		if (r->ft() == FT_JPEG) {
		    r->consume(&jf);
		} else
#endif
		    doSoftwareDecode = 1;
	    }
	}
	if (doSoftwareDecode) {
	    if (restart_interval_ != 0)
		codec_->setRestartInterval(restart_interval_);

	    // decode the frame
	    codec_->decode(bp, cc, typeSpec, rvts_, now_);

	    ndblk_ = codec_->ndblk();
	    render_frame(codec_->frame(), CODEC_JPEG, inq_);
	    codec_->resetndblk();
	}
    }
    pb->release();
}


void MotionJpegDecoder::redraw()
{
	if (codec_ != 0) {
		Decoder::redraw(codec_->frame());
	}
}
