/*
 * tgmb-chunk.cc --
 *
 *      FIXME: This file needs a description here.
 *
 * Copyright (c) 1997-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "tgmb-chunk.h"
#include "mb/mb-items.h"
#include "math.h"
#include "palm-bmp/2bit_munger.h"
#include "tgmb-conn.h"

extern "C" {
#include "tk.h"
}


#define DEFAULT_SHAPE_B 5
#define DEFAULT_SHAPE_C 2


class ChunkFragBase {
public:
	virtual Bool insert(ChunkFrag *frag, ChunkFragDescr descr,
			    const char *recordPtr, int offset)=0;
	virtual void parse(void *value, u_int16_t len,
			   char *recordPtr, int offset)=0;
};


static class ChunkFragString : public ChunkFragBase {
public:
	virtual Bool insert(ChunkFrag *frag, ChunkFragDescr descr,
			    const char *recordPtr, int offset);
	virtual void parse(void *value, u_int16_t len,
			   char *recordPtr, int offset);
} chunkFragString;


static class ChunkFragArrow : public ChunkFragBase {
public:
	virtual Bool insert(ChunkFrag *frag, ChunkFragDescr descr,
			    const char *recordPtr, int offset);
	virtual void parse(void *value, u_int16_t len,
			   char *recordPtr, int offset);

private:
	void calculate(double x1, double y1, double x2, double y2,
		       double width, double shapeB, double shapeC,
		       double &a1, double &b1, double &a2, double &b2);
} chunkFragArrow;


static class ChunkFragColor : public ChunkFragBase {
public:
	virtual Bool insert(ChunkFrag *frag, ChunkFragDescr descr,
			    const char *recordPtr, int offset);
	virtual void parse(void *value, u_int16_t len,
			   char *recordPtr, int offset);

private:
	void calculate(XColor *colorPtr, u_int16_t &color);
} chunkFragColor;


static class ChunkFragInt16 : public ChunkFragBase {
public:
	virtual Bool insert(ChunkFrag *frag, ChunkFragDescr descr,
			    const char *recordPtr, int offset);
	virtual void parse(void *value, u_int16_t len,
			   char *recordPtr, int offset);
} chunkFragInt16;


static class ChunkFragFont : public ChunkFragBase {
public:
	virtual Bool insert(ChunkFrag *frag, ChunkFragDescr descr,
			    const char *recordPtr, int offset);
	virtual void parse(void *value, u_int16_t len,
			   char *recordPtr, int offset);
} chunkFragFont;


static class ChunkFragImage : public ChunkFragBase {
public:
	virtual Bool insert(ChunkFrag *frag, ChunkFragDescr descr,
			    const char *recordPtr, int offset);
	virtual void parse(void *value, u_int16_t len,
			   char *recordPtr, int offset);
} chunkFragImage;


static class ChunkFragPS : public ChunkFragBase {
public:
	virtual Bool insert(ChunkFrag * /*frag*/, ChunkFragDescr /*descr*/,
			    const char * /*recordPtr*/, int /*offset*/) {
		fprintf(stderr, "PostScript data not supported yet\n");
		return FALSE;
	}
	virtual void parse(void * /*value*/, u_int16_t /*len*/,
			   char * /*recordPtr*/, int /*offset*/) {
		fprintf(stderr, "PostScript data not supported yet\n");
	}
} chunkFragPS;


struct ChunkFragConfigSpec {
	size_t offset_;
	ChunkFragDescr descr_;
	ChunkFragBase *handler_;
};



// This is the subsection of line configs that we support
// the ugly casing is to get the correct offset when
// we pass in a PageItem*
#define OFFSET_P2T(type,field) (size_t)(&((type*)((PageItem*)0))->field)


//
// Lines and multi-lines
//
static ChunkFragConfigSpec lineConfigSpecs[] = {
	{ OFFSET_P2T(LineItem, fg_   ), descrColor, &chunkFragColor },
	{ OFFSET_P2T(LineItem, arrow_), descrArrow, &chunkFragArrow },
	{ OFFSET_P2T(LineItem, width_), descrWidth, &chunkFragInt16 },
	{ 0, descrInvalid, NULL }
};


//
// Text
//
static ChunkFragConfigSpec textConfigSpecs[] = {
	{ OFFSET_P2T(TextItem, szText_), descrText,  &chunkFragString },
	{ OFFSET_P2T(TextItem, fillC_),  descrColor, &chunkFragColor  },
	{ OFFSET_P2T(TextItem, font_),   descrFont,  &chunkFragFont   },
	{ 0, descrInvalid, NULL }
};


//
// Image
//
static ChunkFragConfigSpec imageConfigSpecs[] = {
	{ OFFSET_P2T(ImageItem, szImageName_), descrImage, &chunkFragImage },
	{ 0, descrInvalid, NULL }
};


//
// Postscript
//
static ChunkFragConfigSpec psConfigSpecs[] = {
	{ OFFSET_P2T(PSItem, szFileName_), descrImage, &chunkFragPS },
	{ 0, descrInvalid, NULL }
};


//
// This is used for ovals, rects, and other polygons
//
static ChunkFragConfigSpec polyConfigSpecs[] = {
	{ OFFSET_P2T(PolyItem, fillC_   ), descrFill,  &chunkFragColor },
	{ OFFSET_P2T(PolyItem, outlineC_), descrColor, &chunkFragColor },
	{ OFFSET_P2T(PolyItem, width_   ), descrWidth, &chunkFragInt16 },
	{ 0, descrInvalid, NULL }
};


//
// Array of structure so that aConfigSpecs[type] is the correct specs
//
static ChunkFragConfigSpec *aConfigSpecs[] = {
	NULL,                       // none
	lineConfigSpecs,            // line
	polyConfigSpecs,            // rect
	polyConfigSpecs,            // oval
	lineConfigSpecs,            // mline
	textConfigSpecs,            // text
	imageConfigSpecs,           // image
	psConfigSpecs,              // postscript
	NULL                        // invalid
};



Bool
ChunkFragString::insert(ChunkFrag *frag, ChunkFragDescr descr,
			const char *recordPtr, int offset)
{
	char *value = *((char **) (recordPtr+offset));
	value = (value ? value: (char*)"");
	frag->insert_any(descr, value, strlen(value)+1);
	return TRUE;
}


#ifdef NDEBUG
void
ChunkFragString::parse(void *value, u_int16_t /* len */,
		       char *recordPtr, int offset)
#else
void
ChunkFragString::parse(void *value, u_int16_t len,
		       char *recordPtr, int offset)
#endif
{
	assert(len == strlen((char*)value)+1);
	char **pPtr = ((char **) (recordPtr+offset));
	char *newStr;
	AllocNCopy(&newStr, (char*)value);
	delete [] *pPtr;
	*pPtr = newStr;
}


void
ChunkFragArrow::calculate(double x1, double y1, double x2, double y2,
			  double width, double shapeB, double shapeC,
			  double &a1, double &b1, double &a2, double &b2)
{
	double dx, dy, length, sinTheta, cosTheta, t1, t2;

	shapeC += width/2.0;

	dx = x2 - x1;
	dy = y2 - y1;
	length = sqrt(dx*dx + dy*dy);

	if (length == 0) {
	    sinTheta = cosTheta = 0.0;
	} else {
	    sinTheta = dy/length;
	    cosTheta = dx/length;
	}

	t1 = shapeB * cosTheta;
	t2 = shapeC * sinTheta;
	a1 = t1 + t2;
	a2 = t1 - t2;

	t1 = shapeC * cosTheta;
	t2 = shapeB * sinTheta;
	b1 = -t1 + t2;
	b2 =  t1 + t2;

	/*
	 * First arrow: (x1+a1, y1+b1) - (x1,y1) - (x1+a2,y1+b2)
	 * Last arrow:  (x2-a1, y2-b1) - (x2,y2) - (x2-a2,y2-b2)
	 */
}


Bool
ChunkFragArrow::insert(ChunkFrag *frag, ChunkFragDescr descr,
		       const char *recordPtr, int offset)
{
	ArrowType arrowType = *((ArrowType*)(recordPtr+offset));

	/*
	 * structure of the arrow information
	 *
	 * struct {
	 *    u_int16_t type;
	 *    u_int16_t a1;
	 *    u_int16_t b1;
	 *    u_int16_t a2;
	 *    u_int16_t b2;
	 * };
	 */

	const int sizeof_ArrowInfo=10;
	u_char buffer[sizeof_ArrowInfo];
	double a1, b1, a2, b2;
	u_int16_t tmp;

	tmp = htons(arrowType);
	memcpy(buffer+0, &tmp, 2);
	if (arrowType==ArrowNone) {
		a1 = b1 = a2 = b2 = 0.0;
	} else {
		LineItem *linePtr = (LineItem *) recordPtr;
		assert(linePtr->getNumPoints()==2);
		Point *points = linePtr->getPoints();
		calculate(points[0].x, points[0].y, points[1].x, points[1].y,
			  linePtr->getWidth(), DEFAULT_SHAPE_B,DEFAULT_SHAPE_C,
			  a1, b1, a2, b2);
	}

	tmp = htons((u_int16_t) a1);
	memcpy(buffer+2, &tmp, 2);
	tmp = htons((u_int16_t) b1);
	memcpy(buffer+4, &tmp, 2);
	tmp = htons((u_int16_t) a2);
	memcpy(buffer+6, &tmp, 2);
	tmp = htons((u_int16_t) b2);
	memcpy(buffer+8, &tmp, 2);

	frag->insert_any(descr, &buffer, sizeof_ArrowInfo);
	return TRUE;
}


void
ChunkFragArrow::parse(void *value, u_int16_t /*len*/,
		      char *recordPtr, int offset)
{
	ArrowType *pDst;
	u_int16_t arrowType;

	memcpy(&arrowType, value, 2);
	arrowType = ntohs(arrowType);

	pDst  = (ArrowType*)(recordPtr+offset);
	*pDst = (ArrowType) arrowType;
}


void
ChunkFragColor::calculate(XColor *colorPtr, u_int16_t &color)
{
	/*color = (u_int16_t(colorPtr->red)   +
		 u_int16_t(colorPtr->green) +
		 u_int16_t(colorPtr->blue)) / 3;
	color = (color & 0xC0) >> 6; // get the high 2 bits of the color*/


	/*
	 * 0.299 * red + 0.587 * green + 0.114 * blue
	 */

	color =	(u_int16_t) ((colorPtr->red   * 0.299) +
			     (colorPtr->green * 0.587) +
			     (colorPtr->blue  * 0.114));
}


Bool
ChunkFragColor::insert(ChunkFrag *frag, ChunkFragDescr descr,
		       const char *recordPtr, int offset)
{
	u_int16_t color;
	XColor *colorPtr = *((XColor **) (recordPtr+offset));
	if (colorPtr) {
		calculate(colorPtr, color);
		//color = htons(color);
		frag->insert(descr, color);
	}
	return TRUE;
}


void
ChunkFragColor::parse(void *value, u_int16_t /*len*/,
		      char *recordPtr, int offset)
{
	XColor **pColorPtr = (XColor **) (recordPtr+offset);
	u_int16_t color;
	memcpy(&color, value, sizeof(u_int16_t));
	color = ntohs(color);

	XColor xclr;
	xclr.red = xclr.green = xclr.blue = color;
	xclr.flags = DoRed | DoGreen | DoBlue;
	*pColorPtr =
		MB_GetColorByValue(&xclr);
	// calls the Tk_GetColorByValue in mb-tkcompat.c,
	// which ignores its tkwin parameter
}


Bool
ChunkFragInt16::insert(ChunkFrag *frag, ChunkFragDescr descr,
		     const char *recordPtr, int offset)
{
	u_int16_t integer = htons((u_int16_t) (*((int *) (recordPtr+offset))));
	frag->insert_any(descr, &integer, sizeof(u_int16_t));
	return TRUE;
}


#ifdef NDEBUG
void
ChunkFragInt16::parse(void *value, u_int16_t /* len */,
		      char *recordPtr, int offset)
#else
void
ChunkFragInt16::parse(void *value, u_int16_t len,
		      char *recordPtr, int offset)
#endif
{
	assert(len == sizeof(u_int16_t));
	int *intPtr = (int*) (recordPtr+offset);
	u_int16_t integer;
	memcpy(&integer, value, sizeof(u_int16_t));
	*intPtr = ntohs(integer);
}


Bool
ChunkFragFont::insert(ChunkFrag *frag, ChunkFragDescr descr,
		      const char *recordPtr, int offset)
{
	Tk_Font *pFont = (Tk_Font*) (recordPtr+offset);
	char *fontName = MB_NameOfFont(*pFont);
	frag->insert_any(descr, fontName, strlen(fontName)+1);
	return TRUE;
}


#ifdef NDEBUG
void
ChunkFragFont::parse(void *value, u_int16_t /* len */,
		     char *recordPtr, int offset)
#else
void
ChunkFragFont::parse(void *value, u_int16_t len,
		     char *recordPtr, int offset)
#endif
{
	assert(len == strlen((char*)value)+1);
	Tk_Font *pFont = (Tk_Font*) (recordPtr+offset);
	//Tcl_Interp *interp = Tcl::instance().interp();
	*pFont = MB_GetFont((char*)value);
}


Bool
ChunkFragImage::insert(ChunkFrag *frag, ChunkFragDescr descr,
		       const char *recordPtr, int offset)
{
	char **filenamePtr = (char**) (recordPtr + offset);
	Tcl &tcl = Tcl::instance();

	if (!frag->destination_ || !frag->destination_->name()) {
		return TRUE;
	}

	Tcl_Obj *pObj[2];
	Tcl_Obj *result;
	int noOfBits, zoomFactor;

	pObj[0] = Tcl_NewStringObj("extract_file", -1);
	pObj[1] = Tcl_NewStringObj(*filenamePtr, -1);
	int retval = tcl.evalObjs(sizeof(pObj)/sizeof(Tcl_Obj*), pObj);
	if (retval==TCL_ERROR) {
		fprintf(stderr, "failed in extract_file: %s\n%s\n",
			tcl.result(),
			Tcl_GetVar(tcl.interp(), "errorInfo",TCL_GLOBAL_ONLY));
		return FALSE;
	}

	result = Tcl_GetObjResult(tcl.interp());
	unsigned char *data, *out;
	int len, outLen;
	data = (unsigned char*)Tcl_GetStringFromObj(result, &len);
	if (len <= 0) {
		fprintf(stderr, "Invalid input image size %d; "
			"can't handle it\n", len);
		return FALSE;
	}
	zoomFactor = ((TCP_MediaPad*)frag->destination_)->zoom_factor();
	noOfBits=2;
	if (!ConvertTo2Bit(zoomFactor,
			   data, len, &out, &outLen)) {
		fprintf(stderr, "Error occurred in ConvertTo2Bit()");
		return FALSE;
	}

	if (outLen <= 0) {
		fprintf(stderr, "Invalid output image size %d; "
			"can't handle it\n", outLen);
		return FALSE;
	}
	if (outLen >= 65536) {
		fprintf(stderr, "Output image size (%d) is too big; "
			  "can't handle it\n", outLen);
		return FALSE;
	}

	frag->insert_any(descr, out, outLen);
	frag->insert(descrImageZoom, (u_int16_t)zoomFactor);
	frag->insert(descrImageBits, (u_int16_t)noOfBits);

	Free2Bit(out);
	return TRUE;
}


void
ChunkFragImage::parse(void * /*value*/, u_int16_t /*len*/,
		      char * /*recordPtr*/, int /*offset*/)
{
	// ignore this parsing (for now, at least)
}


Bool
ChunkFrag::insert_props(const PageItem *item)
{
	ChunkFragConfigSpec *pSpecs = aConfigSpecs[item->getType()];
	for ( ; pSpecs->descr_!=descrInvalid; pSpecs++) {
		if (pSpecs->handler_->insert(this, pSpecs->descr_,
					     (const char*)item,
					     pSpecs->offset_)==FALSE)
			return FALSE;
	}
	return TRUE;
}


void
ChunkFrag::get_props(PageItem *item)
{
	ChunkFragConfigSpec *pSpecs = aConfigSpecs[item->getType()];
	void *value;
	u_int16_t len;
	for ( ; pSpecs->descr_!=descrInvalid; pSpecs++) {
		if (get_any(pSpecs->descr_, value, len)) {
			pSpecs->handler_->parse(value, len, (char*)item,
						pSpecs->offset_);
		}
	}
}


u_char *
ChunkFrag::allocate_entry(ChunkFragDescr descr, int len)
{
	u_int32_t new_len, alloc_len;
	u_int16_t int16;

	// always allocate an even number of bytes
	alloc_len = ((len % 2) ? len+1 : len);
	new_len = hdr_.len_ + 4 + alloc_len;
	if (!renew(data_, hdr_.len_, new_len)) {
		delete [] data_;
		data_ = NULL;
		hdr_.len_ = 0;
		return NULL;
	}

	int16 = htons(descr);
	memcpy(data_+hdr_.len_, &int16, 2);
	int16 = htons(len);
	memcpy(data_+hdr_.len_+2, &int16, 2);
	hdr_.len_ += 4 + alloc_len;
	*(data_ + hdr_.len_ -1) = '\0';
	// put in a null at the very end, just in case it's an odd # of bytes

	return data_+hdr_.len_-alloc_len;
}


void
ChunkFrag::insert_any(ChunkFragDescr descr, void *data, int len)
{
	u_char *buf;
	buf = allocate_entry(descr, len);
	if (!buf) return;
	memcpy(buf, data, len);
}


void
ChunkFrag::insert(ChunkFragDescr descr, u_int32_t value)
{
	char buf[4];
	value = htonl(value);
	memcpy(buf, &value, 4);
	insert_any(descr, buf, 4);
}


void
ChunkFrag::insert(ChunkFragDescr descr, u_int16_t value)
{
	char buf[2];
	value = htons(value);
	memcpy(buf, &value, 2);
	insert_any(descr, buf, 2);
}


void
ChunkFrag::insertCoords(ChunkFragDescr descr, Point *points,
			int numPoints)
{
	u_char *buf;
	u_int16_t value;
	buf = allocate_entry(descr, numPoints*4);
	if (!buf) return;

	for (int i=0; i < numPoints; i++) {
		value = htons((u_int16_t)points[i].x);
		memcpy(buf, &value, 2);
		buf += 2;
		value = htons((u_int16_t)points[i].y);
		memcpy(buf, &value, 2);
		buf += 2;
	}
}


void
ChunkFrag::replace_any(ChunkFragDescr descr, void *data, int len)
{
	u_char *orig_data;
	void *dummy;
	u_int16_t orig_len;
	if (get_any(descr, dummy, orig_len)==FALSE) {
		insert_any(descr, data, len);
	}
	else {
		orig_data = (u_char*) dummy;
		if (orig_len % 2) orig_len++;

		// check if the two lengths are identical
		// (modulo the odd-byte-padding)
		if (orig_len==len || orig_len==len+1) {
			// just make sure that the very last byte is NULL in
			// case we are copying over an odd number of bytes

			*(orig_data + orig_len - 1) = '\0';
			memcpy(orig_data, data, len);

			// also, change the len field to the new value
			u_int16_t new_len = ntohs((u_int16_t)len);
			memcpy(orig_data-2, &new_len, 2);
		} else {
			u_char *to, *from;
			size_t howMuch;

			to      = orig_data - 4;
			from    = orig_data + orig_len;
			howMuch = hdr_.len_ - (from - data_);
			memmove(to, from, howMuch);
			hdr_.len_ -= (from - to);
			insert_any(descr, data, len);
		}
	}
}


Bool
ChunkFrag::get(ChunkFragDescr descr, u_int16_t &value)
{
	void *ptr;
	u_int16_t len;

	if (get_any(descr, ptr, len)==FALSE) return FALSE;
	if (len!=2) return FALSE;
	memcpy(&value, ptr, 2);
	value = ntohs(value);
	return TRUE;
}


Bool
ChunkFrag::get(ChunkFragDescr descr, u_int32_t &value)
{
	void *ptr;
	u_int16_t len;

	if (get_any(descr, ptr, len)==FALSE) return FALSE;
	if (len!=4) return FALSE;
	memcpy(&value, ptr, 4);
	value = ntohl(value);
	return TRUE;
}


Bool
ChunkFrag::get_any(ChunkFragDescr descr, void *&value, u_int16_t &len)
{
	u_char *ptr=data_, *end=data_+hdr_.len_;
	u_int16_t d, alloc_len;
	while (ptr < end) {
		memcpy(&d, ptr, 2);
		d = ntohs(d);

		memcpy(&len, ptr+2, 2);
		len = ntohs(len);
		alloc_len = ((len % 2) ? len+1 : len);

		if (d==descr) {
			value = ptr+4;
			return TRUE;
		}

		ptr += 4 + alloc_len;
	}

	ptr = NULL;
	len = 0;
	return FALSE;
}


Bool
ChunkFrag::find(ChunkFragDescr descr)
{
	void *ptr;
	u_int16_t len;
	return get_any(descr, ptr, len);
}


void
ChunkFrag::flush()
{
	if (data_) delete [] data_;
	data_ = NULL;
	memset(&hdr_, 0, sizeof(hdr_));
	hdr_.cmd_ = cmdInvalid;
}


void
ChunkFrag::flush_data()
{
	if (data_) delete [] data_;
	data_ = NULL;
	hdr_.len_ = 0;
	hdr_.cmd_ = cmdInvalid;
}
