# ui-windows.tcl --
#
#       UserWindows are toplevel windows containing a VideoWidget and a few
#       menus (for resizing and selecting various switching modes).
#
# Copyright (c) 1993-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
# @(#) $Header: /usr/mash/src/repository/mash/mash-1/tcl/vic/ui-windows.tcl,v 1.72 2002/02/23 03:33:54 chema Exp $


import Switcher CuesReceiver RemoteCamera Configuration TkWindow

#
# UserWindows are toplevel windows containing a VideoWidget and a few menus
# (for resizing and selecting various switching modes).
# The source being decoded by the UserWindow's VideoWidget is set by doing an "ActiveSource attach-window UserWindow";
# therefore switching sources just involves re-attaching to a new ActiveSource.
#
Class UserWindow -superclass Switcher -configuration {
    suppressUserName true
}

#
# This is a hack to get rid of ugly borer when displaying CIF sized
# images in a NTSC sized windows.  <i>d</i> is a Decoder.
#
VideoWindow instproc adjust-voff d {
    set ow [$self width]
    set oh [$self height]
    set iw [$d width]
    set ih [$d height]
    $self voff 0
    if { $ow == 320 && $oh == 240 } {
        if { $iw == 352 && $ih == 288 } {
            $self voff 8
        } elseif { $iw == 176 && $ih == 144 } {
        }
    } elseif { $ow == 640 && $oh == 480 } {
        if { $iw == 352 && $ih == 288 } {
            $self voff 16
        }
    }
}

UserWindow instproc resize-actual { {scale 1} } {
    $self instvar as_

    set src [$as_ set src_]
    set decoder [$src handler]

    set w [expr int($scale * [$decoder width])]
    set h [expr int($scale * [$decoder height])]
    $self resize $w $h
}

UserWindow instproc resize-zoom {scale} {
    $self instvar path_ vw_ as_
    global size$path_

    set w [expr int([[$vw_ window] width] * $scale)]
    set h [expr int([[$vw_ window] height] * $scale)]

    if {($w < 8) || ($h < 8)} {
        return
    }

    set src [$as_ set src_]
    set decoder [$src handler]
    set actualWidth [$decoder width]
    set actualHeight [$decoder height]

    # Make sure the correct menu entry is selected.

    if {($w == $actualWidth) && ($h == $actualHeight)} {
        set size$path_ actual
    } elseif {($w == $actualWidth / 2) && ($h == $actualHeight / 2)} {
        set size$path_ half
    } elseif {($w == $actualWidth * 2) && ($h == $actualHeight * 2)} {
        set size$path_ double
    } else {
        set size$path_ ${w}x${h}
    }

    $self resize $w $h
}

#
# Resize the VideoWindow being displayed in the UserWindow's VideoWidget.
#
UserWindow instproc resize { w h } {
    $self instvar vw_ as_

    $as_ detach-window $self
    [$vw_ window] resize $w $h
    #
    # Force an update so the window gets mapped at
    # the new size before we re-bind the window
    # to the source.
    #
    update idletasks

    $as_ attach-window $self
}

#
# True if w is a top-level viewing window
#
proc viewing_window w {
    if { [string range $w 0 2] == ".vw"} {
        return 1
    } else {
        return 0
    }
}

#
# Initialize some of the state variables and invoke the <i>create-window</i> method. <br>
# <dd>  <i>asm</i>:     the ActiveSourceManager that this UserWindow spouted from
# <dd>  <i>as</i>:      the ActiveSource to be displayed in the UserWindow
# <dd>  <i>usecues</i>: boolean value to indicate whether or not to use cues
# <dd>  <i>cb</i>:      a coordination-bus
#
UserWindow instproc init { asm as usecues cb {w {}} {showMenus 0}} {
    $self next $as
    $self instvar asm_ usecues_
    $self tkvar switched_ timed_ slow_ hw_ drop_even_
    set asm_ $asm
    set usecues_ $usecues
    set switched_ 0
    set timed_ 0
    set slow_ 0
    set hw_ 0
    set drop_even_ 1
    set blocky_ 0
    if { $cb != "" && $cb != "0" } {
        $self create-window $w $as 1 $showMenus
    } else {
        $self create-window $w $as 0 $showMenus
    }

    # Register myself with the ActiveSourceManager's autoplace function.
    $asm_ instvar autoplace_
    if {[info exists autoplace_]} {
        $asm_ autoplace register $as $self
    }
}

#
# Destroy a viewing window but remember where it was
# and what size it was (and since this is the otcl destroy
# method, we also destroy the UserWindow data structure).
#
UserWindow instproc destroy {} {
    $self instvar asm_ as_ path_ vw_
    #FIXME
    set w $path_.frame.video

    $as_ detach-window $self
    $vw_ destroy

    set x [winfo rootx $w]
    set y [winfo rooty $w]
    # adjust for virtual desktops
    incr x [winfo vrootx $w]
    incr y [winfo vrooty $w]
    set top [winfo toplevel $w]
    global userwin_x userwin_y userwin_size size$top
    set userwin_x($as_) $x
    set userwin_y($as_) $y
    set userwin_size($as_) [set size$top]
    destroy $top

    # Remove my ActiveSource from the ActiveSourceManager's autoplace
    # function.
    $asm_ instvar autoplace_
    if {[info exists autoplace_]} {
        $asm_ autoplace remove $as_
    }

    $self instvar cr_
    if [info exists cr_] {
        delete $cr_
    }
    $self next
}

#
# True if this UserWindow is employing a switching-mode to cycle through a set of sources.
#
UserWindow instproc is-switched {} {
    $self tkvar switched_
    return $switched_
}

#
# Delete and create a new renderer on window, <i>w</i> without
# changing anything else.
#
UserWindow instproc reallocate_renderer w {
#    global win_src
#    set src $win_src($w)
    $self instvar as_
    $as_ detach-window $self
    $as_ attach-window $self
}

#
# VideoWidgets are embeddable frames of any size that can be used to display a video stream
# by attaching a decoder for the desired source.
#
Class VideoWidget -superclass TkWindow -configuration {
    stampInterval 1000
}

#
# Create a VideoWidget, <i>w</i>, (as implemented in vw.cc) and
# initialize all the global state associated with a window.  Use
# <i>width</i> and <i>height</i> to specify the resolution of the
# VideoWindow.
#
VideoWidget instproc init { w width height } {
    #FIXME
    $self next $w
    $self instvar window_ is_slow_
    set window_ [new VideoWindow $w $width $height]
    #
    # Use the window "name" as the variable for its mode
    #
    set is_slow_ 0
}

#
# Return the VideoWindow embedded within this VideoWidget.
#
VideoWidget instproc window {} {
    return [$self set window_]
}

#
# Returns true if the update interval is set to slow.
#
VideoWidget instproc is-slow {} {
    return [$self set is_slow_]
}

#
# Redraw the VideoWindow embedded within this VideoWidget.
#
VideoWidget instproc redraw {} {
    [$self set window_] redraw
}

#
# Create colormodel methods that create the appropriate
# renderer for each supported visual/dither type.
#
foreach type { TrueColor/24 TrueColor/16 PseudoColor/8/Dither
               PseudoColor/8/ED PseudoColor/8/Gray PseudoColor/8/Quant } {
    set body "return \[new Renderer/$type \$self \$win \$dec \$heuristics]"
    Colormodel/$type instproc alloc-renderer { win dec {heuristics 0} } $body
    Renderer/$type set nb 0
}

#
# Create a renderer for this widget appropriate for the decoder of this
# <i>src</i> and set up the pipeline.
#
VideoWidget instproc attach-decoder { src colorModel useHW {useHeuristics 0} } {
    set d [$src handler]
    if {$d==""} {
        global src_nickname
        if ![info exists src_nickname($src)] {
            set name [$src sdes cname]
        } else {
            set name $src_nickname($src)
        }
        puts stderr "can't attach-decoder: no handler for src $name;\
                     format is [$src format_name]"
        return
    }
    $self instvar window_ target_ is_slow_
    set target_ ""
    if { $useHW } {
        set fmt [$src format_name]
        if { $fmt == "jpeg" } {
            set fmt $fmt/[$d decimation]
        }
        if ![catch "new assistor/$fmt" v] {
            set target_ $v
            $target_ window $window_
        }
    }
    if { $target_ == "" } {
        set target_ [$colorModel alloc-renderer $window_ [$d decimation] $useHeuristics]
    }
    if $is_slow_ {
        $target_ update-interval [$self get_option stampInterval]
    }
    $window_ adjust-voff $d
    $d attach $target_
}


# 
# Return the target renderer object of this widget.  
# Return "" if it does not exist (if attach-decoder hasn't been called yet.)
#
VideoWidget instproc get_renderer { } {
    $self instvar target_
    if {[info exists target_]} {
	return $target_
    } else {
	return ""
    }
}


#
# Turn on/off the use of image codec heuristics for the renderer
# associated with this VideoWidget.
#
VideoWidget instproc set_heuristics { v } {
    $self instvar target_
    $target_ heuristics $v
}

#
# Set the update interval for the video to be slow.
#
VideoWidget instproc set_slow {} {
    $self instvar is_slow_ target_
    set is_slow_ 1

    if { [info exists target_] } {
        $target_ update-interval [$self get_option stampInterval]
    }
}

#
# Set the update interval for the video to be normal.
#
VideoWidget instproc set_normal {} {
    $self instvar is_slow_ target_
    set is_slow_ 0

    if { [info exists target_] } {
        $target_ update-interval 0
    }
}

#
# Delete the VideoWidget.
#
VideoWidget instproc destroy {} {
    $self instvar target_
    if [info exists target_] {
        delete $target_
        $self next
    }
}

#
# Discontinue the representation of the source, <i>src</i>, within this VideoWidget.
#
VideoWidget instproc detach-decoder src {
	$self instvar target_
	set d [$src handler]
	if {[info exists target_]} {
		$d detach $target_
   	delete $target_
   	unset target_
	}
}

#FIXME this should be derived from TopLevelWindow

#
# Given a widget path, <i>w</i>, embed a new window for viewing video.
# (If <i>w</i> does not yet exist, it is created as a toplevel window.)
#
UserWindow instproc create-window { w as useCB {showMenus 1}} {
    set f [$self get_option smallfont]
    set uid [uniqueID]
    $self instvar asm_ usecues_ path_
    if { $w=={} } {
        set w .vw$uid
        Application toplevel $w
    } else {
        frame $w
    }
    set path_ $w

    catch "wm resizable $w false false"
    frame $w.frame
    $self instvar vw_ as_ path_ controlMenu_
    set as_ $as
    set path_ $w

    global size$w userwin_x userwin_y userwin_size

    set size actual
    if [info exists userwin_x($as)] {
        if { [winfo toplevel $w]==$w } {
            wm geometry $w +$userwin_x($as)+$userwin_y($as)
            wm positionfrom $w user
        }

        set size $userwin_size($as)
    }

    set src [$as set src_]
    set decoder [$src handler]
    set width [$decoder width]
    set height [$decoder height]

    switch -regexp -- $size {
	"half" {
	    set width [expr $width/2]
	    set height [expr $height/2]
	}
	"double" {
	    set width [expr $width*2]
	    set height [expr $height*2]
	}
	"actual" { }
	"[0-9]+x[0-9]+" {
	    set L [split $size x]
	    set width [lindex $L 0]
	    set height [lindex $L 1]
	}
	default {
	    #puts "Ack! unknown size $size"
	    set size "actual"
	}
    }

    set vw_ [new VideoWidget $w.frame.video $width $height]
    set size$w $size

    frame $w.bar
    button $w.bar.dismiss -text Dismiss -font $f -width 8 \
        -highlightthickness 0 -command "$self destroy"

    set m $w.bar.mode.menu
    menubutton $w.bar.mode -text Modes... -menu $m -relief raised \
        -width 8 -font $f
    menu $m

    $m add checkbutton -label {Enable Smoothing} \
	-command "$self set_heuristics" \
	-font $f -variable [$self tkvarname enable_heuristics_]
    $m add checkbutton -label {Save CPU} \
        -command "$self set_slow" \
        -font $f -variable [$self tkvarname slow_]
    $m add checkbutton -label {Use Hardware} \
        -command "$self reallocate_renderer $w.frame.video" \
        -font $f -variable [$self tkvarname hw_]

#    $m add checkbutton -label "Drop Even Fields" \
#	-command "$self set_drop_even" \
#	-font $f -variable [$self tkvarname drop_even_]

    $m add separator

    $m add checkbutton -label {Voice switched} \
        -command "$self set_switched" \
        -font $f -variable [$self tkvarname switched_]
    $m add checkbutton -label {Timer switched} \
        -command "$self set_timed" \
        -font $f -variable [$self tkvarname timed_]

    #
    # A menu for selecting participants to be switchable for
    # voice- and timer-switched windows. Initially, all the
    # participants are selected.
    #

    $m add cascade -label "Switch options..." -menu $m.opt \
        -font $f

    menu $m.opt -tearoff no

    $self instvar switch_list_
    global ${self}_switchname
    foreach s [$asm_ active-sources] {
        set ${self}_switchname($s) 1
        lappend switch_list_ $s
        $m.opt add checkbutton -label \
            [[$asm_ get_activesource $s] name] \
            -command "$self set_switch_list $s" \
            -font $f -variable ${self}_switchname($s)
    }

    if !$useCB {
        $m entryconfigure {Voice switched} -state disabled
    }

    set m $w.bar.size.menu
    menubutton $w.bar.size -text Size... -menu $m -relief raised -width 8 \
        -font $f
    menu $m

    $m add radiobutton -label "Half Size" \
	-command "$self resize-actual 0.5" \
	-font $f -value half -variable size$w
    $m add radiobutton -label "Actual Size" \
	-command "$self resize-actual" \
	-font $f -value actual -variable size$w -accelerator =
    $m add radiobutton -label "Double Size" \
	-command "$self resize-actual 2" \
	-font $f -value double -variable size$w

    set submenu $m.advanced
    menu $submenu
    $m add cascade -label "Other" -font $f -menu $submenu

    $m add separator
    $m add command -label "Reduce" \
        -command "$self resize-zoom 0.5" -font $f -accelerator -
    $m add command -label "Enlarge" \
        -command "$self resize-zoom 2" -font $f -accelerator +
    
    $submenu add radiobutton -label QCIF -command "$self resize 176 144" \
        -font $f -value 176x144 -variable size$w
    $submenu add radiobutton -label CIF -command "$self resize 352 288" \
        -font $f -value 352x288 -variable size$w
    $submenu add radiobutton -label 4CIF -command "$self resize 704 576" \
        -font $f -value 704x576 -variable size$w

    $submenu add separator
    $submenu add radiobutton -label "1/16 NTSC" \
        -command "$self resize 160 120" \
        -font $f -value 160x120 -variable size$w
    $submenu add radiobutton -label "1/4 NTSC" \
        -command "$self resize 320 240" \
        -font $f -value 320x240 -variable size$w
    $submenu add radiobutton -label NTSC \
        -command "$self resize 640 480" \
        -font $f -value 640x480 -variable size$w

    $submenu add separator
    $submenu add radiobutton -label "1/16 NTSC 601" \
        -command "$self resize 180 120" \
        -font $f -value 180x120 -variable size$w
    $submenu add radiobutton -label "1/4 NTSC 601" \
        -command "$self resize 360 240" \
        -font $f -value 360x240 -variable size$w
    $submenu add radiobutton -label "NTSC 601" \
        -command "$self resize 720 480" \
        -font $f -value 720x480 -variable size$w

    $submenu add separator
    $submenu add radiobutton -label "1/16 PAL" \
        -command "$self resize 192 144" \
        -font $f -value 192x144 -variable size$w
    $submenu add radiobutton -label "1/4 PAL" \
        -command "$self resize 384 288" \
        -font $f -value 384x288 -variable size$w
    $submenu add radiobutton -label PAL \
        -command "$self resize 768 576" \
        -font $f -value 768x576 -variable size$w


    label $w.bar.label -text "" -anchor w -relief raised
    pack $w.bar.label -expand 1 -side left -fill both

    # install camera movement controls
    if [$self yesno camctrl] {
        $self instvar camCtrlAgent_
        set camCtrlAgent_ [new RemoteCamera $self $w]
        if ![$camCtrlAgent_ isAllocated] {
            destroy $camCtrlAgent_
            unset camCtrlAgent_
        } else {
            button $w.bar.camctrls -text CamCtrls -font $f \
                -width 8 -highlightthickness 0 \
                -command "$camCtrlAgent_ toggleUI"
            pack $w.bar.camctrls -side left
        }
    }

    pack $w.bar.size $w.bar.mode $w.bar.dismiss -side left -fill y

    pack $w.frame.video -anchor c
    pack $w.frame -expand 1 -fill both

    #
    # Build cues menu if specified by the user
    #
    if $usecues_ {
        frame $w.cues -relief ridge -borderwidth 2
        label $w.cues.l -font $f -text "Cues:"
        pack $w.cues.l -side left -padx 10

        $self instvar cr_
        set cr_ [new CuesReceiver $w.cues lg]
        $cr_ enable [[$as_ set src_] sdes cname]

        pack $w.cues -fill x
    }

    if {$showMenus} {
	pack $w.bar -fill x
    }

    bind $w <Enter> { focus %W }
    #wm focusmodel $w active

    bind $w <d> "$self destroy"
    bind $w <q> "$self destroy"
    $w.bar.dismiss configure -command "$self destroy"
    wm protocol $w WM_DELETE_WINDOW "$self destroy"

    bind $w <Return> "$self forward"
    bind $w <space> "$self forward"
    bind $w <greater> "$self forward"
    bind $w <less> "$self reverse"
    bind $w <comma> "$self reverse"

    bind $w <equal> "$m invoke 1"
    bind $w <minus> "$m invoke 5"
    bind $w <plus> "$m invoke 6"

    #
    # Finally, bind the source to the window.
    #
    $as attach-window $self
}

#
# Return the widget path.
#
UserWindow instproc path {} {
    $self instvar path_
    return $path_
}

#
# Add and/or remove participants in the switchable list
#
UserWindow instproc set_switch_list s {
    $self instvar switch_list_
    global ${self}_switchname
    if [set ${self}_switchname($s)] {
        lappend switch_list_ $s
    } else {
        set i [lsearch $switch_list_ $s]
        set switch_list_ [lreplace $switch_list_ $i $i]
    }
}

#
# Rebuild the switchable list menu.
# This menu gets updated whenever the name of a participant
# changes, or someone gets activated or deactivated.
#
UserWindow instproc rebuild_switch_list_menu { } {
    $self instvar path_ asm_ switch_list_
    set m $path_.bar.mode.menu

    #
    # destroy the old menu first
    destroy $m.opt

    #
    # rebuild options menu
    # button down a name only if it was clicked before
    # remove a name if it is no longer there
    #
    menu $m.opt -tearoff no

    set new_all_list [$asm_ active-sources]

    global ${self}_switchname

    set new_switch_list {}
    foreach s $new_all_list {
        if { [lsearch $switch_list_ $s] != -1 } {
            set ${self}_switchname($s) 1
            lappend new_switch_list $s
        } else {
            set ${self}_switchname($s) 0
        }
        $m.opt add checkbutton -label \
            [[$asm_ get_activesource $s] name] \
            -command "$self set_switch_list $s" \
            -font [$self get_option smallfont] \
            -variable ${self}_switchname($s)
    }
    set switch_list_ $new_switch_list
}


#
# Return the VideoWidget being displayed in this UserWindow.
#
UserWindow instproc video-widget {} {
    return [$self set vw_]
}

#
# Return the ActiveSource attached to this UserWindow.
#
UserWindow instproc attached-source {} {
    return [$self set as_]
}

#
# Update the title of the UserWindow (a toplevel window) and its iconname to include <i>name</i>.
#
UserWindow instproc set-name name {
    $self instvar path_
    set w $path_
    if ![$self yesno suppressUserName] {
        $w.bar.label configure -text $name
    }
    #FIXME this should be in base class
    #puts "todo: move to base class [winfo toplevel $w], $w"

    if { [winfo toplevel $w]==$w } {
        wm iconname $w vic:$name
        wm title $w $name
    }
}

#
# Switch the VideoWindow to display the source, <i>src</i>, in this UserWindow.
#
UserWindow instproc switch src {
    $self instvar as_ asm_ usecues_

    #FIXME map src to active source using UI's data structure
    set as [$asm_ get_activesource $src]
    if { $as_ != $as } {

        #
        # There is a new source in this window, so
        # show this person's cues instead
        #
        if $usecues_ {
                $self instvar cr_
                $cr_ enable [$src sdes cname]
        }

        $as_ detach-window $self
        set as_ $as
        $as_ attach-window $self
    }
}

#
# Returns the source following <i>src</i> in the switchable list.
#
UserWindow instproc next_active_src src {
# FIXME should change these guys to cycle through according
# to the order in the thumbnail display
    $self instvar switch_list_ last_src_

    #
    # If voice- and timer- switched are both enabled
    # then the next participant to switched should be
    # based upon the last timer-switched person instead
    # of the voice-switched one. This way, we won't starve
    # a quite person for being switched
    #

    if { [$self enabled] && [info exists last_src_] } {
        set src $last_src_
    }
    set i [lsearch $switch_list_ $src]
    # (if we didn't find the list element, k will be -1 and incr'd to 0)
    incr i
    if { $i >= [llength $switch_list_] } {
        set i 0
    }
    set next_src [lindex $switch_list_ $i]
    set last_src_ $next_src
    return $next_src
}

#
# Returns the source preceeding <i>src</i> in a list of ActiveSources.
#
UserWindow instproc prev_active_src src {
    $self instvar asm_
    set list [$asm_ active-sources]
    set k [lsearch -exact $list $src]
    if { $k < 0 } {
        set k 0
    } else {
        if { $k == 0 } {
            set k [llength $list]
        }
        incr k -1
    }
    return [lindex $list $k]
}

#
# Enable/disable switching for this UserWindow based on the <i>switched_</i> tkvar.
#
UserWindow instproc set_switched {} {
    $self tkvar switched_
    if $switched_ {
        $self enable
    } else {
        $self disable
    }
}

#
# Start/stop timer-switching this UserWindow based on the <i>timed_</i> tkvar.
#
UserWindow instproc set_timed {} {
    $self tkvar timed_
    if $timed_ {
        $self set_timer
    } else {
        $self cancel_timer
    }
}

#
# Set the speed of the VideoWidget updates based on the <i>slow_</i> tkvar.
#
UserWindow instproc set_slow {} {
    $self tkvar slow_
    $self instvar vw_
    if $slow_ {
        $vw_ set_slow
    } else {
        $vw_ set_normal
    }
}

#
# Turn on/off use of image codec heuristics.
#
UserWindow instproc set_heuristics {} {
    $self instvar vw_
    $vw_ set_heuristics [$self use_heuristics]
}

UserWindow instproc set_drop_even {} {
    $self tkvar drop_even_
    $self instvar as_

    set src [$as_ set src_]
    set fmt [$src format_name]
    if { $fmt == "jpeg" } {
	set d [$src handler]
	$d drop-even $drop_even_
    }
}

#
# Returns whether or not image codec heuristics are enabled or
# disabled based on the <i>enable_heuristics_</i> tkvar.
#
UserWindow instproc use_heuristics {} {
    $self tkvar enable_heuristics_
    if $enable_heuristics_ {
	return 1
    } else {
	return 0
    }
}
