# application-vic.tcl --
#
#       Main vic object (VicApplication) definition.
#
# Copyright (c) 1996-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
# @(#) $Header: /usr/mash/src/repository/mash/mash-1/tcl/vic/application-vic.tcl,v 1.93 2002/02/03 04:39:54 lim Exp $


import RTPApplication Configuration AddressBlock CoordinationBus \
	VideoHandler AudioAgent VicUI ExtOut \
	FontInitializer WidgetResourceInitializer

#
# The base class for the vic application.  Parses the command line
# arguments, creates all the necessary agents and objects, configures
# them, and splices them together.
# Creates a frame using the provided <i>widgetPath</i> and packs the vic ui in it.
#
Class VicApplication -superclass RTPApplication

#
# The vic application constructor.  Parse the command line
# options in argument <i>argv</i> and build the app.
#
VicApplication instproc init {widgetPath argv} {
	$self next vic

	# Initialization of variables and resource db options
	set o [$self options]
	$self init_args $o
	# Search for some good fonts for the user interface and intial
	# the configuration options accordingly.
	new FontInitializer $o
	option add *Font [$o get_option helv12b] startupFile
	$self init_resources $o
	$o load_preferences "rtp vic"
	set argv [$o parse_args $argv]

	# Source the user's hook file if it exists. The function
	# user_hook, which may be defined in this file, will be
	# called at the end of init.
	if {[$o get_option userhookFile] != ""} {
		if {[file isfile [$o get_option userhookFile]] && \
			[file readable [$o get_option userhookFile]]} {
			source [$o get_option userhookFile]
		} else {
			puts stderr "Unable to source \"[$o get_option userhookFile]\". Not a file or not readable."
		}
	}

	set spec [$self check_hostspec $argv [$self get_option megaVideoSession]]
	$self check_rtp_sdes

	# Backward compat. -B flag is kb/s.
	set t [$self get_option maxbw]
	if { $t > 0 } {
		$o add_option maxbw [expr 1000*$t]
	}

	if { $spec != "" } {
		#FIXME need to do this to size the bandwidth slider in the GUI
		set ab [new AddressBlock $spec]
		$o add_option maxbw [$ab set maxbw_(0)]
		delete $ab
	}

	# Create a Coordination Bus
	$self init_confbus

	# Create a VideoAgent, VideoPipeline, and possibly set them up to use SCUBA
	$self instvar agent_ vpipe_ handler_ scuba_sess_
	set handler_ [new VideoHandler $spec]
	set agent_ [$handler_ agent]
	set vpipe_ [$handler_ vpipe]
	if { [$self get_option useScuba] != "" } {
		set scuba_sess_ [$handler_ set scuba_sess_]
	}

	# Create the user interface for this application.
	$self instvar ui_
	set ui_ [$self init_ui $widgetPath $spec]

	# Add audio to this app.
	set aspec [$self get_option audioSessionSpec]
	if { $aspec != "" } {
		$self instvar audioUI_ audioAgent_
		set audioAgent_ [new AudioAgent $self $aspec]
		set audioUI_ [$ui_ create-audio-ui $audioAgent_]
		$audioAgent_ attach $audioUI_
		$o add_option geometry 330x350
	}

	$self user_hook
}

#
# Build the main vic user interface and pack it into a newly created
# window in the default top-level window.  Call init_confbus before this
# method, so that instvars local_chan_ and glob_chan_ are set.
# Creates a frame using the provided <i>widgetPath</i> and packs the vic ui in it.
#
VicApplication instproc init_ui { widgetPath spec } {
	$self instvar agent_ vpipe_ scuba_sess_
	$self instvar local_chan_ glob_chan_

	frame $widgetPath
	set ui [new VicUI $widgetPath $local_chan_ $glob_chan_ $agent_ $vpipe_ "$self exit" $spec [$self get_option useScrollbars] [$self get_option autoPlace]]
	pack $widgetPath -expand 1 -fill both
	update idletasks

	if { [$self get_option useScuba] != "" } {
		$ui set scuba_sess_ $scuba_sess_
	}

	return $ui
}

#
# A hook to gracefully shut down vic.  Shuts down the
# underlying video agent and terminates the application.
#
VicApplication instproc exit {} {
	$self instvar vpipe_
	$self instvar agent_

	$vpipe_ destroy
	$agent_ shutdown
	exit 0
}

#
# Register all of the command line options understood
# by vic with the generic command-line parser.
#
VicApplication private init_args o {
	$o register_option -a audioSessionSpec
	$o register_option -B maxbw
	$o register_option -C conferenceName
	$o register_option -c dither
	$o register_option -D device
	$o register_option -f videoFormat
	$o register_option -F maxfps
	$o register_option -I confBusChannel
	$o register_option -K sessionKey
	$o register_option -M colorFile
	$o register_option -m mtu
	$o register_option -o outfile
	$o register_option -q jpegQfactor
	$o register_option -t defaultTTL
	$o register_option -T softJPEGthresh
	$o register_option -U stampInterval
	$o register_option -u userhookFile
	$o register_option -V visual
	$o register_option -N rtpName
	$o register_list_option -map rtpMap

	$o register_boolean_option -H useHardwareDecode
	$o register_boolean_option -P privateColormap
	$o register_boolean_option -scroll useScrollbars
	$o register_boolean_option -auto autoPlace

	# MeGa resources
	$o register_option -rport megaRecvVideoPort
	$o register_option -ofmt megaVideoFormat
	$o register_option -usemega megaVideoSession
	$o register_option -megactrl megaVideoCtrl
	$o register_option -sspec videoSessionSpec
	$o register_option -maxsbw maxVideoSessionBW
	$o register_option -sbw videoSessionBW
	$o register_option -sloc videoServiceLocation

	$o register_boolean_option -scuba useScuba
	# For MeGa use only.
	$o register_boolean_option -localscuba localScubaScope

	# cues option
	$o register_boolean_option -useCues useCues

	$o register_boolean_option -useRLM useRLM

	# use camera controls
	$o register_boolean_option -camctrl camctrl

	# the local rendezvous address (if specified)
	$o register_option -rendez rendez

}

#
# FIXME: move this out.
#
VicApplication instproc reset { ab } {
	$self instvar handler_ ui_
	if ![info exists handler_] {return}
	$handler_ reset $ab
	if ![info exists ui_] {return}
	$ui_ reset
}

#
# Initialize the configuration data base with all of the default
# options assumed by vic.  These can be overridden.
#
VicApplication private init_resources o {
	global tcl_platform
	new WidgetResourceInitializer

	# used by TopLevelWindow and for VicUI's window-title
	$o add_default iconPrefix vic:

	# used when creating new CoordinationBus; get and set by AddressBlock
	$o add_default defaultTTL 16

	# WAS used by init_confbus
	$o add_default confBusChannel 0

	# used by ControlMenu; get and possibly set by AddressBlock
	$o add_default videoFormat h.261

	# used by VideoPipeline; set by ControlMenu
	$o add_default useJPEGforH261 false

	# used by MetworkMananger; only apps that set this are vic and mui.  maybe it should be add_option
	$o add_default useLayersWindow 1

	$o add_default useRLM 0
	$o add_default camctrl 0

	# used by c++ code
	$o add_default jvColors 32
	$o add_default softJPEGthresh -1
	$o add_default softJPEGcthresh 6

	# used by VicApplication and ControlMenu
	# (scuba) default session bandwidth (kb/s).
	$o add_default maxVideoSessionBW 1000000

	# used by the Info window and ControlMenu
	if {$tcl_platform(platform) != "windows"} {
		option add Vic.background gray85 startupFile
		option add Vic.foreground black startupFile
		option add Vic.disabledForeground gray40 startupFile
		$o add_default background gray85
		$o add_default foreground black
		$o add_default disabledForeground gray40
	} else {
		set b [button .b____dummy____$self]
		$o add_default background [$b cget -background]
		$o add_default foreground [$b cget -foreground]
		$o add_default disabledForeground [$b cget -disabledforeground]
		destroy $b
	}

	# used by VicUI.
	$o add_default useScrollbars false
	$o add_default autoPlace false
}

#
# Create a global bus for communication between tools
# on different machines, and a local bus for tools on
# the same machine.
#
VicApplication instproc init_confbus {} {
	#set channel [$self get_option confBusChannel]
	#$self instvar cb_
	#if { $channel != "" && $channel != 0 } {
	#	set cb_ [new CoordinationBus -channel $channel]
	#} else {
	#	set cb_ ""
	#}

	set channel [$self get_option confBusChannel]
	if {$channel == ""} {set channel 2}

	$self instvar local_chan_ glob_chan_
	set local_chan_ [new CoordinationBus -channel $channel]

	incr channel
	set ttl [$self get_option defaultTTL]
	set glob_chan_ [new CoordinationBus -channel $channel -ttl $ttl]
}

