# user-apps.tcl --
#
#       Code defines how to spawn apps for various media types
#
# Copyright (c) 1997-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

import UserApplication Application RTP/Audio RTP/Video

#
UserApplication private mega_options { prog sspec ofmt } {
    set sname [join [$prog field_value o] :]
    set maxsbw [$self get_option megaMaxBW]
    set sbw [$self get_option megaStartupBW]

    set p [UserApplication set mega_rport_]

    set addr [intoa [lookup_host_addr [$self get_option megaAddrs]]]
    set ctrlrport [expr $p + 2]
    set megactrl $addr/60000:$ctrlrport/1
    set rport $p:$ctrlrport
    UserApplication set mega_rport_ [expr $p+4]

    set o "-usemega $sname -maxsbw $maxsbw -sbw $sbw -ofmt $ofmt \
	    -megactrl $megactrl -rport $rport -sspec $sspec"

    return $o
}

#
UserApplication private map_args { media } {
	set mapargs ""
	foreach a [$media attr_value "rtpmap"] {
		set L [split $a]
		if {[llength $L] != 2} {
			$self warn "bogus rtpmap attribute \"$a\""
			continue
		}
		set pt [lindex $L 0]
		set type [lindex $L 1]
		append mapargs "-map $pt:$type "
	}
	return $mapargs
}

####################
# vat
####################

UserApplication register_media audio
UserApplication register_formats audio PCM DVI GSM LPC
UserApplication register_protos audio RTP/AVP

Class UserApplication/Vat -superclass UserApplication

UserApplication/Vat instproc name {} {
    return "vat"
}

UserApplication/Vat instproc match {prog} {
    # don't use vat if using unified vic unless there is no video
    set app [Application instance]
    set v [[$prog base] media "video"]
    if {[llength $v] > 0 && [$app yesno unifiedVic]} { return "" }

    set title [$prog field_value s]
    set cmds {}
    set media [[$prog base] media "audio"]
    set i 0
    foreach m $media {
	set mapargs [$self map_args $m]

	set port [$m set port_]
	set caddr [$m set caddr_]
	set l [split $caddr "/"]
	if {[llength $l] == 1} {
	    set addr $caddr
	    set ttl 1
	} else {
	    set addr [lindex $l 0]
	    set ttl [lindex $l 1]
	}
	if { [$self get_option megaAddrs] == "" } {
	    set cmd "vat -C \"$title\" $mapargs -t $ttl $addr/$port"
	} else {
	    set spec $addr/$port/$ttl
	    # FIXME
	    set ofmt gsm
	    set mega_args [$self mega_options $prog $spec $ofmt]
	    set cmd "vat -C \"$title\" $mapargs $mega_args"
	}

	set description "vat audio tool"
	if {[llength $media] > 1} {
	    if [$media have_field i] {
		append description " for stream [$media field_value i]"
	    } else {
		append description " for stream $i"
	    }
	    incr i
	}
	lappend cmds [list $description $cmd]

    }
    return $cmds
}

####################
# vic
####################

UserApplication register_media video
UserApplication register_formats video JPEG H261 PVH
#UserApplication register_formats video JPEG H.261 PVH
UserApplication register_protos video RTP/AVP
UserApplication register_attrs video {scuba {}}

Class UserApplication/Vic -superclass UserApplication


UserApplication/Vic private mega_options { msg sspec ofmt mrec mtype} {
    if { $mtype != "video" } {
	    return ""
    }
    # video
    set o "[$self next $msg $sspec $ofmt]"
    if ![$mrec have_attr scuba] {
	    set o "$o -scuba"
    }
    return $o
}

#
# Returns a short name for the application.
#
UserApplication/Vic instproc name {} {
    return "vic"
}

#
# Returns a list of commands (suitable for passing to the tcl
# <i>exec</i> procedure) that might be run for this program.
#
UserApplication/Vic instproc match {prog} {
    set title [$prog field_value s]
    set app [Application instance]

    set aspec ""
    if [$app yesno unifiedVic] {
	set a [[$prog base] media "audio"]
	if {[llength $a] > 1} {
	    puts stderr "warning: multiple audio sessions are present."
	}

	if {[llength $a] > 0} {
	    # get audio parameters
	    set am [lindex $a 0]
	    set aport [$am set port_]
	    set aaddr [lindex [split [$am set caddr_] "/"] 0]
	    set aspec "-a $aaddr/$aport"
	}
    }

    set cmds {}

    # rendezvous channel
    set rspec ""
    set rend [[$prog base] media "data"]
    if {[llength $rend] > 0} {
	set rm [lindex $rend 0]
	set rport [$rm set port_]
	set raddr [lindex [split [$rm set caddr_] "/"] 0]
	set rspec "-rendez $raddr/$rport"
    }

    set media [[$prog base] media "video"]
    set i 0
    foreach m $media {
	set cmd "vic -C \"$title\" [$self map_args $m]"

	set f [lindex [$m set fmt_] 0]
	if [catch {RTP/Video set default_ptoa_($f)} fmt] {
	    set fmt "fmt-$f"
	}

	set port [$m set port_]
	set caddr [$m set caddr_]
	set l [split $caddr "/"]
	set len [llength $l]
	set addr [lindex $l 0]
	set ttl 1
	set count 1
	if {$len > 1} {
	    set ttl [lindex $l 1]
	}

	if {$len > 2} {
	    set count [lindex $l 2]
	}

	set spec "$addr/$port/$fmt/$ttl/$count"
	foreach msg [$prog set msgs_] {
	    #FIXME
	    if {$msg == [$prog base] } { continue }

	    set m [$msg media "video"]
	    if {[llength $m] != 1} {
		$self warn "layered stream with multiple video streams"
		set m [lindex $m 0]
	    }
	    set port [$m set port_]
	    set caddr [[lindex $m 0] set caddr_]
	    set l [split $caddr "/"]
	    set len [llength $l]
	    set addr [lindex $l 0]
	    set count 1
	    set ttl 1
	    if {$len > 1} {
		set ttl [lindex $l 1]
	    }
	    if {$len > 2} {
		set count [lindex $l 2]
	    }
	    append spec ",$addr/$port/$fmt/$ttl/$count"
	}

	#mega
	if {[$m have_attr scuba] && [$m attr_value scuba]} {
	    append cmd " -scuba "
	}
	if { [$self get_option megaAddrs] == "" } {
	    if {[$m have_attr scuba] && [$m attr_value scuba]} {
		  set b [expr 1000*[$m attr_value scuba]]
		  append cmd " -maxsbw $b "
	    }
	    append cmd " $aspec $rspec $spec"
	} else {
	    set spec $addr/$port/$ttl
	    # FIXME
	    set ofmt h261
	    append cmd " [$self mega_options $prog $spec $ofmt $m video]"
	}

	set description "vic video tool"
	if {[llength $media] > 1} {
	    if [$media have_field i] {
		append description " for stream [$media field_value i]"
	    } else {
		append description " for stream $i"
	    }
	    incr i
	}
	lappend cmds [list $description $cmd]
    }

    return $cmds
}

####################
# mb
####################

UserApplication register_media whiteboard
UserApplication register_protos whiteboard udp
UserApplication register_formats whiteboard mb wb

Class UserApplication/MB -superclass UserApplication

#
# Returns a short name for the application.
#
UserApplication/MB instproc name {} {
    return "mb"
}

#
# Returns a list of commands (suitable for passing to the tcl
# <i>exec</i> procedure) that might be run for this program.
#
UserApplication/MB instproc match {prog} {
    set title [$prog field_value s]
    set cmds {}
    set media [[$prog base] media "whiteboard"]
    set i 0
    foreach m $media {
	set port [$m set port_]
	set caddr [$m set caddr_]
	set fmt [lindex [$m set fmt_] 0]
	set l [split $caddr "/"]
	if {[llength $l] == 1} {
	    set addr $caddr
	    set ttl 1
	} else {
	    set addr [lindex $l 0]
	    set ttl [lindex $l 1]
	}

	if {$fmt == "wb"} {
	    set cmd "wb -C \"$title\" -t $ttl $addr/$port"
	} elseif { [$self get_option megaAddrs] == "" } {
	    set cmd "mb -C \"$title\" -sa $addr/$port/$ttl"
	} else {
	    set spec $addr/$port/$ttl
	    set cmd "mb -C \"$title\" [$self mega_options $prog $spec null]"
	}
	set description "mb mediaboard tool"
	if {[llength $media] > 1} {
	    if [$media have_field i] {
		append description " for stream [$media field_value i]"
	    } else {
		append description " for stream $i"
	    }
	    incr i
	}
	lappend cmds [list $description $cmd]
    }
    return $cmds
}


####################
# collaborator
####################

Class UserApplication/Collaborator -superclass UserApplication

UserApplication/Collaborator instproc name {} { return "collaborator" }

#
# collaborator will only appear as an option if the session has at least one of
# the three possible streams (audio, video, and mediaboard).
# It is just launched with minimal arguments for now; stuff like mega,
# dynamic rtp payload types, etc.. won't work.
#
UserApplication/Collaborator instproc match {prog} {
	set options [$self generate_options $prog]

	if { $options == "" } {
		return ""
	} else {
		set description "collaborator integrated mash tool"
		append cmd "collaborator $options"

		if { [$self get_option megaAddrs] != "" } {
			set spec $addr/$port/$ttl
			set mega_args [$self mega_options $prog]
			append cmd " $mega_args"
		}
		return [list [list $description $cmd]]
	}
}

#
# Returns a list of options to supply to the collaborator application.
#
UserApplication/Collaborator public generate_options {prog} {
	set title [$prog field_value s]
	set sdp_message [$prog base]

	#FIXME
	set video [lindex [$sdp_message media "video"] 0]
	set audio [lindex [$sdp_message media "audio"] 0]
	set whiteboard [lindex [$sdp_message media "whiteboard"] 0]
	if { $whiteboard!={} } {
		if { [lindex [$whiteboard set fmt_] 0]!="mb" } {
			set whiteboard {}
		}
	}

	set video_options [$self video2options $video]
	set audio_options [$self audio2options $audio]
	set mb_options [$self mb2options $whiteboard]
	append media_options "$video_options $audio_options $mb_options"

	if { $video_options=={} && $audio_options=={} && $mb_options=={} } {
		return ""
	} else {
		return "-C \"$title\" $media_options"
	}
}


UserApplication/Collaborator instproc mega_options prog {
	# FIXME: not implemented
	return ""
}



#
# Modelled after UserApplication/Vic::match
# Returns {} if media does not exist.
#
UserApplication/Collaborator private video2options { video } {
	if { $video == {} } {
		return {}
	}

	set f [lindex [$video set fmt_] 0]
	if [catch {RTP/Video set default_ptoa_($f)} fmt] {
		set fmt "fmt-$f"
	}

	set port [$video set port_]
	set caddr [$video set caddr_]
	set l [split $caddr "/"]
	set len [llength $l]
	set addr [lindex $l 0]
	set ttl 1
	set count 1
	if {$len > 1} {
		set ttl [lindex $l 1]
	}
	set spec "$addr/$port/$fmt/$ttl"

	if {[$video have_attr scuba] && [$video attr_value scuba]} {
		append cmd " -scuba "
	}
	if {[$video have_attr scuba] && [$video attr_value scuba]} {
		set b [expr 1000*[$video attr_value scuba]]
		append cmd " -vsbw $b "
	}
	append cmd " -video $spec "

	return $cmd
}

#
# Modelled after UserApplication/Vat::match
# Returns {} if media does not exist.
#
UserApplication/Collaborator private audio2options { audio } {
	if { $audio == {} } {
		return {}
	}

	set port [$audio set port_]
	set caddr [$audio set caddr_]
	set l [split $caddr "/"]
	if {[llength $l] == 1} {
	    set addr $caddr
	    set ttl 1
	} else {
	    set addr [lindex $l 0]
	    set ttl [lindex $l 1]
	}

	return "-audio $addr/$port/$ttl"
}

#
# Modelled after UserApplication/MB::match
# Returns {} if media does not exist.
#
UserApplication/Collaborator private mb2options { mb } {
	if { $mb == {} } {
		return {}
	}

	set port [$mb set port_]
	set caddr [$mb set caddr_]
	set fmt [lindex [$mb set fmt_] 0]
	set l [split $caddr "/"]
	if {[llength $l] == 1} {
	    set addr $caddr
	    set ttl 1
	} else {
	    set addr [lindex $l 0]
	    set ttl [lindex $l 1]
	}

	return "-mb $addr/$port/$ttl"
}


####################
# recorder
####################

Class UserApplication/Recorder -superclass UserApplication

UserApplication/Recorder instproc name {} { return "recorder" }

UserApplication/Recorder instproc match {prog} {
	set streams ""
	foreach m [[$prog base] set allmedia_] {
		set addr [lindex [split [$m set caddr_] /] 0]
		set port [$m set port_]
		set spec "$addr/$port"

		set media [$m set mediatype_]
		set proto [$m set proto_]
		set fmt [lindex [$m set fmt_] 0]
		if {$media == "whiteboard" && $proto == "udp" && $fmt == "mb"} {
			append streams " -add \"SRM mediaboard $spec\""
		} elseif {$proto == "RTP/AVP"} {
			append streams " -add \"RTP $media $spec\""
		}
	}
	if {$streams == ""} { return {} }

	set cmd "recorder $streams"
	set dir [$self get_option recordDir]
	if {$dir != ""} {
		append cmd " -directory $dir -noinput"
	}

	return [list [list "mash recording tool" $cmd]]
}


####################
# universalclient
####################

UserApplication register_media data
UserApplication register_protos data udp
UserApplication register_formats data rendezvous

# temporarily callout to separate application; can be
# integrated into other apps after further development

Class UserApplication/Ctrl -superclass UserApplication

UserApplication/Ctrl instproc name {} { return "universalclient" }

UserApplication/Ctrl instproc match {prog} {
    set title [$prog field_value s]
    set base [$prog base]
	set d [$base media "data"]
	if {$d != {} && [lindex [$d set fmt_] 0] == "rendezvous"} {
		set port [$d set port_]
		set caddr [$d set caddr_]
		set l [split $caddr "/"]
		if {[llength $l] == 1} {
			set addr $caddr
			set ttl 1
		} else {
			set addr [lindex $l 0]
			set ttl [lindex $l 1]
		}
		set spec $addr/$port/$ttl
		return [list [list \
			"universal client" "uc -rendez $spec"]]
	}
	return ""
}
