# example.tcl --
#
#       Example of use of FX module: it joins a session, and applies 
#       an effect composing n sources to the first n sources it sees. 
#
#       This script creates the following structure: 
#
#       decoder -> dali renderer -> effect -> dali video-capture -> encoder
#
#       Everything is implemented in C++, and tcl is used as glue. 
#
#
# Copyright (c) 1997-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#

if {[catch {source [file join $env(TCLCL_IMPORT_DIRS) import.tcl]} res]} {
	if {[catch {source ../../../import.tcl} res]} {
		puts "error sourcing the import.tcl initialization script: $res"
		exit 1
	}
}

Import enable
import Application AddressBlock VideoAgent TkWindow
import VideoPipeline
import Device



#
# EffectApplication
#
Class EffectApplication -superclass Application

EffectApplication instproc init {} {
	$self next sv
	$self instvar agent_ srcmgr_

	# this is the default ttl used by the NetworkManager object
	$self add_default defaultTTL 16

	# a VideoAgent object abstracts away the networking
	#set spec 224.2.1.1/22334
	set spec 224.1.2.3/12345
	#set spec 233.0.25.25/22334
	set agent_ [new EffectVideoAgent $self $spec]
	puts "Listening to $spec"

	# let's choose the type of output (h261,420), (jpeg,422), (jpeg,420), ...
	set encoder_type h261; set encoder_decimation 420
	#set encoder_type jpeg; set encoder_decimation 422


	# the Module/VideoEffect/XXX object carries out the effect
	# a. 1-input effects
	#set effect_ [new Module/VideoEffect/Addition $encoder_decimation]
	#set effect_ [new Module/VideoEffect/Copy $encoder_decimation]
	#set effect_ [new Module/VideoEffect/EdgeDetect $encoder_decimation]
	#set effect_ [new Module/VideoEffect/FadeColor $encoder_decimation]
	#set effect_ [new Module/VideoEffect/PanScan $encoder_decimation]
	set effect_ [new Module/VideoEffect/Whirlpool $encoder_decimation]
	#set effect_ [new Module/VideoEffect/ZeroCopy $encoder_decimation]
	# b. 2-input effects
	#set effect_ [new Module/VideoEffect/ChromaKey $encoder_decimation]
	#set effect_ [new Module/VideoEffect/Fade $encoder_decimation]
	#set effect_ [new Module/VideoEffect/Peel $encoder_decimation]
	#set effect_ [new Module/VideoEffect/Pip $encoder_decimation]
	# c. 3-input effects
	#set effect_ [new Module/VideoEffect/Mosaic $encoder_decimation]

	# create a VideoPipeline to be able to send video
	set vpipe_ [new VideoPipeline $agent_]
	$vpipe_ set_quality 99

	# use DaliVideoCapture/Uncompressed as VideoCapture (VideoTap) device
	set device DaliVideoCapture/Uncompressed
	# TBD: we need to create a non-used device to register it as a VideoTap,
	#		which is pretty dirty
	new $device $encoder_decimation

	$vpipe_ select $device $encoder_type
	$vpipe_ start

	# attach the effect to the grabber input
	set grabber_ [[$vpipe_ set tap_] set grabber_]
	#$effect_ attach $grabber_
	$effect_ target $grabber_

	# another object (called EffectSourceManager) is in charge 
	# of listening to the VideoAgent events (sources joining or starting 
	#	transmission) and responding to them
	set srcmgr_ [new EffectSourceManager $agent_ $effect_]

	# request the VideoAgent to send events to the Source Manager
	$agent_ attach $srcmgr_
}



#
# EffectVideoAgent
#
Class EffectVideoAgent -superclass VideoAgent
EffectVideoAgent instproc init {args} {
	eval $self next $args
}
EffectVideoAgent instproc create-decoder {args} {
	puts "EffectVideoAgent::create-decoder{$args}"
	eval $self next $args
}


#
# EffectSourceManager
#
Class EffectSourceManager superclass Observer

EffectSourceManager instproc init {agent effect} {
	$self next
	$self instvar agent_ effect_
	$self instvar local_srcid_ local_addr_
	$self instvar source_list_
	$self instvar input_id_

	# write up the VideoAgent and VideoEffect handlers
	set agent_ $agent
	set effect_ $effect

	# write up the local srcid and addr
	set local_srcid_ [[$agent_ local] srcid]
	set local_addr_ [[$agent_ local] addr]

	# reset the source list
	set source_list_ ""

	# initialize the effect input counter
	set input_id_ 0
}

#
#	EffectSourceManager::trigger_sdes
#
#	trigger_sdes is an event raised by the VideoAgent when somebody 
#	joins our session
#
EffectSourceManager instproc trigger_sdes {src} {
	$self instvar local_srcid_ local_addr_
	$self instvar source_list_

	# check we are not seeing our own source
	set srcid [$src srcid]
	if {$srcid == $local_srcid_} {
		# it's our own source
		return
	}

	# check we are not seeing our own address, which would mean this is the 
	#	fx-produced stream
	set addr [$src addr]
	if {$addr == $local_addr_} {
		# it's the fx-produced stream
		return
	}

	# check we haven't seen this stream before
	foreach source $source_list_ {
		if {($srcid == [lindex $source 0]) && ($addr == [lindex $source 1])} {
			# we've seen this stream before
			return
		}
	}

	lappend source_list_ [list $srcid $addr -1]
}

#
#	EffectSourceManager::trigger_media
#
#	trigger_media is an event raised by the VideoAgent when some source 
#	joined to our session starts transmitting
#
EffectSourceManager instproc trigger_media {src} {
	$self instvar local_srcid_ local_addr_
	$self instvar source_list_
	$self instvar effect_
	$self instvar input_id_

	# check we are not seeing our own source
	set srcid [$src srcid]
	if {$srcid == $local_srcid_} {
		# it's our own source
		return
	}

	# check we are not seeing our own address, which would mean this is the 
	#	fx-produced stream
	set addr [$src addr]
	if {$addr == $local_addr_} {
		# it's the fx-produced stream
		return
	}

	# check if we have seen this stream before
	set stream_id -1
	set i 0
	foreach source $source_list_ {
		if {($srcid == [lindex $source 0]) && ($addr == [lindex $source 1])} {
			# we've seen this stream before
			set stream_id $i
			if {[lindex $source 2] != -1} {
				# the stream has been already initialized
				return
			}
			break
		}
		incr i
	}

	# if it's a new stream, add an entry for it
	if {$stream_id == -1} {
		# first time we see this stream
		set stream_id [llength $source_list_]
		lappend source_list_ [list $srcid $addr -1]
	}

	# check if we have too many inputs for this effect
	if {$input_id_ >= [$effect_ get-inputs]} {
		# we have too many inputs for the effect
		puts "SourceManager: srcid $srcid from $addr ([$src sdes cname]) has\
				been rejected (too many inputs)"
		return
	}

	# new stream: get the new input id
	set input_id $input_id_
	incr input_id_

	# fix the stream list
	set source_list_ [lreplace $source_list_ $stream_id $stream_id \
			[list $srcid $addr $input_id]]
	puts "SourceManager: srcid $srcid from $addr ([$src sdes cname]) has\
			joined the effect as input $input_id"

	# we wait a little to give time for the VideoAgent to create and install 
	# a decoder, and for such decoder to get one packet so that it can 
	# realize the format, size, and color subsampling scheme of the video 
	# stream
	after idle "$self really_activate $src $input_id"
}


#
#	EffectSourceManager::really_activate
#
#	when a source has started transmitting and the decoder is already 
#	installed, really_activate attaches a renderer object to the decoder's 
#	output and to the effect input
#
EffectSourceManager instproc really_activate {src input_id} {
	$self instvar window_ decoder_ decimation_ window_ renderer_
	$self instvar effect_ vpipe_
	$self instvar agent_

	# get the decoder handler
	set decoder_ [$src handler]
	if {$decoder_ == ""} {
		# cannot find a decoder
		puts -nonewline "ESM::really_activate: couldn't find a decoder";
		return
	}

	# create a receiver for the frames (the Renderer object)
	set decimation [$decoder_ decimation]
	set renderer_ [new Renderer/Dali/Uncompressed $decimation]

	# attach the renderer to the decoder's output
	$decoder_ attach $renderer_

	# attach the renderer as an effect input
	$renderer_ attach $effect_ $input_id
}


# get rid of the main window
wm withdraw .

# create the main object
set app [new EffectApplication]

vwait forever

