# xml-to-ui.tcl --
#
#       generates UIs from XML descriptions
#
# Copyright (c) 1998-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.


import Trace PowerSwitchUI UDPChannel XMLParser
import CheckButton EntryObj ScaleObj RadioButtonsObj

#Trace on ; Trace add UIGenerator

#
# generates UIs from XML descriptions <p>
#
# XML DTD accepted by this file described in the tech report: <br>
#
# ``Enabling Smart Spaces: Entity Description and User Interface
#   Generation for a Heterogeneous Component-based Distributed System'' <p>
#
# available on http://www.cs.berkeley.edu/~hodes/research.html
#
# It is also listed here, but will be munged in a HTML browser....
#
#<pre>
#
#<!ELEMENT object (label?, addrspec?, ui*,
#                  method*, object*)>
#<!ATTLIST object
#   name  CDATA   #REQUIRED>
#<!ELEMENT method (param*)>
#<!ATTLIST method
#   name  CDATA   #REQUIRED>
#<!ELEMENT param (#PCDATA)>
#<!ATTLIST param
#   name  CDATA   #REQUIRED
#   lexType  (int | real | boolean | enum
#                 | string | ...) 'string'
#   optional  #BOOLEAN>
#<!ELEMENT label (#PCDATA)>
#<!ELEMENT addrspec (#PCDATA)>
#<!ELEMENT ui (#PCDATA)>
#
#</pre>
#
Class UIGenerator

#
UIGenerator public init {args} {

    # FIXME test harness only -- shouldn't be called with args
    set cnt 0
    foreach f $args {
	puts "parsing `$f'"
	frame .f$cnt
	$self GenerateUIfromFile $f .f$cnt
	pack .f$cnt -side right
	#$self print
	incr cnt
    }
}

# generates a UI from file <i>filename</i> and packs the result
# into window/frame <i>w</i>.  If any agents were allocated (via "new")
# due to <UI> elements in the XML, return a list of them.
#
UIGenerator public GenerateUIfromFile {filename w} {
    $self instvar parseResults_ allocatedObjects_

    set f [open $filename r]
    set x [read $f]
    set parseResults_ [XML::parse $x]
    Trc $class "[XML::pretty_print $parseResults_]"

    $self GenerateUIFromXML $parseResults_ $w

    close $f
    Trc $class "done generating."
    if [info exists allocatedObjects_] {
	set o $allocatedObjects_
	Trc $class "allocated Objects = $o"
	unset allocatedObjects_
	return $o
    }
}

# take a single XML "object" description and dynamically generate a
# corresponding UIAgent
UIGenerator public GenerateUIFromXML {parsedXMLlist w} {
    $self instvar txt_ allocatedObjects_

    $self FlattenXML $parsedXMLlist

    foreach i [lsort [array names txt_]] {
	set obj [lindex [split $i ,] 0]
	set fr $w$obj
	if {[info commands $fr] == ""} {
	    frame $fr -relief groove
	    label $fr.l -text [lindex $txt_($obj,LABEL) 0]
	    pack $fr -side left -expand 1
	    pack $fr.l
	    if {[llength [split $obj .]] == 2} {
		$fr.l configure -font "*-15-*"
	    }
	    if {[info exists txt_($obj,UI)] && \
		    [lindex $txt_($obj,UI) 0] == "LANG mash"} {
		# $self getUIAndStartinFrame $fr FIXME
		# FIXME - instead, for now, assumes have UI local & imported
		Trc $class "new [lindex $txt_($obj,UI) 1] \
			$fr $txt_($obj,ADDRPORT)"
		set o [new [lindex $txt_($obj,UI) 1] $fr $txt_($obj,ADDRPORT)]
		lappend allocatedObjects_ $o
	    } else {
		if {[array names txt_ $obj,METHODS] != ""} {
		    $self buildUIfromMethodList $fr \
			    [list $txt_($obj,METHODS)] $txt_($obj,ADDRPORT)
		}
	    }
	}
    }
    unset txt_
}

#
UIGenerator private buildUIfromMethodList {fr methods addrspec} {
    set mList [lindex $methods 0]    ;# un-listify
    set nameElems ""
    for {set i 0} {$i < [llength $mList]} {incr i} {
	if {[string first NAME [lindex $mList $i]] != -1} {
	    lappend nameElems $i
	}
    }
    lappend nameElems [llength $mList]
    for {set i 0} {$i < [expr [llength $nameElems]-1]} {incr i} {
	set low [lindex $nameElems $i]
	set high [lindex $nameElems [expr $i+1]]
	set mp [lrange $mList $low [expr $high - 1]]
	Trc $class "meths + params = $mp"
	frame $fr.$i -relief groove
	button $fr.$i.b -text [lindex [lindex $mp 0] 1]
	pack $fr.$i $fr.$i.b
	set params [lrange $mp 1 end]
	for {set p 0} {$p < [llength $params]} {incr p} {
	    frame $fr.$i.$p
	    label $fr.$i.$p.l -text [lindex [lindex $params $p] 0]
	    pack $fr.$i.$p $fr.$i.$p.l
	    Trc $class "params = $params"
	    if {[lindex [lindex $params 0] 1] != "LEXTYPE"} {puts "Error!"}
	    set fullLex [lindex [lindex $params $p] 2]
	    set lex [lindex [split $fullLex :] 0]
	    set range [lindex [split $fullLex :] 1]
	    set paramObj$p [$self CreateUIWidgetFromLexType $fr.$i.$p $fullLex]
	}
	set comm "$self SendUDP $addrspec [$fr.$i.b cget -text] "
	for {set j 0} {$j < $p} {incr j} {
	    append comm "\[[set paramObj$j] get_val\] "
	}
	Trc $class "COMM = $comm"
	$fr.$i.b configure -command "$comm"
    }
}

# actually pass on the method invocation ... args were correctly set up
# via XML parse
#
UIGenerator private SendUDP {args} {
    Trc $class "${class}::$proc $args"
    puts "Sending UDP msg: $args"
    set udp [new UDPChannel [lindex $args 0]]
    $udp send "[lrange $args 1 end]"
    delete $udp
}

#
# given a window/frame (w) and lexType string, instantiate some UI
# widgets in w, return an object that accepts a "get_val" method call
# to get the appropriate data.
UIGenerator private CreateUIWidgetFromLexType {w fullLexType} {
    Trc $class "LEXTYPE = $fullLexType"
    set lexType [lindex [split $fullLexType :] 0]
    set lexRange [lindex [split $fullLexType :] 1]
    switch $lexType {
	remoteCall -
	boolean {
	    set obj [new CheckButton $w.widget]
	}
	int {
	    if {$lexRange == ""} {
		set obj [new EntryObj $w.widget]
	    } else {
		set lexRange [lindex [split $lexRange =] 1]
		set low [lindex [split $lexRange -] 0]
		set high [lindex [split $lexRange -] 1]
		set obj [new ScaleObj $w.widget -from $low -to $high \
			-orient horizontal]
	    }
	}
	real {
	    if {$lexRange == ""} {
		set obj [new EntryObj $w.widget]
	    } else {
		set lexRange [lindex [split $lexRange =] 1]
		set low [lindex [split $lexRange -] 0]
		set high [lindex [split $lexRange -] 1]
		set obj [new ScaleObj $w.widget -from $low -to $high \
			-digits 3 -resolution 0.01 -orient horizontal]
	    }
	}
	string {
	    set obj [new EntryObj $w.widget]
	}
	enum {
	    set obj [new RadioButtonsObj $w [split $lexRange ,]]
	    set noPack 1
	}
    }
    # FIXME stupid special case -- RadioButtons
    if [info exists noPack] {
	unset noPack
    } else {
	pack $w.widget -side left
    }

    return $obj
}


# fill txt_ array with "flattened" version suitable for iterative
# parsing into UI commands  (XML must be for objectDescription DTD)
#
UIGenerator private FlattenXML {parsedXMLlist {w {}}} {
    $self instvar txt_ lastelem lastw
    set count 0
    set xml $parsedXMLlist
    if ![info exists lastelem] {
	set lastelem NULL
    }
    foreach {type arg1 arg2 arg3} $xml {
	switch $type {
	    parse:pi -
	    parse:comment -
	    parse:text {
		lappend txt_($lastw,$lastelem) $arg1
	    }
	    parse:elem {
		set lastelem $arg1
		set lastw $w
		if {$arg1 == "METHOD"} {
		    lappend txt_($w,METHODS) $arg2
		    foreach {t a1 a2 a3} $arg3 {
			lappend txt_($w,METHODS) "$a1 $a2"
		    }
		}
		if {$arg1 == "UI"} {
		    if {$arg2 != ""} {lappend txt_($w,UI) $arg2}
		}
		incr count
		$self FlattenXML $arg3 $w.$count
		#incr count -1

	    }
	}
    }
}

#
UIGenerator private parseResults {} {
    $self instvar parseResults_
    return $parseResults_
}

#
UIGenerator private print {} {
    $self instvar txt_
    parray txt_
}






proc parray {a {pattern *}} {
    upvar 1 $a array
    if ![array exists array] {
        error "\"$a\" isn't an array"
    }
    set maxl 0
    foreach name [lsort [array names array $pattern]] {
        if {[string length $name] > $maxl} {
            set maxl [string length $name]
        }
    }
    set maxl [expr {$maxl + [string length $a] + 2}]
    foreach name [lsort [array names array $pattern]] {
        set nameString [format %s(%s) $a $name]
        puts stdout [format "%-*s = %s" $maxl $nameString $array($name)]
    }
}

