# app-collaborator.tcl --
#
#       FIXME: This file needs a description here.
#
# Copyright (c) 1998-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
#  @(#) $Header: /usr/mash/src/repository/mash/mash-1/tcl/applications/collaborator/app-collaborator.tcl,v 1.21 2002/02/03 04:21:46 lim Exp $


import RTPApplication WidgetResourceInitializer FontInitializer \
		CollaboratorUI AudioAgent VideoAgent VideoHandler \
		MTrace Tcl_Player Tcl_Recorder NetworkManager SSAC_Client \
		MBDebugDlg MBUI MBPSInterp MB_Manager MBPageMgr MBNet \
		Program UserApplication/Collaborator SDPParser \
		MPlug/ScrolledUI Observer MBv2Session


Class CollaboratorApp -superclass { RTPApplication Observer }


#
# Invoked with "new CollaboratorApp".  Parse the options supplied by
# <i>argv</i>.  Creates a frame using the provided <i>widgetPath</i> and
# pack the mui application interface in it.
#
CollaboratorApp public init { widgetPath argv } {
	$self next collaborator

	# Initialization of variables and resource db options
	set o [$self options]
	$self init_args $o

	$self init_fonts $o
	$self init_mega  $o

	# load user prefs before process argv
	$o load_preferences "rtp collaborator"
	set argv [$o parse_args $argv]
	append argv [$self parse_sdp_msg]
	# Generate an error for extra args.
	$self check_argv $argv

	# if there is a cnamePrefix arg, then set the cname option
	set cname [$self get_option cname]
	if { $cname == "" } {
		set cname [$self get_option cnamePrefix]
		if { $cname != "" } {
			set cname ${cname}@[localaddr]
			$self add_option cname $cname
		}
	}

	# Retrieving and checking for at least one media spec.
	$self instvar have_video_ have_audio_ have_mb_ have_mbv2_ have_rover_ \
			useMega_
	$self instvar vspec_ aspec_ mbspec_ mbv2spec_
	$self check_hostspecs
	$self check_rtp_sdes

	# now check if we have any rover options
	# do this before calling setup_mega_options,
	# so we can actually verify if a -megactrl flag was passed in or not
	$self setup_rover
	$self setup_mega

	# init resources & load preferences
	$self init_tk_options

	if { $have_video_ } {
		$self init_vic_resources $o
	}
	if { $have_audio_ } {
		$self init_vat_resources $o
	}
	if { $have_mb_ } {
		$self init_mb_resources $o
	}
	if { $have_mbv2_ } {
		$self init_mbv2_resources $o
	}
	$self init_collaborator_resources $o

	# Backward compat. -B flag is kb/s.
	set t [$self get_option maxCollaboratorSessionBWkbps]
	if { $t > 0 } {
		$self add_option maxbw [expr 1000*$t]
	} else {
		$self add_option maxbw \
				[$self get_option maxCollaboratorSessionBW]
	}

	# Create a Coordination Bus
	$self init_confbus

	if { $have_video_ } {
		# Create a VideoAgent, VideoPipeline, and possibly set them
		# up to use SCUBA
		$self instvar videoAgent_ vpipe_ videoHandler_ scuba_sess_
		set videoHandler_ [new VideoHandler $vspec_]
		set videoAgent_ [$videoHandler_ agent]
		set vpipe_ [$videoHandler_ vpipe]
		if { [$self get_option useScuba] != "" } {
			set scuba_sess_ [$videoHandler_ set scuba_sess_]
		}
		#FIXME need to do this to size the bandwidth slider in the GUI
		if { $vspec_ != {} } {
			set ab [new AddressBlock $vspec_]
			$self add_option maxbw [$ab set maxbw_(0)]
			delete $ab
		} elseif $have_rover_ {
			$self instvar ssacClient_
			set spec [$ssacClient_ server_name]
			set ab [new AddressBlock $spec]
			$self add_option maxbw [$ab set maxbw_(0)]
			delete $ab
		}
	}

	if { $have_audio_ } {
		# Create AudioAgent and put it on the conference bus.
		$self instvar audioAgent_ local_chan_ glob_chan_
		set audioAgent_ [new AudioAgent $self $aspec_]
		if { $local_chan_ != {} } {
			$audioAgent_ attach_local_channel $local_chan_
		}
		if { $glob_chan_ != {} } {
			$audioAgent_ attach_global_channel $glob_chan_
		}
	}


	if { $have_mb_ } {
		MTrace init {trcMB}
		# must do this before create ui, because mbManager is
		# required to make ui
		$self init_mb_variables
	}

	# Create the user interface for this application.
	$self instvar ui_
	set ui_ [$self build_ui $widgetPath]

	if { $have_mb_ } {
		$self instvar mgr_ mbnet_
		# must create ui before do this, because need mbui arg
		set mbui [$ui_ get_local_option mbUI]
		set mbnet_ [new MBNet $mbui $mgr_]
		$self instvar session_ mbAgent_
		set session_ [$mbnet_ session]
		set mbAgent_ [$mbnet_ mbAgent]
		$mbnet_ adjust_spec
		$ui_ set_local_option mbSession $session_
		$ui_ set_local_option mbAgent   $mbAgent_
		[$ui_ get_local_option controlmenu,mb] new_hostspec

		set logf [$o get_option playScript]
		if {"none" != $logf} {
			global mb
			set isRT [$config get_option rtPlay]
			set player [new Tcl_Player $logf [$self set sender_] \
					$isRT]
			# wait for 1 second for everything to settle down
			after 15000 $player start
		}
	}

	if { $have_mbv2_ } {
		$self init_mbv2_session $mbv2spec_
	}
}


CollaboratorApp public init_mbv2_session { spec } {
	$self instvar mbv2_session_ ui_
	set mbv2_session_ [new MBv2Session $spec \
			[$self get_option cnamePrefix]]
	$ui_ set_local_option mbv2Session $mbv2_session_
	set o [$ui_ get_local_option observer,mbv2]
	if { $o != {} } { $o session $mbv2_session_ }

	# do the reset before the call to app_info, since we want to
	# set up the observers for the source_update notification
	$self reset {} srmv2 {}

	set cname [$self get_option cname]
	if { $cname == "" } {
		set cname [user_heuristic]@[localaddr]
	}
	$mbv2_session_ app_info [$mbv2_session_ sender] \
			[$self get_option rtpName] $cname
}


CollaboratorApp private setup_rover { } {
	$self instvar have_rover_ ssacClient_ startport_

	# first do the rover stuff
	# check if we have the -megactrl/-unicast/-mars flag
	# if so, we use that spec instead of the one in the catalog file

	set usemega [$self get_option useMegaSession]
	set megactrl [$self get_option myMegaCtrl]
	set startport_ [$self get_option megaStartPort]

	if { $megactrl == {} } \
			{ $self add_default myMegaCtrl 224.4.5.24/50000/31 }

	if { [$self get_option sessionCatalog]=={} && \
			[$self get_option sessionCatalogFile]=={} } {
		set have_rover_ 0
	} else {
		# extract the catalog file
		set ctg [$self get_option sessionCatalog]
		if { $ctg == {} } {
			set filename [$self get_option sessionCatalogFile]
			if [catch {set fd [open $filename]}] {
				puts stderr "could not open file $filename"
				exit
			}

			set ctg [read $fd]
			close $fd
		}

		# figure out what ssg port to use
		# don't bother checking if the megactrl address is unicast
		# or not; the SSAC_Client code will do that when you call
		# "open"

		set ssg_port $startport_
		incr startport_ 2

		# if $megactrl is empty, SSAC_Client will use the
		# default address in the catalog file
		set have_rover_ 1

		# set the "data_multicast" flag to 1 in SSAC_Client if we are
		# using MeGa
		set usemega [$self get_option useMegaSession]
		if { $usemega=={} } { set data_multicast 0 } \
				else {set data_multicast 1 }
		set ssacClient_ [new SSAC_Client $megactrl $ssg_port \
				$data_multicast]
		set media [$ssacClient_ open $ctg 0]

		# add the name of the SSAC server as that for the MeGa service
		# as well
		set ab [new AddressBlock [$ssacClient_ server_name]]
		$self add_option myMegaCtrl [$ab addr]/[$ab sport]/[$ab ttl]
		delete $ab

		$self do_rover_media $media
		$ssacClient_ attach_observer $self
	}
}


CollaboratorApp private do_rover_media { media } {
	$self instvar vspec_ aspec_ mbspec_ mbv2spec_ have_video_ have_audio_ \
			have_mb_ have_mbv2_

	set have_video_ 0
	set have_audio_ 0
	set have_mb_    0
	set have_mbv2_  0
	set vspec_     {}
	set aspec_     {}
	set mbspec_    {}
	set mbv2spec_  {}

	#YYY: need to modify rover to work with mbv2
	foreach m $media {
		switch $m {
			video {
				set vspec_ {}
				set have_video_ 1
			}
			audio {
				set aspec_ {}
				set have_audio_ 1
			}
			mediaboard {
				set mbspec_ {}
				$self add_option mbSessionSpec {}
				set have_mb_ 1
			}
		}
	}
}


CollaboratorApp private setup_mega { } {
	$self instvar have_video_ have_audio_ have_mb_ have_rover_ useMega_ \
			ssacClient_ startport_

	set usemega [$self get_option useMegaSession]
	set megactrl [$self get_option myMegaCtrl]

	if { $usemega == {} } {
		set useMega_ 0
		return
	}

	set useMega_ 1

	# add the following options for all the media types
	set maxsbw [$self get_option maxCollaboratorSessionBW]
	if { $maxsbw=={} } { set maxsbw 1000000 }
	$self add_option maxAudioSessionBW $maxsbw
	$self add_option maxVideoSessionBW $maxsbw
	$self add_option maxMbSessionBW    $maxsbw

	# add the following options only if we don't have rover
	if !$have_rover_ {
		$self add_option megaAudioSession $usemega
		$self add_option megaVideoSession $usemega
		$self add_option megaMbSession $usemega
	}

	# first check whether we are trying to use unicast or multicast
	set megaCtrl [split [$self get_option myMegaCtrl] /]
	set addr [lindex $megaCtrl 0]
	if ![regexp {^[0-9.]+$} $addr] {
		# this looks like a hostname
		set addr [gethostbyname $addr]
	}
	if [in_multicast $addr] {
		set unicast 0
		$self add_option megaRecvVideoPort 0
		$self add_option megaRecvAudioPort 0
		$self add_option megaRecvMbPort    0

		set megaCtrl [$self get_option myMegaCtrl]
		$self add_option megaAudioCtrl $megaCtrl
		$self add_option megaVideoCtrl $megaCtrl
		$self add_option megaMbCtrl    $megaCtrl
	} else {
		set unicast 1

		set port [lindex $megaCtrl 1]
		if { $port == {} } {
			set port [$self get_option megaforPort]
		}

		if $have_video_ {
			$self add_option megaRecvVideoPort "$startport_:[expr \
					$startport_ + 2]"
			$self add_option megaVideoCtrl \
					"$addr/$port:[expr $startport_+2]/1"
			incr startport_ 4
		}
		if $have_audio_ {
			$self add_option megaRecvAudioPort "$startport_:[expr \
					$startport_ + 2]"
			$self add_option megaAudioCtrl \
					"$addr/$port:[expr $startport_+2]/1"
			incr startport_ 4
		}
		if $have_mb_ {
			$self add_option megaRecvMbPort "$startport_:[expr \
					$startport_ + 2]"
			$self add_option megaMbCtrl \
					"$addr/$port:[expr $startport_+2]/1"
			incr startport_ 4
		}
	}

	# FIXME: if we are using MeGa, we *have* to use scuba
	if { [$self get_option useScuba]=={} } {
		$self add_option useScuba 1
		$self add_option localScubaScope 1
	}
}


#
# The commandline arguments that can be used for this application.
#
CollaboratorApp private init_args { o } {
	# collaborator-specific options...
	$o register_option -video videoSessionSpec
	$o register_boolean_option -video videoSessionSpec 224.2.55.22/8000/31
	$o register_option -v videoSessionSpec
	$o register_boolean_option -v videoSessionSpec 224.2.55.22/8000/31
	$o register_option -mb mbSessionSpec
	$o register_boolean_option -mb mbSessionSpec 224.2.55.66/8000/31
	$o register_option -audio audioSessionSpec
	$o register_boolean_option -audio audioSessionSpec 224.2.55.44/8000/31
	$o register_option -a audioSessionSpec
	$o register_boolean_option -a audioSessionSpec 224.2.55.44/8000/31
	$o register_option -mbv2 mbv2SessionSpec
	$o register_boolean_option -mbv2 mbv2SessionSpec 224.2.55.77/8000/31

	# name and cname options
	$o register_option -cnameprefix cnamePrefix
	$o register_option -myname rtpName

	# MPlug will want to pass this app an sdp announcement
	# in the form of a stream of data
	$o register_option -sdp sdpAnnouncement
	# A Helper will want to pass this app an sdp announcement
	# in a file
	$o register_option -sdpfile sdpAnnouncementFile
	# use this option to turn on transmit as soon as we start up
	$o register_boolean_option -xmit xmitVideoOnStartup

	# from vic...
	$o register_option -C conferenceName
	$o register_option -c dither
	$o register_option -D device
	$o register_option -af audioFormat
	$o register_option -vf videoFormat
	$o register_option -F maxfps
	$o register_option -I confBusChannel
	$o register_option -GI globalConfBusChannel
	$o register_option -K sessionKey
	$o register_option -M colorFile
	$o register_option -m mtu
	$o register_option -o outfile
	$o register_option -q jpegQfactor
	$o register_option -t defaultTTL
	$o register_option -T softJPEGthresh
	$o register_option -U stampInterval
	$o register_option -V visual
	$o register_option -N rtpName
	$o register_list_option -map rtpMap

	$o register_boolean_option -H useHardwareDecode
	$o register_boolean_option -P privateColormap

	# MeGa resources
	$o register_option -aofmt megaAudioFormat
	$o register_option -vofmt megaVideoFormat
	$o register_option -mbofmt megaMbFormat
	$o register_option -mbv2ofmt megaMbv2Format

	$o register_option -startport megaStartPort
	$o register_option -usemega useMegaSession
	$o register_boolean_option -usemega useMegaSession [pid]@[localaddr]
	$o register_option -megactrl myMegaCtrl
	$o register_option -unicast  myMegaCtrl
	$o register_option -mars     myMegaCtrl
	###$o register_option -arport megaRecvAudioPort
	###$o register_option -vrport megaRecvVideoPort
	###$o register_option -mbrport megaRecvMbPort

	###$o register_option -ausemega megaAudioSession
	###$o register_option -vusemega megaVideoSession
	###$o register_option -mbusemega megaMbSession

	###$o register_option -amegactrl megaAudioCtrl
	###$o register_option -vmegactrl megaVideoCtrl
	###$o register_option -mbmegactrl megaMbCtrl

	###$o register_option -asspec audioSessionSpec
	###$o register_option -vsspec videoSessionSpec
	###$o register_option -mbsspec mbSessionSpec

	$o register_option -maxsbw maxCollaboratorSessionBW
	# from vic...
	$o register_option -B maxCollaboratorSessionBWkbps

	$o register_option -asbw audioSessionBW
	$o register_option -vsbw videoSessionBW
	$o register_option -mbsbw mbSessionBW

	$o register_option -asloc audioServiceLocation
	$o register_option -vsloc videoServiceLocation
	$o register_option -mbsloc mbServiceLocation

	$o register_boolean_option -scuba useScuba

	# For MeGa use only.
	$o register_boolean_option -localscuba localScubaScope

	# cues option
	$o register_boolean_option -useCues useCues

	# from vat...
	$o register_boolean_option -r compat
	$o register_option -confid confid
	$o register_option -loopback loopback
	$o register_list_option -map rtpMap

	# from mb...
	$o register_option -u    uid
	$o register_option -dbg  debug
	$o register_option -ui   showUI
	$o register_option -drop drop
	$o register_option -play playScript
	$o register_option -rt   rtPlay
	$o register_option -rec  record
	$o register_option -tr   trace
	$o register_option -dp   delayParams
	$o register_option -follow followActive
	$o register_boolean_option -recvonly recvOnly
	$o register_option -geometry geometry

	# rover options
	$o register_option -ctg sessionCatalog
	$o register_option -ctgfile sessionCatalogFile
}


CollaboratorApp private init_mega { o } {
	# used by mega
	$o add_default megaStartPort 10004
	$o add_default megaforPort   60000
	#$o add_default myMegaCtrl    224.4.5.24/50000/31
}


#
# Add default font configurations to the options database.
#
CollaboratorApp private init_fonts { o } {
	new FontInitializer $o

	# from VicApplication...
	option add *Font [$o get_option helv12b] startupFile

	# from VatApplication...
	option add *Radiobutton.font [$o get_option helv12b] 100

	# override the default fonts like mb does...
        $o add_option smallfont [$o get_option helv10]
        $o add_option noAudioFont [$o get_option helv10o]
        $o add_option medfont [$o get_option helv12]
        $o add_option helpFont [$o get_option times12]
}


CollaboratorApp private init_tk_options { } {
	#
	# don't put tearoffs in pull-down menus
	#
	option add *tearOff 0

	option add *Radiobutton.relief flat startupFile
	option add *Checkbutton.anchor w startupFile
	option add *Radiobutton.anchor w startupFile

	if { [winfo depth .] == 1 } {
		# make mono look better
		option add *selectBackground black startupFile
		option add *selectForeground white startupFile
		option add *activeForeground black startupFile
	}
}


#
# Initialize the configuration data base with some of the default
# options assumed by vic.  These can be overridden.
#
CollaboratorApp private init_vic_resources { o } {
        global tcl_platform

	# used by TopLevelWindow and for VicUI's window-title
	$o add_default iconPrefix Collaborator:

	# used when creating new CoordinationBus; get and set by AddressBlock
	$o add_default defaultTTL 31

	# is used by init_confbus
	$o add_default confBusChannel 2
	$o add_default globalConfBusChannel 3

	# used by ControlMenu; get and possibly set by AddressBlock
	$o add_default videoFormat h.261

	# used by VideoPipeline; set by ControlMenu
	$o add_default useJPEGforH261 false

	# used by NetworkMananger; only apps that set this are vic and
	# collaborator. Maybe it should be add_option
	$o add_default useLayersWindow 1

	# used by c++ code
	$o add_default jvColors 32
	$o add_default softJPEGthresh -1
	$o add_default softJPEGcthresh 6

	# used by VicApplication and ControlMenu
	# (scuba) default session bandwidth (kb/s).
	$o add_default maxVideoSessionBW 1000000

        # used by the Info window and ControlMenu
        if {$tcl_platform(platform) != "windows"} {
	        option add Vic.background gray85 startupFile
	        option add Vic.foreground black startupFile
	        option add Vic.disabledForeground gray40 startupFile
	        $o add_default background gray85
                $o add_default foreground black
                $o add_default disabledForeground gray40
        } else {
		set b [button .b____dummy____$self]
		$o add_default background [$b cget -background]
                $o add_default foreground [$b cget -foreground]
                $o add_default disabledForeground [$b cget -disabledforeground]
		destroy $b
	}
}

#
# Initialize the configuration data base with some of the default
# options assumed by vat.  These can be overridden.
#
CollaboratorApp private init_vat_resources { o } {
	# used when creating new CoordinationBus; get and set by AddressBlock
	$o add_default defaultTTL 31

	# used by c++ code
	$o add_default afDevice -1
	$o add_default afBlocks 2
	$o add_default afSoftOuputGain 0
	$o add_default afSoftInputGain 0
	$o add_default audioFileName /dev/audio

	# used by AudioControlMenu; set by AddressBlock
	$o add_default audioFormat PCM2

	# is used by init_confbus
	$o add_default confBusChannel 2
	$o add_default globalConfBusChannel 3
}

#
# Initialize the configuration data base with some of the default
# options assumed by mb.  These can be overridden.
#
CollaboratorApp private init_mb_resources { o } {
        global env
        $self instvar class_

        option add *Radiobutton.relief flat startupFile

        # option add Mb.geometry 700x500 startupFile

	# used by Object
        $o add_default debug 1

        $o add_default playScript none
        $o add_default rtPlay 0
        $o add_default record 0
	$o add_default showUI 1

        $o add_default trace none

	# default added by MBNet, used by AddressBlock
	$o add_default defaultTTL 31
}


CollaboratorApp private init_mbv2_resources { o } {
        option add *Radiobutton.relief flat startupFile
}


#
# Initialize the configuration data base with some of the default
# options specific to collaborator.  These can be overridden.
#
CollaboratorApp private init_collaborator_resources { o } {
        $o add_default transmitOnStartup 0
	option add Collaborator.geometry 1000x680 startupFile
}


#
# Create a global bus for communication between tools
# on different machines, and a local bus for tools on
# the same machine.
#
CollaboratorApp private init_confbus {} {
	#set channel [$self get_option confBusChannel]
	#$self instvar cb_
	#if { $channel != "" && $channel != 0 } {
	#	set cb_ [new CoordinationBus -channel $channel]
	#} else {
	#	set cb_ ""
	#}

        $self instvar local_chan_ glob_chan_
	set lc [$self get_option confBufChannel]
	set gc [$self get_option globalConfBusChannel]

        if [catch {set local_chan_ [new CoordinationBus -channel $lc]}] {
		set local_chan_ ""
	}

	if [$self yesno useCues] {
		# FIXME: cues doesn't work yet!
		set ttl [$self get_option defaultTTL]
		if [catch {set glob_chan_ [new CoordinationBus -channel $gc \
				-ttl $ttl]}] {
			set glob_chan_ ""
		}
	} else {
		set glob_chan_ ""
	}
}


#
# Checks for extraneous arguments.
#
CollaboratorApp private check_argv { argv } {
	if { [llength $argv] > 0 } {
		set extra [lindex $argv 0]
		$self fatal "extra arguments (starting with $extra)"
	}
}



#
# Initializes have_<media>_ instvars.
# For now, at least one of -v <videoSessionSpec> -a <audioSessionSpec> and -mb <mediaboardSessionSpec> are expected.
# The app will be exited if none are provided.
#
CollaboratorApp private check_hostspecs { } {
	$self instvar have_video_ have_audio_ have_mb_ have_mbv2_
	$self instvar vspec_ aspec_ mbspec_ mbv2spec_

	set usemega [$self get_option useMegaSession]
	if { $usemega=={} } {
		set vspec_    [$self get_option videoSessionSpec]
		set aspec_    [$self get_option audioSessionSpec]
		set mbspec_   [$self get_option mbSessionSpec]
		set mbv2spec_ [$self get_option mbv2SessionSpec]
	} else {
		set vspec_    {}
		set aspec_    {}
		set mbspec_   {}
		set mbv2spec_ {}
	}

	set have_video_ [ expr { [$self get_option videoSessionSpec] != "" || \
			[$self get_option megaVideoSession] != "" } ]
	set have_audio_ [ expr { [$self get_option audioSessionSpec] != "" || \
			[$self get_option megaAudioSession] != "" } ]
	set have_mb_    [ expr { [$self get_option mbSessionSpec]    != "" || \
			[$self get_option megaMbSession]    != "" } ]
	set have_mbv2_  [ expr { [$self get_option mbv2SessionSpec]  != "" } ]

	if { $have_mbv2_ && [info commands srm_create_session] == "" } {
		# we don't have mbv2 compiled in the MASH kernel
		$self warn "cannot use mbv2; option isn't compiled into the\
				MASH kernel"
		set have_mbv2_ 0
		set mbv2spec_ {}
		$self add_option mbv2SessionSpec ""
	}

	if { $have_mbv2_ && $usemega!={} } {
		# YYY
		$self warn "cannot use mega with mbv2;\
				will not join the mbv2 session"
		if { !$have_video_ && !$have_audio_ && !$have_mb_ } {
			$self fatal "no other media specified; exiting..."
		}
		set have_mbv2_ 0
		set mbv2spec_ {}
	}
	if { $have_mb_ && $have_mbv2_ } {
		$self warn "can't have mb and mbv2 in the same session"
		$self warn "will join only the mbv2 session"
		set have_mb_ 0
		set mbspec_  {}
	}

	if { !$have_video_ && !$have_audio_ && !$have_mb_ && !$have_mbv2_ } {
		$self add_option videoSessionSpec 224.2.55.22/8000
		$self add_option audioSessionSpec 224.2.55.44/8000
		#$self add_option mbSessionSpec   224.2.55.66/8000
		#$self add_option mbv2SessionSpec 224.2.55.77/8000
		$self check_hostspecs
	}
}


#
# Create the user interface for this application.
# Creates a frame using the provided <i>widgetPath</i> and pack the mui ui in it.
#
CollaboratorApp public build_ui { path } {
	$self instvar videoAgent_ audioAgent_ vpipe_ scuba_sess_ ssacClient_
        $self instvar local_chan_ glob_chan_ mgr_ sender_ sdp_

	$self instvar have_video_ have_audio_ have_mb_ have_mbv2_ have_rover_ \
			useMega_

	set ui [new CollaboratorUI $useMega_ "$self exit"]
	$ui set_local_option have_video $have_video_
	$ui set_local_option have_audio $have_audio_
	$ui set_local_option have_mb    $have_mb_
	$ui set_local_option have_mbv2  $have_mbv2_
	$ui set_local_option have_rover $have_rover_

	if [array exists sdp_] {
		$ui set_local_option sdp [array get sdp_]
	}

	if { $local_chan_!={} } {
		$ui set_local_option can_voiceswitch 1
		$ui set_local_option local_channel $local_chan_
	} else {
		$ui set_local_option can_voiceswitch 0
	}

	if $have_video_ {
		$ui set_local_option videoAgent $videoAgent_
		$ui set_local_option videoPipe $vpipe_
		if [info exists scuba_sess_] {
			$ui set_local_option scubaSession $scuba_sess_
		}
	}
	if $have_audio_ {
		$ui set_local_option audioAgent $audioAgent_
	}
	if $have_mb_ {
		$ui set_local_option mbMgr $mgr_
		$ui set_local_option mbSender $sender_
	}
	if $have_mbv2_ {
		#MMM
	}
	if $have_rover_ {
		$ui set_local_option ssacClient $ssacClient_
	}

	frame $path -class CollaboratorUI
	global mash
	if { $mash(environ) == "mplug" } { pack propagate $path 0 }
	$ui build_widget $path
	pack $path -fill both -expand 1

	#if { [$self get_option useScuba] != "" } {
	#	$ui set scuba_sess_ $scuba_sess_
	#}

	return $ui
}


CollaboratorApp public exit { } {
	$self instvar have_video_ have_audio_ videoAgent_ audioAgent_

	if $have_video_ {
		$videoAgent_ shutdown
	}
	if $have_audio_ {
		$audioAgent_ shutdown
	}

	exit 0
}



# (from MBApp)...
CollaboratorApp private init_mb_variables {} {
        global mb

        $self mbui_init_default
        catch {unset mb}

	$self instvar sender_
        if {[$self get_option record]==1} {
                set sender_ [new Tcl_Recorder MB_Sender]
        } else {
                set sender_ [new MB_Sender]
        }

	$self instvar mgr_
	set mgr_ [new MB_Manager]
	set pageMgr [new MBPageMgr $mgr_]
	$mgr_ attach_page_manager $pageMgr
	$mgr_ attach_sender $sender_
}

# (from MBApp)...
CollaboratorApp private mbui_init_default { } {
	# using 19 as the priority level, since it's one less than
	# widgetDefault, which is the level used by all WidgetClass objects

	option add *Button*HighlightThickness 1 19
	option add *Button*BorderWidth 1 19

	option add *Scrollbar*HighlightThickness 1 19
	option add *Scrollbar*BorderWidth 1 19

	# FIXME: a hack to shrink the scrollbar size
	set scrollbar [scrollbar .___scrollbar___]
	set scrollbarWidth [$scrollbar cget -width]
	set scrollbarWidth [expr {($scrollbarWidth * 2)/3}]
	destroy $scrollbar
	option add *Scrollbar*Width $scrollbarWidth 19

	option add *Entry*HighlightThickness 1 19
	option add *Entry*BorderWidth 2 19
	option add *Entry*Background White 19
	option add *Menubutton*HighlightThickness 1 19
	option add *Menubutton*BorderWidth 1 19

	option add *Mbcanvas*Canvas*width  300
	option add *Mbcanvas*Canvas*height 200

	$self add_option canvWidth 300
	$self add_option canvHeight 200
}


#
# Reset the appropriate objects depending on which media the
# AddressBlock, <i>ab</i>, represents.
#
CollaboratorApp public reset { ab type agent } {
	if { $type == "rtp" } {
		set cl [$agent info class]
		if { $cl=="VideoAgent" || [$cl info heritage VideoAgent]!={}} {
			[$agent video_handler] reset $ab
			set media video
		} else {
			set media audio
		}
	} elseif { $type == "srm" } {
		$self instvar mbAgent_ mbui_ mgr_ session_
		set nm [$mbAgent_ set network_]
		$nm loopback 1

		# do this for the first time only
		$self instvar have_network_
		if {![info exists have_network_] || $have_network_ == 0} {
			$session_ start_timers
			[$self set sender_] attach $mgr_
			set have_network_ 1
		}

		$self instvar ui_
		set mbui [$ui_ get_local_option mbUI]
		# don't reset here! simply activate
		# the default reset method writes something to the window title
		# which we don't want!
		$mbui activate
		set media mb
	} elseif { $type == "srmv2" } {
		$self instvar ui_ mbv2_session_
		set mbv2ui [$ui_ get_local_option mbv2UI]
		$mbv2ui session $mbv2_session_
		set media mbv2
	}

	$self instvar ui_
	if [info exists ui_] { $ui_ reset $media }

	if { $media == "audio" } {
		$self add_option isAudioAgentReset 1
	}
}


#
# Parses the sdp message into mui application options, which are then parsed
# as if they had been provided when CollaboratorApp was instantiated.
# Returns any remaining unprocessed arguments.
# If there is no -sdpfile or -sdp from which to gather an sdp announcement,
# nothing is returned.  The app will then rely on -v -mb -a etc. options.
# Warning: checks are not performed to ensure that -sdp, -sdpfile, or
# {-v, -a, -mb} args are not simultaneously provided to MuiApplication.
# Currently they will just clobber each other.
#
CollaboratorApp private parse_sdp_msg { } {
	# get the sdp announcement
	set sdp_anncmt_file [$self get_option sdpAnnouncementFile]
	if { $sdp_anncmt_file != "" } {
		# open file and read out sdp announcement
		if [catch {open $sdp_anncmt_file r} fileId] {
			puts stderr "Cannot open $sdp_anncmt_file: $fileId"
		} else {
			append sdp_announcement ""
			while {[gets $fileId line] >= 0} {
				append sdp_announcement "$line\n"
			}
			close $fileId
		}
	} else {
		set sdp_announcement [$self get_option sdpAnnouncement]
	}

	set sdp_msg ""
	# get the 1st sdp message out of the sdp announcement
	if { $sdp_announcement != "" } {
		# parse the announcement into a list of sdp messages
		set parser [new SDPParser 1]
		set sdp_msgs [$parser parse $sdp_announcement]
		if { [llength $sdp_msgs] != 1 } {
			puts stderr "Invalid SDP announcement:\
					\n$sdp_announcement"
			exit 0
		} else {
			# get the 1st sdp message
			set sdp_msg [lindex $sdp_msgs 0]
		}
		delete $parser
	}

	# translate the sdp message into appropriate CollaboratorApp options
	# and process these options
	# (modelled after the manner in which nsdr converts an
	#  sdp announcement into options to pass to the mui application)
	if { $sdp_msg != "" } {
		set prog [new Program $sdp_msg]
		set user_app [new UserApplication/Collaborator]
		set options [$user_app generate_options $prog]
		append argv $options
		delete $user_app
		delete $prog

		$self instvar sdp_
		set sdp_(session_name) [$sdp_msg get session_name_]
		set sdp_(session_info) [$sdp_msg get session_info_]
		set sdp_(creator) [$sdp_msg get creator_]
		set sdp_(url) [$sdp_msg get uri_]

		# delete the SDP message objects
		foreach m $sdp_msgs { delete $m }

		# here, options like -v -a -mb etc. should be added to
		# options database
		# and extraneous options are returned
		return [[$self options] parse_args $argv]
	} else {
		return
	}
}


CollaboratorApp public ssac_reset { media spec } {
	$self instvar useMega_ ssac_reset_ videoAgent_ audioAgent_ mbnet_ ui_\
			mbAgent_

	set cm [$ui_ get_local_option controlmenu]
	set ab [new AddressBlock $spec]
	set spec [$ab addr]/[$ab sport]:[$ab rport]/[$ab ttl]
	delete $ab
	if ![info exists ssac_reset_] {
		set ssac_reset_ 1

		$ui_ heard_from mars
		set usemega [$self get_option useMegaSession]
		$self add_option megaAudioSession $usemega
		$self add_option megaVideoSession $usemega
		$self add_option megaMbSession $usemega
	}

	switch $media {
		video {
			if $useMega_ {
				# reset the MeGa ALM
				$self add_option videoSessionSpec $spec
				$videoAgent_ reset_mega
				$cm original_session_spec video $spec
			} else {
				$videoAgent_ reset_spec $spec
			}
		}

		audio {
			if $useMega_ {
				# reset the MeGa ALM
				$self add_option audioSessionSpec $spec
				$audioAgent_ reset_mega
				$cm original_session_spec audio $spec
			} else {
				$audioAgent_ reset_spec $spec
			}
		}

		mediaboard {
			if $useMega_ {
				# reset the MeGa ALM
				$self add_option mbSessionSpec $spec
				$mbnet_ reset_mega
				$cm original_session_spec mb $spec
			} else {
				$mbAgent_ reset_spec $spec
			}
		}
	}
}


CollaboratorApp public activate { start end } {
	$self instvar ui_ slider_activated_
	if ![info exists slider_activated_] {
		set slider_activated_ 1
		$ui_ activate_slider $start $end
	}
}


CollaboratorApp public play_update { state } {
	$self instvar ui_
	$ui_ play_update $state
}


CollaboratorApp public update_title { info1 info2 duration } {
	$self instvar ui_
	$ui_ update_title $info1 $info2 $duration
}


CollaboratorApp public set_offset { offset } {
	$self instvar ui_
	$ui_ set_rover_offset $offset
}




#
# Hooks to make MuiApplication embeddable in a web browser.
#
Class CollaboratorApp/MPlug -superclass CollaboratorApp

#
# This class can be used to startup regular collaborator in scrollable MPlug
# window.
#
CollaboratorApp/MPlug public init { argv } {
	$self instvar mplug_

	$self start_mplug

	# Start up collaborator embedded in the browser.
	#set root [$mplug_ root]
	set root ""
	$self next $root.[pid] $argv
}

#
# Starts MPlug using a scrollable window.
#
CollaboratorApp/MPlug public start_mplug {} {
	$self instvar mplug_

	if { ![info exists mplug_] } {
		# use MPlug instead of MPlug/ScrolledUI
		# since that'll make Collaborator use all of the
		# available screen real estate (-fill both -expand 1)!
		set mplug_ [new MPlug]
	}
}

#
CollaboratorApp/MPlug private do_error { msg } {
	$self instvar mplug_
	label [$mplug_ root].label -text $msg -justify left -anchor w
	pack [$mplug_ root].label  -fill both -expand 1 -padx 10 -pady 10
	vwait forever
}

Class CollaboratorApp/MPlug/SDPMime -superclass CollaboratorApp/MPlug

#
# This class can be used to startup collaborator in scrollable MPlug window
# when the browser is passing an sdp announcement as a stream to MPlug.
# (i.e. "application/x-sd" mimetype)
#
CollaboratorApp/MPlug/SDPMime public init { argv } {
	$self instvar mplug_

	$self start_mplug

	# on success, the following function returns NPRES_DONE which is
	# defined to 0 in a .h file
	# FIXME: how do I pull in constants defined in .h files into tcl?
	if { [$mplug_ wait_for_stream] } {
		puts stderr "Could not retrieve stream"
		exit
	}

	if { [$mplug_ get mimetype] != "application/x-sd" } {
		$self do_error "collaborator cannot handle mimetype:\
				[$mplug_ mimetype]"
	}

	# the stream is expected to be an sdp announcement
	append argv " -sdp {[$mplug_ get stream]} "
	$self next $argv
}


CollaboratorApp proc.public create_pathfinder_app { argv megactrl megafor } {
	set o [$self options]
	$o load_preferences pfinder-local

	if { [$self get_option pfl-nomulticast]=="1" } {
		lappend argv -unicast $megafor
	}

	if { [$self get_option pfl-usemega]=="1" } {
		lappend argv -usemega

		if { [$self get_option pfl-nomulticast]!="1" && \
				$megactrl!={} } {
			lappend argv -megactrl $megactrl
		}

		set maxsbw [$self get_option pfl-maxsbw]
		set srate [$self get_option pfl-srate]
		if { $maxsbw!={} } {
			if { $srate == {} } { set srate [expr $maxsbw/2] }
			if { $srate > $maxsbw } { set srate $maxsbw }
			lappend argv -maxsbw $maxsbw -asbw $srate -vsbw $srate\
					-mbsbw $srate
		}
	}

	global mash
	if { $mash(environ)=="mplug" } {
		new CollaboratorApp/MPlug $argv
	} else {
		pack propagate . false
		new CollaboratorApp .top $argv
	}
}

