/*
 * mb-nethost.h --
 *
 *      Stores all the net2host and host2net overloaded functions
 *
 * Copyright (c) 1996-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * $Header: /usr/mash/src/repository/mash/mash-1/mb/mb-nethost.h,v 1.6 2002/02/03 03:16:30 lim Exp $
 */

#ifndef MB_NETHOST_H
#define MB_NETHOST_H

#include "mb.h"
#include "mb-pkt.h"

/* this determines whether we need to byte swap 64 bit values */
/* initialized in mb-mgr.cc */
extern int cMustByteSwapQuad;

inline unsigned long net2host(unsigned long ul) {
    return ntohl(ul);
}

inline unsigned short net2host(unsigned short us) {
    return ntohs(us);
}

inline long net2host(long l) {
    return (long)net2host((unsigned long)l);
}

inline int net2host(int i) {
    return (int)net2host((unsigned long)i);
}

inline u_int32_t net2host(u_int32_t i) {
    return (int)net2host((unsigned long)i);
}

inline float net2host(float f) {
    unsigned long ul = net2host(*(unsigned long*)&f);
    return *((float*) &ul);
}

inline short net2host(short s) {
    return (short)net2host((unsigned short)s);
}

inline char net2host(char c) {
    return c;
}

inline char net2host(u_char c) {
    return c;
}

/* make ABCDEFGH -> HGFEDCBA */
inline void net2host(const double& dblNet, double& dblHost)  {
    assert(sizeof(double)==sizeof(ulong)*2);
    if (cMustByteSwapQuad) {
        *(((ulong*)&dblHost)+1) = net2host(*(ulong*)(&dblNet));
        *((ulong*)&dblHost) = net2host(*(((ulong*)&dblNet)+1));
    } else {
        *(((ulong*)&dblHost)+1) = net2host(*(((ulong*)&dblNet)+1));
        *((ulong*)&dblHost) = net2host(*(ulong*)(&dblNet));
    }
}

inline void net2host(const SrcId& sidNet, SrcId& sidHost) {
    // NOTE: we store ss_addr as network byte order all the time
    //       no need to change to host order
    sidHost.ss_addr = sidNet.ss_addr;
    sidHost.ss_uid = net2host(sidNet.ss_uid);
}

inline void net2host(const PageId& pidNet, PageId& pidHost) {
    net2host(pidNet.sid, pidHost.sid);
    pidHost.uid=net2host(pidNet.uid);
}

inline void net2host(const MBColor& netC, XColor *&pColor) {
    if (netC.a) {               // transparent
        pColor = NULL;
        return;
    }
    XColor tmpC;
    tmpC.red = net2host(netC.r);
    tmpC.green = net2host(netC.g);
    tmpC.blue = net2host(netC.b);
    tmpC.flags = DoRed | DoGreen | DoBlue;
    pColor = MB_GetColorByValue(&tmpC);
    assert(pColor && "color not found!");
}

/* ----------- */

inline unsigned long host2net(unsigned long ul) {
    return htonl(ul);
}

inline unsigned short host2net(unsigned short us) {
    return htons(us);
}

inline long host2net(long l) {
    return (long)host2net((unsigned long)l);
}

inline int host2net(int i) {
    assert(sizeof(int)==sizeof(unsigned long));
    return (int)host2net((unsigned long)i);
}

inline u_char host2net(u_char c) {
    return c;
}
inline char host2net(char c) {
    return c;
}

inline u_int host2net(u_int i) {
    return (u_int)host2net((unsigned long)i);
}

inline short host2net(short s) {
    return (short)host2net((unsigned short)s);
}

inline float host2net(float f) {
    unsigned long ul = host2net(*((unsigned long*)(&f)));
    return *((float*)&ul);
}

/* make ABCDEFGH -> HGFEDCBA */
inline void host2net(const double& dblHost, double& dblNet)  {
    if (cMustByteSwapQuad) {
        *(((ulong*)&dblNet)+1) = host2net(*(ulong*)(&dblHost));
        *((ulong*)&dblNet) = host2net(*(((ulong*)&dblHost)+1));
    } else {
        *(((ulong*)&dblNet)+1) = host2net(*(((ulong*)&dblHost)+1));
        *((ulong*)&dblNet) = host2net(*(ulong*)(&dblHost));
    }
}

inline void host2net(const SrcId& sidHost, SrcId& sidNet) {
    // NOTE: we store ss_addr as network byte order all the time
    //       no need to change to host order
    sidNet.ss_addr = sidHost.ss_addr;
    sidNet.ss_uid = host2net(sidHost.ss_uid);
}

inline void host2net(const PageId& pidHost, PageId& pidNet) {
    host2net(pidHost.sid, pidNet.sid);
    pidNet.uid=host2net(pidHost.uid);
}

inline void host2net(XColor *pColor, MBColor& netC) {
    // we ignore the values of flags since tk does not seem to be using them
    if (pColor) {
        netC.r = host2net(pColor->red);
        netC.g = host2net(pColor->green);
        netC.b = host2net(pColor->blue);
        netC.a = 0;
    }
    else {
        netC.r = netC.g = netC.b = 0;
        netC.a = 1;             // transparent
    }
}

#endif /* #ifdef MB_NETHOST_H */
