/*
 * mb-basercvr.cc --
 *
 *      FIXME: This file needs a description here.
 *
 * Copyright (c) 1996-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * @(#) $Header: /usr/mash/src/repository/mash/mash-1/mb/mb-basercvr.cc,v 1.29 2002/02/03 03:16:30 lim Exp $
 */

#include "mb/mb-obj.h"
#include "mb/mb-cmd.h"
#include "mb/mb-nethost.h"


MBBaseRcvr::MBBaseRcvr(MBBaseMgr* pMgr, const SrcId& sid)
	: SRM_PacketHandler(0), pMgr_(pMgr), sid_(sid), phtPages_(NULL)
{
	phtPages_ = new Tcl_HashTable;
	if (!phtPages_) {
		SignalError(("Out of Memory!"));
		abort();
	}
	// the following must be true
	// otherwise we would use garbage as part of the keys
	assert((sizeof(PageId)/sizeof(int))*sizeof(int) == sizeof(PageId));
	Tcl_InitHashTable(phtPages_, sizeof(PageId)/sizeof(int));
}


/* virtual */
MBBaseRcvr::~MBBaseRcvr()
{
	Tcl_HashSearch search;
	Tcl_HashEntry *pEntry = Tcl_FirstHashEntry(phtPages_, &search);
	while (pEntry) {
		MBPageObject *pPage =
			(MBPageObject*)Tcl_GetHashValue(pEntry);
		delete pPage;
		pEntry=Tcl_NextHashEntry(&search);
	}
	Tcl_DeleteHashTable(phtPages_);
	delete phtPages_;
}


void
MBBaseRcvr::HandleSA(Pkt_PgStatus* aPgStatus, int numPgs)
{
	Bool schedSA = FALSE;
	// if this is an empty Session announcement, check whether we have
	// any pages, if so send the update SA
	if (numPgs==0) {
		Tcl_HashSearch search;
		Tcl_HashEntry *pEntry = Tcl_FirstHashEntry(phtPages_,
							   &search);

		if (pEntry) {
			MTrace(trcMB|trcVerbose,("couldn't file any page for"
					      "source %u@%s",
					      sid_.ss_uid,
					      intoa(sid_.ss_addr)));
			PageId pid = ((MBPageObject *)
				Tcl_GetHashValue(pEntry))->getId();
			fprintf(stderr, "scheduling a remote SA for %u@%s "
				"(have page %u@%s:%lu)\n",
				sid_.ss_uid, intoa(sid_.ss_addr),
				pid.sid.ss_uid,intoa(pid.sid.ss_addr),pid.uid);

			schedSA = TRUE;
		}
	}
	// note: when numPgs is zero the following loop is skipped
	for (Pkt_PgStatus* pPgStatus= aPgStatus;
	     pPgStatus < aPgStatus+numPgs;
	     pPgStatus++) {
		PageId pgid;
		net2host(pPgStatus->pageid, pgid);
		MBPageObject* pPage = DefinePage(pgid);
		/* FIXME: in win32 we could be in the process of
		 * creating a new page and thus have null value for a
		 * while, change the check into an assert if  */
		if (!pPage) {
			continue;
		}
		ulong maxSn=net2host(pPgStatus->maxSn);

		MTrace(trcMB|trcExcessive, ("pPage_maxseqno: %d, maxSn: %d",
					    pPage->getMaxSeqno(), maxSn));
		if (pPage->Update(getSrcId(), maxSn)) {
			// the page has more data than the source; we should
			// schedule an SA
			schedSA = TRUE;
		}
	}
	if (schedSA) {
		MTrace(trcMB|trcVerbose,("Requesting for SA"));
		pMgr_->RequestSA(this);
	}
}


int
MBBaseRcvr::FillSA(Byte *pb, int len)
{
	MTrace(trcMB|trcExcessive,
	       ("Trying to fill SA for %u@%s",
		getSrcId().ss_uid, intoa(getSrcId().ss_addr)));
	Pkt_SA* pPktSA= (Pkt_SA*) pb;
	host2net(getSrcId(), pPktSA->srcid);
	int index=0;
	Tcl_HashSearch search;
	Tcl_HashEntry *pEntry = Tcl_FirstHashEntry(phtPages_, &search);
	while (pEntry &&
	       int(sizeof(Pkt_SA)+index*sizeof(Pkt_PgStatus)) < len) {
		MBPageObject *pPage = (MBPageObject *)
			Tcl_GetHashValue(pEntry);
		if (pPage) pPage->FillStatus(pPktSA->aPgStatus[index++]);
		pEntry=Tcl_NextHashEntry(&search);
	}
	if (pEntry) perr(("Too many pages, cannot send all SAs in 1 pkt!"));
	pPktSA->numPgs = host2net(index);
	return (sizeof(Pkt_SA)+index*sizeof(Pkt_PgStatus));
}


MBPageObject *
MBBaseRcvr::DefinePage(const PageId &pageId, Bool *newFlag /*=NULL*/)
{
	MTrace(trcMB|trcExcessive,
	       ("Defining Page: (%u@%s):%lx for source %u@%s",
		pageId.sid.ss_uid, intoa(pageId.sid.ss_addr), pageId.uid,
		sid_.ss_uid, intoa(sid_.ss_addr)));
	int created;
	Tcl_HashEntry *pNewEntry =
		Tcl_CreateHashEntry(phtPages_, (char*)&pageId, &created);

	if (newFlag!=NULL) *newFlag = ((created) ? TRUE : FALSE);
	if (created) {
		MBPageObject *pPage = pMgr_->DefinePage(pageId);
		Tcl_SetHashValue(pNewEntry, (ClientData) pPage);
		return pPage;
	} else {
		return (MBPageObject*)Tcl_GetHashValue(pNewEntry);
	}
}


void
MBBaseRcvr::HandleRequest(const SrcId& sidRqtSrc, Pkt_request *pPkt)
{
#ifdef MB_DEBUG
	// sanity check
	SrcId sid;
	net2host(pPkt->pr_sid, sid);
	assert(getSrcId()==sid);
#endif // MB_DEBUG
	PageId pid;
	net2host(pPkt->pr_page, pid);
	ulong snStart = net2host(pPkt->pr_sseq);
	ulong snEnd = net2host(pPkt->pr_eseq);
	MBPageObject* pPage;
	pPage = DefinePage(pid);
	pPage->HandleRequest(sidRqtSrc, getSrcId(), snStart, snEnd);

	char* szPgId = PgId2Str(pid);
	SrcId ssid;
	net2host(pPkt->pr_sid, ssid);
	MBLOG(("r rqt %lx@%s %s %ld %ld fr %s",
	       ssid.ss_uid, intoa(ssid.ss_addr), szPgId,
	       snStart, snEnd, intoa(sidRqtSrc.ss_addr)));
	delete[] szPgId;
}


//
// there is a repair reply for data from this source
// - dispatch request to the page referred to
//
void
MBBaseRcvr::HandleReply(Byte* pb, u_int len)
{
	recvCommands(pb, len, TRUE);
}



// choose either REORDER or LOST
//#define REORDER
//#define LOST

#define TCL_REORDER

#if defined(REORDER) || defined(LOST)
#define COUNT 5
#define RESTART 15
static Byte* ppByte[COUNT+1];
static int ppLen[COUNT+1];
static int count=-1;
static int skip=0,deleted=0;
#endif

/* virtual from SRM_PacketHandler */
void
MBBaseRcvr::recv(Byte *pb, u_int len)
{
	MTrace(trcMB|trcExcessive, ("#### inside MBBaseRcvr::recv"));
#ifdef TCL_REORDER
	Tcl& tcl=Tcl::instance();
	Tcl_Obj* pDebugObj = pMgr_->debugObjName();
	SRM_Source* pSrc = getSrc();
	if (!pSrc) {
		fprintf(stderr,
			"problem: MBBaseRcvr::recv cannot find source");
	}
	if (pSrc && pDebugObj) {
		Tcl_Obj* objv[3];
		objv[0]=pDebugObj;
		objv[1]=Tcl_NewStringObj("recv", -1);
		objv[2]=Tcl_NewStringObj((char*)(pSrc->name()), -1);
		tcl.evalObjs(3, objv);
		// drop message if cont is true
		if (strcmp(tcl.result(),"cont")) {
			MTrace(trcAll,("Dropping pkt of %d\n",len));
			return;
		}
	}
	MTrace(trcMB|trcVerbose,("Receiving pkt of len %d",len));
#endif // TCL_RECORDER

#ifdef REORDER
	if (count==-1) count++;
	else if (!skip) {
		if (count<=COUNT) {
			ppLen[count]=len;
			ppByte[count] = new Byte[len];
			deleted = FALSE;
			memcpy(ppByte[count],pb,len);
			count++;
			return;
		}
		count++;
		if (count>=RESTART && (count-RESTART<=COUNT)) {
			skip=TRUE;
			MTrace(trcAll,("Sending out delayed cmd #%lu",count-RESTART));
			int next = COUNT - (count-RESTART);
			assert(next>=0 && next < COUNT+1);
			MBBaseRcvr::recv(ppByte[next],ppLen[next]);
			skip=FALSE;
		} else if (count > RESTART+COUNT+2) {
			if (!deleted) {
				for(int i=0; i<COUNT+1; i++) delete ppByte[i];
				deleted=TRUE;
			}
			count = 0;
		}
	}
#endif // REORDER

#ifdef LOST
#ifdef REORDER
#error "chooce one of LOST and REORDER"
#endif // REORDER
	if (count<COUNT) {
		count++;                // drop
		return;
	}
	count++;
	if (count > RESTART) count=0;
#endif // LOST

	Pkt_DataHdr* pDataHdr = (Pkt_DataHdr*) pb;
	const u_int cHdrLen = sizeof(Pkt_DataHdr);
	if (MB_PKT_VER < net2host(pDataHdr->ph_version))
		return;                 // may not understand any new data, do nothing
	// possibly corrupted packet
	if (len < cHdrLen) {
		Trace(VERYVERBOSE,("err: ignoring pkt with len(%d) smaller than header "
				   "size(%d)!", len, cHdrLen));
		return;
	}

	recvCommands(pb+cHdrLen, len-cHdrLen, FALSE);
}


/* REVIEW: should pass in rpySrc for replies, so that suppressReplies
   will work correctly */
void
MBBaseRcvr::recvCommands(Byte *pb, u_int len, Bool isReply)
{
	if (len < sizeof(Pkt_PageHdr)) {
		SignalError(("corrupted packet: size too small: len=%d", len));
		return;
	}

	Pkt_PageHdr* pPgHdr=(Pkt_PageHdr*) pb;
	PageId pgid;
	net2host(pPgHdr->pd_page, pgid);
	MBPageObject *pPage = DefinePage(pgid);
	// assume that the packet is valid and switch to that page if needed

	Byte* pbCurr = (Byte*) (pPgHdr+1);
	ulong snStart = net2host(pPgHdr->pd_sseq);
	ulong snEnd = net2host(pPgHdr->pd_eseq);

	if (snStart>snEnd) {
		MTrace(trcAll,
		       ("Packet error: starting sn (%ld) > ending sn (%ld)!",
			snStart, snEnd));
		return;
	}

	char* szPgId = PgId2Str(pgid);
	MBLOG(("r %s %s %d %d", isReply ? "rpy" : "dta",
	       szPgId, snStart, snEnd));
	delete[] szPgId;

	if (isReply) {       // Suppress any pending replies
		pPage->SuppressReplies(getSrcId(), snStart, snEnd);
	}

	for (ulong sn=snStart; sn<=snEnd; sn++) {
		MBCmd* pCmd=MBCmd::Create(sn, pbCurr);
		if (!pCmd) {
			// FIXME: should treat as an error packet and
			// ask for resend?
			SignalError(("corrupted packet: invalid command"));
			return;
		}
		/* don't notify the ui if the packet is a reply (?) */
		if (pPage->HasCmd(pCmd->getSeqno())) {
			char *szStr = ::PgId2Str(pPage->getId());
			MTrace(trcMB|trcVerbose,
			       ("Received duplicate packet #%lu for page %s",
				pCmd->getSeqno(), szStr));
			delete [] szStr;
			break;            // treat as okay
		}
		if (pCmd->Incomplete(pPage)) {
			pPage->AddCmd(pCmd);
			pPage->Defer(pCmd);
			break;
		}
		if (MB_EXE_OK != handleCmd(pCmd, pPage, cMBTimeNegInf,
			       pPage->targetTime())) {
			SignalError(("recv'd corrupted or invalid command"));
			break;
		}
		// REVIEW: verify possible bus erros?
		pbCurr += net2host(((Pkt_CmdHdr*)pbCurr)->dh_len);
		if (pbCurr > pb + len) {
			SignalError(("incorrect pkt: lengths do not match"));
			return;
		}
	}
	assert(pbCurr <= pb + len); // make sure we did not overrun

	// check if we can execute more commands
	pPage->FlushDeferred(this);

	// send out any repair requests pending
	pPage->UpdateRequest(getSrcId(), snStart, snEnd);
}

//
// MBBaseRcvr::handleCmd --
//       this command is called when a command is first received, or
//       when a command has to be reapplied when the target
//       display time changed.
//
// 	 Return FALSE if there is a problem with the command
//
//       If notify is 1, we will notify the UI of the activity
//
// NOTE: since command execution is not idempotent (yet), calling
//       handleCmd multiple times during an execution round could have
//       undefined results.

/* virtual */
int
MBBaseRcvr::handleCmd(MBCmd *pCmd, MBPageObject *pPage,
		      const MBTime& oldTime, const MBTime& newTime)
{
	Bool retCode;
	retCode = executeCmd(pCmd, pPage, oldTime, newTime);

	if (retCode == MB_EXE_OK) {
		MTrace(trcMB|trcExcessive,
		       ("#### 3:Adding cmd %d(%p) to page %p",
			pCmd->getSeqno(), pCmd, pPage));
		pPage->AddCmd(pCmd, 1); /* overwrite prev cmd, if any */
		pMgr_->activity(pPage, pCmd);
	}
	return retCode;
}

// fills the next ADU into pktbuf
int MBBaseRcvr::NextADU(Byte *pb, u_int len, int& nextSize,
                        MBPageObject *pCurrPage)
{
	MTrace(trcMB|trcExcessive,
	       ("Invoking MBBaseRcvr::NextADU for rcvr %d@%s",
		sid_.ss_uid, intoa(sid_.ss_addr)));

	u_int cHdrLen = sizeof(Pkt_DataHdr);
	Byte* pbCurr= pb + cHdrLen;
	if (len < cHdrLen) return 0;
	u_int lenCurr = len - cHdrLen;

	// for now, we will ask each page to fill in data, since the
	// structure only allow commands from one page to be grouped into
	// a packet.
	int outlen=0;
	if (pCurrPage)
		outlen=pCurrPage->NextADU(pbCurr, lenCurr, nextSize);

	if (outlen==0 && nextSize != 0) {
		/* this means the len is too small for the next packet */
		return 0;
	}
	// if no data from current page, ask the rest of the pages to
	// fill in data
	if (!outlen) {
		// REVIEW: the order of asking depends on the hash table by
		// right it should be according to the most recently accessed
		Tcl_HashSearch search;
		Tcl_HashEntry *pEntry = Tcl_FirstHashEntry(getPagesHT(),
							   &search);
		while (!outlen && pEntry) {
			Page *pPage = (Page *)Tcl_GetHashValue(pEntry);
			if (pPage!=pCurrPage)
				outlen = pPage->NextADU(pbCurr, lenCurr,
							nextSize);
			pEntry=Tcl_NextHashEntry(&search);
		}
	}

	if (outlen) {
		Pkt_DataHdr* pDataHdr=(Pkt_DataHdr*) pb;
		pDataHdr->ph_version = host2net((u_short)MB_PKT_VER);
		pDataHdr->ph_type = 0;  // not used
		pDataHdr->ph_flags = 0; // not used
		if (nextSize > 0) nextSize+=cHdrLen;
		return outlen + cHdrLen;
	}
	// no packets
	assert(outlen == 0);
	return 0;
}

SRM_Source* MBBaseRcvr::getSrc() const
{
	SRM_Source* pSrc = pMgr_->getSrc(getSrcId());
	return (pSrc);
}

#ifdef MB_DEBUG
void
MBBaseRcvr::Dump(int dumptype, Tcl_Obj* pObj) {
	Tcl_HashSearch search;
	Tcl_HashEntry *pEntry = Tcl_FirstHashEntry(phtPages_, &search);
	while (pEntry) {
		Page *pPage = (Page *)Tcl_GetHashValue(pEntry);
		pPage->Dump(dumptype, pObj);
		pEntry=Tcl_NextHashEntry(&search);
	}
}
#endif // MB_DEBUG

