/*
 * transcoder-jpeg.cc --
 *
 *      Motion Jpeg Transcoder
 *
 * Copyright (c) 1993-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

static const char rcsid[] =
    "@(#) $Header: /usr/mash/src/repository/mash/mash-1/codec/video/transcoder-jpeg.cc,v 1.12 2002/02/03 03:13:15 lim Exp $";

#include <string.h>
#include "inet.h"
#include "transcoder.h"
#include "jpeg.h"
#include "rtp.h"

#ifdef INFOPAD
#include "ipadchan.h"
#endif /* INFOPAD */

class JpegTranscoder : public VideoTranscoder {
public:
	JpegTranscoder();
	~JpegTranscoder();
protected:
	virtual void configure();
	void process_hdr(const jpeghdr*);
	u_char* reassemble(const rtphdr* rh, const u_char* bp, int& len);

	int decimate_;
	int ndec_;

	/*
	 * Reassembly buffer.  This should be a dma buffer in the
	 * jvdriver case but it's not clear if jvdriver allows buffers
	 * to be shared across sockets. FIXME ask Lance
	 * If it does, then we can pass dma buffers betwee
	 * the decoder and renderers.
	 */
#define JPEG_SLOTS 64
#define JPEG_SLOTMASK (JPEG_SLOTS - 1)
	struct slot {
		int seqno;
		int eof;	/* nbytes in last pkt of frame, o.w. 0 */
		u_int32_t off;
		u_int32_t ts;
	} slots_[JPEG_SLOTS];

	/*
	 * Reassembly buffers.  We do double-buffering, which allows
	 * packets to arrive out of order across frame boundaries,
	 * but not across an entire frame (i.e., we don't want to see
	 * any packets from frame k+2 until we're done with frame k).
	 * We use RTP timestamps to keep track of which frame is
	 * allocated to which buffer.  (RTP guarantees that the
	 * timestamps are constant across a frame, and increase
	 * between succesive frames.  FIXME is latter really true?)
	 */
	struct rbuf {
		int drop;
		u_int32_t ts;
		u_char* bp;
	};
	rbuf rb0_;
	rbuf rb1_;
	int rbsize_;

	JpegDecoder* decoder_;

	int type_;
	int inq_;

	JpegDecoder::config config_;
};

class JpegPixelTranscoder : public JpegTranscoder {
public:
	virtual void configure();
	virtual void recv_data(pktbuf* pb);
};

class JpegDCTTranscoder : public JpegTranscoder {
public:
	virtual void configure();
	virtual void recv_data(pktbuf* pb);
};

static class JpegDCTTranscoderClass : public TclClass {
public:
	JpegDCTTranscoderClass() : TclClass("Transcoder/JPEG/DCT") {}
	TclObject* create(int, const char*const*) {
		return(new JpegDCTTranscoder);
	}
} jpeg_dct_transcoder_class;

static class JpegPixelTranscoderClass : public TclClass {
public:
	JpegPixelTranscoderClass() : TclClass("Transcoder/JPEG/Pixel") {}
	TclObject* create(int, const char*const*) {
		return(new JpegPixelTranscoder);
	}
} jpeg_pixel_transcoder_class;

/*
 * Initial size of each reassembly buffer.
 */
#define JPEG_BUFSIZE (16*1024)

JpegTranscoder::JpegTranscoder()
	       :VideoTranscoder(sizeof(jpeghdr)), decimate_(0), ndec_(0),
		decoder_(0), type_(-1), inq_(-1)
{
	rbsize_ = JPEG_BUFSIZE;
	rb0_.bp = new u_char[2 * JPEG_BUFSIZE];
	rb0_.ts = ~0;
	rb0_.drop = 0;
	rb1_.bp = &rb0_.bp[JPEG_BUFSIZE];
	rb1_.ts = ~0;
	rb1_.drop = 0;
	memset(slots_, 0, sizeof(slots_));

	JpegDecoder::defaults(config_);
}

void JpegTranscoder::configure()
{
	config_.comp[0].hsf = 2;
	config_.comp[0].vsf = (decimation_ == 411) ? 2 : 1;
	config_.comp[1].hsf = 1;
	config_.comp[1].vsf = 1;
	config_.comp[2].hsf = 1;
	config_.comp[2].vsf = 1;
	config_.width = inw_;
	config_.height = inh_;

	JpegDecoder::quantizer(config_, inq_);

}

void JpegDCTTranscoder::configure()
{
	JpegTranscoder::configure();
	crinit(outw_, outh_);

	delete decoder_;
	decoder_ = JpegDCTDecoder::create(config_, outw_, outh_);
	decoder_->thresh(90);
}

void JpegPixelTranscoder::configure()
{
	JpegTranscoder::configure();
	pixel_crinit(outw_, outh_);

	delete decoder_;
	decoder_ = JpegPixelDecoder::create(config_, outw_, outh_);
	decoder_->thresh(6);
}

JpegTranscoder::~JpegTranscoder()
{
	delete decoder_;
	delete rb0_.bp;
}

/*
 * Reassemble an RTP/JPEG stream.  Return a pointer to a buffer
 * each time we encounter an entire frame.  Otherwise, return 0.
 * Set len to the length of the jpeg data in the buffer.
 */
u_char* JpegTranscoder::reassemble(const rtphdr* rh, const u_char* bp, int& len)
{
	jpeghdr* p = (jpeghdr*)(rh + 1);
	int off = (int)ntohl(p->off);
	int cc = len;

	if (off + cc > rbsize_) {
		/*
		 * Grow reassembly buffers.
		 */
		int nsize = rbsize_;
		do {
			nsize <<= 1;
		} while (off + cc > nsize);
		u_char* p = new u_char[2 * nsize];
		memcpy(p, rb0_.bp, rbsize_);
		memcpy(p + nsize, rb1_.bp, rbsize_);
		delete rb0_.bp;
		rb0_.bp = p;
		rb1_.bp = p + nsize;
		rbsize_ = nsize;
	}
	/*
	 * Initialize the slot data structure.
	 */
	int seqno = ntohs(rh->rh_seqno);
	int s = seqno & JPEG_SLOTMASK;
	u_int32_t ts = ntohl(rh->rh_ts);
	slots_[s].seqno = seqno;
	slots_[s].off = off;
	slots_[s].ts = ts;
	/*
	 * Figure out which reassembly-buffer to use.  If we're not
	 * already reassembling this frame, take over the older buffer.
	 */
	rbuf* rb;
	if (ts == rb0_.ts)
		rb = &rb0_;
	else if (ts == rb1_.ts)
		rb = &rb1_;
	else {
		rb = ((int)(rb0_.ts - rb1_.ts) < 0) ? &rb0_ : &rb1_;
		rb->ts = ts;
		rb->drop = 0;
		/*
		 * If we're decimating frames (to save cycles),
		 * remember that we might want to drop the rest
		 * of the packets from this frame.
		 */
		if (decimate_) {
			if (--ndec_ <= 0)
				ndec_ = decimate_;
			else
				rb->drop = 1;
		}
	}
	if (rb->drop)
		return (0);

	memcpy((char*)&rb->bp[off], (char*)bp, cc);

	/*
	 * Check if we're at end-of-frame.  If not, see if we're
	 * filling a hole.  If not, return.  Otherwise, drop out
	 * below and check for an entire frame.  We set cc to be
	 * the entire frame size in the if-else below.
	 */
	if ((ntohs(rh->rh_flags) & RTP_M) != 0) {
		slots_[s].eof = cc;
		cc += off;
	} else {
		slots_[s].eof = 0;
		int ns = s;
		do {
			ns = (ns + 1) & JPEG_SLOTMASK;
			if (slots_[ns].ts != ts || ns == s)
				return (0);
		} while (slots_[ns].eof != 0);
		cc = int(slots_[ns].eof + slots_[ns].off);
	}
	/*
	 * At this point, we know we have an end-of-frame, and
	 * all packets from slot 's' up until the end-of-frame.
	 * Scan backward from slot 's' making sure we have all
	 * packets from the start-of-frame (off == 0) to 's'.
	 */
	int ps = s;
	do {
		ps = (ps - 1) & JPEG_SLOTMASK;
		if (slots_[ps].ts != ts || ps == s)
			return (0);
	} while (slots_[ps].off != 0);

	len = cc;
	return (rb->bp);
}

void JpegTranscoder::process_hdr(const jpeghdr* p)
{
	int reconfig = 0;

	if (p->type != type_) {
		type_ = p->type;
		decimation_ = (type_ == 1) ? 411 : 422;
		reconfig = 1;
	}

	int q = p->q;
	if (q != inq_) {
		JpegDecoder::quantizer(config_, q);
		inq_ = q;
		reconfig = 1;
	}

	int inw = p->width << 3;
	int inh = p->height << 3;
	if (inw_ !=  inw || inh_ != inh) {
		inw_ = inw;
		inh_ = inh;
		Tcl& tcl = Tcl::instance();
		tcl.evalf("%s frame_width %d", encoder_->name(), inw);
		outw_ = atoi(tcl.result());
		tcl.evalf("%s frame_height %d", encoder_->name(), inh);
		outh_ = atoi(tcl.result());
		reconfig = 1;
	}

	if (reconfig)
		configure();
}

void JpegDCTTranscoder::recv_data(pktbuf *pb)
{
	rtphdr* rh = (rtphdr*)pb->dp;
	int cc = pb->len - sizeof(rtphdr);
	const u_char* bp = (const u_char*)(rh + 1);

	jpeghdr* p = (jpeghdr*)(rh + 1);
	process_hdr(p);

	bp += sizeof(jpeghdr);
	cc -= sizeof(jpeghdr);

	bp = reassemble(rh, bp, cc);
	pb->release();
	if (bp == 0) {
		have_frame_ = 0;
		return;
	}

	have_frame_ = 1;
	double now;
	if (!txonly_) {
		if (bps_ == 0)
			return;
		now = gettimeofday();
	}

	if (fc_ <= now || txonly_) {
		/* If we have fallen behind (>200ms), re-sync. */
		if (now - fc_ > 200000.)
			fc_ = now;
		// XXX not necessarily FULL_FRAME, should read
		// from rtp jpeg header
		decoder_->decode(bp, cc, JPEG_FULL_FRAME, crvec_, mark_);
		DCTFrame df(ntohl(rh->rh_ts),
			    ((JpegDCTDecoder*)decoder_)->frame(), crvec_,
			      outw_, outh_);
		int nb = encoder_->nb();
		encoder_->recv(&df);
		obytes_ = encoder_->nb();
		nb = obytes_ - nb;
		double bits = 8 * nb;
		lastfc_ = fc_;
		fc_ += 1e6 * bits / bps_;
		ofrms_++;

		mark_ = age_blocks() | CR_MOTION_BIT | CR_LQ;
	}
}

void JpegPixelTranscoder::recv_data(pktbuf *pb)
{
	rtphdr* rh = (rtphdr*)pb->dp;
	int cc = pb->len - sizeof(rtphdr);
	const u_char* bp = (const u_char*)(rh + 1);

	jpeghdr* p = (jpeghdr*)(rh + 1);
	process_hdr(p);

	bp += sizeof(jpeghdr);
	cc -= sizeof(jpeghdr);

	bp = reassemble(rh, bp, cc);
	if (bp == 0) {
		pb->release();
		have_frame_ = 0;
		return;
	}

	have_frame_ = 1;
	double now;
	if (!txonly_) {
		if (bps_ == 0) {
			pb->release();
			return;
		}
		now = gettimeofday();
	}

	if (fc_ <= now || txonly_) {
		/* If we have fallen behind (>200ms), re-sync. */
		if (now - fc_ > 200000.)
			fc_ = now;
		
		// XXX not necessarily FULL_FRAME, should read
		// from rtp jpeg header
		decoder_->decode(bp, cc, JPEG_FULL_FRAME, crvec_, mark_);
		blk_to_mb_cr();
		YuvFrame f(ntohl(rh->rh_ts),
			   ((JpegPixelDecoder*)decoder_)->frame(), mb_crvec_,
			      outw_, outh_);
		int nb = encoder_->nb();
		encoder_->recv(&f);
		obytes_ = encoder_->nb();
		nb = obytes_ - nb;
		double bits = 8 * nb;
		lastfc_ = fc_;
		fc_ += 1e6 * bits / bps_;
		ofrms_++;

		mark_ = age_blocks() | CR_MOTION_BIT | CR_LQ;
	}
	pb->release();
}

#ifdef INFOPAD
void JpegIpadVQTranscoder::configure(const u_char* p, Network* n)
{
	JpegTranscoder::configure(p, n);

	IpadChannel* c = (IpadChannel*)n;
	encoder_ = new IpadVQEncoder(c, outw_, outh_, decimation_, 0);
	make_decoder();
}

void JpegIpadVQTranscoder::resize(int w, int h)
{
	outw_ = inw_ = w;
	outh_ = inh_ = h;
	/* FIXME */
	u_int bps = encoder_->bps();
	delete encoder_;
	encoder_ = new IpadVQEncoder((IpadChannel*)net_, outw_, outh_,
				     decimation_, 0);
	encoder_->bps(bps);
}

void JpegIpadVQTranscoder::make_decoder(void)
{
	delete decoder_;
	crinit(outw_, outh_);
/*FIXME*/
#ifdef notdef
	Tcl& tcl = Tcl::instance();
	int q = atoi(tcl.attr("softJPEGthresh"));
	if (q < 0)
		q = JpegDecoder::q_to_thresh(inq_);
	decoder_->thresh(q);
	decoder_->cthresh(atoi(tcl.attr("softJPEGcthresh")));
#endif
}

int JpegIpadVQTranscoder::recv(const rtphdr* rh, const u_char* bp, int cc, int)
{
	jpeghdr* p = (jpeghdr*)(rh + 1);

	int reconfig = process_hdr(p);
	if (reconfig)
		reconfigure();

	bp += sizeof(jpeghdr);
	cc -= sizeof(jpeghdr);

	bp = reassemble(rh, bp, cc);
	if (bp == 0)
		return (0);

	if (bps() == 0)
		return (1);

	double now = gettimeofday();
	if (fc_ <= now) {
		/* If we have fallen behind (>200ms), re-sync. */
		if (now - fc_ > 200000.)
			fc_ = now;
		JpegPixelDecoder* d = (JpegPixelDecoder*)decoder_;
		d->decode(bp, cc, crvec_, CR_SEND|CR_MOTION);
		int cc = encoder()->encode(d->frame(), outw_, outh_, crvec_);
		age_blocks();
		double bits = 8 * double(cc);
		obits_ += int(bits);
		ofrms_ += 1;
		opkts_ = encoder()->np();
		lastfc_ = fc_;
		fc_ += 1e6 * bits / double(bps());
	}
	return (1);
}
#endif /* INFOPAD */
