# Copyright (c) 2004 Divmod.
# See LICENSE for details.

import urlparse

from nevow import url
from nevow.testutil import TestCase, FakeRequest
from nevow.flat import flatten

theurl = "http://www.foo.com:80/a/nice/path/?zot=23&zut"

class FR(FakeRequest):
    def prePathURL(Self):
        return theurl


class TestURL(TestCase):
    def test_fromString(self):
        urlpath = url.URL.fromString(theurl)
        self.assertEquals(theurl, str(urlpath))

    def test_fromRequest(self):
        urlpath = url.URL.fromRequest(FR())
        self.assertEquals(theurl, str(urlpath))

    def test_equality(self):
        urlpath = url.URL.fromString(theurl)
        self.failUnlessEqual(urlpath, url.URL.fromString(theurl))
        self.failIfEqual(urlpath, url.URL.fromString('ftp://www.anotherinvaliddomain.com/foo/bar/baz/?zot=21&zut'))

    def test_parent(self):
        urlpath = url.URL.fromString(theurl)
        self.assertEquals("http://www.foo.com:80/a/nice/?zot=23&zut",
                          str(urlpath.parent()))
        urlpath = url.URL.fromString('http://www.foo.com/a')
        self.assertEquals("http://www.foo.com/",
                          str(urlpath.parent()))
        urlpath = url.URL.fromString('http://www.foo.com/a/')
        self.assertEquals("http://www.foo.com/",
                          str(urlpath.parent()))
        urlpath = url.URL.fromString('http://www.foo.com/a/b')
        self.assertEquals("http://www.foo.com/",
                          str(urlpath.parent()))
        urlpath = url.URL.fromString('http://www.foo.com/a/b/')
        self.assertEquals("http://www.foo.com/a/",
                          str(urlpath.parent()))
        urlpath = url.URL.fromString('http://www.foo.com/a/b/c')
        self.assertEquals("http://www.foo.com/a/",
                          str(urlpath.parent()))
        urlpath = url.URL.fromString('http://www.foo.com/a/b/c/')
        self.assertEquals("http://www.foo.com/a/b/",
                          str(urlpath.parent()))
        urlpath = url.URL.fromString('http://www.foo.com/a/b/c/d')
        self.assertEquals("http://www.foo.com/a/b/",
                          str(urlpath.parent()))
        urlpath = url.URL.fromString('http://www.foo.com/a/b/c/d/')
        self.assertEquals("http://www.foo.com/a/b/c/",
                          str(urlpath.parent()))

    def test_parent_root(self):
        urlpath = url.URL.fromString('http://www.foo.com/')
        self.assertEquals("http://www.foo.com/",
                          str(urlpath.parent()))
        self.assertEquals("http://www.foo.com/",
                          str(urlpath.parent().parent()))

    def test_child(self):
        urlpath = url.URL.fromString(theurl)
        self.assertEquals("http://www.foo.com:80/a/nice/path/gong?zot=23&zut",
                          str(urlpath.child('gong')))
        self.assertEquals("http://www.foo.com:80/a/nice/path/gong/?zot=23&zut",
                          str(urlpath.child('gong/')))
        self.assertEquals(
            "http://www.foo.com:80/a/nice/path/gong/double?zot=23&zut",
            str(urlpath.child('gong/double')))
        self.assertEquals(
            "http://www.foo.com:80/a/nice/path/gong/double/?zot=23&zut",
            str(urlpath.child('gong/double/')))

    def test_child_init_tuple(self):
        self.assertEquals(
            "http://www.foo.com/a/b/c",
            str(url.URL(netloc="www.foo.com",
                        pathsegs=['a', 'b']).child("c")))

    def test_child_init_root(self):
        self.assertEquals(
            "http://www.foo.com/c",
            str(url.URL(netloc="www.foo.com").child("c")))

    def test_sibling(self):
        urlpath = url.URL.fromString(theurl)
        self.assertEquals(
            "http://www.foo.com:80/a/nice/path/sister?zot=23&zut",
            str(urlpath.sibling('sister')))
        # use an url without trailing '/' to check child removal
        theurl2 = "http://www.foo.com:80/a/nice/path?zot=23&zut"
        urlpath = url.URL.fromString(theurl2)
        self.assertEquals(
            "http://www.foo.com:80/a/nice/sister?zot=23&zut",
            str(urlpath.sibling('sister')))

    def test_curdir(self):
        urlpath = url.URL.fromString(theurl)
        self.assertEquals(theurl, str(urlpath))
        # use an url without trailing '/' to check object removal
        theurl2 = "http://www.foo.com:80/a/nice/path?zot=23&zut"
        urlpath = url.URL.fromString(theurl2)
        self.assertEquals("http://www.foo.com:80/a/nice/?zot=23&zut",
                          str(urlpath.curdir()))
        
    def test_click(self):
        urlpath = url.URL.fromString(theurl)
        # a null uri should be valid (return here)
        self.assertEquals("http://www.foo.com:80/a/nice/path/?zot=23&zut",
                          str(urlpath.click("")))
        # a simple relative path remove the query
        self.assertEquals("http://www.foo.com:80/a/nice/path/click",
                          str(urlpath.click("click")))
        # an absolute path replace path and query
        self.assertEquals("http://www.foo.com:80/click",
                          str(urlpath.click("/click")))
        # replace just the query
        self.assertEquals("http://www.foo.com:80/a/nice/path/?burp",
                          str(urlpath.click("?burp")))
        # one full url to another should not generate '//' between netloc and pathsegs 
        self.failIfIn("//foobar", str(urlpath.click('http://www.foo.com:80/foobar')))

        # from a url with no query clicking a url with a query,
        # the query should be handled properly
        u = url.URL.fromString('http://www.foo.com:80/me/noquery')
        self.failUnlessEqual('http://www.foo.com:80/me/17?spam=158',
                             str(u.click('/me/17?spam=158')))
        
        
    def test_add(self):
        urlpath = url.URL.fromString(theurl)
        self.assertEquals(
            "http://www.foo.com:80/a/nice/path/?zot=23&zut&burp",
            str(urlpath.add("burp")))
        self.assertEquals(
            "http://www.foo.com:80/a/nice/path/?zot=23&zut&burp=xxx",
            str(urlpath.add("burp", "xxx")))
        self.assertEquals(
            "http://www.foo.com:80/a/nice/path/?zot=23&zut&burp=xxx&zing",
            str(urlpath.add("burp", "xxx").add("zing")))
        # note the inversion!
        self.assertEquals(
            "http://www.foo.com:80/a/nice/path/?zot=23&zut&zing&burp=xxx",
            str(urlpath.add("zing").add("burp", "xxx")))
        # note the two values for the same name
        self.assertEquals(
            "http://www.foo.com:80/a/nice/path/?zot=23&zut&burp=xxx&zot=32",
            str(urlpath.add("burp", "xxx").add("zot", 32)))

    def test_add_noquery(self):
        # fromString is a different code path, test them both
        self.assertEquals(
            "http://www.foo.com:80/a/nice/path/?foo=bar",
            str(url.URL.fromString("http://www.foo.com:80/a/nice/path/")
                .add("foo", "bar")))
        self.assertEquals(
            "http://www.foo.com/?foo=bar",
            str(url.URL(netloc="www.foo.com").add("foo", "bar")))

    def test_replace(self):
        urlpath = url.URL.fromString(theurl)    
        self.assertEquals(
            "http://www.foo.com:80/a/nice/path/?zot=32&zut",
            str(urlpath.replace("zot", 32)))
        # replace name without value with name/value and vice-versa
        self.assertEquals(
            "http://www.foo.com:80/a/nice/path/?zot&zut=itworked",
            str(urlpath.replace("zot").replace("zut", "itworked")))
        # Q: what happens when the query has two values and we replace?
        # A: we replace both values with a single one
        self.assertEquals(
            "http://www.foo.com:80/a/nice/path/?zot=32&zut",
            str(urlpath.add("zot", "xxx").replace("zot", 32)))

    def test_clear(self):
        urlpath = url.URL.fromString(theurl)    
        self.assertEquals(
            "http://www.foo.com:80/a/nice/path/?zut",
            str(urlpath.clear("zot")))
        self.assertEquals(
            "http://www.foo.com:80/a/nice/path/?zot=23",
            str(urlpath.clear("zut")))
        # something stranger, query with two values, both should get cleared
        self.assertEquals(
            "http://www.foo.com:80/a/nice/path/?zut",
            str(urlpath.add("zot", 1971).clear("zot")))
        # two ways to clear the whole query
        self.assertEquals(
            "http://www.foo.com:80/a/nice/path/",
            str(urlpath.clear("zut").clear("zot")))
        self.assertEquals(
            "http://www.foo.com:80/a/nice/path/",
            str(urlpath.clear()))

    def test_secure(self):
        self.assertEquals(str(url.URL.fromString('http://localhost/').secure()), 'https://localhost/')
        self.assertEquals(str(url.URL.fromString('http://localhost/').secure(True)), 'https://localhost/')
        self.assertEquals(str(url.URL.fromString('https://localhost/').secure()), 'https://localhost/')
        self.assertEquals(str(url.URL.fromString('https://localhost/').secure(False)), 'http://localhost/')
        self.assertEquals(str(url.URL.fromString('http://localhost/').secure(False)), 'http://localhost/')
        self.assertEquals(str(url.URL.fromString('http://localhost/foo').secure()), 'https://localhost/foo')
        self.assertEquals(str(url.URL.fromString('http://localhost/foo?bar=1').secure()), 'https://localhost/foo?bar=1')
        self.assertEquals(str(url.URL.fromString('http://localhost/').secure(port=443)), 'https://localhost/')
        self.assertEquals(str(url.URL.fromString('http://localhost:8080/').secure(port=8443)), 'https://localhost:8443/')
        self.assertEquals(str(url.URL.fromString('https://localhost:8443/').secure(False, 8080)), 'http://localhost:8080/')
        

class Serialization(TestCase):
    def testQuoting(self):
        context = None
        scheme = 'http'
        loc = 'localhost'
        path = ('baz', 'buz')
        query = [("foo", "bar"), ("baz", "=quux"), ("foobar", "?")]
        fragment = 'futz'
        u = url.URL(scheme, loc, path, query, fragment)
        s = flatten(url.URL(scheme, loc, path, query, fragment))
        
        parsedScheme, parsedLoc, parsedPath, parsedQuery, parsedFragment = urlparse.urlsplit(s)
        
        self.assertEquals(scheme, parsedScheme)
        self.assertEquals(loc, parsedLoc)
        self.assertEquals('/' + '/'.join(path), parsedPath)
        self.assertEquals(query, url.unquerify(parsedQuery))
        self.assertEquals(fragment, parsedFragment)
