# Copyright (c) 2004 Divmod.
# See LICENSE for details.


"""Standard IData introspection adapters.

Classes and functions in this module are responsible for resolving
data directives into the actual data to be rendered.

By default, Nevow knows how to I{look inside} Python's mapping (dict)
and sequence (tuple, list) types as well as call any functions and
methods found in the stan tree.
"""


from nevow import compy

from nevow.inevow import IGettable, ISettable, IContainer, IData
from nevow.stan import directive
from nevow import util

def convertToData(data, context):
    """Recursively resolve the data until either a Twisted deferred or
    something that does not implement the IGettable interface is
    found.
    """
    newdata = IGettable(data, persist=False, default=None)
    if newdata is not None:
        newdata = newdata.get(context)
        if isinstance(newdata, util.Deferred):
            return newdata.addCallback(convertToData, context)
        else:
            return convertToData(newdata, context)
    else:
        return data

class NoAccessor(NotImplementedError):
    pass


class DirectiveAccessor(compy.Adapter):
    __implements__ = IGettable

    def get(self, context):
        data = context.locate(IData)
        container = IContainer(data, None, persist=False)
        if container is None:
            raise NoAccessor, "%s does not implement IContainer, and there is no registered adapter." % data
        child = container.child(context, self.original.name)
        return child

class FunctionAccessor(compy.Adapter):
    __implements__ = IGettable
    def get(self, context):
        return self.original(context, context.locate(IData))

class DictionaryContainer(compy.Adapter):
    __implements__ = IContainer
    
    def child(self, context, name):
        return self.original[name]


def intOrNone(s):
    try:
        return int(s)
    except ValueError:
        return None


class ListContainer(compy.Adapter):
    __implements__ = IContainer

    def child(self, context, name):
        if ':' in name:
            return self.original[slice(*[intOrNone(x) for x in name.split(':')])]
        return self.original[int(name)]

