/*

Copyright (C) 2000, 2001, 2002 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#include <netdude/nd.h>
#include <netdude/nd_gettext.h>
#include <netdude/nd_packet_iterator.h>
#include <netdude/nd_packet.h>
#include <netdude/nd_protocol.h>

/* The Checksum Fix plugin. It calls the fix_packet()
 * callback for all protocols existing in the active
 * packets (i.e. all packets if select-to-all mode is
 * active, or just the selected ones, otherwise).
*/

typedef struct nd_cksum_data
{
  int          tested;
  int          modified;
  int          index;
  gboolean     changed;
} ND_CksumData;


const char *
name(void)
{
  return _("Checksum Fix Plugin");
}


const char *
description(void)
{
  return _("The Checksum Fix plugin. It fixes the headers of "
	   "all protocols in the packets the plugin is applied "
	   "to. The actual functionality depends on the way "
	   "the protocols implement it.");
}


const char *
author(void)
{
  return _("Christian Kreibich, <christian@whoop.org>");
}

const char *
version(void)
{
  return _("0.1.0");
}


static void
fix_packet_cb(ND_Packet *p, ND_ProtoData *pd, void *user_data)
{
  ND_CksumData *data = (ND_CksumData *) user_data;

  if (!p || !pd)
    return;

  /* Fix up the protocol data, depending on how the
   * protocol plugin implements that operation.
   * If the packet was modified, set a flag to let our
   * called know about it.
   */
  if (pd->inst.proto->fix_packet(p, data->index))
    data->changed = TRUE;
}

void
run(ND_Trace *trace)
{
  ND_PacketIterator  pit;
  ND_CksumData       data;
  char               message[MAXPATHLEN];

  if (!trace)
    return;

  /* Initialize our helper struct: */
  memset(&data, 0, sizeof(ND_CksumData));
  
  /* Use a packet iterator for selected packets on the given trace */
  for (nd_pit_init(&pit, trace, TRUE); nd_pit_get(&pit); nd_pit_next(&pit))
    {
      /* Update our data structure -- we maintain
       * investigated and modified packets.
       */
      data.index = nd_pit_get_index(&pit);
      data.tested++;
      data.changed = FALSE;

      /* Iterate over the protocols contained in the packet,
       * from inner- to outermost, and call our callback for
       * each protocol which will then correct any checksums.
       */
      nd_packet_foreach_proto_backward(nd_pit_get(&pit),
				       fix_packet_cb,
				       &data);

      if (data.changed)
	data.modified++;
    }

  /* Some GUI candy to let the user see what's going on */
  g_snprintf(message, MAXPATHLEN, _("%i packets checked, %i fixed."),
	     data.tested, data.modified);

  nd_gui_statusbar_set(message);
}
