#!/bin/sh

#
# Connector/ODBC-3.51
# Script to create a Windows src package
#

version=@VERSION@
export version
SOURCE=`pwd`
CP="cp -p"

DEBUG=0
SILENT=0
SUFFIX=""
OUTTAR=0

#
# This script must run from Connector/ODBC top directory
#

if [ ! -f scripts/make_win_src_distribution ]; then
  echo "ERROR : You must run this script from the Connector/ODBC top-level directory"
  exit 1
fi

#
# Debug print of the status
#

print_debug() 
{
  for statement 
  do
    if [ "$DEBUG" = "1" ] ; then
      echo $statement
    fi
  done
}

#
# Usage of the script
#

show_usage() 
{
  echo "Connector/ODBC utility script to create a Windows src package, and it takes"
  echo "the following arguments:"
  echo ""
  echo "  --debug   Debug, without creating the package"
  echo "  --tmp     Specify the temporary location"
  echo "  --silent  Do not list verbosely files processed"
  echo "  --tar     Create tar.gz package instead of .zip"
  echo "  --help    Show this help message"

  exit 0
}

#
# Parse the input arguments
#

parse_arguments() {
  for arg do
    case "$arg" in
      --debug)    DEBUG=1;;
      --tmp=*)    TMP=`echo "$arg" | sed -e "s;--tmp=;;"` ;;
      --suffix=*) SUFFIX=`echo "$arg" | sed -e "s;--suffix=;;"` ;;
      --silent)   SILENT=1 ;;
      --tar)      OUTTAR=1 ;;
      --help)     show_usage ;;
      *)
  echo "Unknown argument '$arg'"
  exit 1
      ;;
    esac
  done
}

parse_arguments "$@"

#
# Assign the tmp directory if it was set from the environment variables
#

for i in $TMP $TMPDIR $TEMPDIR $TEMP /tmp
do
  if [ "$i" ]; then 
    print_debug "Setting TMP to '$i'"
    TMP=$i 
    break
  fi
done

    
#

#
# Create a tmp dest directory to copy files
#

BASE=$TMP/MYODBC_WIN_DIST-$version

if [ -d $BASE ] ; then
  print_debug "Destination directory '$BASE' already exists, deleting it"
  rm -r -f $BASE
fi
mkdir $BASE $BASE/win32

cp $SOURCE/win32/myodbc3.dsp $BASE/win32/
cp $SOURCE/win32/myodbc3.dsw $BASE/win32/

#
# Setup Windows VC workspace files
#

(
find $BASE/ \( -name "*.dsp" -o -name "*.dsw" \) -and -not -path \*SCCS\* -print
)|(
  while read v 
  do
    print_debug "Replacing LF -> CRLF from '$v'"

    # ^M -> type CTRL V + CTRL M 
    cat $v | sed 's/
//' | sed 's/$/
/' > $v.tmp
    rm $v
    mv $v.tmp $v
  done
)

#
# Clean up if we did this from a bk tree
#

if [ -d $BASE/SCCS ]  
then
  find $BASE/ -type d -name SCCS -printf " \"%p\"" | xargs rm -r -f
fi


#
# Copy directory files
#

copy_dir_files() {
  
  for arg do
    print_debug "Copying files from directory '$arg'"
    
    if [ ! -d $BASE/$arg ]; then
      mkdir $BASE/$arg
    fi
    
    cd $SOURCE/$arg/
    
    for i in *.c *.h *.def *.rc *.rsp *.bmp \
             README* INSTALL* *.hlp *.ini 
    do 
      if [ -f $i ] 
      then
        print_debug " $i"
        $CP $SOURCE/$arg/$i $BASE/$arg/$i
      fi
    done
  done
}


#
# Input directories to be copied
#

for i in driver samples test win32
do
  copy_dir_files $i
done

#
# Create dummy configure header
#

if [ -f $BASE/include/myconf.h ]; then
  rm -f $BASE/include/myconf.h
fi
touch $BASE/driver/myconf.h


#
# Copy miscellaneous files
#
cp $SOURCE/samples/WIN-Makefile $BASE/samples/Makefile
cp $SOURCE/test/WIN-Makefile $BASE/test/Makefile
cp $SOURCE/win32/Makefile $BASE/win32/Makefile
cp $SOURCE/win32/Makefile_debug $BASE/win32/Makefile_debug

cd $SOURCE

for i in COPYING \
         ChangeLog \
         README \
         INSTALL-SOURCE \
         INSTALL-BINARY \
         INSTALL-WIN-SOURCE
do
  print_debug "Copying file '$i'"
  if [ -f $i ] 
  then
    $CP $i $BASE/$i
  fi
done

#
# Specify the distribution package name and copy it
#

NEW_FILE_NAME=connector-odbc-$version
BASE2=$TMP/$NEW_FILE_NAME
rm -r -f $BASE2
mv $BASE $BASE2
BASE=$BASE2

#
# If debugging, don't create a zip/tar/gz
#

if [ "$DEBUG" = "1" ] ; then
  echo "Please check the distribution files from $BASE"
  echo "Exiting (without creating the package).."
  exit
fi

#
# This is needed to prefere gnu tar instead of tar because tar can't
# always handle long filenames
#

PATH_DIRS=`echo $PATH | sed -e 's/^:/. /' -e 's/:$/ ./' -e 's/::/ . /g' -e 's/:/ /g' `
which_1 ()
{
  for cmd
  do
    for d in $PATH_DIRS
    do
      for file in $d/$cmd
      do
	if test -x $file -a ! -d $file
	then
	  echo $file
	  exit 0
	fi
      done
    done
  done
  exit 1
}

#
# Create the result zip/tar file
#

set_tarzip_options() 
{
  for arg 
  do
    if [ "$arg" = "tar" ]; then
      ZIPFILE1=gnutar
      ZIPFILE2=gtar
      OPT=cvf
      EXT=".tar"
      NEED_COMPRESS=1
      if [ "$SILENT" = "1" ] ; then
        OPT=cf
      fi
    else
      ZIPFILE1=zip
      ZIPFILE2=""
      OPT="-vr"
      EXT=".zip"
      NEED_COMPRESS=0
      if [ "$SILENT" = "1" ] ; then
        OPT="-r"
      fi
    fi
  done
}

if [ "$OUTTAR" = "1" ]; then
  set_tarzip_options 'tar'
else
  set_tarzip_options 'zip'
fi

tar=`which_1 $ZIPFILE1 $ZIPFILE2`
if test "$?" = "1" -o "$tar" = ""
then
  print_debug "Search failed for '$ZIPFILE1', '$ZIPFILE2', using default 'tar'"
  tar=tar
  set_tarzip_options 'tar'
fi

#
# Create the archive
#

print_debug "Using $tar to create archive"

cd $TMP

$tar $OPT $SOURCE/$NEW_FILE_NAME$EXT $NEW_FILE_NAME
cd $SOURCE

if [ "$NEED_COMPRESS" = "1" ]
then
  print_debug "Compressing archive"
  gzip -9 $NEW_FILE_NAME$EXT
  EXT="$EXT.gz"
fi

print_debug "Removing temporary directory"
rm -r -f $BASE

if [ "$SILENT" = "0" ]
then
  echo "$NEW_FILE_NAME$EXT created successfully !!"
fi

# End of script
