/* Copyright (C) 2000-2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   There are special exceptions to the terms and conditions of the GPL as it
   is applied to this software. View the full text of the exception in file
   EXCEPTIONS in the directory of this software distribution.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

#include "MYODBCSetupDataSourceDialog.h"

#include "MySQLSide.xpm"

#include <qstring.h>

MYODBCSetupDataSourceDialog::MYODBCSetupDataSourceDialog( QWidget *pwidgetParent, WORD nRequest, LPCSTR pszDriver, MYODBC_NAMEVALUE **aNameValues )
    : QTabDialog( pwidgetParent, "MYODBCSetupDataSourceDialog" )
{
    this->nRequest          = nRequest;
    this->pszDriver         = pszDriver;
    this->pszDataSourceName = NULL;
    this->aNameValues       = aNameValues;

    // do some lookups
    nDataSourceName = MYODBCSetupGetAttributeIndex( aNameValues, "DSN" );
    nDescription    = MYODBCSetupGetAttributeIndex( aNameValues, "DESCRIPTION" );
    nServer         = MYODBCSetupGetAttributeIndex( aNameValues, "SERVER" );
    nUser           = MYODBCSetupGetAttributeIndex( aNameValues, "USER" );
    nPassword       = MYODBCSetupGetAttributeIndex( aNameValues, "PASSWORD" );
    nDatabase       = MYODBCSetupGetAttributeIndex( aNameValues, "DATABASE" );  
    nPort           = MYODBCSetupGetAttributeIndex( aNameValues, "PORT" );
    nStmt           = MYODBCSetupGetAttributeIndex( aNameValues, "STMT" );
    nSocket         = MYODBCSetupGetAttributeIndex( aNameValues, "SOCKET" );
    nOption         = MYODBCSetupGetAttributeIndex( aNameValues, "OPTION" );

    if ( nDataSourceName >= 0 )
        this->pszDataSourceName = aNameValues[nDataSourceName]->pszValue;

    doInit();
}

MYODBCSetupDataSourceDialog::MYODBCSetupDataSourceDialog( QWidget *pwidgetParent, WORD nRequest, LPCSTR pszDriver, LPCSTR pszDataSourceName )
    : QTabDialog( pwidgetParent )
{
    this->nRequest          = nRequest;
    this->pszDriver         = pszDriver;
    this->pszDataSourceName = pszDataSourceName;
    this->aNameValues       = NULL;

    // do some lookups
    nDataSourceName = -1;
    nDescription    = -1;
    nServer         = -1;
    nUser           = -1;
    nPassword       = -1;
    nDatabase       = -1;
    nPort           = -1;
    nStmt           = -1;
    nSocket         = -1;
    nOption         = -1;

    doInit();
}

void MYODBCSetupDataSourceDialog::slotTest()
{
    SQLHENV     hEnv;
    SQLHDBC     hDbc;
    SQLRETURN   nReturn;
    QString     stringConnectIn;

    /* build connect string in */
#ifdef Q_WS_MACX
    /*!
        \note OSX
 
              Yet another place where iODBC fails to act like the ODBC
              specification. SQLDriverConnect() should accept the 
              driver *description* and should allow for the curly
              braces but fails on both accounts deciding instead to
              interpret the whole value as a driver file name.

              Its possible to do the xref here, for SQLDriverConnect,
              but due to time constraints the file is hard-coded.
    */ 
    stringConnectIn  = "DRIVER=/usr/lib/libmyodbc3.dylib";
#else
    stringConnectIn  = "DRIVER={" + QString( pszDriver ) + "}";
#endif
    stringConnectIn += ";UID=" + plineeditUser->text();
    stringConnectIn += ";PWD=" + plineeditPassword->text();
    stringConnectIn += ";SERVER=" + plineeditServer->text();
    if ( !pcomboboxDatabase->lineEdit()->text().isEmpty() )
        stringConnectIn += ";DATABASE=" + pcomboboxDatabase->lineEdit()->text();
    if ( !plineeditPort->text().isEmpty() )
        stringConnectIn += ";PORT=" + plineeditPort->text();
    if ( !plineeditSocket->text().isEmpty() )
        stringConnectIn += ";SOCKET=" + plineeditSocket->text();
    if ( !plineeditOptions->text().isEmpty() )
        stringConnectIn += ";OPTION=" + plineeditOptions->text();
    if ( !plineeditStmt->text().isEmpty() )
        stringConnectIn += ";STMT=" + plineeditStmt->text();

    nReturn = SQLAllocHandle( SQL_HANDLE_ENV, NULL, &hEnv );
    if ( nReturn != SQL_SUCCESS ) 
        doShowDiagnostics( nReturn, SQL_HANDLE_ENV, NULL );
    if ( !SQL_SUCCEEDED(nReturn))
        return;

    nReturn = SQLSetEnvAttr( hEnv, SQL_ATTR_ODBC_VERSION, (SQLPOINTER)SQL_OV_ODBC3, 0 );
    if ( nReturn != SQL_SUCCESS ) 
        doShowDiagnostics( nReturn, SQL_HANDLE_ENV, NULL );
    if ( !SQL_SUCCEEDED(nReturn))
        goto slotTestExit1;

    nReturn = SQLAllocHandle( SQL_HANDLE_DBC, hEnv, &hDbc );
    if ( nReturn != SQL_SUCCESS ) 
        doShowDiagnostics( nReturn, SQL_HANDLE_ENV, hEnv );
    if ( !SQL_SUCCEEDED(nReturn))
        goto slotTestExit1;

    nReturn = SQLDriverConnect( hDbc, NULL, (SQLCHAR*)stringConnectIn.latin1(), SQL_NTS, NULL, 0, NULL, SQL_DRIVER_NOPROMPT );
    if ( nReturn != SQL_SUCCESS ) 
        doShowDiagnostics( nReturn, SQL_HANDLE_DBC, hDbc );
    if ( !SQL_SUCCEEDED(nReturn))
        goto slotTestExit2;

    QMessageBox::information( this, "MyODBC", "Success; connection was made!\n", QMessageBox::Ok, QMessageBox::NoButton, QMessageBox::NoButton );

    nReturn = SQLDisconnect( hDbc );

slotTestExit2:
    nReturn = SQLFreeHandle( SQL_HANDLE_DBC, hDbc );
slotTestExit1:
    nReturn = SQLFreeHandle( SQL_HANDLE_ENV, hEnv );
}

void MYODBCSetupDataSourceDialog::slotHelp()
{
    QProcess  * pprocess    = new QProcess( this );
    QString     stringURL   = "http://dev.mysql.com/doc/connector/odbc/en/manual_toc.html";

#ifdef Q_WS_WIN
    pprocess->addArgument( "c:\\program files\\internet explorer\\IEXPLORE.EXE" );
    pprocess->addArgument( stringURL );
    if ( !pprocess->start() )
    {
        QMessageBox::warning( this, "MyODBC Help", "Failed to execute a browser.\n\nPlease visit\n" + stringURL, QMessageBox::Ok, QMessageBox::NoButton, QMessageBox::NoButton );
        delete pprocess;
    }
#else
#   ifdef Q_WS_MACX
    pprocess->addArgument( "open" );
//    pprocess->addArgument( "/Applications/Safari.app" );
    pprocess->addArgument( stringURL );
    if ( !pprocess->start() )
    {
        QMessageBox::warning( this, "MyODBC Help", "Failed to execute a browser.\n\nPlease visit\n" + stringURL, QMessageBox::Ok, QMessageBox::NoButton, QMessageBox::NoButton );
        delete pprocess;
    }
#   else
    pprocess->addArgument( "netscape" );            // generic
    pprocess->addArgument( stringURL );
    if ( !pprocess->start() )
    {
        delete pprocess;
        pprocess = new QProcess( this );
        pprocess->addArgument( "konqueror" );       // KDE folks
        pprocess->addArgument( stringURL );
        if ( !pprocess->start() )
        {
            delete pprocess;
            pprocess = new QProcess( this );
            pprocess->addArgument( "htmlview" );    // GNOME folks
            pprocess->addArgument( stringURL );
            if ( !pprocess->start() )
            {
                QMessageBox::warning( this, "MyODBC Help", "Failed to execute a browser.\n\nPlease visit\n" + stringURL, QMessageBox::Ok, QMessageBox::NoButton, QMessageBox::NoButton );
                delete pprocess;
            }
        }
    }
#   endif
#endif

}

void MYODBCSetupDataSourceDialog::slotApply()
{
    char *  pszAttributes;
    int     nAttributesLength;
    int     nDataSourceName;
    int     nDescription;
    int     nServer;
    int     nUser;
    int     nPassword;
    int     nDatabase;
    int     nPort;
    int     nStmt;
    int     nSocket;
    int     nOptions;
    int     n = 0;

    nAttributesLength = 68; /* 'DSN=\0SERVER=\0USER=\0PASSWORD=\0DATABASE=\0PORT=\0OPTION=\0SOCKET=\0STMT=\0\0' */ 

    nAttributesLength += (nDataSourceName = plineeditDataSourceName->text().length() + 1);
    nAttributesLength += (nDescription = plineeditDescription->text().length() + 1);
    nAttributesLength += (nServer = plineeditServer->text().length() + 1);
    nAttributesLength += (nUser = plineeditUser->text().length() + 1);
    nAttributesLength += (nPassword = plineeditPassword->text().length() + 1);
    nAttributesLength += (nDatabase = pcomboboxDatabase->lineEdit()->text().length() + 1);
    nAttributesLength += (nPort = plineeditPort->text().length() + 1);
    nAttributesLength += (nStmt = plineeditStmt->text().length() + 1);
    nAttributesLength += (nSocket = plineeditSocket->text().length() + 1);
    nAttributesLength += (nOptions = plineeditOptions->text().length() + 1);

    pszAttributes = (char*)malloc( nAttributesLength );

    if ( !plineeditDataSourceName->text().isEmpty() )
    {
        memcpy( &(pszAttributes[n]), "DSN=", 4 );
        n += 4;
        memcpy( &(pszAttributes[n]), plineeditDataSourceName->text().latin1(), nDataSourceName );
        n += nDataSourceName;
    }
    
    if ( !plineeditDescription->text().isEmpty() )
    {
        memcpy( &(pszAttributes[n]), "DESCRIPTION=", 12 );
        n += 12;
        memcpy( &(pszAttributes[n]), plineeditDescription->text().latin1(), nDescription );
        n += nDescription;
    }

    if ( !plineeditServer->text().isEmpty() )
    {
        memcpy( &(pszAttributes[n]), "SERVER=", 7 );
        n += 7;
        memcpy( &(pszAttributes[n]), plineeditServer->text().latin1(), nServer );
        n += nServer;
    }

    if ( !plineeditUser->text().isEmpty() )
    {
        memcpy( &(pszAttributes[n]), "USER=", 5 );
        n += 5;
        memcpy( &(pszAttributes[n]), plineeditUser->text().latin1(), nUser );
        n += nUser;
    }

    if ( !plineeditPassword->text().isEmpty() )
    {
        strcpy( &(pszAttributes[n]), "PASSWORD=" );
        n += 9;
        memcpy( &(pszAttributes[n]), plineeditPassword->text().latin1(), nPassword );
        n += nPassword;
    }

    if ( !pcomboboxDatabase->lineEdit()->text().isEmpty() )
    {
        strcpy( &(pszAttributes[n]), "DATABASE=" );
        n += 9;
        memcpy( &(pszAttributes[n]), pcomboboxDatabase->lineEdit()->text().latin1(), nDatabase );
        n += nDatabase;
    }

    if ( !plineeditPort->text().isEmpty() )
    {
        strcpy( &(pszAttributes[n]), "PORT=" );
        n += 5;
        memcpy( &(pszAttributes[n]), plineeditPort->text().latin1(), nPort );
        n += nPort;
    }

    if ( !plineeditStmt->text().isEmpty() )
    {
        strcpy( &(pszAttributes[n]), "STMT=" );
        n += 5;
        memcpy( &(pszAttributes[n]), plineeditStmt->text().latin1(), nStmt );
        n += nStmt;
    }

    if ( !plineeditSocket->text().isEmpty() )
    {
        strcpy( &(pszAttributes[n]), "SOCKET=" );
        n += 7;
        memcpy( &(pszAttributes[n]), plineeditSocket->text().latin1(), nSocket );
        n += nSocket;
    }

    if ( !plineeditOptions->text().isEmpty() )
    {
        strcpy( &(pszAttributes[n]), "OPTION=" );
        n += 7;
        memcpy( &(pszAttributes[n]), plineeditOptions->text().latin1(), nOptions );
        n += nOptions;
    }

/*
char *  ptr;
ptr = pszAttributes;

while ( *ptr )
{
    QMessageBox::information( this, "MYODBCConfig", ptr, QMessageBox::Ok, QMessageBox::NoButton, QMessageBox::NoButton );
    ptr += strlen( ptr ) + 1;
}
*/
    if ( nRequest == ODBC_ADD_DSN )
    {
        if ( ConfigDSN( (HWND)0, nRequest, pszDriver, pszAttributes ) )
        {
            free( pszAttributes );
            return;
        }
    }
    else
    {
        if ( ConfigDSN( (HWND)0, nRequest, pszDriver, pszAttributes ) )
        {
            free( pszAttributes );
            return;
        }
    }
    doShowInstallerError();
    bDone = FALSE;
}

void MYODBCSetupDataSourceDialog::doInit()
{
    bDone = TRUE;

    doInitTab1();

    QString stringRequest = "?";
    if ( nRequest == ODBC_CONFIG_DSN )
    {
        doSetValuesFromSystemInfo();
        stringRequest = "Edit";
    }
    else
    {
        nRequest = ODBC_ADD_DSN;
        stringRequest = "Add";
    }
    /* given attributes have precedence over existing during an edit */
    doSetValuesFromGivenAttributes();

    if ( pszDriver )
        setCaption( QString( pszDriver ) + " - DSN " + stringRequest + " (" + QString( pszDataSourceName ) + ")" );
    else
        setCaption( "MyODBC - DSN " + stringRequest + " (" + QString( pszDataSourceName ) + ")" );

    setCancelButton();
    setHelpButton();

    connect( this, SIGNAL( helpButtonPressed() ), SLOT( slotHelp() ) );
    connect( this, SIGNAL( applyButtonPressed() ), SLOT( slotApply() ) );
}

void MYODBCSetupDataSourceDialog::doInitTab1()
{
    QWidget *pwidgetPage = new QWidget( this );

    QHBoxLayout *playoutTop = new QHBoxLayout( pwidgetPage );
    playoutTop->setMargin( 5 );

    plabelSideImage = new QLabel( pwidgetPage );
    plabelSideImage->setBackgroundPixmap( MySQLSide_xpm );
    plabelSideImage->setScaledContents( FALSE ); // w=163
    plabelSideImage->setMinimumSize( 163, 311 );
    plabelSideImage->setMaximumSize( 163, 311 );
    QToolTip::add( plabelSideImage, "Brought to you by the database elves." );
    playoutTop->addWidget( plabelSideImage );
    playoutTop->setStretchFactor( plabelSideImage, 10 );

    QGridLayout *playoutFields = new QGridLayout( pwidgetPage, 11, 5 );
    playoutTop->addLayout( playoutFields );
    playoutFields->setMargin( 20 );
    playoutFields->setSpacing( 5 );

    QLabel *plabel;
    int nRow = 0;
    int nColLabel = 1;
    int nColField = 3;

    plabel = new QLabel( "Data Source Name", pwidgetPage );
    plineeditDataSourceName = new QLineEdit( pwidgetPage );
    QToolTip::add( plineeditDataSourceName, "A unique name for this data source.\nOptional: No\nDefault: myodbc" );
    playoutFields->addWidget( plabel, nRow, nColLabel );
    playoutFields->addWidget( plineeditDataSourceName, nRow, nColField );
    nRow++;

    plabel = new QLabel( "Description", pwidgetPage );
    plineeditDescription = new QLineEdit( pwidgetPage );
    QToolTip::add( plineeditDescription, "A brief description of this data source.\nOptional: Yes\nDefault: <empty>" );
    playoutFields->addWidget( plabel, nRow, nColLabel );
    playoutFields->addWidget( plineeditDescription, nRow, nColField );
    nRow++;

    plabel = new QLabel( "Server", pwidgetPage );
    plineeditServer = new QLineEdit( "localhost", pwidgetPage );
    QToolTip::add( plineeditServer, "The hostname of the MySQL server.\nOptional: Yes (silently uses default)\nDefault: localhost" );
    playoutFields->addWidget( plabel, nRow, nColLabel );
    playoutFields->addWidget( plineeditServer, nRow, nColField );
    nRow++;

    plabel = new QLabel( "User", pwidgetPage );
    plineeditUser = new QLineEdit( pwidgetPage );
    QToolTip::add( plineeditUser, "The username used to connect to MySQL.\nOptional: Yes (silently uses default)\nDefault (Windows only): ODBC" );
    playoutFields->addWidget( plabel, nRow, nColLabel );
    playoutFields->addWidget( plineeditUser, nRow, nColField );
    nRow++;

    plabel = new QLabel( "Password", pwidgetPage );
    plineeditPassword = new QLineEdit( pwidgetPage );
    plineeditPassword->setEchoMode( QLineEdit::Password );
    QToolTip::add( plineeditPassword, "The password for the server user combination.\nOptional: Yes\nDefault: <empty>" );
    playoutFields->addWidget( plabel, nRow, nColLabel );
    playoutFields->addWidget( plineeditPassword, nRow, nColField );
    nRow++;

    plabel = new QLabel( "Database", pwidgetPage );
    pcomboboxDatabase = new QComboBox( TRUE, pwidgetPage );
    QToolTip::add( pcomboboxDatabase, "The database to be current upon connect.\nOptional: Yes\nDefault: <none>" );
    playoutFields->addWidget( plabel, nRow, nColLabel );
    playoutFields->addWidget( pcomboboxDatabase, nRow, nColField );
    nRow++;

    plabel = new QLabel( "Port", pwidgetPage );
    plineeditPort = new QLineEdit( pwidgetPage );
    QToolTip::add( plineeditPort, "The TCP/IP port to use if server is not localhost.\nOptional: Yes (silently uses default)\nDefault: 3306" );
    playoutFields->addWidget( plabel, nRow, nColLabel );
    playoutFields->addWidget( plineeditPort, nRow, nColField );
    nRow++;

    plabel = new QLabel( "Socket", pwidgetPage );
    plineeditSocket = new QLineEdit( pwidgetPage );
    QToolTip::add( plineeditSocket, "The socket or Windows pipe to connect to.\nOptional: Yes\nDefault: <empty>" );
    playoutFields->addWidget( plabel, nRow, nColLabel );
    playoutFields->addWidget( plineeditSocket, nRow, nColField );
    nRow++;

    plabel = new QLabel( "Options", pwidgetPage );
    plineeditOptions = new QLineEdit( pwidgetPage );
    QToolTip::add( plineeditOptions, "A integer by which you can specify how Connector/ODBC should work.\nIf you want to have many options, you should add the flags.\nFor example setting option to 12 (4+8) gives you debugging without package limits.\nSee Help for a list of flags.\nOptional: Yes\nDefault: 0\n\n" );
    playoutFields->addWidget( plabel, nRow, nColLabel );
    playoutFields->addWidget( plineeditOptions, nRow, nColField );
    nRow++;

    plabel = new QLabel( "Stmt", pwidgetPage );
    plineeditStmt = new QLineEdit( pwidgetPage );
    QToolTip::add( plineeditStmt, "A statement that will be executed when connection to MySQL.\nOptional: Yes\nDefault: <empty>" );
    playoutFields->addWidget( plabel, nRow, nColLabel );
    playoutFields->addWidget( plineeditStmt, nRow, nColField );
    nRow++;

    ppushbuttonTest = new QPushButton( "Test", pwidgetPage );
    playoutFields->addWidget( ppushbuttonTest, nRow, nColField );
    QToolTip::add( ppushbuttonTest, "Enter the basic connect data required and then\nclick here to try to connect to the server." );
    connect( ppushbuttonTest, SIGNAL( clicked() ), this, SLOT( slotTest() ) );

    if ( nDataSourceName >= 0 && aNameValues[nDataSourceName]->pszValue )
        addTab( pwidgetPage, aNameValues[nDataSourceName]->pszValue );
    else
        addTab( pwidgetPage, "Data Source Name" );
}

void MYODBCSetupDataSourceDialog::doSetValuesFromSystemInfo()
{
    char    szEntryNames[SQL_MAX_DSN_LENGTH * MYODBC_MAX_DSN_NAMES];
    char *  pszEntryName;
    char    szValue[4096];

    if ( !pszDataSourceName || !(*pszDataSourceName) )
        return;

#if defined(Q_WS_MACX)    
    /*!
        \note   OSX

                SQLGetPrivateProfileString is the proper call and is available - but
                at this time it appears utterly broken. So we call an alternative
                instead. 
    */
    if ( GetPrivateProfileString( pszDataSourceName, NULL, NULL, szEntryNames, sizeof( szEntryNames ) - 1, "ODBC.INI" ) < 1 )
#else
    if ( SQLGetPrivateProfileString( pszDataSourceName, NULL, "", szEntryNames, sizeof( szEntryNames ) - 1, "ODBC.INI" ) < 1 )
#endif
        return; 

    /*!
        Scan result and return TRUE if we find a match.
    */    
    pszEntryName = szEntryNames;
    while( *pszEntryName )
    {
        QString stringEntry( (QString( pszEntryName )).upper() );

        *szValue = '\0';
#if defined(Q_WS_MACX)    
        if ( GetPrivateProfileString( pszDataSourceName, pszEntryName, NULL, szValue, sizeof( szValue ) - 1, "ODBC.INI" ) > 0 )
#else
        if ( SQLGetPrivateProfileString( pszDataSourceName, pszEntryName, "", szValue, sizeof( szValue ) - 1, "ODBC.INI" ) > 0 )
#endif
        {
            QString stringValue( (QString( szValue )).upper() );

            if ( stringEntry == "DESCRIPTION" )
                plineeditDescription->setText( szValue );
            else if ( stringEntry == "SERVER" )
                plineeditServer->setText( szValue );
            else if ( stringEntry == "USER" )
                plineeditUser->setText( szValue );
            else if ( stringEntry == "PASSWORD" )
                plineeditPassword->setText( szValue );
            else if ( stringEntry == "DATABASE" )
                pcomboboxDatabase->lineEdit()->setText( szValue );
            else if ( stringEntry == "PORT" )
                plineeditPort->setText( szValue );
            else if ( stringEntry == "STMT" )
                plineeditStmt->setText( szValue );
            else if ( stringEntry == "SOCKET" )
                plineeditSocket->setText( szValue );
            else if ( stringEntry == "OPTION" )
                plineeditOptions->setText( szValue );
            else
            {
                /* 
                    Tempted to put a message in here? the dm may have entries in here so its
                    ok if we do not know all entries.
                */  
            }
        }
        pszEntryName += strlen( pszEntryName ) + 1;
    } /* while */
}

void MYODBCSetupDataSourceDialog::doSetValuesFromGivenAttributes()
{
    if ( pszDataSourceName )
        plineeditDataSourceName->setText( pszDataSourceName );

    if ( nDescription >= 0 )
        plineeditDescription->setText( aNameValues[nDescription]->pszValue );

    if ( nServer >= 0 )
        plineeditServer->setText( aNameValues[nServer]->pszValue );

    if ( nUser >= 0 )
        plineeditUser->setText( aNameValues[nUser]->pszValue );

    if ( nPassword >= 0 )
        plineeditPassword->setText( aNameValues[nPassword]->pszValue );

    if ( nDatabase >= 0 )
        pcomboboxDatabase->lineEdit()->setText( aNameValues[nDatabase]->pszValue );

    if ( nPort >= 0 )
        plineeditPort->setText( aNameValues[nPort]->pszValue );

    if ( nSocket >= 0 )
        plineeditSocket->setText( aNameValues[nSocket]->pszValue );

    if ( nOption >= 0 )
        plineeditOptions->setText( aNameValues[nOption]->pszValue );

    if ( nStmt >= 0 )
        plineeditStmt->setText( aNameValues[nStmt]->pszValue );
}

void MYODBCSetupDataSourceDialog::doShowDiagnostics( SQLRETURN nReturn, SQLSMALLINT nHandleType, SQLHANDLE h )
{
    BOOL bMessageShown = FALSE;

    if ( h )
    {
        SQLSMALLINT nRec = 1;
        SQLCHAR     szSQLState[6];
        SQLINTEGER  nNative;
        SQLCHAR     szMessage[SQL_MAX_MESSAGE_LENGTH];
        SQLSMALLINT nMessage;

        *szSQLState = '\0';
        *szMessage  = '\0';

        while ( SQL_SUCCEEDED( SQLGetDiagRec( nHandleType,
                                              h,
                                              nRec,
                                              szSQLState,
                                              &nNative,
                                              szMessage,
                                              SQL_MAX_MESSAGE_LENGTH,
                                              &nMessage ) ) )
        {
            szSQLState[5]               = '\0';
            szMessage[SQL_MAX_MESSAGE_LENGTH - 1]  = '\0';

            if ( nReturn == SQL_ERROR )
                QMessageBox::critical( this, "MYODBCConfig", (char*)szMessage, QMessageBox::Ok, QMessageBox::NoButton, QMessageBox::NoButton );
            else if ( nReturn == SQL_SUCCESS_WITH_INFO )
                QMessageBox::warning( this, "MYODBCConfig", (char*)szMessage, QMessageBox::Ok, QMessageBox::NoButton, QMessageBox::NoButton );
            else
                QMessageBox::information( this, "MYODBCConfig", (char*)szMessage, QMessageBox::Ok, QMessageBox::NoButton, QMessageBox::NoButton );

            bMessageShown = TRUE;
            nRec++;

            *szSQLState = '\0';
            *szMessage  = '\0';
        }
    }

    if ( !bMessageShown )
    {
        switch ( nReturn )
        {
            case SQL_ERROR:
                QMessageBox::critical( this, "MYODBCConfig", "Request returned with SQL_ERROR. Sorry; no diagnostic information.", QMessageBox::Ok, QMessageBox::NoButton, QMessageBox::NoButton );
                break;
            case SQL_SUCCESS_WITH_INFO:
                QMessageBox::warning( this, "MYODBCConfig", "Request return with SQL_SUCCESS_WITH_INFO. Sorry; no diagnostic information.", QMessageBox::Ok, QMessageBox::NoButton, QMessageBox::NoButton );
                break;
            case SQL_INVALID_HANDLE:
                QMessageBox::critical( this, "MYODBCConfig", "Request returned with SQL_INVALID_HANDLE.", QMessageBox::Ok, QMessageBox::NoButton, QMessageBox::NoButton );
                break;
            default:
                QMessageBox::information( this, "MYODBCConfig", "Request did not return with SQL_SUCCESS. Sorry; no diagnostic information.", QMessageBox::Ok, QMessageBox::NoButton, QMessageBox::NoButton );
        }
    }
}

void MYODBCSetupDataSourceDialog::doShowInstallerError()
{
    WORD      nRecord = 1;
    DWORD     nError;
    char      szError[SQL_MAX_MESSAGE_LENGTH];
    RETCODE   nReturn;

    nReturn = SQLInstallerError( nRecord, &nError, szError, SQL_MAX_MESSAGE_LENGTH - 1, 0 );
    if ( SQL_SUCCEEDED( nReturn ) )
        QMessageBox::critical( this, "MYODBCConfig", (char*)szError, QMessageBox::Ok, QMessageBox::NoButton, QMessageBox::NoButton );
    else
        QMessageBox::critical( this, "MYODBCConfig", "ODBC Installer error (unknown)", QMessageBox::Ok, QMessageBox::NoButton, QMessageBox::NoButton );
}


