/***************************************************************************
                          gnumarkedfiles.cpp  -  description
                             -------------------
    begin                : Sun Jan 19 2003
    copyright            : (C) 2003 by Max Zaitsev
    email                : maksik@gmx.co.uk
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "mutella.h"
#include "structures.h"

#include "conversions.h"

#include "gnumarkedfiles.h"

#include "event.h"
#include "messages.h"
#include "common.h"


MGnuMarkedFiles::MGnuMarkedFiles()
{
}

MGnuMarkedFiles::~MGnuMarkedFiles()
{
}

bool MGnuMarkedFiles::AddFile(MarkedFileType type, long nSize, const SHA1Hash& sha1, const CString& sSearch, long nHandle /*=0*/)
{
	MLock lock(m_mutex);
	// check file size
	tMap::iterator it = m_mapFiles.find(nSize);
	if (it == m_mapFiles.end())
	{
		// we have to create new FileSizeClass
		MFileSizeClass& fsc = m_mapFiles[nSize];
		SMarkedFile mf;
		mf.sha1        = sha1;
		mf.setSearch.insert(sSearch);
		mf.nSize       = nSize;
		mf.nHandle     = nHandle;
		mf.nType       = type;
		mf.nUpdateTime = xtime();
		fsc.vectorMF.push_back(mf);
		fsc.UpdateMaps();
	}
	else
	{
		// we have to update an existing type-class
		// try to locate similar file item
		long nFileIndex = -1;
		int nMatch = MFR_Flags & it->second.Match(sha1, sSearch, &nFileIndex);
		switch (nMatch) {
			case MFR_PrbNoSha | MFR_Probable :
				if (sha1.isValid())
					break;
			case MFR_Exact : {
				ASSERT(nFileIndex >= 0);
				ASSERT(it->second.vectorMF.size() >  nFileIndex);
				ASSERT(it->second.vectorMF[nFileIndex].sha1 == sha1);
				SMarkedFile& mf = it->second.vectorMF[nFileIndex];
				int nOldSetSize = mf.setSearch.size();
				mf.nUpdateTime = xtime();
				mf.setSearch.insert(sSearch);
				int nOldType = mf.nType;
				// handle handles here
				if (nHandle != 0 && nHandle != mf.nHandle)
				{
#ifdef _DEBUG
					cout << "MGnuMarkedFiles::AddFile() : handle conflict \n";
					cout << "    nHandle = " << nHandle << "  mf.nHandle = " << mf.nHandle << endl;
					cout << "    type = " << type << "  mf.nType = " << mf.nType << endl;
#endif
					mf.nHandle = nHandle;
				}
				// promote types here
				if (type == MFT_LibraryActual && (mf.nType & MFT_LibraryActual))
				{
					mf.nType |= MFT_LibraryActMul;
				}
				else if (type == MFT_LibraryActual && ((mf.nType & MFT_Attempted) || (mf.nType & MFT_Partial)))
				{
					mf.nType &= ~MFT_Attempted;
					mf.nType &= ~MFT_Partial;
					mf.nType |= MFT_LibraryActual;
				}				
				else if (type == MFT_Partial && (mf.nType & MFT_Attempted))
				{
					mf.nType &= ~MFT_Attempted;
					mf.nType |= MFT_Partial;
				}
				else
					mf.nType |= type;
				//
				if (nOldSetSize != mf.setSearch.size())
				{
					it->second.UpdateMaps();
					return true;
				}
				return nOldType != mf.nType;
			}
			case MFR_Probable :
			case MFR_Possible :
			default:
				break;
		}
		SMarkedFile mf;
		mf.sha1        = sha1;
		mf.setSearch.insert(sSearch);
		mf.nSize       = nSize;
		mf.nHandle     = nHandle;
		mf.nType       = type;
		mf.nUpdateTime = xtime();
		it->second.vectorMF.push_back(mf);
		it->second.UpdateMaps();
	}
	return true;
}

bool MGnuMarkedFiles::RemoveFile(long nSize, const SHA1Hash& sha1, const CString& sSearch, bool removeAll /*= false*/ )
{
#warning MGnuMarkedFiles::RemoveFile() needs an update
	MLock lock(m_mutex);
	// check file size
	tMap::iterator it = m_mapFiles.find(nSize);
	if (it == m_mapFiles.end())
		return false; // nothing to remove
    long nMFIndex;
	long nMatch = it->second.Match(sha1, sSearch, &nMFIndex);
	if ( MFR_Exact == (MFR_Flags & nMatch) )
	{
		it->second.vectorMF.erase(it->second.vectorMF.begin() + nMFIndex);
		it->second.RebuildMaps();
		if (removeAll)
			nMatch = it->second.Match(sha1, sSearch, &nMFIndex);
	}
	if ( removeAll && (MFR_Flags & nMatch) )
	{
		// TODO: i do realize that this implementation is suboptimal -- fix it
		do {
			it->second.vectorMF.erase(it->second.vectorMF.begin() + nMFIndex);
			it->second.RebuildMaps();
		} while ( MFR_Flags & it->second.Match(sha1, sSearch, &nMFIndex) );
	}
	return true;
}

long MGnuMarkedFiles::MatchFile(long nSize, const SHA1Hash& sha1, const CString& sSearch)
{
	MLock lock(m_mutex);
	// check file size
	tMap::iterator it = m_mapFiles.find(nSize);
	if (it == m_mapFiles.end())
		return MFT_NoMatch;
	// check other things
	return it->second.Match(sha1, sSearch);
}

bool MGnuMarkedFiles::LoadList(const CString& sPath)
{
	FILE* f = fopen(ExpandPath(sPath).c_str(),"r");
	if (!f)
	{
		POST_ERROR(ES_UNIMPORTANT, CString("Failed to open file \'") + sPath + "\' -- marked files database will not be loaded");
		TRACE3("Failed to open file \'", sPath, "\' -- marked files database will not be loaded");
		return false;
	}
	//
	MLock lock(m_mutex);
	//
	char tmp[1024];
	char* t;
	SMarkedFile mf;
	bool bSha1, bHandle, bSize, bType, bTime;
	bSha1 = bSize = bType = bTime = false;
	while (!feof(f) && !ferror(f))
	{
		if (NULL!=fgets(tmp,1024,f))
		{
			tmp[1023] = '\0';
			t = StripWhite(tmp);
			if (strlen(t))
			{
				if (*t != '#')
				{
					if (0 == strncmp(t, "String:", 7))
						mf.setSearch.insert(StripWhite(t+7));
					else if (0 == strncmp(t, "Sha1:", 5))
					{
						bSha1 = mf.sha1.fromStr( StripWhite(t+5));
					}
					else if (0 == strncmp(t, "Handle:", 7))
					{
						mf.nHandle = atol( StripWhite(t+7) );
						bHandle = (mf.nHandle > 0);
					}
					else if (0 == strncmp(t, "Size:", 5))
					{
						mf.nSize = atol( StripWhite(t+5) );
						bSize = (mf.nSize > 0);
					}
					else if (0 == strncmp(t, "Type:", 5))
					{
						t = StripWhite(t+5);
						mf.nType = 0;
						if (strstr(t, "Library"))
						{
							mf.nType |= MFT_LibraryStored;
							bType = true;
						}
						if (strstr(t, "Attempted"))
						{
							mf.nType |= MFT_Attempted;
							bType = true;
						}
						if (strstr(t, "User"))
						{
							mf.nType |= MFT_User;
							bType = true;
						}
					}
					else if (0 == strncmp(t, "Time:", 5))
					{
						mf.nUpdateTime = atol( StripWhite(t+5) );
						bTime = (mf.nUpdateTime > 0);
					}
				}
				else
				{
					mf.setSearch.clear();
					bSha1 = bSize = bType = bTime = false;
				}
			}
			else
			{
				// decode sEntry and create a file entry
				if (bTime && bType && bSize && ( bSha1 || mf.setSearch.size()))
				{
					// check handle
					if (!bHandle)
						mf.nHandle = 0;
					// clean the bogus combinations
					if (mf.nType & MFT_LibraryStored)
						mf.nType &= ~MFT_Attempted;
					// place to add item
					// check file size
					tMap::iterator it = m_mapFiles.find(mf.nSize);
					if (it == m_mapFiles.end())
					{
						// we have to create new FileSizeClass
						MFileSizeClass& fsc = m_mapFiles[mf.nSize];
						fsc.vectorMF.push_back(mf);
					}
					else
					{
						// avoid creating duplicates (due to earlier bugs)
						if (it->second.vectorMF.back().setSearch != mf.setSearch &&
							it->second.vectorMF.back().sha1      != mf.sha1      &&
							it->second.vectorMF.back().nType     != mf.nType     )
								it->second.vectorMF.push_back(mf);
					}
				}
				mf.setSearch.clear();
				bSha1 = bSize = bType = false;
			}
		}
	}
	fclose(f);
	// now that all is loaded go and build indexes
	for (tMap::iterator itM = m_mapFiles.begin(); itM != m_mapFiles.end(); ++itM)
		itM->second.RebuildMaps();
	return true;
}

bool MGnuMarkedFiles::StoreList(const CString& sPath)
{
	// save list in the following format:
	//   String: search string
	//   Sha1: SHA1_base32
	//   Size: size
	//   Type: type
	//   <empty line>
	FILE* f = fopen(ExpandPath(sPath).c_str(),"w");
	if (!f)
	{
		POST_ERROR(ES_IMPORTANT, CString("Failed to create file \'") + sPath + "\' -- marked file database will not be stored");
		TRACE3("Failed to create file \'", sPath, "\' -- marked files database will not be stored");
		return false;
	}
	fprintf(f,"# this is mutella\'s \"known files\" database\n");
	fprintf(f,"# it is created automatically at the end\n");
	fprintf(f,"# of mutella session and updated periodically\n");
	fprintf(f,"# during the session\n");
	fprintf(f,"#\n\n");
	//
	
	MLock lock(m_mutex);

	//for each item
	for (tMap::iterator itM = m_mapFiles.begin(); itM != m_mapFiles.end(); ++itM)
		for (tMFVec::iterator itV = itM->second.vectorMF.begin(); itV != itM->second.vectorMF.end(); ++itV)
		{
			// search strings
			for (set<CString>::iterator its = itV->setSearch.begin(); its != itV->setSearch.end(); ++its)
				fprintf(f,"String: %s\n", its->c_str());
			// sha1
			fprintf(f,"Sha1: %s\n", itV->sha1.toStr().c_str());
			// handle
			fprintf(f,"Handle: %d\n", itV->nHandle);
			// size
			fprintf(f,"Size: %d\n", itV->nSize);
			// type
			fprintf(f,"Type:");
			if ( itV->nType & (MFT_LibraryActual|MFT_LibraryStored|MFT_LibraryActMul) )
				fprintf(f," Library");
			if ( itV->nType & (MFT_Attempted|MFT_Partial) )
				fprintf(f," Attempted");
			if ( itV->nType & MFT_User )
				fprintf(f," User");
			fprintf(f,"\n"); // end of type line
			fprintf(f, "Time: %d\n", itV->nUpdateTime);
			// empty line
			fprintf(f,"\n");
		}
	fclose(f);
	return true;
}

long MGnuMarkedFiles::GetFreeFileHandle (long nSize)
{
	MLock lock(m_mutex);
	// check file size
	tMap::iterator it = m_mapFiles.find(nSize);
	if (it == m_mapFiles.end())
		return 1; // nothing to remove
    return it->second.GetFreeHandle();
}

bool MGnuMarkedFiles::AddFileFlag(long nSize, long nHandle, long nType)
{
	MLock lock(m_mutex);
	// check file size
	tMap::iterator it = m_mapFiles.find(nSize);
	if (it == m_mapFiles.end())
		return false;
	it->second.AddFlag(nHandle, nType);
	return true;
}

void MGnuMarkedFiles::RemoveFile(long nSize, long nHandle, long nType)
{
	MLock lock(m_mutex);
	// check file size
	tMap::iterator it = m_mapFiles.find(nSize);
	if (it == m_mapFiles.end())
		return;
	it->second.Remove(nHandle, nType);
	if (!it->second.Size())
		m_mapFiles.erase(it);
}

/////////////////////////////////////////////////////////////////////
// MFileSizeClass implementation

MFileSizeClass::MFileSizeClass()
{
	nNextWIndex = 0;
}

void MFileSizeClass::UpdateMapsForSingleElement(long nIndex, const SMarkedFile& mf)
{
	if (mf.sha1.isValid())
	{
		ASSERT(mapSHA1_To_MFInd.find(mf.sha1)==mapSHA1_To_MFInd.end() || mapSHA1_To_MFInd.find(mf.sha1)->second==nIndex);  // TODO: Fix this assertion for repeated files
		mapSHA1_To_MFInd[mf.sha1] = nIndex;
	}
	//
	if (mf.nHandle > 0)
	{
		mapHandle_To_MFInd[mf.nHandle].insert(nIndex);
	}
	//
	if (mf.setSearch.empty())
		return;
	for (set<CString>::iterator its = mf.setSearch.begin(); its != mf.setSearch.end(); ++its)
	{
		char* szSearch = (char*) alloca(its->length()+1);
		strcpy(szSearch, its->c_str());
		set<CString> setWords;
		MakeWordList(szSearch, setWords);
		set<long> setWInd;
		map<CString, long>::iterator itWI;
		for(set<CString>::iterator it=setWords.begin(); it!=setWords.end(); ++it)
		{
			itWI = mapWord_To_WIndex.find(*it);
			if (itWI != mapWord_To_WIndex.end())
				setWInd.insert(itWI->second);
			else
			{
				mapWord_To_WIndex[*it] = nNextWIndex;
				setWInd.insert(nNextWIndex);
				++nNextWIndex;
			}
		}
		if(setWInd.empty())
			continue;
		mapIndexSet_To_MFInd[setWInd].push_back(nIndex);
	}
}

long MFileSizeClass::GetFreeHandle()
{
	int nHandle=1;
	for (map< long, set<long> >::iterator it=mapHandle_To_MFInd.begin(); it!=mapHandle_To_MFInd.end(); ++it)
	{
		if (it->first>nHandle)
			return nHandle;
		++nHandle;
	}
	return nHandle;
}

void MFileSizeClass::UpdateMaps()
{
	// called after addition of an element
	int nInd = vectorMF.size() - 1;
	ASSERT(nInd >= 0);
	UpdateMapsForSingleElement(nInd, vectorMF.back());
}

void MFileSizeClass::UpdateMaps(long nInd)
{
	ASSERT(nInd >= 0);
	ASSERT(nInd < vectorMF.size());
	UpdateMapsForSingleElement(nInd, vectorMF[nInd]);
}

void MFileSizeClass::RebuildMaps()
{
	// called after element removal or multiple element addition
	mapSHA1_To_MFInd.clear();
	mapWord_To_WIndex.clear();
	nNextWIndex = 0;
	mapIndexSet_To_MFInd.clear();
	mapHandle_To_MFInd.clear();
	// build everything
	for (long i=0; i<vectorMF.size(); ++i)
		UpdateMapsForSingleElement(i, vectorMF[i]);
}

long MFileSizeClass::Match(const SHA1Hash& sha1, const CString& sSearch, long* pnMFIndex /*=NULL*/)
{
	// check for exact match (sha1)
	if (sha1.isValid())
	{
		map<SHA1Hash, long>::iterator it = mapSHA1_To_MFInd.find(sha1);
		if (it != mapSHA1_To_MFInd.end())
		{
			if (pnMFIndex)
				*pnMFIndex = it->second;
			return MFR_Exact | vectorMF[it->second].nType;
		}
	}
	// parse search string
	char* szSearch = (char*) alloca(sSearch.length()+1);
	strcpy(szSearch, sSearch.c_str());
	set<CString> setWords;
	MakeWordList(szSearch, setWords);
	set<long> setWInd;
	map<CString, long>::iterator itWI;
	bool bHasUnknownWords = false;
	for(set<CString>::iterator it=setWords.begin(); it!=setWords.end(); ++it)
	{
		itWI = mapWord_To_WIndex.find(*it);
		if (itWI != mapWord_To_WIndex.end())
			setWInd.insert(itWI->second);
		else
			bHasUnknownWords = true;
	}
	if(setWInd.size())
	{
		// check for probable/possible match
		map< set<long>, list<long> >::iterator it = mapIndexSet_To_MFInd.find(setWInd);
		if (it != mapIndexSet_To_MFInd.end())
		{
			// possible multiple match
			int nFlags = 0;
			bool bHaveSha1 = false;
			for (list<long>::iterator iti = it->second.begin(); iti != it->second.end(); ++iti)
			{
				ASSERT(*iti >= 0);
				ASSERT(*iti < vectorMF.size());
				nFlags |= vectorMF[*iti].nType;
				bHaveSha1 = bHaveSha1 || vectorMF[*iti].sha1.isValid();
			}
			if (pnMFIndex && it->second.size()==1)
				*pnMFIndex = it->second.front();
			if (!bHaveSha1)
				nFlags | MFR_PrbNoSha;
			if (!bHasUnknownWords)
				return nFlags | MFR_Probable;
			else
				return nFlags | MFR_Possible;
		}
		else
		{
			// check for 'possible' match
			int nFlags = 0;
			for (it = mapIndexSet_To_MFInd.begin(); it != mapIndexSet_To_MFInd.end(); ++it)
			{
				if (!bHasUnknownWords && includes(it->first.begin(), it->first.end(), setWInd.begin(), setWInd.end()) ||
					includes(setWInd.begin(), setWInd.end(), it->first.begin(), it->first.end()) )
				{
					// either some other set includes us entirely or
					// we include entirely some other set
					for (list<long>::iterator iti = it->second.begin(); iti != it->second.end(); ++iti)
					{
						ASSERT(*iti >= 0);
						ASSERT(*iti < vectorMF.size());
						nFlags |= vectorMF[*iti].nType;
					}
				}
			}
			if (nFlags)
				return nFlags | MFR_Possible;
		}
	}
	// return no-match
	return MFT_NoMatch;
}

void MFileSizeClass::AddFlag(long nHandle, long nType)
{
	if (nHandle == 0 || nType == 0)
		return;
		
	map< long, set<long> >::iterator it = mapHandle_To_MFInd.find(nHandle);
	if (it == mapHandle_To_MFInd.end())
		return;
	
	for(set<long>::iterator its=it->second.begin(); its!=it->second.end(); ++its)
	{
		ASSERT(*its >= 0);
		ASSERT(*its < vectorMF.size());
		tMFVec::iterator itv = vectorMF.begin() + *its;
		itv->nType = (itv->nType | nType) & MFT_All;
	}
}

void MFileSizeClass::Remove(long nHandle, long nType)
{
	if (nHandle == 0 || nType == 0)
		return;
		
	bool bNeedUpdate = false;
	map< long, set<long> >::iterator it = mapHandle_To_MFInd.find(nHandle);
	if (it == mapHandle_To_MFInd.end())
		return;
	
	for(set<long>::reverse_iterator its=it->second.rbegin(); its!=it->second.rend(); ++its)
	{
		ASSERT(*its >= 0);
		ASSERT(*its < vectorMF.size());
		tMFVec::iterator itv = vectorMF.begin() + *its;
		itv->nType = (itv->nType & ~nType) & MFT_All;
		if (!itv->nType)
		{
			bNeedUpdate = true;
			// we are using reverse iterator here, so it's safe to remove elements
			vectorMF.erase(itv);
		}
	}
	
	if (bNeedUpdate)
		RebuildMaps();
}


