/*

    mp_iface.c

    Interface.

    mp - Programmer Text Editor

    Copyright (C) 1991-2001 Angel Ortega <angel@triptico.com>

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

    http://www.triptico.com

*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "mp_core.h"
#include "mp_video.h"
#include "mp_synhi.h"
#include "mp_lang.h"
#include "mp_iface.h"


/********************
	 Data
 ********************/

/* though used in video drivers, these are defined here */
int _mpv_x_size;
int _mpv_y_size;

/* word buffer */
static char _draw_word[128];

/* insert flag */
int _mpi_insert=1;

/* text to search */
static char _mpi_search_text[4096]="";

/* special shift */
int _mpi_shift=0;

/* reposition flag */
#ifdef ABSOLUTELY_NO_COLORS
int _mpi_reposition_cursor=1;
#else
int _mpi_reposition_cursor=0;
#endif

char * MP_LICENSE=
"\nMinimum Profit " VERSION " - Programmer Text Editor\n\n\
Copyright (C) 1991-2002 Angel Ortega <angel@triptico.com>\n\
\n\
This program is free software; you can redistribute it and/or\n\
modify it under the terms of the GNU General Public License\n\
as published by the Free Software Foundation; either version 2\n\
of the License, or (at your option) any later version.\n\
\n\
This program is distributed in the hope that it will be useful,\n\
but WITHOUT ANY WARRANTY; without even the implied warranty of\n\
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n\
GNU General Public License for more details.\n\
\n\
You should have received a copy of the GNU General Public License\n\
along with this program; if not, write to the Free Software\n\
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.\n\
\n\
http://www.triptico.com\n";


/* macro */

int _mpi_macro[1024];
int _mpi_macro_index=-1;

/* instant position (mouse click or so) */
int _mpi_instant_x=-1;
int _mpi_instant_y=-1;

/* mark column 80 */
int _mpi_mark_column_80=0;

/* element selected from a list */
int _mpi_list_selection=-1;

/* exit requested */
int _mpi_exit_requested=0;

/* the template file */
char _mpi_template_file[1024]="";


/* experimental ftt code */

static char * _mpi_ftt_key="\x20\x5f\x2f\x7c\x5c\x28\x29\x6f\x2d";

static char * _mpi_ftt_challenge=
	"\x5b\x46\x65\x61\x72\x20\x74\x68\x65\x20\x54\x72\x69\x63\x65\x72"
	"\x61\x74\x6f\x70\x73\x5d";

static unsigned char * _mpi_ftt_data=
	"\x00\x00\x11\x11\x11\x11\x00\x00\x00\x12\x00\x00\x00\x00\x41\x00"
	"\x02\x00\x34\x00\x00\x23\x00\x40\x30\x00\x30\x40\x02\x03\x00\x03"
	"\x30\x00\x57\x00\x00\x76\x00\x03\x04\x10\x04\x02\x40\x20\x01\x20"
	"\x03\x04\x10\x20\x04\x01\x20\x30\x03\x00\x33\x08\x80\x33\x00\x30"
	"\x03\x00\x30\x41\x12\x03\x00\x30\x56\x56\x56\x00\x00\x56\x56\x56\xFF";

/* number of lines to move up over current visible line,
   to take into account possible start of comments
   many lines above */
#if defined(ABSOLUTELY_NO_COLORS) || defined(ANCIENT_HARDWARE)
int _mpi_preread_lines=0;
#else
int _mpi_preread_lines=60;
#endif

#ifndef ANCIENT_HARDWARE
#define ANCIENT_HARDWARE 0
#endif

int _mpi_ancient_hardware=ANCIENT_HARDWARE;

/* alternate color set */
#ifndef ALT_COLOR
#define ALT_COLOR 0
#endif
int mpi_alt_color=ALT_COLOR;

/* use mouse to position cursor or not (if applicable) */
int mpi_mouse=0;

/* monochrome mode */
int mpi_monochrome=0;

/* use transparent mode (if applicable) */
int mpi_transp_mode=1;

/* move selecting flag */
int mpi_move_selecting=0;


/*******************
	Code
*******************/

/**
 * mpi_plain_draw_all - Draws the document window (simple version)
 *
 * Stripped down version of mpi_color_draw_all() that doesn't draw
 * any color other than the cursor and the selected block. It's
 * used when 'plain' or no syntax highlighter is selected, as
 * it's also faster.
 */
void mpi_plain_draw_all(mp_txt * txt)
{
	int vx,vy;
	mp_txt * wrk;
	int n,m;
	int c,color;
	int ry;
	int xcursor=0,ycursor=0;

	mpv_title(NULL);

	mp_adjust(txt,_mpv_x_size,_mpv_y_size-1);

	vx=txt->vx;
	vy=txt->vy;

	wrk=mp_get_tmp_txt(txt);

	mp_move_bol(wrk);

	/* moves up to first visible line */
	while(wrk->y > vy)
	{
		if(! mp_move_up(wrk))
			break;
	}

	/* line loop */
	while(wrk->y < vy + _mpv_y_size)
	{
		/* column loop */

		ry=wrk->y;

		/* move drawing cursor to beginning of line */
		mpv_goto(0,ry - vy);

		c='\0';

		for(m=0;m < vx + _mpv_x_size;m++)
		{
			c=mp_get_char(wrk);

			/* if visible... */
			if(m >= vx)
			{
				color=MP_COLOR_NORMAL;

				/* if inside selection block... */
				if(ry>txt->mby && ry<=txt->mey)
					color=MP_COLOR_SELECTED;

				if(ry==txt->mby && m>=txt->mbx)
					color=MP_COLOR_SELECTED;

				if(ry==txt->mey && m>txt->mex)
					color=MP_COLOR_NORMAL;

				/* if over the cursor... */
				if(m==txt->x && ry==txt->y)
				{
					xcursor=m-vx;
					ycursor=ry-vy;
					color=MP_COLOR_CURSOR;
				}

				if(txt->type == MP_TYPE_LIST && ry == txt->y)
					color=MP_COLOR_CURSOR;

				/* finally draws */
				mpv_char((c=='\n' || c=='\0') ? ' ' : c,color);
			}

			if(c=='\n' || c=='\0')
				break;
		}

		/* spaces to end of line */
		if(m<vx)
			m=vx;

		for(;m < vx+_mpv_x_size;m++)
		{
			if(txt->type == MP_TYPE_LIST && ry == txt->y)
				mpv_char(' ',MP_COLOR_CURSOR);
			else
				mpv_char(' ',MP_COLOR_NORMAL);
		}

		/* if last read char is '\0', it's the end */
		if(c=='\0')
			break;

		/* if we are not at the end of the line, move there */
		if(c!='\n')
		{
			mp_move_bol(wrk);

			if(! mp_move_down(wrk))
				break;
		}
	}

	/* the rest of lines are drawn as blanks */
	for(n=wrk->y - vy;n <= _mpv_y_size;n++)
	{
		mpv_goto(0,n+1);

		for(m=0;m < _mpv_x_size;m++)
			mpv_char(' ',MP_COLOR_NORMAL);
	}

	mp_end_tmp_txt();

	/* scrollbar */
	mpv_scrollbar(txt->vy+1,_mpv_y_size,txt->lasty+1);

	/* status line */
	sprintf(_draw_word,"%s %d,%d [%d] %c %c %s",
		txt->mod?"*":"",
		txt->x+1,txt->y+1,txt->lasty+1,
		_mpi_shift?'^':' ',
		_mpi_macro_index==-1 ? ' ':'R',
		txt->synhi?_mps_synhi[txt->synhi-1].type:"");
	mpv_status_line(_draw_word);

	mpv_filetabs();

	mpv_goto(0,_mpv_y_size);
	if(_mpi_reposition_cursor) mpv_cursor(xcursor,ycursor);

	mpv_refresh();
}


/**
 * mpi_color_draw_all - Draws the document window using colors.
 * @txt: the text
 *
 * Draws the document window, with syntax highlighting decorations and
 * other colors, if applicable.
 */
void mpi_color_draw_all(mp_txt * txt)
{
	int vx,vy;
	mp_txt * wrk;
	int n,m;
	int c,wc,color,rcolor;
	int wi;
	int spcs;
	int ry;
	int xcursor=0,ycursor=0;

	mpv_title(NULL);

	mp_adjust(txt,_mpv_x_size,_mpv_y_size-1);

	vx=txt->vx;
	vy=txt->vy;

	wrk=mp_get_tmp_txt(txt);

	mp_move_bol(wrk);

	/* moves up to first visible line */
	while(wrk->y > (vy - _mpi_preread_lines))
	{
		if(! mp_move_up(wrk))
			break;
	}

	_in_comment=0;

	/* line loop */
	for(;;)
	{
		/* column loop */

		ry=wrk->y;

		/* move this outside the loop to allow
		   multiline quoted strings (you shouldn't) */
		_draw_quoting='\0';

		/* end if below the last line */
		if(ry >= (vy + _mpv_y_size))
			break;

		c=mp_get_char(wrk);

		/* move drawing cursor to beginning of line */
		if(ry >= vy)
			mpv_goto(0,ry - vy);

		for(m=0;;)
		{
			/* word loop */

			wi=0;
			spcs=mps_is_sep(c,txt->synhi);

			while(wi<sizeof(_draw_word)-1)
			{
				if(c=='\n' || c=='\0')
					break;

				_draw_word[wi++]=c;

				c=mp_get_char(wrk);

				if(spcs != mps_is_sep(c,txt->synhi))
				{
					if(_draw_word[wi - 1]!='\\')
						break;
				}
			}

			_draw_word[wi]='\0';

			if(spcs)
				rcolor=MP_COLOR_NORMAL;
			else
				rcolor=mps_word_color(txt->synhi,_draw_word, ry);

			if(c=='\n' || c=='\0')
			{
				_draw_word[wi++]=' ';
				_draw_word[wi]='\0';
			}

			/* draws the visible chars of the word */
			for(wi=0;(wc=_draw_word[wi])!='\0';wi++,m++)
			{
				color=mps_quoting(wc,rcolor,txt->synhi);

				/* if visible... */
				if(m>=vx && m<vx+_mpv_x_size && ry >= vy)
				{
					/* if inside selection block... */
					if(ry>txt->mby &&
						ry<=txt->mey)
						color=MP_COLOR_SELECTED;

					if(ry==txt->mby &&
						m>=txt->mbx)
						color=MP_COLOR_SELECTED;

					if(ry==txt->mey &&
						m>txt->mex)
						color=rcolor;

					/* if over the cursor... */
					if(m==txt->x &&
						ry==txt->y)
					{
						xcursor=m-vx;
						ycursor=ry-vy;
						color=MP_COLOR_CURSOR;
					}

					if(_mpi_mark_column_80 && m==80)
						color=MP_COLOR_SELECTED;

					if(txt->type == MP_TYPE_LIST &&
					   ry == txt->y)
						color=MP_COLOR_CURSOR;

					/* finally draws */
					mpv_char(wc,color);
				}
			}

			if(c=='\n' || c=='\0')
				break;
		}

		/* spaces to end of line */
		if(m<vx)
			m=vx;

		for(;m<vx+_mpv_x_size;m++)
		{
			if(_mpi_mark_column_80 && m==80)
				mpv_char(' ',MP_COLOR_SELECTED);
			else
			if(txt->type == MP_TYPE_LIST &&
			   ry == txt->y)
				mpv_char(' ',MP_COLOR_CURSOR);
			else
				mpv_char(' ',MP_COLOR_NORMAL);
		}

		/* if last read char is '\0', it's the end */
		if(c=='\0')
			break;

		/* if we are not at the end of the line, move there */
		if(c!='\n')
		{
			mp_move_bol(wrk);

			if(! mp_move_down(wrk))
				break;
		}
	}

	/* the rest of lines are drawn as blanks */
	for(n=ry - vy;n <= _mpv_y_size;n++)
	{
		mpv_goto(0,n+1);

		for(m=0;m < _mpv_x_size;m++)
		{
			if(_mpi_mark_column_80 && m==80)
				mpv_char(' ',MP_COLOR_SELECTED);
			else
				mpv_char(' ',MP_COLOR_NORMAL);
		}
	}

	mp_end_tmp_txt();

	/* scrollbar */
	mpv_scrollbar(txt->vy+1,_mpv_y_size,txt->lasty+1);

	/* status line */
	sprintf(_draw_word,"%s %d,%d [%d] %c %c %s",
		txt->mod?"*":"",
		txt->x+1,txt->y+1,txt->lasty+1,
		_mpi_shift?'^':' ',
		_mpi_macro_index==-1 ? ' ':'R',
		txt->synhi?_mps_synhi[txt->synhi-1].type:"");
	mpv_status_line(_draw_word);

	mpv_filetabs();

	mpv_goto(0,_mpv_y_size);
	if(_mpi_reposition_cursor) mpv_cursor(xcursor,ycursor);

	mpv_refresh();
}


/**
 * mpi_draw_all - Draws the document window.
 * @txt: the text
 *
 * Draws the document window.
 */
void mpi_draw_all(mp_txt * txt)
{
	if(!_mpi_ancient_hardware &&
	   (_mpi_mark_column_80 || txt->synhi > 1))
		mpi_color_draw_all(txt);
	else
		mpi_plain_draw_all(txt);
}


/**
 * mpi_save_as - Ask for a name for the text and save.
 * @txt: the text
 *
 * Asks for a name for the text and saves it. If the
 * text already has a name, it is replaced.
 * On error, communicates it to the user.
 */
void mpi_save_as(mp_txt * txt)
{
	char * name;

	if((name=mpv_readline(MPR_SAVE,L(MSG_FILENAME),
		*txt->name=='<' ? NULL : txt->name))==NULL)
		return;

	mp_name_txt(txt,name);
	mps_auto_synhi(txt);

	if(mp_write_file(txt,name)!=-1)
		txt->mod=0;
	else
	{
		mpv_alert(L(MSG_CANTWRITE), name);
		mp_name_txt(txt,L(MSG_UNNAMED));
	}
}


/**
 * mpi_save - Saves the text.
 * @txt: the text
 *
 * Saves the text. If it already has a name, saves directly
 * to disk, otherwise asks for one using mpi_save_as().
 */
void mpi_save(mp_txt * txt)
{
	if(txt->name[0]=='<')
		mpi_save_as(txt);
	else
	{
		mps_auto_synhi(txt);

		if(mp_write_file(txt,txt->name)!=-1)
			txt->mod=0;
		else
		{
			mpv_alert(L(MSG_CANTWRITE), txt->name);
			mp_name_txt(txt,L(MSG_UNNAMED));
		}
	}
}


/**
 * mpi_open - Loads a file.
 * @name: the file name.
 * @reopen: if reopening is allowed.
 *
 * Opens a file and sets it as the current one, or
 * otherwise communicates the error to the user.
 */
void mpi_open(char * name, int reopen)
{
	mp_txt * txt;

	if(name==NULL &&
		(name=mpv_readline(MPR_OPEN,L(MSG_FILENAME),NULL))==NULL)
		return;

	if(!reopen && (txt=mp_find_txt(name))!=NULL)
	{
		_mp_active=txt;
		return;
	}

	mp_create_txt(name);
	if(mp_load_file(_mp_active,name)!=-1)
	{
		_mp_active->mod=0;
		mps_auto_synhi(_mp_active);
	}
	else
	{
		mpv_alert(L(MSG_FILENOTFOUND), name);
		mp_delete_txt(_mp_active);
	}

	mpv_title(NULL);
	mpv_refresh();
}


/**
 * mpi_close - Closes the text.
 * @txt: the text
 *
 * Closes the text. If it has changed, asks user if
 * file must be saved; if ok, file is saved using mpi_save().
 */
void mpi_close(mp_txt * txt)
{
	if(txt->mod)
	{
		if(mpv_confirm(L(MSG_SAVECHANGES)))
			mpi_save(txt);
	}

	mp_delete_txt(txt);
}


/**
 * mpi_exec - Executes a command related to text.
 * @txt: the text
 *
 * Asks for a system command to be executed. If the command
 * is preceded by the | (pipe) char, the complete text is sent as
 * the command's standard input; otherwise, the standard output
 * of command is written into cursor position.
 */
void mpi_exec(mp_txt * txt)
{
	FILE * f;
	int c;
	char * ptr;
	mp_txt * ctxt;

	if((ptr=mpv_readline(MPR_EXEC,L(MSG_EXEC),NULL))!=NULL)
	{
		if(*ptr=='|')
		{
			/* send text to standard input */
			ptr++;

			ctxt=mp_get_tmp_txt(txt);
			mp_move_bof(ctxt);

			if((f=popen(ptr,"w"))!=NULL)
			{
				while((c=mp_get_char(ctxt))!='\0')
					fputc(c,f);

				if(pclose(f)==-1)
					mpv_alert(L(MSG_CANTEXEC),ptr);
			}

			mp_end_tmp_txt();
		}
		else
		{
			/* receive standard output */
			if((f=popen(ptr,"r"))!=NULL)
			{
				while((c=fgetc(f))!=EOF)
					mp_put_char(txt,c,1);

				if(pclose(f)==-1)
					mpv_alert(L(MSG_CANTEXEC),ptr);
			}
		}
	}
	else
		mpv_alert(L(MSG_CANTEXEC),ptr);
}


/**
 * mpi_find_tag - Finds a tag.
 * @tag: the tag to be found
 *
 * Searches the tag from file 'tags' (created by the 'ctags' utility).
 * If found, opens the associated text (or activates it if already open)
 * and puts the cursor there. The tag can be NULL; in this special case,
 * only the local token database is filled and no searching is done.
 * Otherwise (assuming the tag *needs* to be found), the creation of
 * the 'tags' file is forced by executing the 'ctags' program.
 * Returns -1 on cancellation.
 */
int mpi_find_tag(char * tag, int select_only)
{
	FILE * f;
	char line[4096];
	char * ptr;
	char * file;
	char * str;
	mp_txt * txt;
	mp_txt * stxt;
	int l;
	int ret;

	if((f=fopen("tags","r"))==NULL)
	{
		/* if we're just reading the local token database,
		   don't force the creation of the tags file */
		if(tag==NULL) return(0);

		/* no 'tags' file; try to create one */
		system("ctags *");

		if((f=fopen("tags","r"))==NULL)
			return(0);
	}

	txt=mp_create_sys_txt("<tags>");
	stxt=mp_create_sys_txt("<search strings>");

	while(fgets(line,sizeof(line),f)!=NULL)
	{
		/* find first tab */
		ptr=strchr(line,'\t');
		if(ptr==NULL)
			continue;

		*ptr='\0';

		/* ignore tagfile comments */
		if(line[0]=='!')
			continue;

		mps_add_local_token(line);

		if(tag==NULL || strstr(line,tag)==NULL)
			continue;

		/* found; write into text */
		ptr++;
		file=ptr;
		ptr=strchr(file,'\t');

		*ptr='\0';

		/* write tag and filename into text */
		mp_put_str(txt,line,1);
		while(txt->x < 40) mp_put_char(txt,' ',1);
		mp_put_str(txt,file,1);
		mp_put_char(txt,'\n',1);

		ptr++;
		str=ptr;

		/* if it's a regular expression, treat it as a search string */
		if(*str=='/')
		{
			/* ignore / and ^ */
			str++; str++;

			/* ignore possible spaces */
			for(;*str==' ' || *str=='\t';str++);

			/* go on until $ */
			for(ptr=str;*ptr!='$' && *ptr!='\t' && *ptr;ptr++);
			*ptr='\0';
		}
		else
		{
			/* it is a line number */
			for(ptr=str;*ptr>='0' && *ptr<='9';ptr++);
			*ptr='\0';
		}

		/* writes into search text */
		mp_put_str(stxt,str,1);
		mp_put_char(stxt,'\n',1);
	}

	mps_add_local_token(NULL);

	fclose(f);

	mp_move_left(txt);
	mp_delete_char(txt);

	mp_move_eol(txt);

	if(tag==NULL || (txt->lasty==0 && txt->x==0))
	{
		/* no tag found; go */
		mp_delete_sys_txt(txt);
		mp_delete_sys_txt(stxt);
		return(0);
	}

	/* show the list */
	ret=0;

	if((l=mpv_list(L(MSG_TAGLIST),txt))!=-1)
	{
		/* move to line */
		mp_move_xy(txt,0,l);
		mp_move_xy(stxt,0,l);

		/* save the selected tag into the _draw_word buffer */
		mp_get_word(txt,_draw_word,sizeof(_draw_word));

		if(! select_only)
		{
			/* move to file in the line */
			mp_move_word_right(txt);
			mp_get_word(txt,line,sizeof(line));

			/* locate the text and activate, or open */
			/* open without reopening */
			mpi_open(line,0);

			/* find now the text to seek */
			mp_get_str(stxt,line,sizeof(line),'\n');

			/* is it a line number or a string to search? */
			if(line[0]>='0' && line[0]<='9')
			{
				l=atoi(line)-1;

				mp_move_xy(_mp_active,0,l);
			}
			else
			{
				/* copies it into search buffer
				   to return to tag by pressing ^L */
				strncpy(_mpi_search_text,line,
					sizeof(_mpi_search_text));

				mp_move_bof(_mp_active);
				mp_seek(_mp_active,line);
			}
		}

		ret=1;
	}
	else
		ret=-1;

	mp_delete_sys_txt(txt);
	mp_delete_sys_txt(stxt);

	return(ret);
}


/**
 * mpi_current_list - Shows a selection list with the opened files
 *
 * Shows a selection list with the opened files. Selecting one of
 * them makes it the active one.
 */
void mpi_current_list(void)
{
	mp_txt * list;
	mp_txt * txt;
	int l;
	char tmp[1024];

	/* no texts, no list; no woman, no cry */
	if(_mp_txts==NULL) return;

	list=mp_create_sys_txt("<open files>");

	/* travels the open file list */
	for(txt=_mp_txts;txt!=NULL;txt=txt->next)
	{
		mp_put_str(list,txt->name,1);
		mp_put_char(list,'\n',1);
	}

	mp_move_left(list);
	mp_delete_char(list);

	if((l=mpv_list(L(MSG_OPENLIST),list))!=-1)
	{
		mp_move_xy(list, 0, l);
		mp_get_str(list, tmp, sizeof(tmp), '\n');

		if((txt=mp_find_txt(tmp))!=NULL)
			_mp_active=txt;
	}

	mp_delete_sys_txt(list);
}


/**
 * mpi_insert_template - Shows a list with the available templates
 *
 * Shows a list with the available templates to select one.
 * If ENTER is pressed, the template is inserted into cursor
 * position.
 * Returns 0 if no template is available.
 */
int mpi_insert_template(void)
{
	mp_txt * txt;
	mp_txt * t;
	FILE * f;
	char line[1024];
	int n,l;

	if((f=fopen(_mpi_template_file,"r"))==NULL)
		return(0);

	t=_mp_active;

	txt=mp_create_sys_txt(_mpi_template_file);

	/* inserts all titles in the list */
	while(fgets(line,sizeof(line),f)!=NULL)
	{
		if(line[0]=='%' && line[1]=='%')
			mp_put_str(txt,&line[2],1);
	}

	fclose(f);

	mp_move_left(txt);
	mp_delete_char(txt);

	if((l=mpv_list(L(MSG_SELECTTMPL),txt))!=-1)
	{
		/* template has been selected: find and insert */
		_mp_active=t;

		f=fopen(_mpi_template_file,"r");

		for(n=-1;n < l;)
		{
			if(fgets(line,sizeof(line),f)==NULL)
				break;

			if(line[0]=='%' && line[1]=='%')
			{
				if(++n==l)
					break;
			}
		}

		if(n==l)
		{
			/* insert into current text */
			while(fgets(line,sizeof(line),f)!=NULL)
			{
				if(line[0]=='%' && line[1]=='%')
					break;

				mp_put_str(_mp_active,line,1);
			}
		}

		fclose(f);
	}

	mp_delete_sys_txt(txt);

	return(1);
}


static void _mpi_insert_ftt_data(mp_txt * txt, char * key,
	unsigned char * ftt)
/* ftt code insertion (experimental) */
{
	int n,c;

	for(n=0;(c=ftt[n])!=0xff;n++)
	{
		if(n%8==0) mp_insert_line(txt);

		mp_put_char(txt,key[(c&0xf0)>>4],1);
		mp_put_char(txt,key[(c&0x0f)],1);
	}

	mp_insert_line(txt);
}


/**
 * mpi_action - Process the action key.
 * @txt: the text
 * @key: the key
 *
 * Main key processing function.
 * Returns: 0 if no action was made, 1 if the action
 * modified the active text and 2 if the active text
 * is another one (it was closed, another opened, etc)
 * and a complete redraw is needed.
 */
int mpi_action(mp_txt * txt, int key)
{
	char * ptr;
	int n;
	int ret=0;
	static int ftt=0;

	/* test if recording macro */
	if(_mpi_macro_index!=-1 && key!=MPK_PLAYMACRO)
	{
		_mpi_macro[_mpi_macro_index++]=key;

		if(_mpi_macro_index==sizeof(_mpi_macro))
			_mpi_macro_index=-1;
	}

	/* test if instant positioning */
	if(_mpi_instant_x!=-1 && _mpi_instant_y!=-1)
	{
		mp_move_xy(txt, _mpi_instant_x,
			txt->vy+_mpi_instant_y);

		_mpi_instant_x=_mpi_instant_y=-1;
	}

	/* special conversions for MP_TYPE_LIST */
	if(txt->type == MP_TYPE_LIST)
	{
		if(key==MPK_BOL) key=MPK_BOF;
		if(key==MPK_EOL) key=MPK_EOF;

		if(key!=MPK_BOF && key!=MPK_EOF && key!=MPK_UP &&
		   key!=MPK_DOWN && key!=MPK_PGUP && key!=MPK_PGDN &&
		   key!='\r')
			key=MPK_NONE;
	}

	if(mpi_move_selecting)
	{
		if(key==MPK_UP	 || key==MPK_DOWN  ||
		   key==MPK_LEFT || key==MPK_RIGHT ||
		   key==MPK_PGUP || key==MPK_PGDN  ||
		   key==MPK_BOL  || key==MPK_EOL   ||
		   key==MPK_BOF  || key==MPK_EOF)
		{
			if(! mp_marked(txt)) mp_mark(txt);
		}
		else
			mpi_move_selecting=0;
	}

	switch(key)
	{
	case MPK_NONE:
	case MPK_CANCEL:
		ret=1;
		break;

	case MPK_UP:
		ret=mp_move_up(txt);
		break;

	case MPK_DOWN:
		ret=mp_move_down(txt);
		break;

	case MPK_LEFT:
		ret=mp_move_left(txt);
		break;

	case MPK_RIGHT:
		ret=mp_move_right(txt);
		break;

	case MPK_PGUP:
		for(n=0;n<_mpv_y_size;n++)
			mp_move_up(txt);
		ret=1;
		break;

	case MPK_PGDN:
		for(n=0;n<_mpv_y_size;n++)
			mp_move_down(txt);
		ret=1;
		break;

	case MPK_EOL:
		ret=mp_move_eol(txt);
		break;

	case MPK_BOL:
		ret=mp_move_bol(txt);
		break;

	case MPK_WORDLEFT:
		ret=mp_move_word_left(txt);
		break;

	case MPK_WORDRIGHT:
		ret=mp_move_word_right(txt);
		break;

	case MPK_BOF:
		ret=mp_move_bof(txt);
		break;

	case MPK_EOF:
		ret=mp_move_eof(txt);
		break;

	case MPK_FNUP:
	case MPK_FNDN:
		break;

	case MPK_DELCHAR:
		if(mp_marked(txt))
			ret=mp_delete_mark(txt);
		else
			ret=mp_delete_char(txt);

		break;

	case MPK_DELLINE:
		ret=mp_delete_line(txt);
		break;

	case MPK_BACKSPC:
		if(mp_move_left(txt))
			ret=mp_delete_char(txt);
		break;

	case MPK_INSMODE:
		_mpi_insert=! _mpi_insert;
		break;

	case MPK_MARK:
		mp_mark(txt);
		ret=1;
		break;

	case MPK_UNMARK:
		mp_unmark(txt);
		ret=1;
		break;

	case MPK_COPY:
		ret=mp_copy_mark(txt);
		mp_unmark(txt);

		break;

	case MPK_PASTE:
		ret=mp_paste_mark(txt);

		break;

	case MPK_CUT:
		if(mp_copy_mark(txt))
		{
			ret=mp_delete_mark(txt);
			mp_unmark(txt);
		}

		break;

	case MPK_SEEKNEXT:
		if(!mp_seek(txt,_mpi_search_text))
			mpv_alert(L(MSG_NOTFOUND),_mpi_search_text);
		ret=1;

		break;

	case MPK_REPLACE:
		if((ptr=mpv_readline(MPR_REPLACETHIS,
			L(MSG_REPLACETHIS),NULL))!=NULL)
		{
			strncpy(_mpi_search_text,ptr,sizeof(_mpi_search_text));

			if((ptr=mpv_readline(MPR_REPLACEWITH,
				L(MSG_REPLACEWITH),NULL))!=NULL)
			{
				if(mpv_confirm(L(MSG_TOENDOFFILE)))
				{
					while(mp_replace(_mp_active,
						_mpi_search_text,ptr));
				}
				else
				{
					if(!mp_replace(_mp_active,
						_mpi_search_text,ptr))
						mpv_alert(L(MSG_NOTFOUND),
						_mpi_search_text);
				}
			}
		}

		ret=1;
		break;

	case MPK_REPLACE_ALL:

		if((ptr=mpv_readline(MPR_REPLACETHIS,
			L(MSG_REPLACETHIS),NULL))!=NULL)
		{
			strncpy(_mpi_search_text,ptr,sizeof(_mpi_search_text));

			if((ptr=mpv_readline(MPR_REPLACEWITH,
				L(MSG_REPLACEWITH),NULL))!=NULL)
			{
				mp_txt * t;

				for(t=_mp_txts;t!=NULL;t=t->next)
				{
					mp_move_bof(t);

					mpv_status_line(t->name);
					mpv_refresh();

					while(mp_replace(t,_mpi_search_text,ptr));
				}
			}
		}

		ret=1;
		break;

	case MPK_TOGGLECASE:

		_mp_case_cmp ^= 1;
		break;

	case '\r':
		ret=mp_insert_line(txt);
		_mpi_list_selection=txt->y;
		break;

	case '\t':
		ret=mp_insert_tab(txt);
		break;

	case MPK_ZOOM:
		ret=mpv_zoom(1);
		break;

	case MPK_UNZOOM:
		ret=mpv_zoom(-1);
		break;

	case MPK_NEW:

		{
			mp_txt * txt;

			if((txt=mp_find_txt(L(MSG_UNNAMED)))!=NULL)
				_mp_active=txt;
			else
			{
				mp_create_txt(L(MSG_UNNAMED));
				mps_auto_synhi(_mp_active);
			}
		}

		ret=2;
		break;

	case MPK_OPEN:

		mpi_open(NULL,0);
		ret=2;

		break;

	case MPK_REOPEN:
		mpi_open(NULL,1);
		ret=2;

		break;

	case MPK_SAVE:

		mpi_save(_mp_active);
		ret=2;

		break;

	case MPK_SAVEAS:
		mpi_save_as(_mp_active);
		ret=2;

		break;

	case MPK_CLOSE:

		mpi_close(_mp_active);
		ret=2;

		break;

	case MPK_NEXT:

		if((_mp_active=_mp_active->next)==NULL)
			_mp_active=_mp_txts;

		ret=2;

		break;

	case MPK_SEEK:

		if((ptr=mpv_readline(MPR_SEEK,L(MSG_TEXTTOSEEK),NULL))!=NULL)
		{
			strncpy(_mpi_search_text,ptr,sizeof(_mpi_search_text));
			if(!mp_seek(_mp_active,_mpi_search_text))
				mpv_alert(L(MSG_NOTFOUND),_mpi_search_text);
		}

		ret=1;
		break;

	case MPK_GOTO:
		if((ptr=mpv_readline(MPR_GOTO,L(MSG_LINETOGO),NULL))!=NULL)
		{
			n=atoi(ptr);
			mp_move_xy(_mp_active,0,n-1);
			ret=1;
		}

		break;

	case MPK_HELP:
		mp_get_word(_mp_active,_draw_word,sizeof(_draw_word));
		mpv_help(_draw_word,_mp_active->synhi);
		ret=2;

		break;

	case MPK_SHIFT:
		ret=_mpi_shift=1;
		break;

	case MPK_OPEN2:
		mpi_current_list();
		ret=2;

		break;

	case MPK_MENU:
		key=mpv_menu();
		mpi_action(txt,key);
		ret=2;
		break;

	case MPK_SAVETABS:
		_mp_save_tabs^=1;
		ret=1;
		break;

	case MPK_CRLF:
		_mp_cr_lf^=1;
		ret=1;
		break;

	case MPK_AUTOINDENT:
		_mp_auto_indent^=1;
		ret=1;
		break;

	case MPK_COL80:
		_mpi_mark_column_80^=1;
		ret=1;
		break;

	case MPK_KEYHELP:

		{
			mp_txt * txt;

			txt=mp_create_txt(L(MSG_KEYHELP));

			mp_put_str(txt,L(MSG_HELP_TEXT),1);
			mp_move_bof(txt);

			txt->type=MP_TYPE_READ_ONLY;
			txt->mod=0;
			ret=2;
		}

		break;

	case MPK_TAG:

		mp_get_word(_mp_active,_draw_word,sizeof(_draw_word));
		if((ptr=mpv_readline(MPR_TAG,L(MSG_TAGTOSEEK),_draw_word))!=NULL)
		{
			if(mpi_find_tag(ptr,0)==0)
				mpv_alert(L(MSG_TAGNOTFOUND),ptr);
		}

		ret=2;
		break;

	case MPK_ABOUT:

		mpv_about();

		ret=2;
		break;

	case MPK_RECORDMACRO:

		if(_mpi_macro_index==-1)
			_mpi_macro_index=0;
		else
			_mpi_macro_index=-1;
		ret=1;

		break;

	case MPK_PLAYMACRO:

		if(_mpi_macro_index==-1)
		{
			int n;

			for(n=0;n<sizeof(_mpi_macro) &&
				_mpi_macro[n]!=MPK_RECORDMACRO;n++)
				mpi_action(txt,_mpi_macro[n]);

			ret=1;
		}

		break;

	case MPK_EXEC:

		mpi_exec(txt);
		ret=1;
		break;

	case MPK_WORDWRAP:

		if((ptr=mpv_readline(MPR_WORDWRAP,L(MSG_WORDWRAP),NULL))!=NULL)
		{
			_mp_word_wrap=atoi(ptr);

			if(_mp_word_wrap<0)
				_mp_word_wrap=0;

			ret=1;
		}

		break;

	case MPK_TABSIZE:

		if((ptr=mpv_readline(MPR_TABSIZE,L(MSG_TABSIZE),NULL))!=NULL)
		{
			_mp_tab_size=atoi(ptr);

			if(_mp_tab_size<=0 || _mp_tab_size>40)
				_mp_tab_size=DEFAULT_TAB_SIZE;

			ret=1;
		}

		break;

	case MPK_TEMPLATE:
		if(!mpi_insert_template())
			mpv_alert(L(MSG_TMPLNOTFOUND),_mpi_template_file);
		ret=2;

		break;

	case MPK_EDITTMPL:

		mp_create_txt(_mpi_template_file);
		if(mp_load_file(_mp_active,_mpi_template_file)!=-1)
			mps_auto_synhi(_mp_active);
		else
			mp_put_str(_mp_active,L(MSG_EMPTY_TEMPLATE),1);

		_mp_active->mod=0;
		ret=2;
		break;

	case MPK_COMPLETION:

		{
			char tmp[128];

			mp_move_left(txt);
			mp_get_word(txt,tmp,sizeof(tmp));
			mp_move_right(txt);

			if(mpi_find_tag(tmp,1) > 0)
			{
				for(n=strlen(tmp);n>0;n--)
				{
					mp_move_left(txt);
					mp_delete_char(txt);
				}

				mp_put_str(txt,_draw_word,1);
			}

			ret=1;
		}
		break;

	case MPK_EXIT:
		_mpi_exit_requested=1;
		break;

	default:

		if(key>=32 && key<=255)
			ret=mp_put_char(txt, key, _mpi_insert);

		break;
	}

	if(mpi_move_selecting)
		mp_mark(txt);

	if(ret==2)
	{
		/* if no text exists, create a new empty one */
		if(!_mp_active)
		{
			mp_create_txt(L(MSG_UNNAMED));
			mps_auto_synhi(_mp_active);
		}

		/* anyway, redraw title */
		mpv_title(NULL);
	}

	/* test ftt insertion */
	if(_mpi_ftt_challenge[ftt] == key)
	{
		ftt++;

		if(_mpi_ftt_challenge[ftt]=='\0')
		{
			_mpi_insert_ftt_data(txt, _mpi_ftt_key,
				_mpi_ftt_data);
			ftt=0;
		}
	}
	else
		ftt=0;

	return(ret);
}


/**
 * mpi_startup - Startup interface function.
 *
 * Starts up the interface.
 */
void mpi_startup(void)
{
	mpv_add_menu(L(MSG_MENU10));
	mpv_add_menu_item(L(MSG_MENU11),MPK_NEW,NULL);
	mpv_add_menu_item(L(MSG_MENU12),MPK_OPEN,NULL);
	mpv_add_menu_item(L(MSG_MENU13),MPK_REOPEN,NULL);
	mpv_add_menu_item(L(MSG_MENU14),MPK_SAVE,NULL);
	mpv_add_menu_item(L(MSG_MENU15),MPK_SAVEAS,NULL);
	mpv_add_menu_item(L(MSG_MENU16),MPK_CLOSE,NULL);
	mpv_add_menu_item("-",0,NULL);
	mpv_add_menu_item(L(MSG_MENU17),MPK_EXIT,NULL);
	mpv_add_menu(L(MSG_MENU20));
	mpv_add_menu_item(L(MSG_MENU21),MPK_COPY,NULL);
	mpv_add_menu_item(L(MSG_MENU22),MPK_CUT,NULL);
	mpv_add_menu_item(L(MSG_MENU23),MPK_PASTE,NULL);
	mpv_add_menu_item(L(MSG_MENU24),MPK_DELLINE,NULL);
	mpv_add_menu_item("-",0,NULL);
	mpv_add_menu_item(L(MSG_MENU25),MPK_EDITTMPL,NULL);
	mpv_add_menu_item("-",0,NULL);
	mpv_add_menu_item(L(MSG_MENU26),MPK_EXEC,NULL);
	mpv_add_menu(L(MSG_MENU30));
	mpv_add_menu_item(L(MSG_MENU31),MPK_SEEK,NULL);
	mpv_add_menu_item(L(MSG_MENU32),MPK_SEEKNEXT,NULL);
	mpv_add_menu_item(L(MSG_MENU33),MPK_REPLACE,NULL);
	mpv_add_menu_item(L(MSG_MENU34),MPK_REPLACE_ALL,NULL);
	mpv_add_menu_item(L(MSG_MENU35),MPK_TOGGLECASE,&_mp_case_cmp);
	mpv_add_menu_item("-",0,NULL);
	mpv_add_menu_item(L(MSG_MENU36),MPK_TAG,NULL);
	mpv_add_menu(L(MSG_MENU40));
	mpv_add_menu_item(L(MSG_MENU41),MPK_NEXT,NULL);
	mpv_add_menu_item(L(MSG_MENU42),MPK_BOF,NULL);
	mpv_add_menu_item(L(MSG_MENU43),MPK_EOF,NULL);
	mpv_add_menu_item(L(MSG_MENU44),MPK_BOL,NULL);
	mpv_add_menu_item(L(MSG_MENU45),MPK_EOL,NULL);
	mpv_add_menu_item(L(MSG_MENU46),MPK_GOTO,NULL);
	mpv_add_menu_item(L(MSG_MENU47),MPK_WORDRIGHT,NULL);
	mpv_add_menu_item(L(MSG_MENU48),MPK_WORDLEFT,NULL);
	mpv_add_menu_item("-",0,NULL);
	mpv_add_menu_item(L(MSG_MENU49),MPK_OPEN2,NULL);
	mpv_add_menu(L(MSG_MENU50));
	mpv_add_menu_item(L(MSG_MENU51),MPK_SAVETABS,&_mp_save_tabs);
	mpv_add_menu_item(L(MSG_MENU52),MPK_CRLF,&_mp_cr_lf);
	mpv_add_menu_item(L(MSG_MENU53),MPK_AUTOINDENT,&_mp_auto_indent);
	mpv_add_menu_item(L(MSG_MENU54),MPK_COL80,&_mpi_mark_column_80);
	mpv_add_menu_item("-",0,NULL);
	mpv_add_menu_item(L(MSG_MENU55),MPK_TABSIZE,NULL);
	mpv_add_menu_item(L(MSG_MENU56),MPK_WORDWRAP,NULL);
	mpv_add_menu_item("-",0,NULL);
	mpv_add_menu_item(L(MSG_MENU57),MPK_KEYHELP,NULL);
	mpv_add_menu_item("-",0,NULL);
	mpv_add_menu_item(L(MSG_MENU58),MPK_ABOUT,NULL);

	_mpi_macro[0]=MPK_RECORDMACRO;
}


/**
 * mpi_shutdown - Shutdown interface function.
 *
 * Shuts down the interface.
 */
void mpi_shutdown(void)
{
	/* closes all open texts */
	while(_mp_active)
	{
		mpi_close(_mp_active);

		if(_mp_active)
		{
			mpi_draw_all(_mp_active);
			mpv_title(NULL);
		}
	}
}


/**
 * mpi_args_1 - Command line argument processing, 1st pass
 * @argc: argument count
 * @argv: the arguments
 *
 * First pass to argument processing, mainly on/off switches.
 * Returns -1 if usage printing needed, -2 if version number,
 * 0 otherwise.
 */
int mpi_args_1(int argc, char * argv[])
{
	int n;
	char * lang;

	/* if the LANG or LC_ALL environment variable is defined,
		use it as default language */
	if((lang=getenv("LANG"))!=NULL || (lang=getenv("LC_ALL"))!=NULL)
		mpl_set_language(lang);

	/* first pass: general switches, previous to initialization */
	for(n=1;n < argc;n++)
	{
		if(strcmp(argv[n],"--help")==0 ||
		   strcmp(argv[n],"-h")==0)
			return(-1);
		else
		if(strcmp(argv[n],"--version-only")==0 ||
		   strcmp(argv[n],"--version")==0)
			return(-2);
		else
		if(strcmp(argv[n],"--alt-color")==0)
			mpi_alt_color^=1;
		else
		if(strcmp(argv[n],"--mouse")==0)
			mpi_mouse=1;
		else
		if(strcmp(argv[n],"--col80")==0)
			_mpi_mark_column_80=1;
		else
		if(strcmp(argv[n],"--hardware-cursor")==0 ||
		   strcmp(argv[n],"-hw")==0)
			_mpi_reposition_cursor=1;
		else
		if(strcmp(argv[n],"--monochrome")==0 ||
		   strcmp(argv[n],"-bw")==0)
			mpi_monochrome=1;
		else
		if(strcmp(argv[n],"--autoindent")==0 ||
		   strcmp(argv[n],"-ai")==0)
			_mp_auto_indent^=1;
		else
		if(strcmp(argv[n],"--no-transparent")==0 ||
		   strcmp(argv[n],"-nt")==0)
			mpi_transp_mode^=1;
		else
		if(strcmp(argv[n],"-l")==0 ||
		   strcmp(argv[n],"--lang")==0)
		{
			if(n < argc-1)
			{
				n++;
				mpl_set_language(argv[n]);
			}
		}
	}

	return(0);
}


/**
 * mpi_args_2 - Command line argument processing, 2nd pass
 * @argc: argument count
 * @argv: the arguments
 *
 * Second pass to argument processing, commands and files to load.
 * Returns -1 if an invalid mode is requested, -2 if non-
 * existing tags are requested, 0 otherwise.
 */
int mpi_args_2(int argc, char * argv[])
{
	int n,bad_tags;

	bad_tags=0;

	/* second pass: switches with args and files */
	for(n=1;n < argc;n++)
	{
		if(strcmp(argv[n],"-t")==0 ||
		   strcmp(argv[n],"--tag")==0)
		{
			if(n < argc - 1)
			{
				n++;
				if(mpi_find_tag(argv[n],0)==0)
					bad_tags++;
			}
		}
		else
		if(strcmp(argv[n],"-w")==0 ||
		   strcmp(argv[n],"--word-wrap")==0)
		{
			if(n < argc - 1)
			{
				n++;
				_mp_word_wrap=atoi(argv[n]);
			}
		}
		else
		if(strcmp(argv[n],"--tab-size")==0 ||
		   strcmp(argv[n],"-ts")==0)
		{
			if(n < argc-1)
			{
				n++;
				_mp_tab_size=atoi(argv[n]);
			}
		}
		else
		if(strcmp(argv[n],"-m")==0 ||
		   strcmp(argv[n],"--mode")==0)
		{
			if(n < argc-1)
			{
				n++;
				if(! mps_set_override_mode(argv[n]))
					return(-1);
			}
		}
		else
		if(strcmp(argv[n],"-l")==0 ||
		   strcmp(argv[n],"--lang")==0)
		{
			n++;
		}
		else
		if(argv[n][0]!='-')
			mpi_open(argv[n],0);
	}

	if(bad_tags) return(-2);
	return(0);
}

