/* 
 *   Creation Date: <1999/11/16 00:49:26 samuel>
 *   Time-stamp: <2002/11/24 15:14:05 samuel>
 *   
 *	<init.c>
 *	
 *	Newworld booter (Mac OS 8.6 and later)
 *   
 *   Copyright (C) 1999, 2000, 2001, 2002 Samuel & David Rydh 
 #      (samuel@ibrium.se, dary@lindesign.se)
 *   
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation
 *   
 */

#include "of.h"

#include "ofmem.h"
#include "prom.h"
#include "phandles.h"


/************************************************************************/
/*	phandles							*/
/************************************************************************/

static struct {
	char *node, *prop;
	char *real_path;
} phfixtab[] = {
	{ "/chosen", "interrupt-controller", "/pci/pci-bridge/mac-io/interrupt-controller" },
	{ NULL, NULL, NULL }
};

/*
 * Fix the phandle identifier for some nodes (the original phandle id is not 
 * normally available, thus we must do this manually).
 */
static void
fix_phandles( void )
{
	mol_phandle_t new_ph;
	int i;

	for( i=0; phfixtab[i].node; i++ ) {
		if( prom_get_prop_by_path( phfixtab[i].node, phfixtab[i].prop, (char*)&new_ph, 4 ) != 4 ) {
			printm("Property %s:%s not found or invalid\n", phfixtab[i].node, phfixtab[i].prop );
			continue;
		}
		if( prom_change_phandle( prom_find_device(phfixtab[i].real_path), new_ph ) < 0 )
			printm("prom_change_phandle returned an error\n");
	}
}


/************************************************************************/
/*	ihandles							*/
/************************************************************************/

/* ihandles:
 *	top 16 bit == molih identifier
 *	low 16 bits == file descriptor
 */

static struct {
	char	*node, *prop;
	int     molih;
	char	*path;
} preopen_ih[] = {
	{ "/chosen", "memory",	molih_memory,	"/memory" },
	{ "/chosen", "mmu",	molih_mmu,	"/cpus/@0" },
	{ "/chosen", "stdin",	molih_kbd,	"/psuedo-hid/keyboard" },
	{ "/chosen", "stdout",  molih_stdout,	NULL /*?*/ },
	{ "/chosen", "nvram",	molih_nvram,	"/pci/pci-bridge/mac-io/nvram" },
};

static struct {
	int	molih;
	char	*path;
} node_to_molih_tab[] = {
	{ molih_memory, 	"/memory" },
	{ molih_mmu, 		"/cpus/@0" },
	{ molih_nvram,		"/pci/pci-bridge/mac-io/nvram" },
	{ molih_kbd,		"/psuedo-hid/keyboard" },
	{ molih_rtas,		"/rtas" },
	{ molih_pseudofs,	"/pseudo-blk/disk" },
	{ molih_blockdev,	"/mol-blk/disk" },
	{ molih_powermgt,	"/pci/pci-bridge/mac-io/power-mgt" },
};

static struct {
	int	molih;
	char	*device_type;
} devtype_to_molih_tab[] = {
	{ molih_video,		"display" },
};


#define MAX_NUM_IHANDLES	64

/* node <-> molih table */
typedef struct {
	ulong		ihandle;	/* high word is the molih number, low word fd */
	ulong		phandle;
	void		*data;
} ih_entry_t;

static struct {
	ih_entry_t	ihtab[ MAX_NUM_IHANDLES ];
	int		num_ihandles;
	int		next_molih;
	int		next_fd;
	void		*data;		/* data associated to this instance */
} ih;

/* associate molih, ihandle and phandle */
int
allocate_ihandle( int molih, mol_phandle_t phandle, void *data )
{
	ih_entry_t *ihp = &ih.ihtab[ ih.num_ihandles ];
	int ih2;
	
	if( ih.num_ihandles >= MAX_NUM_IHANDLES ) {
		printm("MAX_NUM_IHANDLES exceeded!\n");
		return 0;
	}
	if( !molih ) {
		if( !(molih=phandle_to_molih( phandle )) )
			molih = ih.next_molih++;
	} else {
		/* make certain that phandle is not bound to a different molih already */
		if( (ih2=phandle_to_molih(phandle)) && ih2 != molih ) {
			printm("Error: conflicting molih bindings\n");
			molih = ih2;
		}
	}
	ih.num_ihandles++;
	ihp->ihandle = (molih << 16) | ih.next_fd++;
	ihp->phandle = phandle;
	ihp->data = data;

	/* printm("MOLIH: %08X IHANDLE: %08lX PHANDLE: %08lX\n", molih, ihp->ihandle, ihp->phandle ); */
	return ihp->ihandle;
}

void
free_ihandle( int ihandle )
{
	int i;
	for( i=0; i<ih.num_ihandles && ih.ihtab[i].ihandle != ihandle; i++ )
		break;
	if( i<ih.num_ihandles ) {
		memmove( &ih.ihtab[i], &ih.ihtab[i+1], sizeof(ih.ihtab[0]) );
		ih.num_ihandles--;
	} else {
		printm("free_ihandle: ihandle unregistered\n" );
	}
}

mol_phandle_t
ihandle_to_phandle( int ihandle )
{
	int i;

	for( i=0; i<ih.num_ihandles; i++ )
		if( ih.ihtab[i].ihandle  == ihandle )
			return ih.ihtab[i].phandle;
	return 0;
}

int
ihandle_to_molih( int ihandle )
{
	return ihandle >> 16;
}

int
phandle_to_molih( mol_phandle_t ph )
{
	char buf[64];
	int i;

	if( ph == -1 )
		return 0;

	for( i=0; i < sizeof(node_to_molih_tab)/sizeof(node_to_molih_tab[0]); i++ )
		if( prom_find_device(node_to_molih_tab[i].path) == ph )
			return node_to_molih_tab[i].molih;

	if( prom_get_prop(ph, "device_type", buf, sizeof(buf)) <= 0 )
		return 0;
	
	for( i=0; i < sizeof(devtype_to_molih_tab)/sizeof(devtype_to_molih_tab[0]); i++ )
		if( !strcmp( buf, devtype_to_molih_tab[i].device_type ) )
			return devtype_to_molih_tab[i].molih;
	return 0;
}

int
get_ihandle_data( int ihandle, void **retdata )
{
	int i;
	for( i=0; i<ih.num_ihandles; i++ )
		if( ih.ihtab[i].ihandle == ihandle ) {
			if( retdata )
				*retdata = ih.ihtab[i].data;
			return 0;
		}
	return 1;
}

static void
ihandles_init( void )
{
	int i, ph, ihandle;

	ih.next_molih = molih__next_free;
	ih.next_fd = 1;

	/* preopen certain packages (the ihandles are typically found under /chosen) */
	for( i=0; i < sizeof(preopen_ih)/sizeof(preopen_ih[0]); i++ ) {
		ph = prom_find_device( preopen_ih[i].path );
		ihandle = allocate_ihandle( preopen_ih[i].molih, ph, NULL );

		if( (ph=prom_find_device(preopen_ih[i].node)) == -1 )
			ph = prom_create_node(preopen_ih[i].node);
		if( prom_set_prop( ph, preopen_ih[i].prop, (char*)&ihandle, 4 ) != 4 )
			printm("error setting ihandle property\n");
	}
}

static void
ihandles_cleanup( void )
{
	ih.num_ihandles = 0;
}


/************************************************************************/
/*	init / cleanup							*/
/************************************************************************/

void
of_startup( void )
{
	mol_phandle_t ph;
	
	ofmem_init();
	fix_phandles();
	ihandles_init();

	if( (ph=prom_find_device("/rtas")) == -1 )
		printm("Warning: No /rtas node\n");
	else {
		ulong size = 0x1000;
		while( size < (ulong)of_rtas_end - (ulong)of_rtas_start )
			size *= 2;
		prom_set_prop( ph, "rtas-size", (char*)&size, sizeof(size) );
	}
}

void
of_cleanup( void )
{
	ihandles_cleanup();
	ofmem_cleanup();
}
