/*
** Modular Logfile Analyzer
** Copyright 2000 Jan Kneschke <jan@kneschke.de>
**
** Homepage: http://www.modlogan.org
**

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version, and provided that the above
    copyright and permission notice is included with all distributed
    copies of this or derived software.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA

**
** $Id: mrecord.c,v 1.33 2004/03/18 02:31:50 ostborn Exp $
*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>

#include "mconfig.h"
#include "mrecord.h"
#include "misc.h"

/* Record handling */

/*
 * free the record
 *
 * frees all the extension and the actual record pointer
 *
 * @param record the record which should be freed
 */

void mrecord_free_ext(mlogrec *record) {
	if (!record || !record->ext) return;
	
	switch(record->ext_type) {
	case M_RECORD_TYPE_WEB:
		mrecord_free_web(record->ext);
		break;
	case M_RECORD_TYPE_TELECOM:
		mrecord_free_telecom(record->ext);
		break;
	case M_RECORD_TYPE_TRAFFIC:
		mrecord_free_traffic(record->ext);
		break;
	case M_RECORD_TYPE_MAIL:
		mrecord_free_mail(record->ext);
		break;
	case M_RECORD_TYPE_UNSET:
		fprintf(stderr, "%s.%d: can't free unset record: %d\n", __FILE__, __LINE__, record->ext_type);
		break;
	default:
		fprintf(stderr, "%s.%d: Unknown record type: %d\n", __FILE__, __LINE__, record->ext_type);
	}
}

void mrecord_free(mlogrec *record) {
	if (!record) return;

	mrecord_free_ext(record);

	free(record);
}

void mrecord_reset_web(mlogrec_web *record) {
	if (!record) return;

	if (record->ext != NULL) {
		switch(record->ext_type) {
		case M_RECORD_TYPE_WEB_EXTCLF:
			mrecord_free_web_extclf(record->ext);
			break;
		case M_RECORD_TYPE_WEB_SQUID:
			mrecord_free_web_squid(record->ext);
			break;
		case M_RECORD_TYPE_WEB_FTP:
			mrecord_free_web_ftp(record->ext);
			break;
		default:
			fprintf(stderr, "%s.%d: Unknown record type: %d\n", __FILE__, __LINE__, record->ext_type);
		}
	}
	
#define CLEAN(x) x->used = 0;
	CLEAN(record->req_host_ip);
	CLEAN(record->req_host_name);
	CLEAN(record->req_user);
	CLEAN(record->req_url);
	CLEAN(record->req_method);
	CLEAN(record->req_protocol);
	CLEAN(record->req_getvars);
#undef CLEAN
	
	record->ext_type = M_RECORD_TYPE_WEB_UNSET;
	record->ext = NULL;
}

/*
 * reset the record
 *
 * should save some allocations, use it instead of mrecord_init + mrecord_free
 *
 * @param record the record which should be reset
 */

void mrecord_reset(mlogrec *record) {
	if (!record) return;

	if (record->ext != NULL) {
		switch(record->ext_type) {
		case M_RECORD_TYPE_WEB:
			mrecord_reset_web(record->ext);
			break;
		case M_RECORD_TYPE_TELECOM:
			mrecord_free_telecom(record->ext);
			record->ext = NULL;
			record->ext_type = M_RECORD_TYPE_UNSET;
			break;
		case M_RECORD_TYPE_TRAFFIC:
			mrecord_free_traffic(record->ext);
			record->ext = NULL;
			record->ext_type = M_RECORD_TYPE_UNSET;
			break;
		case M_RECORD_TYPE_MAIL:
			mrecord_free_mail(record->ext);
			record->ext = NULL;
			record->ext_type = M_RECORD_TYPE_UNSET;
			break;
		case M_RECORD_TYPE_UNSET:
			fprintf(stderr, "%s.%d: can't free unset record: %d\n", __FILE__, __LINE__, record->ext_type);
			break;
		default:
			fprintf(stderr, "%s.%d: Unknown record type: %d\n", __FILE__, __LINE__, record->ext_type);
		}
	}
	
	record->timestamp = 0;
}

/*
 * init the record
 *
 * allocated the neccesary memory and sets the attributes to init values
 *
 * @return a fresh and clean record
 */

mlogrec *mrecord_init() {
	mlogrec *record = malloc(sizeof(mlogrec));
	
	if (record != NULL) {
		record->timestamp = 0;

		record->ext = NULL;
		record->ext_type = M_RECORD_TYPE_UNSET;
	}

	return record;
}


void mrecord_free_web(mlogrec_web *record) {
	if (!record) return;

	if (record->ext != NULL) {
		switch(record->ext_type) {
		case M_RECORD_TYPE_WEB_EXTCLF:
			mrecord_free_web_extclf(record->ext);
			break;
		case M_RECORD_TYPE_WEB_SQUID:
			mrecord_free_web_squid(record->ext);
			break;
		case M_RECORD_TYPE_WEB_FTP:
			mrecord_free_web_ftp(record->ext);
			break;
		default:
			fprintf(stderr, "%s.%d: Unknown record type: %d\n", __FILE__, __LINE__, record->ext_type);
		}
	}

	buffer_free(record->req_host_ip);
	buffer_free(record->req_host_name);
	buffer_free(record->req_user);
	buffer_free(record->req_url);
	buffer_free(record->req_method);
	buffer_free(record->req_protocol);
	buffer_free(record->req_getvars);

	free(record);
}

mlogrec_web *mrecord_init_web() {
	mlogrec_web *record = malloc(sizeof(mlogrec_web));
	
	if (record != NULL) {
		record->req_host_ip	= buffer_init();
		record->req_host_name	= buffer_init();
		record->req_user	= buffer_init();
		record->req_protocol	= buffer_init();
		record->req_method	= buffer_init();
		record->req_url		= buffer_init();
		record->req_getvars	= buffer_init();
		record->req_status	= 0;
		record->xfersize	= 0;

		record->ext = NULL;
		record->ext_type = 0;
	}

	return record;
}

void mrecord_free_telecom(mlogrec_telecom *record) {
	if (!record) return;

	if (record->ext != NULL) {
		switch(record->ext_type) {
		case M_RECORD_TYPE_TELECOM_INTERNAL:
			mrecord_free_telecom_internal(record->ext);
			break;
		default:
			fprintf(stderr, "%s.%d: Unknown record type: %d\n", __FILE__, __LINE__, record->ext_type);
		}
	}

	if (record->called_number)	free(record->called_number);
	if (record->calling_number)	free(record->calling_number);

	free(record);
}

mlogrec_telecom *mrecord_init_telecom() {
	mlogrec_telecom *record = malloc(sizeof(mlogrec_telecom));

	if (record != NULL) {
		record->called_number	= NULL;
		record->calling_number	= NULL;
		record->direction	= M_RECORD_TELECOM_DIRECTION_UNSET;
		record->duration	= 0;

		record->ext = NULL;
		record->ext_type = 0;
	}

	return record;
}

void mrecord_free_traffic(mlogrec_traffic *record) {
	if (!record) return;

	if (record->ext != NULL) {
		switch(record->ext_type) {
		case M_RECORD_TYPE_TRAFFIC_FLOW:
			mrecord_free_traffic_flow(record->ext);
			break;
		case M_RECORD_TYPE_TRAFFIC_IPCHAINS:
			mrecord_free_traffic_ipchains(record->ext);
			break;
		case M_RECORD_TYPE_TRAFFIC_IPPL:
			mrecord_free_traffic_ippl(record->ext);
			break;
		default:
			fprintf(stderr, "%s.%d: Unknown record type: %d\n", __FILE__, __LINE__, record->ext_type);
		}
	}

	if (record->src)	free(record->src);
	if (record->dst)	free(record->dst);

	free(record);
}

mlogrec_traffic *mrecord_init_traffic() {
	mlogrec_traffic *record = malloc(sizeof(mlogrec_traffic));

	if (record != NULL) {
		record->src		= NULL;
		record->dst		= NULL;
		record->xfer_incoming	= 0;
		record->xfer_outgoing	= 0;

		record->ext = NULL;
		record->ext_type = 0;
	}

	return record;
}

void mrecord_free_web_extclf(mlogrec_web_extclf *record) {
	if (!record) return;

	buffer_free(record->ref_url);
	buffer_free(record->ref_getvars);
	buffer_free(record->req_useragent);
	buffer_free(record->req_useros);
	
	buffer_free(record->srv_host);
	buffer_free(record->srv_port);

	free(record);
}

mlogrec_web_extclf *mrecord_init_web_extclf() {
	mlogrec_web_extclf *record = malloc(sizeof(mlogrec_web_extclf));

	if (record != NULL) {
		record->ref_url		= buffer_init();
		record->ref_getvars	= buffer_init();
		record->req_useragent	= buffer_init();
		record->req_useros	= buffer_init();

		record->srv_port	= buffer_init();
		record->srv_host	= buffer_init();

		record->duration	= 0;
	}

	return record;
}

void mrecord_free_web_ftp(mlogrec_web_ftp *record) {
	if (!record) return;

	if (record->req_group) buffer_free(record->req_group);

	free(record);
}

mlogrec_web_ftp *mrecord_init_web_ftp() {
	mlogrec_web_ftp *record = malloc(sizeof(mlogrec_web_ftp));

	if (record != NULL) {
		record->req_group	= buffer_init();
		record->trans_duration	= 0;
		record->trans_command   = M_RECORD_FTP_COMMAND_UNSET;
		record->trans_mode	= M_RECORD_FTP_MODE_UNSET;
	}

	return record;
}

void mrecord_free_web_squid(mlogrec_web_squid *record) {
	if (!record) return;

	free(record);
}

mlogrec_web_squid *mrecord_init_web_squid() {
	mlogrec_web_squid *record = malloc(sizeof(mlogrec_web_squid));

	if (record != NULL) {
		record->log_tag		= M_RECORD_SQUID_LOG_UNSET;
		record->data_tag	= M_RECORD_SQUID_DATA_UNSET;
	}

	return record;
}

void mrecord_free_telecom_internal(mlogrec_telecom_internal *record) {
	if (!record) return;

	if (record->user_id)		free(record->user_id);
	if (record->provider)		free(record->provider);

	free(record);
}

mlogrec_telecom_internal *mrecord_init_telecom_internal() {
	mlogrec_telecom_internal *record = malloc(sizeof(mlogrec_telecom_internal));

	if (record != NULL) {
		record->user_id		= NULL;
		record->provider	= NULL;
		record->units_to_pay	= 0;
	}

	return record;
}

void mrecord_free_traffic_flow(mlogrec_traffic_flow *record) {
	if (!record) return;

	free(record);
}

mlogrec_traffic_flow *mrecord_init_traffic_flow() {
	mlogrec_traffic_flow *record;

	record = malloc(sizeof(mlogrec_traffic_flow));
	assert(record);
	memset(record, 0, sizeof(mlogrec_traffic_flow));

	return record;
}

void mrecord_free_traffic_ipchains(mlogrec_traffic_ipchains *record) {
	if (!record) return;

	if (record->device)	free(record->device);
	if (record->host)	free(record->host);
	if (record->chain)	free(record->chain);

	free(record);
}

void mrecord_free_traffic_ippl(mlogrec_traffic_ippl *record) {
	if (!record) return;

	record->src_port	= 0;
	record->dst_port	= 0;
	record->conn_state	= M_RECORD_IPPL_CONNSTATE_UNSET;
	record->prototype	= M_RECORD_IPPL_PROTOCOL_UNSET;
	record->has_ipopts	= 0;
	if (record->remident)	free(record->remident);
	if (record->remhost)	free(record->remhost);
	if (record->protoname)	free(record->protoname);

	free(record);
}

mlogrec_traffic_ipchains *mrecord_init_traffic_ipchains() {
	mlogrec_traffic_ipchains *record = malloc(sizeof(mlogrec_traffic_ipchains));

	if (record != NULL) {
		record->protocol	= M_RECORD_TRAFFIC_PROTOCOL_UNSET;
		record->device		= NULL;
		record->host		= NULL;
		record->chain		= NULL;
		record->action		= M_RECORD_IPCHAINS_ACTION_UNSET;
		record->src_port	= 0;
		record->dst_port	= 0;
		record->rule		= -1;
	}

	return record;
}

mlogrec_traffic_ippl *mrecord_init_traffic_ippl() {
	mlogrec_traffic_ippl *record = malloc(sizeof(mlogrec_traffic_ippl));

	if (record != NULL) {
		record->protoname	= NULL;
		record->prototype	= M_RECORD_IPPL_PROTOCOL_UNSET;
		record->has_ipopts	= 0;
		record->src_port	= 0;
		record->dst_port	= 0;
		record->conn_state	= M_RECORD_IPPL_CONNSTATE_UNSET;
		record->remident	= NULL;
		record->remhost		= NULL;
	}

	return record;
}


void mrecord_free_mail(mlogrec_mail *record) {
	if (!record) return;

	if (record->ext != NULL) {
		switch(record->ext_type) {
		case M_RECORD_TYPE_MAIL_QMAIL_STATUS:
			mrecord_free_mail_qmail_status(record->ext);
			break;
		case M_RECORD_TYPE_MAIL_VIRUS:
			mrecord_free_mail_virus(record->ext);
			break;
		default:
			fprintf(stderr, "%s.%d: Unknown record type: %d\n", __FILE__, __LINE__, record->ext_type);
		}
	}

	if (record->receipient)		free(record->receipient);
	if (record->sender)		free(record->sender);
	if (record->status_text)	free(record->status_text);

	free(record);
}

mlogrec_mail *mrecord_init_mail() {
	mlogrec_mail *record = malloc(sizeof(mlogrec_mail));
	memset(record, 0, sizeof(mlogrec_mail));

	return record;
}


void mrecord_free_mail_qmail_status (mlogrec_mail_qmail_status *record) {
	if (!record) return;

	free(record);
}

mlogrec_mail_qmail_status *mrecord_init_mail_qmail_status () {
	mlogrec_mail_qmail_status  *record = malloc(sizeof(mlogrec_mail_qmail_status));
	memset(record, 0, sizeof(mlogrec_mail_qmail_status));

	return record;
}

void mrecord_free_mail_virus (mlogrec_mail_virus *record) {
	if (!record) return;

	if (record->scanner) free(record->scanner);
	if (record->subject) free(record->subject);
	if (record->virus) free(record->virus);

	free(record);
}

mlogrec_mail_virus *mrecord_init_mail_virus () {
	mlogrec_mail_virus  *record = malloc(sizeof(mlogrec_mail_virus));

	if (record != NULL) {
		record->scanner = NULL;
		record->subject = NULL;
		record->virus = NULL;
	}

	return record;
}

int mrecord_copy_web_extclf(mlogrec_web_extclf *dst, mlogrec_web_extclf *src) {
	if (!dst) return -1;
	if (!src) return -1;

#define FIELD_CPY(x) \
	if (src->x->used) buffer_copy_string_len(dst->x, src->x->ptr, src->x->used);

	FIELD_CPY(ref_url);
	FIELD_CPY(ref_getvars);
	FIELD_CPY(req_useragent);
	FIELD_CPY(req_useros);
	FIELD_CPY(srv_host);
	FIELD_CPY(srv_port);
#undef FIELD_CPY

	dst->duration = src->duration;

	return 0;
}

int mrecord_copy_web_squid(mlogrec_web_squid *dst, mlogrec_web_squid *src) {
	if (!dst) return -1;
	if (!src) return -1;

	dst->log_tag = src->log_tag;
	dst->data_tag = src->data_tag;

	return 0;
}

int mrecord_copy_mail_qmail_status (mlogrec_mail_qmail_status *dst, mlogrec_mail_qmail_status *src) {
	if (!dst) return -1;
	if (!src) return -1;

	dst->local_cur	= src->local_cur;
	dst->local_max	= src->local_max;
	dst->remote_cur	= src->remote_cur;
	dst->remote_max	= src->remote_max;

	dst->queue_cur	= src->queue_cur;
	dst->deliver_cur	= src->deliver_cur;

	return 0;
}

int mrecord_copy_mail_virus (mlogrec_mail_virus *dst, mlogrec_mail_virus *src) {
	if (!dst) return -1;
	if (!src) return -1;

#define FIELD_CPY(x) \
	if (src->x) {\
		dst->x = strdup(src->x); \
		assert(dst->x); \
	}

	FIELD_CPY(subject);
	FIELD_CPY(virus);
	FIELD_CPY(scanner);
#undef FIELD_CPY

	return 0;
}


int mrecord_copy_mail(mlogrec_mail *dst, mlogrec_mail *src) {
	if (!dst) return -1;
	if (!src) return -1;

#define FIELD_CPY(x) \
	if (src->x) {\
		dst->x = strdup(src->x); \
		assert(dst->x); \
	}

	FIELD_CPY(receipient);
	FIELD_CPY(sender);
#undef FIELD_CPY

	dst->duration = src->duration;
	dst->bytes_out = src->bytes_out;
	dst->bytes_in = src->bytes_in;
	dst->ext_type = src->ext_type;

	switch (dst->ext_type) {
		case M_RECORD_TYPE_MAIL_QMAIL_STATUS:
			dst->ext = mrecord_init_mail_qmail_status();

			mrecord_copy_mail_qmail_status(dst->ext, src->ext);
			break;
		case M_RECORD_TYPE_MAIL_VIRUS:
			dst->ext = mrecord_init_mail_virus();

			mrecord_copy_mail_virus(dst->ext, src->ext);
			break;
		case M_RECORD_TYPE_MAIL_UNSET:
			break;
		default:
			M_DEBUG1(M_DEBUG_LEVEL_VERBOSE, M_DEBUG_SECTION_INTERNALS, M_DEBUG_LEVEL_VERBOSE,
				 "can't copy datatype %d\n", dst->ext_type);
	}

	return 0;
}

int mrecord_copy_web_ftp(mlogrec_web_ftp *dst, mlogrec_web_ftp *src) {
	if (!dst) return -1;
	if (!src) return -1;

#define FIELD_CPY(x) \
	if (src->x->used) buffer_copy_string_len(dst->x, src->x->ptr, src->x->used);

	FIELD_CPY(req_group);
#undef FIELD_CPY

	dst->trans_command = src->trans_command;
	dst->trans_duration = src->trans_duration;
	dst->trans_mode = src->trans_mode;

	return 0;
}


int mrecord_copy_web(mlogrec_web *dst, mlogrec_web *src) {
	if (!dst) return -1;
	if (!src) return -1;

#define FIELD_CPY(x) \
	if (src->x->used) buffer_copy_string_len(dst->x, src->x->ptr, src->x->used);

	FIELD_CPY(req_host_ip);
	FIELD_CPY(req_host_name);
	FIELD_CPY(req_user);
	FIELD_CPY(req_protocol);
	FIELD_CPY(req_url);
	FIELD_CPY(req_method);
	FIELD_CPY(req_getvars);
#undef FIELD_CPY

	dst->req_status = src->req_status;
	dst->xfersize = src->xfersize;
	dst->ext_type = src->ext_type;

	switch (dst->ext_type) {
		case M_RECORD_TYPE_WEB_EXTCLF:
			dst->ext = mrecord_init_web_extclf();

			mrecord_copy_web_extclf(dst->ext, src->ext);
			break;
		case M_RECORD_TYPE_WEB_SQUID:
			dst->ext = mrecord_init_web_squid();

			mrecord_copy_web_squid(dst->ext, src->ext);
			break;
		case M_RECORD_TYPE_WEB_FTP:
			dst->ext = mrecord_init_web_ftp();

			mrecord_copy_web_ftp(dst->ext, src->ext);
			break;
		case M_RECORD_TYPE_WEB_UNSET:
			break;
		default:
			M_DEBUG1(M_DEBUG_LEVEL_VERBOSE, M_DEBUG_SECTION_INTERNALS, M_DEBUG_LEVEL_VERBOSE,
				 "can't copy datatype %d\n", dst->ext_type);
	}

	return 0;
}

int mrecord_copy_traffic_flow(mlogrec_traffic_flow *dst, mlogrec_traffic_flow *src) {
	if (!dst) return -1;
	if (!src) return -1;

	dst->protocol = src->protocol;
	dst->packets = src->packets;
	dst->src_port = src->src_port;
	dst->dst_port = src->dst_port;
	dst->src_as = src->src_as;
	dst->dst_as = src->dst_as;

	return 0;
}

int mrecord_copy_traffic_ippl(mlogrec_traffic_ippl *dst, mlogrec_traffic_ippl *src) {
	if (!dst) return -1;
	if (!src) return -1;

	dst->prototype 	= src->prototype;
	dst->src_port 	= src->src_port;
	dst->dst_port 	= src->dst_port;
	dst->conn_state	= src->conn_state;
	dst->has_ipopts	= src->has_ipopts;
	dst->remident	= src->remident ? strdup(src->remident) : NULL;
	dst->remhost	= src->remhost ? strdup(src->remhost) : NULL;
	dst->protoname	= src->protoname ? strdup(src->protoname) : NULL;
	
	return 0;
}

int mrecord_copy_traffic(mlogrec_traffic *dst, mlogrec_traffic *src) {
	if (!dst) return -1;
	if (!src) return -1;

#define FIELD_CPY(x) \
	if (src->x) {\
		dst->x = strdup(src->x); \
		assert(dst->x); \
	}

	FIELD_CPY(src);
	FIELD_CPY(dst);
#undef FIELD_CPY

	dst->xfer_incoming = src->xfer_incoming;
	dst->xfer_outgoing = src->xfer_outgoing;
	dst->ext_type = src->ext_type;

	switch (dst->ext_type) {
		case M_RECORD_TYPE_TRAFFIC_FLOW :
			dst->ext = mrecord_init_traffic_flow();

			mrecord_copy_traffic_flow(dst->ext, src->ext);
			break;
		case M_RECORD_TYPE_TRAFFIC_IPPL :
			dst->ext = mrecord_init_traffic_ippl();

			mrecord_copy_traffic_ippl(dst->ext, src->ext);
			break;
		case M_RECORD_TYPE_TRAFFIC_UNSET:
			break;
		default:
			fprintf(stderr, "%s.%d: no copy function for traffic subtype %d\n",__FILE__, __LINE__, dst->ext_type);
	}



	return 0;
}

/*
 * copy the dataset of the src-record to the dst-record
 *
 * dst gets a copy of the dataset of the src-record. Both record are
 * indepented afterwards.
 *
 * @param dst destination-record
 * @param src source-record
 * @return 0 on success, -1 on NULL pointers
 */

int mrecord_copy(mlogrec *dst, mlogrec *src) {
	if (!dst) return -1;
	if (!src) return -1;

	dst->timestamp = src->timestamp;

	dst->ext_type = src->ext_type;

	switch (dst->ext_type) {
	case M_RECORD_TYPE_WEB:
		dst->ext = mrecord_init_web();

		mrecord_copy_web(dst->ext, src->ext);
		break;
	case M_RECORD_TYPE_TRAFFIC:
		dst->ext = mrecord_init_traffic();

		mrecord_copy_traffic(dst->ext, src->ext);
		break;
	case M_RECORD_TYPE_MAIL:
		dst->ext = mrecord_init_mail();

		mrecord_copy_mail(dst->ext, src->ext);
		break;
	case M_RECORD_TYPE_UNSET:
		break;
	default:
		M_DEBUG1(M_DEBUG_LEVEL_VERBOSE, M_DEBUG_SECTION_INTERNALS, M_DEBUG_LEVEL_VERBOSE,
				 "can't copy datatype %d\n", dst->ext_type);
	}

	return 0;
}

/*
 * transfers the dataset of the src-record to the dst-record
 *
 * the dataset is moved which means that the dst gets the
 * dataset of the src-record and the src-record is reset.
 * No copy operations !
 *
 * @param dst destination-record
 * @param src source-record
 * @return 0 on success, -1 on NULL pointers
 */

int mrecord_move(mlogrec *dst, mlogrec *src) {
	void *ext;
	int ext_type;
	
	if (!dst) return -1;
	if (!src) return -1;

	mrecord_reset(dst);
	
	ext            = dst->ext;
	ext_type       = dst->ext_type;

	dst->timestamp = src->timestamp;
	dst->ext_type  = src->ext_type;
	dst->ext       = src->ext;

	src->ext       = ext;
	src->ext_type  = ext_type;
	src->timestamp = 0;

	return 0;
}
