/*======================================================================*\
|*		Editor mined						*|
|*		display output functions				*|
\*======================================================================*/

#include "mined.h"
#include "io.h"
#include "pc437.map"

#include <errno.h>


/*======================================================================*\
|*			Scroll bar display				*|
\*======================================================================*/

static
void
disp_scrollbar_cellend ()
{
}

static
void
disp_scrollbar_end ()
{
  if (! standout_glitch) {
	disp_scrollbar_off ();
  }
}

/* mined displays last_y + 1 lines in a display area of YMAX lines */
/* the file has (approx.?) total_lines lines */
/* the file position was in line # line_number */

/* debug traces: */
#define dont_debug_scrollbar
#define dont_debug_scrollbar_calc
#define dont_debug_partial_scroll
#define dont_debug_dirty
/* visible debug scrollbar: */
#define dont_debug_lazy_scrollbar

static int prev_disp_start = 0;
static int prev_disp_end = 0;

static FLAG scrollbar_dirty = True;	/* does scrollbar need refresh ? */
static int first_dirty = -1;
static int last_dirty = -1;

#ifdef debug_dirty
#define printf_dirty(s, n)	printf ("%s %d - (prev %d..%d) [%d dirty %d..%d]\n", s, n, prev_disp_start, prev_disp_end, scrollbar_dirty, first_dirty, last_dirty);
#else
#define printf_dirty(s, n)	
#endif


static
void
set_scrollbar_dirty (scry)
  int scry;
{
  scrollbar_dirty = True;
  if (scry < first_dirty || first_dirty < 0) {
	first_dirty = scry;
  }
  if (scry > last_dirty) {
	last_dirty = scry;
  }
#ifdef debug_partial_scroll
	printf ("scrollbar_dirty @ %d, -> dirty %d..%d\n", scry, first_dirty, last_dirty);
#endif
	printf_dirty ("set_dirty", scry);
}

void
scrollbar_scroll_up (from)
  int from;
{
	int unit = utf8_screen && fine_scrollbar ? 8 : 1;

	prev_disp_start -= unit;
	prev_disp_end -= unit;
	set_scrollbar_dirty (SCREENMAX - 1);
#ifdef debug_partial_scroll
	printf ("scrollbar_scroll_up from %d, -> prev %d..%d\n", from, prev_disp_start, prev_disp_end);
#endif
	printf_dirty ("scroll_up", from);
}

void
scrollbar_scroll_down (from)
  int from;
{
	int unit = utf8_screen && fine_scrollbar ? 8 : 1;

	prev_disp_start += unit;
	prev_disp_end += unit;
	set_scrollbar_dirty (from);
#ifdef debug_partial_scroll
	printf ("scrollbar_scroll_down from %d, -> prev %d..%d\n", from, prev_disp_start, prev_disp_end);
#endif
	printf_dirty ("scroll_down", from);
}

static
FLAG
fine_grained_scrollbar (force)
  FLAG force;
{
/* calculate scroll bar display using Unicode 
   character cell vertical eighth blocks U+2581..U+2587 ▁▂▃▄▅▆▇
   as follows:
 * screen has lines 0...last_y, screen_lines = last_y + 1,
   with fields = screen_lines * 8,
   e.g. 10 lines 0...9, 80 fields 0...79;
 * to be mapped to buffer line numbers 1...total_lines;
   scroll block size in fields is s / fields = screen_lines / total_lines,
   s = screen_lines * fields / total_lines, 
   minimum 1 (display) / 7 (Unicode) / 8 (algorithm),
   max_start = fields - s;
 * scroll block start position is in range 0...(80 - s),
   to be mapped to position of last screen line 
   last_line_number = (line_number - y + last_y):
   if last_line_number <= screen_lines ==> p = 0
   if last_line_number == total_lines ==> p = max_start
   so roughly:
   p / max_start
   	= (last_line_number - screen_lines) 
   	  / (total_lines - screen_lines);
   but to accomodate rounding problems and avoid zero division map range
   last_line_number = screen_lines + 1 ==> p = start1,
   last_line_number = total_lines - 1 ==> p = max_start - 1,
   where start1 corresponds to the delta caused by scrolling by 1 line;
   start1 / fields = 1 / (total_lines - screen_lines), min 1,
   but replace this with an adjustment (see below) for better computation;
 - distribute equally, so map
   (max_start - start1) fields to (total_lines - 1 - screen_lines) lines
   where both will taken to zero by an offset for scaling:
   fields start1 ... max_start-1 
   	==> consider result to be p - start1
   lines screen_lines + 1 ... total_lines - 1 
   	==> consider ref_last/total = last/total_line_number - (screen_lines + 1):
   p - start1 / max_start
   	= (last_line_number - screen_lines - 1) / (total_line_number - screen_lines - 1)
 * scroll block extends from p to p + s - 1
*/

  int unit = utf8_screen && fine_scrollbar ? 8 : 1;

  int screen_lines;
  long fields;
  int s;
  int max_start;
  long last_line_number;
  int disp_start;
  int disp_end;
  int i;
  int scri;
  FLAG painted = False;
  int slices = 0;
  int oldslices = 0;
  FLAG klopsinslot;
  FLAG oldklopsinslot;

  screen_lines = last_y + 1;
  fields = screen_lines * unit;
  s = fields * screen_lines / total_lines;
  if (s < unit) {
	s = unit;
  }
  max_start = fields - s;
  last_line_number = line_number - y + last_y;
  if (last_line_number <= screen_lines) {
	disp_start = 0;
  } else if (last_line_number == total_lines) {
	disp_start = max_start;
  } else {
	/* compensate by + 1 for adjustment at the beginning */
	disp_start = max_start * 
			(last_line_number - screen_lines - 1 + 1) 
			/ (total_lines - screen_lines - 1);
	/* assure distance if not quite at beginning or end */
	if (disp_start == 0) {
		disp_start = 1;
	} else if (disp_start >= max_start) {
		disp_start = max_start - 1;
	}
  }
  disp_end = disp_start + s - 1;

#ifdef debug_scrollbar_calc
	printf ("last_line_number %d/%d, screen_lines %d (0-%d)\n", last_line_number, total_lines, screen_lines, last_y);
	printf (" size %d, maxstart %d, ~ * %d / %d, pos %d-%d/%d\n", 
		s, max_start, 
		last_line_number - screen_lines, total_lines - screen_lines, 
		disp_start, disp_end, fields);
#endif

  if (disp_scrollbar) {
#ifdef debug_scrollbar
	if (update_scrollbar_lazy) {
		printf ("scrollbar (%d) %d - %d (prev. %d - %d)\n", force, disp_start, disp_end, prev_disp_start, prev_disp_end);
	}
#endif

/* last_y / SCREENMAX */
	for (i = 0; i < fields; i ++) {
		if (i >= disp_start && i <= disp_end) {
			slices ++;
			klopsinslot = True;
		} else {
			klopsinslot = False;
		}
		if (i >= prev_disp_start && i <= prev_disp_end) {
			oldslices ++;
			oldklopsinslot = True;
		} else {
			oldklopsinslot = False;
		}
		if (((i + 1) % unit) == 0) {
		    scri = i / unit;
		    if (slices != oldslices 
			|| klopsinslot != oldklopsinslot
			|| force
			|| (scrollbar_dirty && scri >= first_dirty && scri <= last_dirty)
		       ) {
			painted = True;
			set_cursor (XMAX, scri);
			if (slices == 0) {
				disp_scrollbar_background ();
				putchar (' ');
				disp_scrollbar_cellend ();
			} else if (slices == unit) {
				disp_scrollbar_foreground ();
				putchar (' ');
				disp_scrollbar_cellend ();
#ifdef debug_scrollbar_calc
				if (klopsinslot == False) {
					printf ("@ %d (%d) ", scri, i / unit * unit);
				}
				printf ("%d", unit);
#endif
			} else {
				/* choose among the eighths blocks */
				/* U+2581..U+2587 ▁▂▃▄▅▆▇ */
				if (klopsinslot) {
					/* display top segment part */
					disp_scrollbar_background ();
					put_unichar (0x2580 + slices);
					disp_scrollbar_cellend ();
#ifdef debug_scrollbar_calc
					if (! klopsinslot) {
						printf ("@ %d (%d) ", scri, i + 1 - slices);
					}
					printf ("%d", slices);
#endif
				} else {
					/* display bottom segment */
					disp_scrollbar_foreground ();
					put_unichar (0x2588 - slices);
					disp_scrollbar_cellend ();
#ifdef debug_scrollbar_calc
					printf ("%d", slices);
#endif
				}
			}
#ifdef debug_lazy_scrollbar
		    } else {
			painted = True;
			set_cursor (XMAX, scri);
			if (slices > 0) {
				disp_scrollbar_foreground ();
			} else {
				disp_scrollbar_background ();
			}
			putchar ('X');
			disp_scrollbar_cellend ();
#endif
		    }
		    slices = 0;
		    oldslices = 0;
		}
	}
	disp_scrollbar_end ();
#ifdef debug_scrollbar_calc
	printf ("\n");
#endif
  }

  printf_dirty ("scrollbar", force);
  prev_disp_start = disp_start;
  prev_disp_end = disp_end;
  scrollbar_dirty = False;
  first_dirty = SCREENMAX;
  last_dirty = -1;
  printf_dirty ("> scrollbar", force);
  return painted;
}

#ifdef use_cell_scrollbar

static
FLAG
cell_grained_scrollbar (force)
  FLAG force;
{
/* last_y / YMAX */
  int disp_start = ((long) line_number - y) * last_y / total_lines;
  int disp_end = ((long) line_number - y + last_y + 1) * last_y / total_lines;
  int i;
  FLAG painted = False;
  FLAG isin_newklops, isin_oldklops;

  if (disp_scrollbar) {
#ifdef debug_scrollbar
	if (update_scrollbar_lazy) {
		printf ("scrollbar (%d) %d - %d (prev. %d - %d)\n", force, disp_start, disp_end, prev_disp_start, prev_disp_end);
	}
#endif

/* last_y / SCREENMAX */
	for (i = 0; i <= last_y; i ++) {
	    isin_newklops = i >= disp_start && i <= disp_end;
	    isin_oldklops = i >= prev_disp_start && i <= prev_disp_end;
	    if (isin_newklops != isin_oldklops || force) {
		painted = True;
		set_cursor (XMAX, i);
		if (isin_newklops) {
#ifdef CJKterm_not_coloured
			if (cjk_term) {
				reverse_on ();
				putchar ('<');
				reverse_off ();
			} else
#endif
			{
				disp_scrollbar_foreground ();
				putchar (' ');
				disp_scrollbar_cellend ();
			}
		} else {
			disp_scrollbar_background ();
			putchar (' ');
			disp_scrollbar_cellend ();
		}
#ifdef debug_lazy_scrollbar
	    } else {
		painted = True;
		set_cursor (XMAX, i);
		if (isin_newklops) {
			disp_scrollbar_foreground ();
		} else {
			disp_scrollbar_background ();
		}
		putchar ('X');
		disp_scrollbar_cellend ();
#endif
	    }
	}
	disp_scrollbar_end ();
  }

  prev_disp_start = disp_start;
  prev_disp_end = disp_end;
  scrollbar_dirty = False;
  first_dirty = SCREENMAX;
  last_dirty = -1;
  return painted;
}

FLAG
display_scrollbar (update)
  FLAG update;
{
  if (utf8_screen && fine_scrollbar) {
	return fine_grained_scrollbar (update == False);
  } else {
	return cell_grained_scrollbar (update == False);
  }
}

#else

FLAG
display_scrollbar (update)
  FLAG update;
{
  return fine_grained_scrollbar (update == False);
}

#endif


/*======================================================================*\
|*			Buffered output					*|
\*======================================================================*/

/*
 * Bad_write () is called when a write failed. Notify the user.
 */
void
bad_write (fd)
  int fd;
{
  if (fd == output_fd) {	/* Cannot write to terminal? */
	panicio ("Write error on terminal", serror ());
  }

  clear_buffer (); /* out_count = 0; */
  ring_bell ();
  error2 ("Write aborted (File incomplete): ", serror ());
}

/*
 * Flush the I/O buffer on filedescriptor fd.
flush () is (void) flush_buffer (output_fd) unless with curses
 */
int
flush_buffer (fd)
  int fd;
{
  if (out_count <= 0) {		/* There is nothing to flush */
	return FINE;
  }
#ifdef conio
  if (fd == output_fd) {
	cputs (screen);
  }
#else
#ifdef BorlandC
  if (fd == output_fd) {
	screen [out_count] = '\0';
/* don't ask me why that crazy compiler doesn't work with write () below */
	printf ("%s", screen);
  }
#endif
#endif
  else {
	char * writepoi = screen;
	int written = 0;
	int none_count = 0;
/*	int less_count = 0;	*/

	while (out_count > 0) {
		written = write (fd, writepoi, out_count);
		if (written == -1) {
			if (geterrno () == EINTR && winchg) {
				/* try again */
			} else {
				bad_write (fd);
				return ERRORS;
			}
		} else if (written == 0) {
			none_count ++;
			if (none_count > 20) {
				bad_write (fd);
				return ERRORS;
			}
		} else {
			out_count -= written;
			writepoi += written;
		}
	}
  }
  clear_buffer (); /* Empty buffer: out_count = 0; */
  return FINE;
}

/*
 * putoutchar does a buffered output to screen.
   (see also writechar which does the same)
 */
int
putoutchar (c)
  character c;
{
  char clow;

  if (c == '\n') {
	if (putoutchar ('\r') == ERRORS) {
		return ERRORS;
	}
  }
  if (translate_output) {
	if ((c & '\200') != '\0') {
		altcset_on ();
		clow = (c & '\177');
		if ((int) clow < translen) {
			putoutchar (transout [(int) clow]);
		} else {
			putoutchar (clow);
		}
		altcset_off ();
		return FINE;
	}
  }
  screen [out_count ++] = c;
  if (out_count == screen_BUFL) {
	return flush_buffer (output_fd);
  }
#ifdef DEBUG
  flush ();
#endif
  return FINE;
}

/*
 * Putoutstring writes the given string out to the terminal.
 * (buffered via misused screen buffer!)
putstring (str) does putoutstring (str) unless with curses or conio
 */
void
putoutstring (text)
  register char * text;
{
  while (* text != '\0') {
	(void) putoutchar (* text ++);
  }
}


/*======================================================================*\
|*			Marked character output				*|
\*======================================================================*/

static
void
putnarrowchar (unichar)
  unsigned long unichar;
{
  if (iswide (unichar)) {
	if (unichar == 0xB7) {
		put_unichar ('.');
	} else if (unichar == 0x2028) {
		put_unichar ('');
	} else if (unichar == 0x2029) {
		put_unichar ('P');
	} else if (unichar == (unsigned long) '') {
		put_unichar ('0');
	} else {
		put_unichar (' ');
	}
  } else {
	put_unichar (unichar);
  }
}

static
void
putnarrowutf (utfmark)
  char * utfmark;
{
  unsigned long unichar;
  int utflen;

  utf8_info (utfmark, & utflen, & unichar);
  if (iswide (unichar)) {
	putnarrowchar (unichar);
  } else {
	put1utfchar (utfmark);
  }
}


#define indicate_illegal_UTF indicate_marked
#define indicate_UTF indicate_marked

/**
   Indicate highlighted special character property (e.g. illegal code).
 */
static
void
indicate_marked (c)
  character c;
{
  if (char_on_status_line) {
	reverse_off ();
  }
  unidisp_on ();
  putnarrowchar ((unsigned long) c);
  unidisp_off ();
  if (char_on_status_line) {
	reverse_on ();
  }
}

/**
   Indicate coulored special character.
 */
static
void
indicate_special (c)
  character c;
{
  if (char_on_status_line) {
	reverse_off ();
  }
  unimarkdisp_on ();
  putnarrowchar ((unsigned long) c);
  unimarkdisp_off ();
  if (char_on_status_line) {
	reverse_on ();
  }
}

/*
 * put mark character, possibly switching alternate character set
 * putmarkmode and endmarkmode are only called by putmark and for 
 * display of TAB markers
 */
FLAG dim_mode = False;
FLAG mark_mode = False;
FLAG use_extracset = False;

static
void
putmarkmode (c, utfmark)
  character c;
  char * utfmark;
{
  if (dim_mode == False) {
	dim_on ();
	dim_mode = True;
  }

  if (utf8_screen) {
	if (utfmark != NIL_PTR && * utfmark != '\0') {
		putnarrowutf (utfmark);
	} else {
		putnarrowchar ((unsigned long) c);
	}
  } else if ((c & '\200') != '\0') {
	if (mark_mode) {
		end_extracset ();
		mark_mode = False;
	}
	if (pc_term) {
		put_unichar (c);
	} else {
		putchar (c);
	}
  } else {
	if (mark_mode == False && use_extracset) {
		start_extracset ();
		mark_mode = True;
	}
	putchar (c);
  }
}

static
void
endmarkmode ()
{
  if (dim_mode) {
	dim_off ();
	dim_mode = False;
  }
  if (mark_mode) {
	end_extracset ();
	mark_mode = False;
  }
}

/*
   Output a line status marker.
 */
void
putmark (c, utfmark)
  char c;
  char * utfmark;
{
  putmarkmode (c, utfmark);
  endmarkmode ();
}

static
void
putCJKmark (umark)
  unsigned long umark;
{
  unsigned long mark = cjk (umark);
  dim_on ();
  if (no_char (mark)) {
	put_cjkchar ('?');
	put_cjkchar ('?');
  } else {
	put_cjkchar (mark);
  }
  dim_off ();
}

static
void
putUmark (c, utfmark)
  character c;
  char * utfmark;
{
  unimarkdisp_on ();
  if (utf8_screen) {
	if (utf8_text && utfmark != NIL_PTR && * utfmark != '\0') {
		putnarrowutf (utfmark);
	} else {
		putnarrowchar ((unsigned long) c);
	}
  } else {
	if (pc_term) {
		put_unichar (c);
	} else {
		putchar (c);
	}
  }
  unimarkdisp_off ();
}

static
void
putret (type)
  unsigned char type;
{
  if (type == lineend_LF) {
	putmark (RET_MARK, UTF_RET);
  } else if (type == lineend_CRLF) {
	/* MSDOS line separator */
	putmark (DOSRET_MARK, UTF_DOSRET);
  } else if (type == lineend_CR) {
	/* Mac line separator */
/*	putmark ('', "¥");	*/
/*	putmark ('', "¦");	*/
	putmark ('@', "@");
  } else if (type == lineend_NONE) {
	putmark ('', "¬");
  } else if (type == lineend_NUL) {
	putmark ('', "º");
  } else if (type == lineend_LS) {
	/* Unicode line separator 2028:   */
	putUmark (RET_MARK, UTF_RET);
  } else if (type == lineend_PS) {
	/* Unicode paragraph separator 2029:   */
	putUmark (PARA_MARK, UTF_PARA);
  } else {
	putmark ('', "¤");
  }
}


/*======================================================================*\
|*			Character output				*|
\*======================================================================*/

static int utfcount = 0;
static unsigned long unichar;
static unsigned short prev_cjkchar = 0L;

static
void
put_uniend ()
{
  if (utf8_text && utfcount > 0) {
	/* previous character not terminated properly */
	indicate_illegal_UTF ('');
	utfcount = 0;
  }
}

/*
 * put Unicode (UCS-4, actually) character to screen;
 * put as UTF-8 on UTF-8 terminal, or as Latin-1 otherwise
	7 bits	0xxxxxxx
	 8..11	110xxxxx 10xxxxxx
	12..16	1110xxxx 10xxxxxx 10xxxxxx
	17..21	11110xxx 10xxxxxx 10xxxxxx 10xxxxxx
	22..26	111110xx 10xxxxxx 10xxxxxx 10xxxxxx 10xxxxxx
	27..31	1111110x 10xxxxxx 10xxxxxx 10xxxxxx 10xxxxxx 10xxxxxx
 */
void
_put_unichar (unichar)
  unsigned long unichar;
{
  character c;

  if (mapped_text && unichar == 0) {
	indicate_marked ('?');
  } else if (unichar < ' ') {
	indicate_UTF (unichar + '@');
  } else if (unichar == 0x7F) {
	indicate_UTF ('?');
  } else if (unichar >= 0x80 && unichar <= 0x9F) {
	indicate_UTF (unichar + '@');
  } else if (unichar == 0xA0 /* nbsp */ && (utf8_text || (! mapped_text && ! pc_term))) {
	indicate_special ('');
  } else if (unichar == 0xFFFE) {
	indicate_illegal_UTF ('e');
  } else if (unichar == 0xFFFF) {
	indicate_illegal_UTF ('f');
  } else if (utf8_screen) {
	if (unichar < 0x80) {
		__putchar (unichar);
	} else if (unichar < 0x800) {
		__putchar (0xC0 | (unichar >> 6));
		__putchar (0x80 | (unichar & 0x3F));
	} else if (unichar < 0x10000) {
		__putchar (0xE0 | (unichar >> 12));
		__putchar (0x80 | ((unichar >> 6) & 0x3F));
		__putchar (0x80 | (unichar & 0x3F));
	} else if (unichar < 0x200000) {
		__putchar (0xF0 | (unichar >> 18));
		__putchar (0x80 | ((unichar >> 12) & 0x3F));
		__putchar (0x80 | ((unichar >> 6) & 0x3F));
		__putchar (0x80 | (unichar & 0x3F));
	} else if (unichar < 0x4000000) {
		__putchar (0xF8 | (unichar >> 24));
		__putchar (0x80 | ((unichar >> 18) & 0x3F));
		__putchar (0x80 | ((unichar >> 12) & 0x3F));
		__putchar (0x80 | ((unichar >> 6) & 0x3F));
		__putchar (0x80 | (unichar & 0x3F));
	} else if (unichar < 0x80000000) {
		__putchar (0xFC | (unichar >> 30));
		__putchar (0x80 | ((unichar >> 24) & 0x3F));
		__putchar (0x80 | ((unichar >> 18) & 0x3F));
		__putchar (0x80 | ((unichar >> 12) & 0x3F));
		__putchar (0x80 | ((unichar >> 6) & 0x3F));
		__putchar (0x80 | (unichar & 0x3F));
	} else {
		/* special encoding of 2 Unicode chars, mapped from 1 JIS character */
		put_unichar (unichar & 0xFFFF);
		unichar = (unichar >> 16) & 0x7FFF;
		if (combining_screen || ! iscombining (unichar)) {
			put_unichar (unichar);
		} else {
			/* if it's a combining char but the screen 
			   cannot handle it, just ignore it for now
			*/
		}
	}
  } else {
	if (unichar < 0x100) {
		if (pc_term) {
			if (unichar < 0x80) {
				__putchar (unichar);
			} else {
				c = (character) pc_charmap [unichar - 0x80];
				if (c < 0x80) {
					indicate_UTF (c);
				} else {
					__putchar (c);
				}
			}
		} else {
			__putchar (unichar);
		}
	} else {
		if (iscombining (unichar)) {
			indicate_UTF ('\'');
		} else {
			if (unichar >= 0xFF01 && unichar <= 0xFF5E) {
				/* FULLWIDTH forms */
				indicate_UTF (unichar - 0xFEE0);
			} else if (unichar == 0xFFE0) {
				/* FULLWIDTH CENT SIGN */
				indicate_UTF (0xA2);
			} else if (unichar == 0xFFE1) {
				/* FULLWIDTH POUND SIGN */
				indicate_UTF (0xA3);
			} else if (unichar == 0xFFE2) {
				/* FULLWIDTH NOT SIGN */
				indicate_UTF (0xAC);
			} else if (unichar == 0xFFE3) {
				/* FULLWIDTH MACRON */
				indicate_UTF (0xAF);
			} else if (unichar == 0xFFE4) {
				/* FULLWIDTH BROKEN BAR */
				indicate_UTF (0xA6);
			} else if (unichar == 0xFFE5) {
				/* FULLWIDTH YEN SIGN */
				indicate_UTF (0xA5);
			} else if (unichar == 0x20AC) {
				/* EURO SIGN */
				indicate_UTF ('E');
			} else {
				indicate_UTF (UNI_MARK);
			}
		}
		if (iswide (unichar)) {
			indicate_UTF (' ');
		}
	}
  }
}

/**
   put CJK character to screen;
   to be called only if cjk_term
   returns screen length
 */
int
_put_cjkchar (cjkchar)
  unsigned long cjkchar;
{
  character cjkbytes [5];
  character * cp;
  unsigned long unichar;

  (void) cjkencode (cjkchar, cjkbytes);

  if (! valid_cjk (cjkchar, cjkbytes) && (suppress_invalid_cjk || ! cjk_term)) {
	/* character code does not follow encoding pattern */
	indicate_marked ('#');
	if (cjk_term && ! cjk_uni_term) {
		indicate_marked (' ');
		return 2;
	} else if (utf_cjk_wide_padding && cjkchar >= 0x100) {
		indicate_marked (' ');
		return 2;
	}

  } else if (cjk_term) {
	if (suppress_extended_cjk &&
	    ((cjk_encoding == 'G' && cjkchar >= 0x80000000 && gb18030_term == False)
	  || (cjk_encoding == 'J' && cjkchar >= 0x8F0000 && euc3_term == False)
	  || (cjk_encoding == 'C' && cjkchar >= 0x8E000000 && euc4_term == False)
	  || (cjklow_term == False &&
	      ((cjkchar >= 0x8000 && cjkchar < 0xA000
	       && (cjk_encoding != 'J' || (cjkchar >> 8) != 0x8E)
	       )
	       || ((cjkchar & 0xFF) >= 0x80 && (cjkchar & 0xFF) < 0xA0)
	      )
	     )
	   )) {
		/* character code is in extended encoding range and 
		   is assumed not to be displayable by terminal */
		indicate_special ('#');
		indicate_special (' ');
		return 2;
	} else if (suppress_unknown_cjk && lookup_cjk (cjkchar) == 0) {
		/* character code has no mapping to Unicode */
		indicate_marked ('?');
		indicate_marked (' ');
		return 2;
	} else {
		cp = cjkbytes;
		while (* cp != '\0') {
			putchar (* cp ++);
		}
		if (cjkchar >= 0x100) {
			if (cjk_encoding == 'J' && cjkchar >> 8 == 0x8E) {
				return 1;
			} else {
				return 2;
			}
		}
	}

  } else {
	unichar = lookup_cjk (cjkchar);
	if (unichar != 0) {
		put_unichar (unichar);
		if (iswide (unichar)) {
			return 2;
		} else if (utf_cjk_wide_padding && cjkchar >= 0x100) {
			/* simulate double screen width of CJK 
			   character even if corresponding 
			   Unicode character has only single width
			*/
			__putchar (' ');
			return 2;
		}
	} else {
		indicate_marked ('?');
		if (utf_cjk_wide_padding && cjkchar >= 0x100) {
			indicate_marked (' ');
			return 2;
		}
	}
  }

  return 1;
}

/*
 * put_char () puts a character byte to the screen (via buffer)
 * If UTF-8 text is being output through put_char () but the screen 
 * is not in UTF-8 mode, put_char () transforms the byte sequences.
 * In CJK mode, this function should not be called anymore with 
 * multibyte CJK codes (Shift-JIS single-byte is OK).

putchar (c) is __putchar (c)
__putchar (c) does (void) putoutchar (c) unless with curses or conio
 */
static
void
put_char (c)
  character c;
{
  if (utf8_text) {
	if (c < 0x80) {
		put_uniend ();
		__putchar (c);
	} else if ((c & 0xC0) == 0x80) {
		if (utfcount == 0) {
			indicate_illegal_UTF ('8');
			return;
		}
		unichar = (unichar << 6) | (c & 0x3F);
		utfcount --;

		if (utfcount == 0) {
			/* final UTF-8 byte */
			put_unichar (unichar);
		} else {
			/* character continues */
			return;
		}
	} else { /* first UTF-8 byte */
		if (utfcount > 0) {
			/* previous character not terminated properly */
			indicate_illegal_UTF ('');
			utfcount = 0;
		}
		if ((c & 0xE0) == 0xC0) {
			utfcount = 2;
			unichar = c & 0x1F;
		} else if ((c & 0xF0) == 0xE0) {
			utfcount = 3;
			unichar = c & 0x0F;
		} else if ((c & 0xF8) == 0xF0) {
			utfcount = 4;
			unichar = c & 0x07;
		} else if ((c & 0xFC) == 0xF8) {
			utfcount = 5;
			unichar = c & 0x03;
		} else if ((c & 0xFE) == 0xFC) {
			utfcount = 6;
			unichar = c & 0x01;
		} else {
			/* illegal UTF-8 code 254 (0xFE) or 255 (0xFF) */
			indicate_illegal_UTF ('4' + (c & 1));
			return;
		}
		utfcount --;
		return;
	}
  } else if (cjk_text && cjk_term == False) {
	/* prev_cjkchar mechanism obsolete (would not suffice anymore...) */
	if (prev_cjkchar != 0 || (c >= 0x80 && ! multichar (c))) {
		put_cjkchar (prev_cjkchar << 8 | c);
		prev_cjkchar = 0;
	} else if (multichar (c)) {
		prev_cjkchar = c;
	} else {
		__putchar (c);
	}
  } else {
	if (! pc_term && c == 0xA0 /* nbsp */) {
		indicate_special ('');
	} else {
		__putchar (c);
	}
  }
}

/*
 * putcharacter (c) puts one non-UTF character to the screen
 */
void
putcharacter (c)
  character c;
{
  if (mapped_text) {
	put_unichar (lookup_cjk (c));
  } else if (utf8_screen && utf8_text == False) {
	put_unichar (c);
  } else {
	put_char (c);
  }
}

/*
 * put 1 UTF-8 character (from string)
 */
void
put1utfchar (s)
  char * s;
{
  unsigned long unichar;
  int utflen;

  utf8_info (s, & utflen, & unichar);
  put_unichar (unichar);
}


/*======================================================================*\
|*			Text display					*|
\*======================================================================*/

#define default_script -999

static
char syntax_marker = syntax_none;
static
int script_colour = default_script;


void
put_blanks (endpos)
  int endpos;
{
  int startpos = 0;
  while (startpos ++ <= endpos) {
	putchar (' ');
  }
}

static
void
clear_wholeline ()
{
  if (can_clear_eol) {
	clear_eol ();
  } else {
	put_blanks (XMAX);
  }
}

void
clear_lastline ()
{
  if (can_clear_eol) {
	clear_eol ();
  } else {
	put_blanks (XMAX - 1);
  }
}


static
void
disp_syntax (syntax_marker_was, syntax_marker)
  char syntax_marker_was;
  char syntax_marker;
{
  if (syntax_marker != syntax_marker_was) {
	if (syntax_marker & syntax_JSP) {
		dispHTML_jsp ();
	} else if (syntax_marker & syntax_comment) {
		dispHTML_comment ();
	} else if (syntax_marker & syntax_HTML) {
		dispHTML_code ();
	} else {
		dispHTML_off ();
	}
  }
}


static
void
disp_script (script_colour)
  int script_colour;
{
  if (script_colour == default_script) {
	disp_normal ();
  } else {
	disp_colour (script_colour);
  }
}

static
struct colour_mapping {
	char * scriptname;
	int colour;
	int colour0;
} colour_table [] =
{
#include "colours.h"
};

/**
   Determine display colour for character script.
 */
static
void
map_script_colour (scriptname, sc, sc0)
  char * scriptname;
  int * sc;
  int * sc0;
{
  int min = 0;
  int max = sizeof (colour_table) / sizeof (struct colour_mapping) - 1;
  int mid;
  int cmp;

  /* binary search in table */
  while (max >= min) {
    mid = (min + max) / 2;
    cmp = strcmp (colour_table [mid].scriptname, scriptname);
    if (cmp < 0) {
      min = mid + 1;
    } else if (cmp > 0) {
      max = mid - 1;
    } else {
      * sc = colour_table [mid].colour;
      * sc0 = colour_table [mid].colour0;
      return;
    }
  }

  * sc = default_script;
  * sc0 = -1;
}


static
void
restore_attr ()
{
  if (dim_HTML) {
	disp_syntax (0, syntax_marker);
  }
  if (script_colour > 0) {
	disp_script (script_colour);
  }
}


#define dont_debug_put_line
#ifdef debug_put_line
#define trace_put_line(params)	printf params
#else
#define trace_put_line(params)	
#endif


/*
 * Put_line prints the given line on the standard output.
 * If offset is not zero, printing will start at that x-coordinate.
 * If the FLAG clear_line is True, then the screen line will be cleared
 * when the end of the line has been reached.
line_print (ly, line) is put_line (ly, line, 0, True, False)
 */
void
put_line (ly, line, offset, clear_line, positioning)
  int ly;		/* current screen line */
  LINE * line;		/* Line to print */
  int offset;		/* Offset to start if positioning == False */
			/* position if positioning == True */
  FLAG clear_line;	/* Clear to eoln if True */
  FLAG positioning;	/* positioning inside line if True (for prop. fonts) */
{
  char * textp = line->text;
  char * parap;
  int count = get_shift (line->shift_count) * - SHIFT_SIZE;
  int count_ini = count;
  int tab_count;			/* Used in tab expansion */
  int tab_mid;
  int offset_start;
  int offset_stop;
  character c;
  unsigned long unichar;
  int follow;
  int charwidth;
  FLAG disp_separate_combining;
  FLAG disp_separate_joining;
  char syntax_marker_was;
  char syntax_marker_new;
  char syntax_byte_printed;
  int script_colour0;
  int last_script_colour;

  int len;
  unsigned long cjkchar;
  int charlen;
  character cjkbytes [5];

  syntax_marker = line->prev->syntax_marker;
  syntax_marker_was = syntax_none;
  syntax_marker_new = syntax_marker;
  script_colour = default_script;

  if (positioning) {
	offset_start = 0;
	offset_stop = offset;
  } else {
	offset_start = offset;
	offset_stop = XBREAK;
  }

/* Skip all chars as indicated by the offset_start and the shift_count field */
  while (count < offset_start) {
	syntax_marker_new = syntax_state (syntax_marker_new, textp);
	if (* textp == '<' || * textp == '>') {
		syntax_marker_was = syntax_marker;
		syntax_marker = syntax_marker_new;
	}
	advance_char_scr (& textp, & count, line->text);
  }

  if (standout_glitch) {
	clear_eol ();
	set_scrollbar_dirty (ly);
	disp_normal ();
  }

  if (count == 0 && count_ini < 0 && SHIFT_BEG != '\0') {
	putmark (SHIFT_BEG, NIL_PTR);
	if (! is_tab (* textp)) {
		syntax_marker_new = syntax_state (syntax_marker_new, textp);
		if (* textp == '<' || * textp == '>') {
			syntax_marker_was = syntax_marker;
			syntax_marker = syntax_marker_new;
		}
		advance_char_scr (& textp, & count, line->text);
	} else {
		count ++;
	}
  }
/* Check for shift-out left across double-width character boundary */
  if (count_ini < 0 && count > offset_start) {
	putmark (TABchar, NIL_PTR);
  }

  if (utf8_text) {
	utf8_info (textp, & follow, & unichar);
	charwidth = uniscrwidth (unichar, textp, line->text);
  } else if (cjk_text && multichar (* textp) && (* (textp + 1)) != '\n') {
	cjkchar = charvalue (textp);
	charwidth = cjkscrwidth (cjkchar, textp, line->text);
  } else if (mapped_text) {
	charwidth = cjkscrwidth ((character) * textp, textp, line->text);
  } else {
	charwidth = 1;
  }

  restore_attr ();

  while (* textp != '\n' && count < offset_stop + 1 - charwidth) {
	if (dim_HTML) {
		syntax_marker_new = syntax_state (syntax_marker_new, textp);
		if (* textp == '<' || * textp == '>') {
			syntax_marker_was = syntax_marker;
			syntax_marker = syntax_marker_new;
		}
		if (* textp == '<') {
			disp_syntax (syntax_marker_was, syntax_marker);
		}
	}

	if (is_tab (* textp)) {		/* Expand tabs to spaces */
		put_uniend ();
		tab_count = tab (count);
		tab_mid = count + (tab_count - count) / 2 + 1;
		if (cjk_term) {
			if ((count & 1) == 1) {
				count ++;
				if (cjk_tab_width == 1) {
					putCJKmark (CJK_TAB_MARK);
				} else {
					__putchar (' ');
				}
			}
			while (count < offset_stop && count < tab_count) {
				putCJKmark (CJK_TAB_MARK);
				count ++;
				if (cjk_tab_width == 2) {
					count ++;
				}
			}
		} else if (TABchar0 != '\0') {
			if (TABchar0 != TABchar) {
				count ++;
				putmarkmode (TABchar0, UTF_TAB0);
			}
			while (count < offset_stop && count < tab_count - 1) {
				count ++;
				putmarkmode (TABchar, UTF_TAB);
			}
			if (count < offset_stop && count < tab_count) {
				count ++;
				if (TABchar2 != '\0') {
					putmarkmode (TABchar2, UTF_TAB2);
				} else {
					putmarkmode (TABchar, UTF_TAB);
				}
			}
		} else while (count < offset_stop && count < tab_count) {
			count ++;
			if (count == tab_mid && TABcharmid != '\0') {
				putmarkmode (TABcharmid, UTF_TABmid);
			} else {
				putmarkmode (TABchar, UTF_TAB);
			}
		}
		endmarkmode ();
		restore_attr ();

		textp ++;
	} else if (iscontrol (* textp)
		|| ((character) * textp == 0xA0 /* nbsp */
		    && ! pc_term
		    && utf8_text == False && cjk_text == False
		    && ! mapped_text
		   )
		)
	{
		put_uniend ();
		if (use_vt100_block_graphics && display_block_graphics
		    && (
			   * textp == 1
			|| (* textp >= 0x0B && * textp <= 0x0F)
			|| * textp == 0x12
			|| (* textp >= 0x15 && * textp <= 0x19)
		)) {
			altcset_on ();
			putcharacter (* textp + 0x5F);
			altcset_off ();
		} else if ((character) * textp == 0xA0 /* nbsp */) {
			indicate_special ('');
		} else {
			ctrldisp_on ();
			putcharacter (controlchar (* textp));
			ctrldisp_off ();
		}
		restore_attr ();

		textp ++;
		count ++;
	} else {
		syntax_byte_printed = * textp;

		if (utf8_text) {
			follow --;
			c = * textp ++;

			last_script_colour = script_colour;
			map_script_colour (script (unichar), 
				& script_colour, & script_colour0);
			if (script_colour != last_script_colour) {
				if (script_colour0 > 0) {
					disp_script (script_colour0);
				}
				if (colours_256 || script_colour0 <= 0) {
					disp_script (script_colour);
				}
			}

			disp_separate_combining = False;
			disp_separate_joining = False;
			if (iscombined (unichar, textp - 1, line->text)) {
			    if (combining_screen && ! combining_mode) {
				/* enforce separated display */
				disp_separate_combining = True;
				combiningdisp_on ();

				if (iswide (unichar)) {
					put_unichar (0x3000);
					count += 2;
				} else {
					if (isjoined (unichar, textp - 1, line->text)) {
						disp_separate_joining = True;
#ifdef mlterm_fixed
						/* Separate display of 
						   Arabic ligature components;
						   don't output base blank 
						   and try to suppress 
						   Arabic ligature shaping
						   with one of
						   200B;ZERO WIDTH SPACE
						   200C;ZERO WIDTH NON-JOINER
						   200D;ZERO WIDTH JOINER
						   FEFF;ZERO WIDTH NO-BREAK SPACE
						   - none works in mlterm
						 */
						put_unichar (0x200C);
#else
						/* Transform character 
						   to its isolated form;
						   see below
						 */
#endif
					} else {
						put_char (' ');
					}
					count += 1;
				}
			    } else {
				if (! combining_screen) {
					disp_separate_combining = True;
					combiningdisp_on ();
					count += charwidth;
				}
			    }
			} else {
				count += charwidth;
			}

#ifndef mlterm_fixed
			if (disp_separate_joining) {
				/* prevent terminal ligature by 
				   substituting joining character 
				   with its ISOLATED FORM */
				put_unichar (isolated_alef (unichar));
				/* skip rest of joining char */
				while (follow > 0 && (* textp & 0xC0) == 0x80) {
					textp ++;
					follow --;
				}
			}
			else
#endif
			{
				put_char (c);
				while (follow > 0 && (* textp & 0xC0) == 0x80) {
					put_char (* textp ++);
					follow --;
				}
			}
			if (disp_separate_combining) {
				combiningdisp_off ();
				restore_attr ();
			}

		} else if (cjk_text) {
			if (multichar (* textp)) {
				len = CJK_len (textp);
				if (utf_cjk_wide_padding || ! utf8_screen) {
					cjkchar = charvalue (textp);
				}
				charlen = cjkencode (cjkchar, cjkbytes);
				trace_put_line (("%04X %d + %d ", cjkchar, count, charwidth));

				while (len > 0 && * textp != '\n' && * textp != '\0') {
					textp ++;
					len --;
					charlen --;
				}
				if (len != 0 || charlen != 0) {
					/* incomplete CJK char */
					indicate_marked ('<');
					count ++;
					if (utf_cjk_wide_padding) {
						if (charlen == 0 || charlen == -3) {
							indicate_marked (' ');
							count ++;
						}
					}
				} else {
					trace_put_line (("- put %04X ", cjkchar));

					unichar = lookup_cjk (cjkchar);
					disp_separate_combining = False;
					if (iscombining (unichar)) {
					    if (combining_screen && ! combining_mode) {
						/* enforce separated display */
						disp_separate_combining = True;
						combiningdisp_on ();

						if (charwidth == 2) {
							put_cjkchar (cjk (0x300A));
						} else {
							put_char (' ');
						}
					    } else {
						if (! combining_screen) {
							disp_separate_combining = True;
							combiningdisp_on ();
						}
					    }
					}

					put_cjkchar (cjkchar);

					if (disp_separate_combining) {
						combiningdisp_off ();
						restore_attr ();
					}

					count += charwidth;
				}
				trace_put_line (("-> %d\n", count));
			} else {
				put_cjkchar ((character) * textp);
				textp ++;
				count ++;
			}
		} else {
			c = * textp ++;

			disp_separate_combining = False;
			if (mapped_text) {
				unichar = lookup_cjk (c);
				if (iscombining (unichar)) {
				    if (combining_screen && ! combining_mode) {
					/* enforce separated display */
					disp_separate_combining = True;
					combiningdisp_on ();

					put_char (' ');
				    } else {
					if (! combining_screen) {
						disp_separate_combining = True;
						combiningdisp_on ();
					}
				    }
				}
			}

			if (mapped_text && unichar == 0xA0 /* nbsp */) {
				indicate_special ('');
			} else {
				putcharacter (c);
			}

			if (disp_separate_combining) {
				combiningdisp_off ();
				restore_attr ();
			}

			count ++;
		}

		if (dim_HTML && syntax_byte_printed == '>') {
			disp_syntax (syntax_marker_was, syntax_marker);
		}
	}

	if (utf8_text) {
		utf8_info (textp, & follow, & unichar);
		charwidth = uniscrwidth (unichar, textp, line->text);
	} else if (cjk_text && multichar (* textp) && (* (textp + 1)) != '\n') {
		cjkchar = charvalue (textp);
		charwidth = cjkscrwidth (cjkchar, textp, line->text);
	} else if (mapped_text) {
		charwidth = cjkscrwidth ((character) * textp, textp, line->text);
	} else {
		charwidth = 1;
	}
  }
  put_uniend ();

  if (script_colour != default_script) {
	disp_script (default_script);
  }

  if (dim_HTML) {
	/* now reset HTML display attribute */
	disp_syntax (syntax_marker, 0);
  }

  if (positioning) {
	/* self-made cursor for terminals (such as xterm)
	   which have display problems with proportional screen fonts
	   and their cursor */
	reverse_on ();
	if (* textp != '\n') {
		if (utf8_text) {
			put1utfchar (textp);
			put_uniend ();
		} else if (cjk_text) {
			put_cjkchar (charvalue (textp));
		} else {
			putcharacter (* textp);
		}
	} else if (cjk_term) {
		putCJKmark (0x300A);	/* 《 */
	} else if (RET_MARK != '\0') {
		if (PARA_MARK != '\0' && paradisp
		 && line->return_type != lineend_LS
		 && line->return_type != lineend_PS) {
			parap = textp;
			parap --;
			if (* parap == ' ' || * parap == '\t') {
				putret (line->return_type);
			} else {
				putmark (PARA_MARK, UTF_PARA);
			}
		} else {
			putret (line->return_type);
		}
	} else {
		putchar (' ');
	}
	reverse_off ();
	set_cursor (0, 0);
  } else /* (positioning == False) */ {
	/* If line is longer than XBREAK chars, print the shift_mark */
	if (count <= XBREAK && * textp != '\n') {
		if (cjk_term) {
			/*putCJKmark (0x300B);	/* 》 */
			putmark ('>', NIL_PTR);
			count ++;
		} else {
			putmark (SHIFT_MARK, NIL_PTR);
		}
		count ++;
		if (count == XBREAK) {
			putchar (' ');
			count ++;
		}
	}

	/* Mark end of line if so desired */
	if (* textp == '\n') {
	    if (cjk_term) {
		putCJKmark (0x300A);	/* 《 */
		if (cjk_lineend_width == 2) {
			count ++;
		}
		/* to assure it fits, scrollbar_width is set to 1 
		   in CJK terminal mode even if scrollbar is switched off */
	    } else if (RET_MARK != '\0') {
		if (PARA_MARK != '\0' && paradisp
		 && line->return_type != lineend_LS
		 && line->return_type != lineend_PS) {
			parap = textp;
			parap --;
			if (* parap == ' ') {
				putret (line->return_type);
			} else {
				putmark (PARA_MARK, UTF_PARA);
			}
		} else {
			putret (line->return_type);
			if (standout_glitch && 
			    (line->return_type == lineend_LS
			    || line->return_type == lineend_PS)) {
				putchar (' ');
				count ++;
				if (count > XBREAK) {
					set_scrollbar_dirty (ly);
				}
			}
		}

		count ++;
		if (RET_BLANK) {
			while (count < XBREAK) {
				putmark (RET_BLANK, UTF_RETblank);
				count ++;
			}
			if (RET_BLANK2 && count <= XBREAK) {
				putmark (RET_BLANK2, UTF_RETblank2);
				count ++;
			}
		}
	    }
	}

	/* Clear the rest of the line if clear_line is True */
	if (clear_line) {
		set_scrollbar_dirty (ly);
		if (can_clear_eol) {
			if (count <= XBREAK) {
				clear_eol ();
			}
		} else {
			while (count ++ <= XBREAK) {
				/* clear up to XMAX */
				putchar (' ');
			}
		}
	}
  }
}

/*
 * set_cursor_xy sets the cursor by either directly calling set_cursor
 * or, in the case of proportional font support, reprinting the line
 * up to the x position
 */
void
set_cursor_xy ()
{
  if (proportional) {
	set_cursor (0, y);
	if (x != 0) {
		put_line (y, cur_line, x, False, True);
	}
	/* cur_line may still be undefined if x == 0 */
  } else {
	set_cursor (x, y);
  }
}

/*
 * Display line at screen line y_pos if it lies between y_min and y_max.
 * If it is no text line (end of file), clear screen line.
 */
static
void
display_line_at (y_pos, line, y_min, y_max, first)
  int y_pos;
  int y_min;
  int y_max;
  register LINE * line;
  FLAG first;
{
  line = proceed (line, y_pos - y_min);
  if (y_pos >= y_min && y_pos <= y_max) {
	set_cursor (0, y_pos);
	if (line == tail) {
		clear_wholeline ();
	} else {
		if (first == False) {
			if (display_delay >= 0) {
				flush ();
			}
#ifdef msdos
			if (display_delay > 0) {
				delay (display_delay);
			}
#else
#ifdef use_usleep
			if (display_delay > 0) {
				(void) usleep (1000 * display_delay);
			}
#else
			if (display_delay > 0) {
				(void) char_ready_within (display_delay);
			}
#endif
#endif
		}
		line_print (y_pos, line);
	}
  }
}

/*
 * Display () shows count + 1 lines on the terminal starting at the given 
 * coordinates. At end of file, the rest of the screen is blanked.
 * When count is negative, a backwards print from 'line' will be done.
 */
void
display (y_coord, line, count, new_pos)
  int y_coord;
  register LINE * line;
  register int count;
  int new_pos;
{
  int y_max = y_coord + count;
  int y_off;

/* Find new startline if count is negative */
  if (count < 0) {
	line = proceed (line, count);
	count = - count;
  }

#ifdef obsolete
  clean_menus ();
  This led to recursive calls of display () resulting in wrong display;
  menus are now cleared before invoking functions from menu.
#endif

  display_line_at (new_pos, line, y_coord, y_max, True);
  y_off = 0;
  while (y_off < count) {
	y_off ++;
	display_line_at (new_pos - y_off, line, y_coord, y_max, False);
	display_line_at (new_pos + y_off, line, y_coord, y_max, False);
  }
#ifdef UNUSED
/* old code, building the display from top to bottom (how boring): */
/* with this code, XBREAK must be set to XMAX - 1 */
/* Print the lines */
  set_cursor (0, y_coord);
  while (line != tail && count -- >= 0) {
	line_print (y_coord, line);
	line = line->next;
	y_coord ++;
  }

/* Print the blank lines (if any) */
  if (loading == False) {
	while (count -- >= 0) {
		clear_eol ();
		putchar ('\n');
	}
  }
#endif
}


/*======================================================================*\
|*				End					*|
\*======================================================================*/
