/* mg-canvas-field.c
 *
 * Copyright (C) 2002 - 2004 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "mg-canvas.h"
#include "mg-canvas-field.h"
#include <libmergeant/mg-defs.h>
#include <libmergeant/mg-field.h>
#include <libmergeant/mg-db-table.h>
#include <libmergeant/mg-db-field.h>
#include <libmergeant/mg-query.h>

static void mg_canvas_field_class_init (MgCanvasFieldClass * class);
static void mg_canvas_field_init       (MgCanvasField * drag);
static void mg_canvas_field_finalize   (GObject   * object);

static void mg_canvas_field_set_property (GObject              *object,
					  guint                 param_id,
					  const GValue         *value,
					  GParamSpec           *pspec);
static void mg_canvas_field_get_property (GObject              *object,
					  guint                 param_id,
					  GValue               *value,
					  GParamSpec           *pspec);

static void mg_canvas_field_extra_event  (MgCanvasItem * citem, GdkEventType event_type);
enum
{
	PROP_0,
	PROP_FIELD,
};

struct _MgCanvasFieldPrivate
{
	MgField *field;
};

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass *field_parent_class = NULL;

guint
mg_canvas_field_get_type (void)
{
	static GType type = 0;

        if (!type) {
		static const GTypeInfo info = {
			sizeof (MgCanvasFieldClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) mg_canvas_field_class_init,
			NULL,
			NULL,
			sizeof (MgCanvasField),
			0,
			(GInstanceInitFunc) mg_canvas_field_init
		};		

		type = g_type_register_static (MG_CANVAS_TEXT_TYPE, "MgCanvasField", &info, 0);
	}

	return type;
}

	

static void
mg_canvas_field_class_init (MgCanvasFieldClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);

	field_parent_class = g_type_class_peek_parent (class);

	object_class->finalize = mg_canvas_field_finalize;
	MG_CANVAS_ITEM_CLASS (class)->extra_event = mg_canvas_field_extra_event;

	/* Properties */
	object_class->set_property = mg_canvas_field_set_property;
	object_class->get_property = mg_canvas_field_get_property;

	g_object_class_install_property
                (object_class, PROP_FIELD,
                 g_param_spec_pointer ("field", NULL, NULL, (G_PARAM_READABLE | G_PARAM_WRITABLE)));
}

static void
mg_canvas_field_init (MgCanvasField * field)
{
	field->priv = g_new0 (MgCanvasFieldPrivate, 1);
	field->priv->field = NULL;
}

static void
field_nullified_cb (MgField *field, MgCanvasField *cf)
{
	g_signal_handlers_disconnect_by_func (G_OBJECT (field),
					      G_CALLBACK (field_nullified_cb), cf);
	gtk_object_destroy (GTK_OBJECT (cf));
}

static void
mg_canvas_field_finalize (GObject   * object)
{
	MgCanvasField *cf;
	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_MG_CANVAS_FIELD (object));

	cf = MG_CANVAS_FIELD (object);
	if (cf->priv) {
		if (cf->priv->field) {
			g_signal_handlers_disconnect_by_func (G_OBJECT (cf->priv->field),
							      G_CALLBACK (field_nullified_cb), cf);
			cf->priv->field = NULL;
		}
		g_free (cf->priv);
		cf->priv = NULL;
	}

	/* for the parent class */
	field_parent_class->finalize (object);
}

static void 
mg_canvas_field_set_property    (GObject              *object,
				 guint                 param_id,
				 const GValue         *value,
				 GParamSpec           *pspec)
{
	MgCanvasField *cf;
	gpointer ptr;
	GString *string;
	MgServerDataType *dtype;

	cf = MG_CANVAS_FIELD (object);

	switch (param_id) {
	case PROP_FIELD:
		ptr = g_value_get_pointer (value);
		MgEntity *ent;
		g_return_if_fail (IS_MG_FIELD (ptr));

		/* signals management */
		if (cf->priv->field) {
			g_signal_handlers_disconnect_by_func (G_OBJECT (ptr),
							      G_CALLBACK (field_nullified_cb), cf);
			cf->priv->field = NULL;
		}

		cf->priv->field = MG_FIELD (ptr);
		g_signal_connect (G_OBJECT (ptr), "nullified",
				  G_CALLBACK (field_nullified_cb), cf);
		
		/* attributes setting */
		string = g_string_new ("");
		dtype = mg_field_get_data_type (MG_FIELD (ptr));
		g_string_append_printf (string, _("Type: %s"), mg_base_get_name (MG_BASE (dtype)));

		g_object_set (object, "mg_base", ptr, NULL);
		ent = mg_field_get_entity (MG_FIELD (ptr));
		if (IS_MG_DB_TABLE (ent)) {
			gboolean nullok = mg_db_field_is_null_allowed (MG_DB_FIELD (ptr));
			gboolean pkpart = mg_db_field_is_pkey_part (MG_DB_FIELD (ptr));
			g_object_set (object, 
				      "highlight_color", MG_CANVAS_DB_TABLE_COLOR, 
				      "text_underline", !nullok,
				      "text_bold", pkpart,
				      NULL);
			if (mg_db_field_is_pkey_alone (MG_DB_FIELD (ptr)))
				g_string_append (string, _("\nPrimary key"));
			else
				if (pkpart)
					g_string_append (string, _("\nPart of primary key"));
				
		}
		if (IS_MG_QUERY (ent))
			g_object_set (object, "highlight_color", MG_CANVAS_QUERY_COLOR, NULL);

		g_object_set (object, "tip_text", string->str, NULL);
		g_string_free (string, TRUE);
		break;
	}
}

static void 
mg_canvas_field_get_property    (GObject              *object,
				 guint                 param_id,
				 GValue               *value,
				 GParamSpec           *pspec)
{
	MgCanvasField *cf;
	gpointer ptr;

	cf = MG_CANVAS_FIELD (object);

	switch (param_id) {
	case PROP_FIELD:
		g_object_get (object, "mg_base", &ptr, NULL);
		g_value_set_pointer (value, ptr);
		break;
	}
}

static void
mg_canvas_field_extra_event  (MgCanvasItem *citem, GdkEventType event_type)
{
	if (event_type == GDK_LEAVE_NOTIFY)
		mg_canvas_text_set_highlight (MG_CANVAS_TEXT (citem), FALSE);
}


/**
 * mg_canvas_field_get_field
 * @cfield: a #MgCanvasField object
 *
 * Get the #MgField which @cfield represents
 *
 * Returns: the object implementing the #MgField interface
 */
MgField *
mg_canvas_field_get_field (MgCanvasField *cfield)
{
	g_return_val_if_fail (cfield && IS_MG_CANVAS_FIELD (cfield), NULL);
	g_return_val_if_fail (cfield->priv, NULL);

	return cfield->priv->field;
}

