/*
 * medussa - a distributed cracking system
 * Copyright (C) 1999 Kostas Evangelinos <kos@bastard.net>
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

/*
 * $Id: cli.c,v 1.11 2000/11/14 03:57:02 kos Exp $
 *
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <unistd.h>
#if HAVE_LIBREADLINE
#include <readline/readline.h>
#include <readline/history.h>
#endif

#include "cli.h"
#include "xmalloc.h"
#include "array.h"
#include "common.h"

cli_t *
cli_init(void *up) {
  cli_t *c;

  c = (cli_t *)xcalloc(1, sizeof(cli_t));
  
  c->up = up;
  c->cmds = array_init(sizeof(cmd_t));
  strncpy(c->prompt, CLI_DEF_PROMPT, CLI_LINELEN);
  strncpy(c->parsesep, CLI_DEF_PARSESEP, CLI_LINELEN);
  strncpy(c->lastcommand, "?", CLI_LINELEN);
  return c;
}

int
cli_destroy(cli_t *c) {
  array_nuke(c->cmds);
  free(c);
  return 0;
}

int
cli_register(cli_t *c, int type, cmd_proc proc, char *name, ...) {
  cmd_t cmd;
  va_list vl;
  char *p;

  va_start(vl, name);
  strncpy(cmd.name, name, CLI_LINELEN);
  strncpy(cmd.help, "No help", CLI_LINELEN);
  cmd.proc = proc;
  cmd.type = type;

  switch(type) {
  case CLI_PARSE:
    cmd.match = 0;
    cmd.argmin = va_arg(vl, int);
    cmd.argmax = va_arg(vl, int);
    if((p = va_arg(vl, char *)))
      strncpy(cmd.help, p, CLI_LINELEN);
    break;
  case CLI_RAW:
    break;
  default:
    va_end(vl);
    return 2;
  }

  va_end(vl);
  array_add(c->cmds, &cmd);
  return 0;
}

int
cli_set(cli_t *c, int type, ...) {
  va_list vl;

  va_start(vl, type);
  switch(type) {
  case CLI_PROMPT:    
    strncpy(c->prompt, va_arg(vl, char *), CLI_LINELEN);
    break;
  case CLI_PARSESEP:
    strncpy(c->parsesep, va_arg(vl, char *), CLI_LINELEN);
    break;
  default:
    va_end(vl);
    return 1;
    break;
  }

  va_end(vl);
  return 0;
}

int
cli_parse(cli_t *c, char *line, char **args, int *nargs) {
  
  *nargs = 0;
  for(args[(*nargs)++]=strtok(line, " \t\n"); (args[(*nargs)]=strtok(NULL, " \t\n")); (*nargs)++)
    ;

  return 0;
}

int
cli_zap_matches(cli_t *c) {
  int i;
  cmd_t *cmd;

  for(i=0; (cmd=array_get(c->cmds, i)); i++)
    cmd->match = 0;
  return 0;
}

int
cli_find_matches(cli_t *c, char *what) {
  cmd_t *cmd;
  int i;
  int len;
  int nmatches;

  nmatches = 0;
  len = strlen(what);
  for(i=0; (cmd=array_get(c->cmds, i)); i++)
    if(!strncmp(cmd->name, what, MIN(len, CLI_LINELEN))) {
      cmd->match = 1;
      nmatches++;
    }
  return nmatches;
}

int
cli_despatch(cli_t *c, char **args, int nargs) {
  cmd_t *cmd;
  int i;

  if(!nargs)
    return 2;
    
  if(!strcmp(args[0], "?")) {
    if(nargs==2) {
      for(i=0; (cmd=array_get(c->cmds, i)); i++) {
	if(!strcmp(cmd->name, args[1]))
	  fprintf(stdout, "%-12s %-67s\n", cmd->name, cmd->help);
      }
    } else {
      for(i=0; (cmd=array_get(c->cmds, i)); i++) {
	fprintf(stdout, "%-12s %-67s\n", cmd->name, cmd->help);
      }
    }
    return 0;
  }

  cli_zap_matches(c);
  switch(cli_find_matches(c, args[0])) {
  case 0:
    fprintf(stdout, "Unknown command \"%s\"\n", args[0]);
    return 1;
    break;
  case 1:
    for(i=0; (cmd=array_get(c->cmds, i)); i++) {
      if(cmd->match) {
	if(cmd->argmin && nargs < cmd->argmin) {
	  fprintf(stdout, "Command %s needs at least %d arguments\n", cmd->name, cmd->argmin);
	  return 2;
	}
	if(cmd->argmax && nargs > cmd->argmax) {
	  fprintf(stdout, "Command %s needs at most %d arguments\n", cmd->name, cmd->argmax);
	  return 3;
	}
	if(cmd->proc(c->up, nargs, args))
	  return -1;
	return 0;
      }
    }
  default:
    fprintf(stdout, "Ambiguous command \"%s\": ", args[0]);
    for(i=0; (cmd=array_get(c->cmds, i)); i++) {
      if(cmd->match)
	fprintf(stdout, "%s ", cmd->name);
    }
    fprintf(stdout, "\n");
    return 4;
  }
}

int
cli_main(cli_t *c) {
#if HAVE_LIBREADLINE
  char *pline;
#endif
  char line[CLI_LINELEN];
  int nargs;
  char *args[CLI_MAXARGS];
  int result;
  
#if HAVE_LIBREADLINE
  if(!(pline = readline(c->prompt)))
    return 1;
  strncpy(line, pline, CLI_LINELEN);
  free(pline);
#else
  if(isatty(0)) {
    fwrite(c->prompt, strlen(c->prompt), 1, stdout);
    fflush(stdout);
  }
    
  if(!fgets(line, CLI_LINELEN, stdin))
    return 1;    
#endif

  zap_whitespace(line, CLI_LINELEN, ZAP_LEFT|ZAP_RIGHT);
  if(line[0] == '\0')
    strncpy(line, c->lastcommand, CLI_LINELEN);
  else
    strncpy(c->lastcommand, line, CLI_LINELEN);
  
#if HAVE_LIBREADLINE
  add_history(line);
#endif
  cli_parse(c, line, args, &nargs);
  result = cli_despatch(c, args, nargs);
  
  if(result < 0)
    return 1;
  else
    return 0;
}

#ifdef _CLI_DEBUG
int
command1(int nargs, char **args) {
  printf("This is command1\n");
}

int
command2(int nargs, char **args) {
  printf("This is command2\n");
}

int
main(int argc, char **argv) {
  cli_t cli;

  cli_new(&cli);
  cli_set(&cli, CLI_PROMPT, "test> ");
  cli_set(&cli, CLI_PARSESEP, " ");

  cli_register(&cli, CLI_PARSE, command1, "command1", 1, 1);
  cli_register(&cli, CLI_RAW, command2, "command2");
  cli_main(&cli);
  
  cli_free(&cli);
}
#endif
