/*
  MeCab -- Yet Another Part-of-Speech and Morphological Analyzer
 
  $Id: dictionary.h,v 1.7 2004/03/08 07:40:51 taku-ku Exp $;

  Copyright (C) 2001-2004 Taku Kudo <taku-ku@is.aist-nara.ac.jp>
  This is free software with ABSOLUTELY NO WARRANTY.
  
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/  
#ifndef _MECAB_DICTIONARY_H
#define _MECAB_DICTIONARY_H

#include "mecab.h"
#include "darts.h"
#include "mmap.h"
#include "common.h"

namespace MeCab {
   
  // Language Independent Dictionary 
  class Dictionary 
  {
  private:
    Mmap<Token> tmmap;
    Mmap<char>  immap;
    Mmap<char>  dmmap;
    Darts::DoubleArray da;
    Token *result   [512];
    Darts::DoubleArray::result_type daresult [256];
    std::string _what;

  public:
    bool open (const char *prefix)
    {
      try {
	std::string dmapfileName = createFileName (std::string (prefix), DOUBLE_ARRAY_FILE);
	if (! dmmap.open (dmapfileName.c_str())) throw std::runtime_error (dmmap.what());
	da.setArray ((void *)dmmap.begin());

	std::string tmapfileName = createFileName (std::string (prefix), TOKEN_FILE);
	if (! tmmap.open (tmapfileName.c_str())) throw std::runtime_error (tmmap.what());

	std::string imapfileName = createFileName (std::string (prefix), POS_FILE);
	if (! immap.open (imapfileName.c_str())) throw std::runtime_error (immap.what());

	return true;
      }

      catch (exception &e) {
	close ();
	_what =  std::string ("Dictionary::open(): ") + e.what ();
	return false;
      }
    }

    bool close () 
    {
       dmmap.close();
       tmmap.close();
       immap.close();
       return true;
    }

    inline Token **commonPrefixSearch (const char* str, unsigned int len) 
    {
      unsigned int size = 0;
      unsigned int n = da.commonPrefixSearch (str, daresult, 256, len);

      for (unsigned int i = 0; i < n; i++) {
	unsigned int k = 0xff & daresult[i];
	unsigned int p = daresult[i] >> 8;
	for (unsigned int j = 0; j < k; j++) result[size++] = &tmmap[p++];
      }

      result[size] = 0; // null terminate
      return result;
    }

    inline Token **exactMatchSearch (const char* str, unsigned int len) 
    {
      unsigned int size = 0;
      int n = da.exactMatchSearch (str, len);

      if (n != -1) {
	unsigned int k = 0xff & n;
	unsigned int p = n >> 8;
	for (unsigned int i = 0; i < k; i++) result[size++] = &tmmap[p++];
      }

      result[size] = 0; // null terminate
      return result;
    }

    const char *getFeature () { return immap.begin(); } // ad-hoc.. mmm.
    const char *what       () { return _what.c_str(); } 

    Dictionary (const char *prefix)
    {
      if (! open (prefix)) throw std::runtime_error (_what);
    }

    Dictionary   () {};
    ~Dictionary  () { this->close (); };
  };
}
#endif
