/*
 * PipeExec.xs
 *
 * $Header: //sapdb/TOOLSRC/develop/sys/src/tt/cpan/SAPDB/Install/PipeExec.xs#1 $
 * $DateTime: 2002/04/11 13:12:05 $
 * $Change: 19335 $
 */

static char szWhatHeader[] =
        "@(#) $Header: //sapdb/TOOLSRC/develop/sys/src/tt/cpan/SAPDB/Install/PipeExec.xs#1 $";

#ifndef __cplusplus
#ifndef HAS_BOOL
typedef char bool;
#endif
#endif

#include "EXTERN.h"
#include "perl.h"
#include "XSUB.h"

/*
 * expect compiler warning 4113 when using Microsoft C
 * in c code generated by xs preprocessor
 */
#if _MSC_VER >= 1200
#pragma warning(disable : 4113)
#endif

#ifndef WIN32
#define UNIX UNIX
#endif

#define BUFFERSIZE (16 * 1024)

typedef struct {
#ifdef UNIX
	int pid;
	int fd_stdin;
	int fd_stdout;
#endif
#ifdef WIN32
	HANDLE h_process;
	HANDLE h_thread;
	HANDLE h_stdin;
	HANDLE h_stdout;
#endif
	char *errtext;
	char buffer [BUFFERSIZE + 1];
} sess_t;

#if defined _WIN64
typedef __int64 ptr_t; 
#else
typedef long ptr_t; 
#endif

static int open_pipe (sess_t *, char *);
static int readline_pipe (sess_t *, char *buffer);
static int close_pipe (sess_t *);

#ifdef UNIX
#include <errno.h>
#include <fcntl.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include <utime.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <pwd.h>

#ifdef _AIX
#define NEED_PRAGMA_ALLOCA NEED_PRAGMA_ALLOCA
#endif
#if defined UNIX && !defined _AIX && !defined SNI
#define NEED_ALLOCA_H NEED_ALLOCA_H
#endif

#ifdef NEED_ALLOCA_H
#include <alloca.h>
#endif

#ifdef NEED_PRAGMA_ALLOCA
#pragma alloca
#endif

#if defined __hpux || defined sun || defined _AIX
#define NEED_SYS_ERRLIST_PROTOTYPE NEED_SYS_ERRLIST_PROTOTYPE
#endif

#ifdef NEED_SYS_ERRLIST_PROTOTYPE
extern char *sys_errlist[];
#endif

#ifndef HAS_SETEUID
#ifdef HAS_SETRESUID
static
int seteuid (uid_t euid) {
	return setresuid (-1, euid, -1);
}
#endif
#endif

#ifndef HAS_SETEGID
#ifdef HAS_SETRESGID
static
int setegid (gid_t egid) {
	return setresgid (-1, egid, -1);
}
#endif
#endif

/*
 * prepare argv form cmdline
 * parameter will have reverse order when poping from perl stack
 */
#define prepare_argv(cmd) { \
	int rc; \
	int i; \
	char *ptr; \
	\
	dSP; \
	ENTER; \
	SAVETMPS; \
	PUSHMARK (sp); \
	XPUSHs (sv_2mortal (newSVpv (cmd, strlen (cmd)))); \
	PUTBACK; \
	\
	rc = perl_call_pv ( \
		"SAPDB::Install::PipeExec::PrepareArgv", G_ARRAY); \
	\
	SPAGAIN; \
	\
	if (rc >= 2) { \
		rc--; \
		argv = alloca ((rc + 1) * sizeof (char *)); \
		\
		argv[rc] = 0; \
		\
		for (i = 0; i < rc; i++ ) { \
			ptr = POPp; \
			argv[rc - i- 1] = alloca (strlen (ptr) + 1); \
			strcpy (argv[rc - i - 1], ptr); \
		}; \
		file = alloca (strlen (ptr) + 1); \
		strcpy (file, ptr); \
		errmsg = 0; \
	} else { \
		ptr = POPp; \
		errmsg = alloca (strlen (ptr) + 1); \
		strcpy (errmsg, ptr); \
		argv = 0; \
		file = 0; \
	} \
	\
	PUTBACK; \
	FREETMPS; \
	LEAVE; \
}

static sess_t *
new_pipe () {
	sess_t *sess;

	sess = (sess_t *) malloc (sizeof (sess_t));
	if (sess == 0)
		return 0;

	memset (sess, 0, sizeof (sess_t));
	sess->fd_stdout = -1;
	sess->fd_stdin = -1;
	return (sess);
}

static int
open_pipe (sess_t *sess, char *cmd) {
	int pid;
	int fd0[2];
	int fd1[2];
	int rc;
	char **argv;
	char *file;
	char *errmsg;
	uid_t uid;
	uid_t euid;
	gid_t gid;
	gid_t egid;
	struct passwd * pwd;

	rc = pipe (fd0);
	if (rc != 0) {
		char *msg = "cannot create pipe0\n";
		sess->errtext = malloc (strlen (msg) + 1);
		strcpy (sess->errtext, msg);
		return -1;
	}

	rc = pipe (fd1);
	if (rc != 0) {
		char *msg = "cannot create pipe1\n";
		sess->errtext = malloc (strlen (msg) + 1);
		strcpy (sess->errtext, msg);
		return -1;
	}

	uid = getuid ();
	euid = geteuid ();

	gid = getgid ();
	egid = getegid ();

	/*
	 * prepare argv
	 */
	prepare_argv (cmd);
	if (file == 0) {
		char *msg0 = ": cannot execute\n";
		char *msg1 = "\n";
		if (errmsg != 0 && strlen (errmsg) > 0) {
			sess->errtext = malloc 
			(strlen (errmsg) + (strlen (msg1)) + 1);
			strcpy (sess->errtext, errmsg);
			strcat (sess->errtext, msg1);
		} else {
			sess->errtext = malloc 
			(strlen (msg0) + strlen (cmd) + 1);
			strcpy (sess->errtext, cmd);
			strcat (sess->errtext, msg0);
		}
		return -1;
	}
	
	pid = fork ();
	if (pid < 0) {
		/* fork failed */
		char *msg0 = "fork failed: ";
		char *msg1 = "\n";
		malloc (strlen (msg0) + strlen (msg1) + strlen (cmd) + 1);
		strcpy (sess->errtext, msg0);
		strcat (sess->errtext, sys_errlist[errno]);
		strcat (sess->errtext, msg1);
		return -1;
	}
	
	if (pid == 0) {
		/* child */

		/*
	 	 * readirect stdin, stdout and stderr
		 */
		close (fd1[0]);
		close (fd0[1]);
		
		close (0);
		close (1);
		close (2);

		dup2 (fd0[0], 0);
		dup2 (fd1[1], 1);
		dup2 (fd1[1], 2);
		
		/* 
		 * if euid or egid say we should execute program 
		 * as unprivileged user or group switch real uid 
		 * and real gid to unprivileged user and group
		 */
		if ((uid == 0) && (gid != egid || uid != euid)) {
									
			if (seteuid (0) != 0) {
				printf ("cannot set euid to 0: %s\n",
				        sys_errlist[errno]);
				exit (-1);
			}
			if (setegid (0) != 0) {
				printf ("cannot set egid to 0: %s\n", 
				        sys_errlist[errno]);
				exit (-1);
			}
			if (setgid (egid) != 0) {
				printf ("cannot set gid to %d: %s\n", 
				        egid, sys_errlist[errno]);
				exit (-1);
			}
	
			
			pwd = getpwuid(euid);
            
			if(pwd == 0){
				printf("cannot get user name: %s",sys_errlist[errno]);
                		exit (-1);
            		}
            
			if(initgroups(pwd->pw_name, egid) != 0){
				printf("cannot initialize groups: %s\n",sys_errlist[errno]);
				exit (-1);
			}
		
			if (setuid (euid) != 0) {
				printf ("cannot set uid to %d: %s\n", 
				        euid, sys_errlist[errno]);
				exit (-1);
			}
		}	

		/*
		 * exec program
		 */
		execv (file, argv);
		printf ("%s: %s\n", file, sys_errlist[errno]);
		exit (-1);
	} else {
		/* parent */
		close (fd1[1]);
		close (fd0[0]);

		sess->fd_stdin = fd0[1];
		sess->fd_stdout = fd1[0];
		sess->buffer[0] = '\0';
		sess->pid = pid;
		return (0);
	}
}

static int
close_pipe (sess_t *sess) {
	int wpid;
	int status;

	close (sess->fd_stdin);
	close (sess->fd_stdout);

	wpid = waitpid (sess->pid, &status, WNOHANG);
	if (wpid == 0) {
		kill (sess->pid, SIGTERM);
		wpid = waitpid (sess->pid, &status, 0);
		if (wpid < 0) {
			return -1;
		}
	}

	if (WIFSIGNALED (status)) {
		char *msg0 = "signal %2d";
		char *msg1 = " core dumped\n";
		char *msg2 = "\n";
		if (status & 0x80) {
			sess->errtext = 
			malloc (strlen (msg0) + strlen (msg1) + 1);
			sprintf (sess->errtext, msg0, status & 0x7f);
			strcat (sess->errtext, msg1);
		} else {
			sess->errtext = 
			malloc (strlen (msg0) + strlen (msg2) + 1);
			sprintf (sess->errtext, msg0, status & 0x7f);
			strcat (sess->errtext, msg2);
		}
		return -1;
	}
	
	return (WEXITSTATUS (status));
}

static int
readline_pipe (sess_t *sess, char *buffer) {
	int rc;
	char *ptr;
	int len;

	ptr = strchr (sess->buffer, (int) '\n');
	if (ptr == 0) {
		ptr = sess->buffer + strlen (sess->buffer);
		rc = read (sess->fd_stdout, ptr, BUFFERSIZE);
		if (rc  < 0) {
			char *msg0 = "read error: ";
			char *msg1 = "\n";
			sess->errtext = 
			malloc (strlen (msg0) + strlen (msg1) + 
			        strlen (sys_errlist[errno]) + 1);
			strcpy (sess->errtext, msg0);
			strcat (sess->errtext, sys_errlist[errno]);
			strcat (sess->errtext, msg1);
			return -1;
		}
		ptr[rc] = '\0';
	}

	ptr = strchr (sess->buffer, '\n');
	if (ptr != 0) {
		len = ptr - sess->buffer;
		memcpy (buffer, sess->buffer, len);
		buffer[len] = '\0'; 
		memcpy (sess->buffer, sess->buffer + len + 1, BUFFERSIZE - len);
		strcat (buffer, "\n");
		len++;
	} else if (strlen (sess->buffer) != 0) {
		strcpy (buffer, sess->buffer);
		sess->buffer[0] = '\0';
		strcat (buffer, "\n");
		len = strlen (buffer);
	} else {
		buffer[0] = '\0';
		len = 0;
	}

	return (len);	
}

#elif defined WIN32

#include <string.h>

static char *get_errtxt ();
static void free_errtxt (char *);
static int put_err (sess_t *, char *, char *);

static sess_t *
new_pipe () {
	sess_t *sess;

	sess = (sess_t *) malloc (sizeof (sess_t));
	if (sess == 0) {
		return 0;
	}

	memset (sess, 0, sizeof (sess_t));
	sess->h_stdout = INVALID_HANDLE_VALUE;
	sess->h_stdin = INVALID_HANDLE_VALUE;
	sess->h_process = INVALID_HANDLE_VALUE;
	sess->h_thread = INVALID_HANDLE_VALUE;

	return (sess);
}

static int
open_pipe (sess_t *sess, char *cmd) {
	int rc;
	SECURITY_ATTRIBUTES sa_pipe;
	HANDLE h_child_stdout_rd, h_child_stdout_rd_dup;
	HANDLE h_child_stdin_rd, h_child_stdin_wr;
	HANDLE h_child_stdin_wr_dup, h_child_stdout_wr;
	HANDLE h_stdin_sav, h_stdout_sav, h_stderr_sav;
	PROCESS_INFORMATION pi;
	STARTUPINFO si;

	/*
	 * create stdout/stderr pipe
	 */
	h_stdout_sav = GetStdHandle (STD_OUTPUT_HANDLE); 
	h_stderr_sav = GetStdHandle (STD_ERROR_HANDLE); 

	memset (&sa_pipe, 0, sizeof (SECURITY_ATTRIBUTES));
	sa_pipe.nLength = sizeof (SECURITY_ATTRIBUTES);
	sa_pipe.lpSecurityDescriptor = 0;
	sa_pipe.bInheritHandle = TRUE;

	rc = CreatePipe (&h_child_stdout_rd, &h_child_stdout_wr, &sa_pipe, 0);
	if (rc == 0) {
		put_err (sess, cmd, "cannot create stdout pipe");
		return -1;
	}

	rc = SetStdHandle (STD_OUTPUT_HANDLE, h_child_stdout_wr);
	if (rc == 0) {
		put_err (sess, cmd, "cannot redirect stdout to pipe");
		return -1;
	}

	rc = SetStdHandle (STD_ERROR_HANDLE, h_child_stdout_wr);
	if (rc == 0) {
		put_err (sess, cmd, "cannot redirect stderr to pipe");
		return -1;
	}

	rc = DuplicateHandle (
	     GetCurrentProcess (), h_child_stdout_rd,
	     GetCurrentProcess (), &h_child_stdout_rd_dup,
		 0, FALSE, DUPLICATE_SAME_ACCESS);
	if (rc == 0) {
		put_err (sess, cmd, "cannot set stdout pipe to noninheritable");
		return -1;
	}

	rc = CloseHandle (h_child_stdout_rd);
	if (rc == 0) {
		put_err (sess, cmd, "cannot close inheritable stdout pipe");
		return -1;
	}

	/*
	 * create stdin pipe
	 */
	h_stdin_sav = GetStdHandle (STD_INPUT_HANDLE); 

	memset (&sa_pipe, 0, sizeof (SECURITY_ATTRIBUTES));
	sa_pipe.nLength = sizeof (SECURITY_ATTRIBUTES);
	sa_pipe.lpSecurityDescriptor = 0;
	sa_pipe.bInheritHandle = TRUE;

	rc = CreatePipe (&h_child_stdin_rd, &h_child_stdin_wr, &sa_pipe, 0);
	if (rc == 0) {
		put_err (sess, cmd, "cannot create stdin pipe");
		return -1;
	}

	rc = SetStdHandle (STD_INPUT_HANDLE, h_child_stdin_rd); 
	if (rc == 0) {
		put_err (sess, cmd, "cannot redirect stdin to pipe");
		return -1;
	}

	rc = DuplicateHandle (
	     GetCurrentProcess (), h_child_stdin_wr, 
	     GetCurrentProcess (), &h_child_stdin_wr_dup,
	     0, FALSE, DUPLICATE_SAME_ACCESS); 
	if (rc == 0) {
		put_err (sess, cmd, "cannot set stdout pipe to noninheritable");
		return -1;
	}
	
	rc = CloseHandle (h_child_stdin_wr);
	if (rc == 0) {
		put_err (sess, cmd,  "cannot close inheritable stdin pipe\n");
		return -1;
	}

	/*
	 * create child process
	 */
	memset (&si, 0, sizeof(si)); 
	si.cb = sizeof(si);

	rc = CreateProcess (0, cmd, 0, 0, 
	                   TRUE, 0, 0, 0, &si, &pi);
	if (rc == 0) {
		put_err (sess, cmd, "cannot create process");
		return -1;
	}

	/*
	 * restore handles
	 */
	rc = SetStdHandle (STD_INPUT_HANDLE, h_stdin_sav);
	if (rc == 0) {
		put_err (sess, cmd, "cannot restore stdin handle");
		return -1;
	}
 
	rc = SetStdHandle (STD_OUTPUT_HANDLE, h_stdout_sav);
	if (rc == 0) {
		put_err (sess, cmd, "cannot restore stdout handle");
		return -1;
	}

	rc = SetStdHandle (STD_ERROR_HANDLE, h_stderr_sav);
	if (rc == 0) {
		put_err (sess, cmd, "cannot restore stderr handle");
		return -1;
	}

	/*
	 * close childs end of pipe
	 */
	rc = CloseHandle (h_child_stdout_wr);
	if (rc == 0) {
		put_err (sess, cmd, "cannot close childs end of stdout pipe");
		return -1;
	}

	rc = CloseHandle (h_child_stdin_rd);
	if (rc == 0) {
		put_err (sess, cmd, "cannot close childs end of stdin pipe");
		return -1;
	}

	/*
	 * thats it
	 */
	sess->h_process = pi.hProcess;
	sess->h_thread = pi.hThread;
	sess->h_stdout = h_child_stdout_rd_dup;
	sess->h_stdin = h_child_stdin_wr_dup;

	return 0;
}

static int
close_pipe (sess_t *sess) {
	DWORD exit_code;
	DWORD rc;


	if (sess->h_process == INVALID_HANDLE_VALUE) {
		return -1;
	}

	/*
	 * wait for process termination to catch exit code of child process 
	 */
	for (;;) {
		rc = WaitForSingleObject (sess->h_process, INFINITE);
		if (rc == WAIT_FAILED) {
			exit_code = -1;
			break;
		}

		rc = GetExitCodeProcess (sess->h_process, &exit_code);
		if (rc == 0) {
			exit_code = -1;
			break;
		}

		if (exit_code != STILL_ACTIVE) {
			break;
		}
	}

	/*
	 * put away all handles
	 */
    CloseHandle (sess->h_process);
	sess->h_process = INVALID_HANDLE_VALUE;

    CloseHandle (sess->h_thread);
	sess->h_thread = INVALID_HANDLE_VALUE;

    CloseHandle (sess->h_stdout);
	sess->h_stdout = INVALID_HANDLE_VALUE;

	CloseHandle (sess->h_stdin);
	sess->h_stdin = INVALID_HANDLE_VALUE;

	return (exit_code);
}

static int
readline_pipe (sess_t *sess, char *buffer) {
	int rc;
	char *ptr;
	int len;
	DWORD want;
	DWORD got;

	ptr = strchr (sess->buffer, (int) '\n');
	if (ptr == 0) {
		
		ptr = sess->buffer + strlen (sess->buffer);
		want = BUFFERSIZE;
		rc = ReadFile (sess->h_stdout, ptr, want, &got, 0);

		/* 
		 * check for broken pipe
		 */
		if ((rc == 0) && (GetLastError () == ERROR_BROKEN_PIPE)) {
			return 0;
		}

		/* 
		 * check for EOF
		 */
		if ((rc != 0) && (got == 0)) {
			return 0;
		}

		/*
		 * unexpected error
		 */
		if (rc  == 0) {
			char *msg = "read error\n";
			sess->errtext = (char *) malloc (strlen (msg) + 1);
			strcpy (sess->errtext, msg);
			return -1;
		}
	}

	ptr = strchr (sess->buffer, '\n');
	if (ptr != 0) {
		len = ptr - sess->buffer;
		memcpy (buffer, sess->buffer, len);
		buffer[len] = '\0';
		memcpy (sess->buffer, sess->buffer + len + 1, BUFFERSIZE - len);
		strcat (buffer, "\n");
		len++;
	} else if (strlen (sess->buffer) != 0) {
		strcpy (buffer, sess->buffer);
		sess->buffer[0] = '\0';
		strcat (buffer, "\n");
		len = strlen (buffer);
	} else {
		buffer[0] = '\0';
		len = 0;
	}
	return (len);	
}

static char *
get_errtxt () {
	int rc;
	DWORD lang;
	char *buffer;

	rc = FormatMessage (FORMAT_MESSAGE_IGNORE_INSERTS |
		FORMAT_MESSAGE_FROM_SYSTEM | FORMAT_MESSAGE_ALLOCATE_BUFFER,
		0, GetLastError (), 0, (LPTSTR) &buffer, 0, 0);

	if  (rc == 0) {
		return 0;
	}
	
	return (buffer);
}

static void
free_errtxt (char *buffer) {
	LocalFree (buffer);
}

static int
put_err (sess_t *sess, char *msg0, char *msg2) {
	char *msg1 = ": ";
	char *msg3 = ", ";
	char *msg4;

	msg4 = get_errtxt ();

	sess->errtext =	(char *) malloc (strlen (msg0) + strlen (msg1) +
	                        strlen (msg2) + strlen (msg3) + 
	                        strlen (msg4) + 1);

	strcpy (sess->errtext, msg0);
	strcat (sess->errtext, msg1);
	strcat (sess->errtext, msg2);
	strcat (sess->errtext, msg3);
	strcat (sess->errtext, msg4);
	free_errtxt (msg4);

	for (;;) {
		int len;

		len = strlen (sess->errtext);
		if (len <= 1) {
			break;
		}	

		len--;

		switch (sess->errtext[len]) {
		case '\n':
		case '\r':
			sess->errtext[len] = '\0';
			break;
		default:
			return 0;
		}
	}

	return 0;
}

#endif

#include "PipeExec.h"

MODULE = SAPDB::Install::PipeExec	PACKAGE = SAPDB::Install::PipeExec

PROTOTYPES: DISABLE

BOOT:
	perl_eval_pv ((char *) sz_text, 1);

void
newXS (...)
PREINIT:
	sess_t *sess;
PPCODE:
	sess = new_pipe ();
	XSRETURN_IV ((ptr_t) sess);

void
OpenXS (...)
PREINIT:
	char *cmd;
	sess_t *sess;
	int rc;
PPCODE:
	if (items != 2) {
		XSRETURN_UNDEF;
	}

	sess = (sess_t *) SvIV (ST(0));
	if (sess == 0) {
		XSRETURN_UNDEF;
	}

	cmd = SvPV (ST(1), PL_na);
	if (strlen (cmd) == 0) {
		XSRETURN_UNDEF;
	}
	
	rc = open_pipe (sess, cmd);
	if (rc < 0) {
		XSRETURN_UNDEF;
	}
	XSRETURN_IV (0);

void 
CloseXS (...)
PREINIT:
	int rc;
	sess_t *sess;
PPCODE:
	if (items != 1) {
		XSRETURN_UNDEF;
	}

	sess = (sess_t *) SvIV (ST(0));
	if (sess == 0) {
		XSRETURN_UNDEF;
	}

	rc = close_pipe (sess);
	if (sess->errtext) {
		free (sess->errtext);
	}

	free (sess);
	if (rc < 0) {
		XSRETURN_UNDEF;
	}
	XSRETURN_IV (rc);

void
ReadlineXS (...)
PREINIT:
	sess_t *sess;
	int rc;
	char buffer[BUFFERSIZE + 1];
PPCODE:
	if (items != 1) {
		XSRETURN_UNDEF;
	}

	sess = (sess_t *) SvIV (ST(0));
	if (sess == 0) {
		XSRETURN_UNDEF;
	}

	rc = readline_pipe (sess, buffer);

	if (rc <= 0) {
		XSRETURN_UNDEF;
	}
	
	XPUSHs (sv_2mortal (newSVpv (buffer, strlen (buffer))));
	XSRETURN (1);

void
GetErrorXS (...)
PREINIT:
	sess_t *sess;
	int rc;
PPCODE:
	if (items != 1) {
		XSRETURN_UNDEF;
	}

	sess = (sess_t *) SvIV (ST(0));
	if (sess == 0) {
		XSRETURN_UNDEF;
	}

	if (sess->errtext == 0) {
		XSRETURN_UNDEF;
	}
	
	XPUSHs (sv_2mortal (newSVpv (sess->errtext, strlen (sess->errtext))));
	XSRETURN (1);

