/*!
  @file           veo49.c
  @author         JoergM
  @ingroup        Security
  @brief          Security specific functions

\if EMIT_LICENCE

    ========== licence begin  GPL
    Copyright (c) 2000-2004 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end


\endif
*/




/*===========================================================================*
 *  INCLUDES                                                                 *
 *===========================================================================*/

#include "gsp00.h"
#include "geo49.h"

#if defined(WIN32)
#include "gos00.h" /* nocheck */
#include <aclapi.h>
#else
#define _POSIX_PTHREAD_SEMANTICS
#include <pwd.h>
#include <grp.h>
#include <sys/stat.h>
#include <fcntl.h>
#endif

/*===========================================================================*
 *  DEFINES                                                                  *
 *===========================================================================*/


/*===========================================================================*
 *  MACROS                                                                   *
 *===========================================================================*/



/*===========================================================================*
 *  LOCAL CLASSES, STRUCTURES, TYPES, UNIONS ...                             *
 *===========================================================================*/



/*===========================================================================*
 *  STATIC/INLINE FUNCTIONS (PROTOTYPES)                                     *
 *===========================================================================*/



/*===========================================================================*
 *  PUBLIC FUNCTIONS (CODE)                                                  *
 *===========================================================================*/

externC bool eo49GetFileOwner(
const teo49_FileHandle FileHandle, 
tsp00_Namec OwnerName)
{
#if defined(WIN32)
    TCHAR Account[sizeof(tsp00_KnlIdentifierc)];
    TCHAR Domain[sizeof(tsp00_KnlIdentifierc)];
    PSID OwnerSID;
    PSECURITY_DESCRIPTOR SecurityDescriptor;
    SID_NAME_USE aUser = SidTypeUser;

    if ( sql02_get_platform_id() != VER_PLATFORM_WIN32_NT )
    {
        DWORD NameSize = sizeof(tsp00_Namec);
        return ( GetUserName( (char *)OwnerName, &NameSize) != 0 );
    }
    else
    {
        bool ok;
        DWORD AccountSize = sizeof(tsp00_KnlIdentifier);
        DWORD DomainSize = sizeof(tsp00_KnlIdentifier);

        if ( GetSecurityInfo( FileHandle,
                              SE_FILE_OBJECT,
                              OWNER_SECURITY_INFORMATION,
                             &OwnerSID,
                              NULL, // Group SID not needed
                              NULL, // DACL not needed
                              NULL, // SACL not needed
                             &SecurityDescriptor ) != ERROR_SUCCESS )
        {
            return false;
        }

        ok = ( LookupAccountSid( NULL,
                               OwnerSID,
                               Account,
                              &AccountSize,
                               Domain,
                              &DomainSize,
                              &aUser ) != false );

        LocalFree(SecurityDescriptor);

        if ( ok )
        {
            if ( strcmp(Domain, "BUILTIN") != 0 )
            {
                ok = (bool)( (strlen(Account) + strlen(Domain)) < sizeof(tsp00_Namec) );
                if ( ok )
                {
                    strcpy(OwnerName, Domain);
                    strcat(OwnerName, "\\");
                    strcat(OwnerName, Account);
                }
            }
            else
            {
                strcpy(OwnerName, Account);
            }
        }
        return ok;
    }
#else
    struct stat statBuffer;

    if ( fstat(FileHandle, &statBuffer) == 0 )
    {
        struct passwd passwordEntry;
        struct passwd *pEntry = &passwordEntry;
        char buffer[4096];

        if ( 0 != getpwuid_r(statBuffer.st_uid, &passwordEntry, buffer, 4096, &pEntry) )
        {
            return false;
        }
        if ( strlen(pEntry->pw_name) > sizeof(tsp00_Namec) )
        {
            return false;
        }
        strcpy(OwnerName, pEntry->pw_name);
        return true;
    }
    return false;    
#endif
}

externC bool eo49SetFileGroup(
const teo49_FileHandle FileHandle, 
tsp00_Namec const GroupName)
{
#if defined(WIN32)
    return true;
#else
    struct stat statBuffer;
    struct group *grEntry = getgrnam (&GroupName[0]);

    if ( grEntry && fstat(FileHandle, &statBuffer) == 0 )
    {
        return ( fchown(FileHandle, statBuffer.st_uid, grEntry->gr_gid) == 0 );
    }
    return false;    
#endif
}

externC bool eo49GetFileGroup(
const teo49_FileHandle FileHandle, 
tsp00_Namec GroupName)
{
#if defined(WIN32)
    memset(GroupName, 0, sizeof(tsp00_Namec));
    return true;
#else
    struct stat statBuffer;

    if ( fstat(FileHandle, &statBuffer) == 0 )
    {
        struct group *pEntry = getgrgid(statBuffer.st_gid);
        if ( !pEntry )
        {
            return false;
        }
        if ( strlen(pEntry->gr_name) > sizeof(tsp00_Namec) )
        {
            return false;
        }
        strcpy(GroupName, pEntry->gr_name);
        return true;
    }
    return false;    
#endif
}

/*===========================================================================*
 *  LOCAL/PRIVATE/PROTECTED FUNCTIONS (CODE)                                 *
 *===========================================================================*/

/*===========================================================================*
 *  END OF CODE                                                              *
 *===========================================================================*/
