/*! 
  -----------------------------------------------------------------------------
 
  module: DBMSrvBHist_History.hpp
 
  -----------------------------------------------------------------------------
 
  responsible:  TiloH
 
  special area: backup history handling by the DBMServer
   
  description:  a class for managing the information of the backup history
                files

  version:      7.4.4 and higher 
  -----------------------------------------------------------------------------
 
                          Copyright (c) 2002-2004 SAP AG
 
  -----------------------------------------------------------------------------



    ========== licence begin  GPL
    Copyright (c) 2002-2004 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end



*/

#if !defined(DBMSRVBHIST_HISTORY_HPP)
#define DBMSRVBHIST_HISTORY_HPP


//
//-----------------------------------------------------------------------------
// includes
//-----------------------------------------------------------------------------
//

#include "SAPDB/ToolsCommon/Tools_List.hpp"
#include "SAPDB/DBM/Srv/BackupHistory/DBMSrvBHist_Backup.hpp"
#include "SAPDB/DBM/Srv/BackupHistory/DBMSrvBHist_ListProducer.hpp"
#include "gcn00.h"
#include "gcn003.h"
#include "hcn36.h"


//-----------------------------------------------------------------------------
// global variable
//-----------------------------------------------------------------------------

const teo00_Uint8 DBMSrvBHist_History_NOLOG=(0x7FFFFFFF);


//
//-----------------------------------------------------------------------------
// class DBMSrvBHist_History
//-----------------------------------------------------------------------------
//

class DBMSrvBHist_History: public DBMSrvBHist_ListProducer
{
  public:
    DBMSrvBHist_History();
    ~DBMSrvBHist_History();

    void Free();

    tcn00_Error BackupHistoryDate(VControlDataT * vcontrol,
                                  CommandT      * command,
                                  char          * replyData,
                                  int           * replyLen,
                                  int             replyLenMax);

    tcn00_Error BackupHistoryOpen(VControlDataT * vcontrol,
                                  CommandT      * command,
                                  char          * replyData,
                                  int           * replyLen,
                                  int             replyLenMax);

    tcn00_Error BackupHistoryOpen(VControlDataT * vcontrol,
                                  int             WithEBIDs,
                                  char          * replyData,
                                  int           * replyLen,
                                  int             replyLenMax);

    tcn00_Error BackupHistoryList(VControlDataT * vcontrol,
                                  CommandT      * command,
                                  char          * replyData,
                                  int           * replyLen,
                                  int             replyLenMax);

    tcn00_Error BackupHistoryListNext(VControlDataT * vcontrol,
                                      CommandT      * command,
                                      char          * replyData,
                                      int           * replyLen,
                                      int             replyLenMax);

    tcn00_Error BackupHistoryClose(VControlDataT * vcontrol,
                                   CommandT      * command,
                                   char          * replyData,
                                   int           * replyLen,
                                   int             replyLenMax);

    tcn00_Error AnalyzeLogBackup(tcn003_BackupResult * Result);

    tcn00_Error ReadBasicHistories(const char * DBName,
                                   char       * replyData,
                                   int        * replyLen);

    void        ReduceToFileLogBackupsWithName(const char * AbsoluteName, const char * DefaultDirectory);
    int         ContainsFileBackup(const char                 *  BaseName,
                                   const tcn36_DateTimeString  & DateTime,
                                   const char                 *& Key,
                                   const char                 *& Label);

    const Tools_List<DBMSrvBHist_Backup *> & GetHistoryList() const;

  private:
    enum CloseFileFlag{ReportError, DoNotReportError};

    tcn00_Error CheckHistoryFile(bool & Exists, char Date[64]);
    void SetVControlAndReplyBuffer(VControlDataT * NewVControl,
                                   char          * NewReplyData,
                                   int           * NewReplyLen,
                                   int             NewReplyLenMax,
                                   int             ResetErrorHandler);
    tcn00_Error InitListParameters();

    tcn00_Error ReadHistories();
    tcn00_Error OpenFile(const tsp00_Pathc & FileType, tsp00_Int4 & File);
    tcn00_Error ReadNextTextLine(tsp00_Int4 & File, char *& Line, size_t & LineLength, int & ReachedEndOfFile);
    tcn00_Error ReadHistory(Tools_List<DBMSrvBHist_Backup *> & BackupList, char *& Line, size_t & LineLength);
    tcn00_Error ReadMediaHistory(Tools_List<DBMSrvBHist_Part *> & MediaList, char *& Line, size_t & LineLength);
    tcn00_Error ReadEBIDHistory(Tools_List<DBMSrvBHist_Part *> & EBIDList, char *& Line, size_t & LineLength);
    tcn00_Error CloseFile(tsp00_Int4 File, CloseFileFlag ReportErrorFlag);

    tcn00_Error JoinHistoryWithMediaHistory(Tools_List<DBMSrvBHist_Part *> & MediaList);
    tcn00_Error JoinHistoryWithEBIDHistory(Tools_List<DBMSrvBHist_Part *> & EBIDList);
    tcn00_Error JoinEBIDHistoryWithToolData(const tcn35_BackupToolConnectorEnum   ToolType,
                                            Tools_List<DBMSrvBHist_Part *>      & EBIDList,
                                            Tools_List<DBMSrvBHist_Part *>      & EBIDsFromTool);

    tcn00_Error CheckAvailabilityOfExternalBackups(Tools_List<DBMSrvBHist_Part *> & EBIDList);
    tcn00_Error FindFirstRestoreItem();
    tcn00_Error CreateOutputList();
    tcn00_Error BasicListHistory();
    bool        PrintCurrentBackupInto(char * Buffer, size_t MaxUsableBufferSize);
    void        EnlargeColumnWidths();

    void        IfRCOKPrintOKIntoReply(tcn00_Error Status);


    Tools_List<DBMSrvBHist_Backup*> History;
    Tools_List<DBMSrvBHist_Backup*> OutputList;

    Tools_List<DBMSrvBHist_Backup*>::iterator FirstBackup;      //!< pointing to the first backup that must be used from the list History

    Tools_List<DBMSrvBHist_Backup*>::iterator CurrentBackup;    //!< element from OutputList that is listed next
    Tools_List<DBMSrvBHist_Backup*>::iterator StartBackup;      //!< first element of OutputList to be listed
    Tools_List<DBMSrvBHist_Backup*>::iterator EndBackup;        //!< last element of OutputList to be listed

    char * ColumnsAsString;
    char * Key;
    char * Label;
    char * Action;
    char * Restore;
    char * Until;

    bool ForRestore;
    bool RcOKOnly;
    bool WithMedia;
    bool WithEBIDs;
    bool ListInverted;

    teo00_Uint8 UsedLogPage;
    teo00_Uint8 FirstLogPage;
    bool Restartable;

    VControlDataT * VControl;
    char          * ReplyData;
    int           * ReplyLen;
    int             ReplyLenMax;

    Tools_List<DBMSrvBHist_Backup::Column> Columns;

    DBMSrvBHist_BackupColumnWidths    ColumnWidthsBackup;
    DBMSrvBHist_PartMediaColumnWidths ColumnWidthsMedia;
};

#endif
