/*!
    @file     SharedSQL_IHandles.hpp
    @ingroup  SharedSQL
    @author   DirkT
    @brief    Handles for SharedSQL_Commands
    @see            

\if EMIT_LICENCE
    ========== licence begin  GPL
    Copyright (c) 2000-2004 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end
\endif
*/

#ifndef SHAREDSQL_IHANDLES_HPP
#define SHAREDSQL_IHANDLES_HPP

/*===========================================================================*
 *  INCLUDES                                                                 *
 *===========================================================================*/

#include "SQLManager/SharedSQL/SharedSQL_Types.hpp"

#include "SAPDBCommon/SAPDB_Types.hpp"

/*===========================================================================*
 *  CLASSES, STRUCTURES, TYPES, UNIONS ...                                   *
 *===========================================================================*/

/*---------------------------------------------------------------------------*/
/*!
    @interface      SharedSQL_IPrepareHandle
    @brief          Interface to SharedSQL_PrepareHandle
 */
class SharedSQL_IPrepareHandle
{
public:

    /*!
        @brief  Returns wheather or not the command must be prepared
        @param  Status [out] - The (old) status before the handle was created. It may be NEW, INVALID or DROPPED if the command has to be prepared or PREPARED if not.
        @return SAPDB_Bool
    */
    virtual SAPDB_Bool MustPrepare( SharedSQL_CommandStatus& Status ) const = 0;

    /*!
        @brief  Returns the master parse id. e.g. the parse id used to prepare the command.
        @return const SharedSQL_ParseID*
    */
    virtual const SharedSQL_ParseID* GetMasterParseID( void ) const = 0;

    /*!
        @brief  Store the execution plan
        @param  MPID [in] - Parse ID used to prepare the plan. Will be stored as MasterParseID within the command
        @param  P    [in] - Pointer to the execution plan
        @param  Size [in] - Size of the execution plan
        @return SharedSQL_Error
    */
	virtual SharedSQL_Error	PutPlan( SharedSQL_ParseID MPID, void *const P, SAPDB_ULong Size ) const = 0;

    /*!
        @brief  Returns a pointer to the execution plan
        @param  Key [in]
        @return const void*
    */
	virtual const void*	GetPlan( void* Key ) const = 0;

    /*!
        @brief  To be called when an error during preparation occurs. Will set the status to INVALID.
    */
	virtual void ErrorWhilePreparing( void ) const = 0;

    /*!
        @brief  Release the Handle. Will set the status to PREPARED if a plan was stored otherwise to INVALID.
        @param  ParseTimeSec      [in] - Time needed for preparation (sec)
        @param  ParseTimeMicroSec [in] - Time needed for preparation (microsec)
    */
    virtual void ReleaseHandle( SAPDB_UInt8 ParseTimeSec, SAPDB_UInt8 ParseTimeMicroSec ) = 0;
};  // SharedSQL_IPrepareHandle

/*---------------------------------------------------------------------------*/
/*!
    @interface      SharedSQL_IExecuteHandle
    @brief          Interface to SharedSQL_ExecuteHandle
 */
class SharedSQL_IExecuteHandle
{
public:

    /*!
        @brief  Returns the master parse id. e.g. the parse id used to prepare the command.
        @return const SharedSQL_ParseID*
    */
    virtual const SharedSQL_ParseID* GetMasterParseID( void ) const = 0;

    /*!
        @brief  Returns a pointer to the execution plan
        @param  Key [in]
        @return const void*
    */
	virtual const void*	GetPlan( void* Key ) const = 0;

    /*!
        @brief  Sets the execution plan to INVALID if an error during excution occurs.
    */
	virtual void SetPlanInvalid( void ) const = 0;

    /*!
        @brief  Returns if the execution plan is valid
        @param  M9 [out] - Flag to indicate wheather -8 or -9 should be returned to the application
        @return SAPDB_Bool
    */
	virtual SAPDB_Bool IsPlanValid(  SAPDB_Bool& M9  ) const = 0;

    /*!
        @brief  Returns the sql statement and its context
        @param  Context [out]
        @return const SharedSQL_SQLStmt
    */
	virtual const SharedSQL_SQLStmt GetSQLStmt( const SharedSQL_SQLContext* &Context ) const = 0;

    /*!
        @brief  Release the ExecutionHandle 
    */
    virtual void ReleaseHandle( void ) = 0;
}; // SharedSQL_IExecuteHandle

#endif  /* SHAREDSQL_IHANDLES_HPP */
