/*! @defgroup ActionHandling Action handling */
/*!
    @ingroup ActionHandling
    @file    Log_IAction.hpp
	@author  UweH
	@brief   Defines the interface Log_IAction for all actions.

\if EMIT_LICENCE

    ========== licence begin  GPL
    Copyright (c) 2000-2004 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end


\endif
 */
#ifndef Log_IAction_H
#define Log_IAction_H

#include "SAPDBCommon/SAPDB_Types.hpp"
#include "ggg00.h"

class   SAPDBMem_IRawAllocator;
class   Data_SplitSpaceWriter;
class   Data_SplitSpaceReader;

/*!
   @brief Baseclass for all actions.

   - Actions define an operation, something, which ist executed.
   - The persistent format describes what is to be done.
   - To undo an action, the inverse action must be created and written to an undo file.
 */
class Log_IAction
{
public:
    /// constructor
    Log_IAction()
    : m_Type (Log_NoOp)
    {}
    /// This returns the actions type.
    Log_ActionType GetType () const
    {
        return m_Type;
    }
    /// Some special actions need to be written with no restrictions like LOGFULL. These entries may return true.
	// PTS 1117320 UH 2002-10-28 added
    virtual bool ForceLogWriting () const
    {
        return false;
    }
    /// Some special actions need a log position. These entries may return true. e.g. commit, rollback, savepoint
	// PTS 1117320 UH 2002-10-28 added
    virtual bool WaitForLogWriting () const
    {
        return false;
    }
    /// if this returns false a local redo entry cash must be flushed
    virtual bool LocalRedoEntryCashingPossible() const
    {
        return true;
    }
    /// This returns the persistent length of the actions image, which must not be split.
    virtual SAPDB_UInt GetMinimalPersistentLength () const = 0;
    /// This returns the total persistent length of the actions image.
    virtual SAPDB_UInt GetPersistentLength () const = 0;
    /// Everything is written which is needed to re-execute this action.
    virtual void WritePersistentFormat (Data_SplitSpaceWriter &Writer,
                                        bool                  &isOK) const = 0;
    /// Everything is read which is needed to re-execute this action.
    virtual void ReadPersistentFormat (Data_SplitSpaceReader  &Reader,
                                       SAPDBMem_IRawAllocator &Allocator,
                                       bool                   &isOK) = 0;
    /// Executes the action.
    virtual void Execute (tgg00_TransContext &TransContext) const
    {}
    /// The action is written to the trace file.
    virtual void WriteToTrace (const char * Title) const
    {}
protected:
    /// The actions type.
    Log_ActionType m_Type;
};
#endif  /* Log_IAction_H */
