/*!
	========== licence begin GPL
    Copyright (C) 2002-2003 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end
*/
package com.sap.dbtech.jdbc.translators;

import java.io.InputStream;
import java.io.Reader;
import java.math.BigDecimal;
import java.sql.Blob;
import java.sql.Clob;
import java.sql.Date;
import java.sql.SQLException;
import java.sql.Time;
import java.sql.Timestamp;
import java.util.Calendar;
import java.util.Vector;

import com.sap.dbtech.jdbc.exceptions.SQLExceptionSapDB;
import com.sap.dbtech.jdbc.packet.PartNotFound;
import com.sap.dbtech.jdbc.packet.ReplyPacket;
import com.sap.dbtech.util.MessageKey;
import com.sap.dbtech.util.MessageTranslator;
import com.sap.dbtech.util.StructuredBytes;
import com.sap.dbtech.vsp001.PartKind;

/**
 * Base class for the 'get' operation on OMS Streams 
 */
public abstract class AbstractABAPStreamGetval 
	extends ABAPStreamDescriptor {
	
	private   Vector          replyBuffers;
	protected int             elementCount;
	protected StructuredBytes data;
		
	public AbstractABAPStreamGetval(ABAPStreamTranslator translator) {
		super(translator);
		replyBuffers = new Vector();
		elementCount = 0;
	}
	
	public boolean addReplyData(ReplyPacket replyPacket) 
		throws SQLException {
		
		try { 
			replyPacket.findPart(PartKind.Data_C);
		} catch(PartNotFound partNotFound) {
			throw new SQLExceptionSapDB(MessageTranslator.translate(MessageKey.ERROR_STREAM_NODATA));
		}
		int partArguments = replyPacket.partArguments();
		if(partArguments < 0) {
			int remainder = partArguments & 0x7FFF;
			remainder += 32768;
			partArguments = remainder;
		}
		elementCount += partArguments;
		if(partArguments != 0) {
			replyBuffers.add(replyPacket.getBytes(replyPacket.getPartDataPos(),
			    			                      replyPacket.partLength()));
		}
		return partArguments == 0;
	}

	/**
	 * Coalesce the reply into one big byte array.
	 */
	public void coalesceReply() {
		int replySize = 0;
		for(int i=0; i<replyBuffers.size(); ++i) {
			replySize += ((byte[])replyBuffers.elementAt(i)).length;	
		}
		this.data = new StructuredBytes(replySize);
		int currentStart = 0;
		for(int i=0; i<replyBuffers.size(); ++i) {
			byte[] currentBuffer = (byte[]) replyBuffers.elementAt(i);
			System.arraycopy(currentBuffer, 0, this.data.bytes(), currentStart, currentBuffer.length);
			currentStart += currentBuffer.length;
		}
		this.replyBuffers = null;
	}

	/**
	 * Retrieve the 0-based index in the list of columns/output parameters.
	 * @return The 0-based index of the column that is related to this
	 *   GETVAL object
	 */
	public int getColIndex() {
		return this.translator.getColIndex();
	}

	public void throwIllegalConversion(String javatype) throws SQLException {
		throw new ConversionExceptionSapDB
			(MessageTranslator.translate(MessageKey.ERROR_CONVERSIONSQLJAVA, 
				translator.getColumnTypeName(), javatype));

	}

	public String getString() 
		throws SQLException {
		throwIllegalConversion(String.class.getName());
		return null;
	}
		
	public InputStream getAsciiStream()
		throws SQLException {
		throwIllegalConversion(InputStream.class.getName());
		return null;
	}
	
	public  BigDecimal getBigDecimal(int scale)
		throws SQLException {
		throwIllegalConversion(BigDecimal.class.getName());
		return null;
	}
	
	public  BigDecimal getBigDecimal()
		throws SQLException {
		throwIllegalConversion(BigDecimal.class.getName());
		return null;
	}
	
	public  InputStream getBinaryStream()
		throws SQLException {
		throwIllegalConversion(InputStream.class.getName());
		return null;
	}

	public  Blob getBlob() 
		throws SQLException {
		throwIllegalConversion(Blob.class.getName());
		return null;			
	}

	public  boolean getBoolean() 
		throws SQLException {
		throwIllegalConversion("boolean");
		return false;
	}
	
	public  byte getByte()
		throws SQLException {
		throwIllegalConversion("byte");
		return 0;
	}
		
	public  byte[] getBytes()
		throws SQLException {
		throwIllegalConversion("byte[]");
		return null;
	}

	public  Date getDate(Calendar cal)
		throws SQLException {
		throwIllegalConversion(java.sql.Date.class.getName());
		return null;
	}

	public  double getDouble()
		throws SQLException {
		throwIllegalConversion("double");
		return 0.0;
	}

	public  Reader getCharacterStream() 
		throws SQLException {
		throwIllegalConversion(java.io.Reader.class.getName());
		return null;
	}

	public  Clob getClob()
		throws SQLException {
		throwIllegalConversion(Clob.class.getName());
		return null;
	}

	public  float getFloat() 
		throws SQLException {
		throwIllegalConversion("float");
		return 0.0f;
	}

	public  int getInt() 
		throws SQLException {
		throwIllegalConversion("int");
		return 0;
	}

	public  long getLong()
		throws SQLException {
		throwIllegalConversion("long");
		return 0;
	}
	public  Object getObject()
		throws SQLException {
		throwIllegalConversion(Object.class.getName());
		return null;
	}

	public  Time getTime(Calendar cal)
		throws SQLException {
		throwIllegalConversion(Time.class.getName());
		return null;
	}

	public  Timestamp getTimestamp(Calendar cal) 
		throws SQLException {
		throwIllegalConversion(Timestamp.class.getName());
		return null;
	}
		
}
