#!/bin/sh
src=`pwd`
tmp=$src/tests-tmp
rm -rf $tmp
mkdir -p $tmp
PATH="$src:/bin:/usr/bin:/usr/local/bin"
tests_failed=0
tests_count=0
sfqmail() { $src/smtpfront-qmail 2>/dev/null | tail -n +2; }
sfecho() { $src/smtpfront-echo 2>/dev/null | tail -n +2; }
pfauth() { $src/pop3front-auth "$@" echo Yes. 2>/dev/null | tail -n +2; }
ifauth() { $src/imapfront-auth sh -c 'echo Yes: $IMAPLOGINTAG' 2>/dev/null | tail -n +2; }
pfmaildir() { $src/pop3front-maildir "$@" 2>/dev/null | tail -n +2; }
maildir=$tmp/Maildir
maildir() {
  rm -rf $maildir
  mkdir -p $maildir/{cur,new,tmp}
}
tstmsg() {
  fn=$1
  {
    echo "Header: foo"
    echo
    echo "body"
  } >$maildir/$fn
}

PROTO=TEST
TESTLOCALIP=1.2.3.4
TESTREMOTEIP=5.6.7.8
TESTLOCALHOST=local.host
TESTREMOTEHOST=remote.host

export PROTO TESTLOCALIP TESTREMOTEIP TESTLOCALHOST TESTREMOTEHOST

cat >$tmp/cvm <<EOF
#!/bin/sh
req="\`cat -v\`"
if [ "x\$req" = 'x^Atestuser^@^@testpass^@^@' ]; then
  printf '\\0\\0010500\\0\\1testuser\\0\\0002500\\0\\0003500\\0\\4Test User\\0\\5$tmp\\0\\6/bin/false\\0\\7testgroup\\0\\0'
  exit 0
fi
printf \\\\0144
exit 100
EOF
chmod +x $tmp/cvm

run_compare_test() {
  local name=$1
  shift
  sed -e "s:@SOURCE@:$src:g" >$tmp/expected
  ( runtest "$@" ) 2>&1 >$tmp/actual-raw
  cat -v $tmp/actual-raw >$tmp/actual
  if ! cmp $tmp/expected $tmp/actual >/dev/null 2>&1
  then
    echo "Test $name $@ failed:"
	( cd $tmp; diff -u9999 expected actual | tail -n +3; echo; )
	let tests_failed=tests_failed+1
  fi
  rm -f $tmp/expected $tmp/actual
  let tests_count=tests_count+1
}

##### Test tests/rules-header-add #####

runtest() {
cat >$tmp/rules <<EOF
ka:b:K1:::HEADER_ADD="X-Header: testing!!!"
EOF

MAILRULES=$tmp/rules sfecho <<EOF
MAIL FROM:<a>
RCPT TO:<b>
DATA
.
EOF
MAILRULES=$tmp/rules sfecho <<EOF
MAIL FROM:<a>
RCPT TO:<c>
DATA
.
EOF

rm -f $tmp/rules
}
run_compare_test tests/rules-header-add  <<END_OF_TEST_RESULTS
250 Sender='a'.^M
250 K1^M
354 End your message with a period.^M
250 Received 151 bytes.^M
250 Sender='a'.^M
250 Recipient='c'.^M
354 End your message with a period.^M
250 Received 128 bytes.^M
END_OF_TEST_RESULTS


##### Test tests/rules-empty #####

runtest() {
cat >$tmp/rules <<EOF
k:*:K1
k*:*:K2
EOF

MAILRULES=$tmp/rules sfecho <<EOF
MAIL FROM:<>
MAIL FROM:<foo>
EOF

rm -f $tmp/rules
}
run_compare_test tests/rules-empty  <<END_OF_TEST_RESULTS
250 K1^M
250 K2^M
END_OF_TEST_RESULTS


##### Test tests/rules-recip #####

runtest() {
cat >$tmp/rules <<EOF
k*:one@one:KK
d*:two@two:DD
z*:three@three:ZZ
zx@y:four@four:ZZZ
p*:five@five:PP
EOF

MAILRULES=$tmp/rules sfecho <<EOF
MAIL FROM:<nobody@nobody>
RCPT TO:<one@one>
RCPT TO:<two@two>
RCPT TO:<three@three>
RCPT TO:<four@four>
RCPT TO:<five@five>
EOF

rm -f $tmp/rules
}
run_compare_test tests/rules-recip  <<END_OF_TEST_RESULTS
250 Sender='nobody@nobody'.^M
250 KK^M
553 DD^M
451 ZZ^M
250 Recipient='four@four'.^M
250 Recipient='five@five'.^M
END_OF_TEST_RESULTS


##### Test tests/pop3front-maildir-last #####

runtest() {
pflast() { echo LAST | pfmaildir $maildir; }
maildir
tstmsg new/1
tstmsg new/2
pflast
mv $maildir/new/1 $maildir/cur/1
pflast
mv $maildir/cur/1 $maildir/cur/1:2,S
pflast
mv $maildir/new/2 $maildir/cur/2:2,S
pflast
mv $maildir/cur/1:2,S $maildir/new/1
pflast
}
run_compare_test tests/pop3front-maildir-last  <<END_OF_TEST_RESULTS
+OK 0^M
+OK 0^M
+OK 1^M
+OK 2^M
+OK 2^M
END_OF_TEST_RESULTS


##### Test tests/pop3front-maildir-sort #####

runtest() {
maildir
tstmsg new/99
tstmsg cur/100
tstmsg new/101
tstmsg cur/98

echo UIDL | pfmaildir $maildir
}
run_compare_test tests/pop3front-maildir-sort  <<END_OF_TEST_RESULTS
+OK ^M
1 98^M
2 99^M
3 100^M
4 101^M
.^M
END_OF_TEST_RESULTS


##### Test tests/rules-negate #####

runtest() {
cat >$tmp/rules <<EOF
k!a@a:*:A
k!!a@a:*:B
EOF

MAILRULES=$tmp/rules sfecho <<EOF
MAIL FROM:<a@a>
MAIL FROM:<b@b>
EOF

rm -f $tmp/rules
}
run_compare_test tests/rules-negate  <<END_OF_TEST_RESULTS
250 B^M
250 A^M
END_OF_TEST_RESULTS


##### Test tests/rules-cdb #####

runtest() {
cat >$tmp/rules <<EOF
k[[$tmp/list.cdb]]:*:LIST
k[[@$tmp/atlist.cdb]]:*:ATLIST
d*:*:DD
EOF

cat <<EOF | cdbmake $tmp/list.cdb $tmp/list.tmp
+3,0:a@a->
+2,0:@c->

EOF

cat <<EOF | cdbmake $tmp/atlist.cdb $tmp/atlist.tmp
+1,0:b->

EOF

MAILRULES=$tmp/rules sfecho <<EOF
MAIL FROM:<a@a>
MAIL FROM:<b@a>
MAIL FROM:<a@b>
MAIL FROM:<b@B>
MAIL FROM:<c@c>
MAIL FROM:<c@C>
MAIL FROM:<d@d>
EOF

rm -f $tmp/rules $tmp/list.cdb $tmp/atlist.cdb
}
run_compare_test tests/rules-cdb  <<END_OF_TEST_RESULTS
250 LIST^M
553 DD^M
250 ATLIST^M
250 ATLIST^M
250 LIST^M
250 LIST^M
553 DD^M
END_OF_TEST_RESULTS


##### Test tests/rules-sender #####

runtest() {
cat >$tmp/rules <<EOF
kone@one:*:KK
dtwo@two:*:DD
zthree@three:*:ZZ
zfour@four:one@one:ZZ
pfive@five:*:PP
EOF

MAILRULES=$tmp/rules sfecho <<EOF
MAIL FROM:<one@one>
MAIL FROM:<two@two>
MAIL FROM:<three@three>
MAIL FROM:<four@four>
MAIL FROM:<five@five>
EOF

rm -f $tmp/rules
}
run_compare_test tests/rules-sender  <<END_OF_TEST_RESULTS
250 KK^M
553 DD^M
451 ZZ^M
250 Sender='four@four'.^M
250 Sender='five@five'.^M
END_OF_TEST_RESULTS


##### Test tests/imapfront-auth #####

runtest() {
CVM_SASL_PLAIN=$tmp/cvm
export CVM_SASL_PLAIN

ifauth <<EOF
1 LOGIN
2 LOGIN A
3 LOGIN A B C
4 LoGiN A B
5 LOGIN testuser testpass
EOF

ifauth <<EOF
6 login "testuser" "testpass"
EOF

ifauth <<EOF
7 login {8}
testuser{8}
testpass
EOF

unset CVM_SASL_PLAIN
}
run_compare_test tests/imapfront-auth  <<END_OF_TEST_RESULTS
1 BAD Syntax error: command requires arguments^M
2 BAD LOGIN command requires exactly two arguments^M
3 BAD LOGIN command requires exactly two arguments^M
4 NO LOGIN failed^M
Yes: 5
Yes: 6
+ OK^M
+ OK^M
Yes: 7
END_OF_TEST_RESULTS


##### Test tests/smtpfront-maxnotimpl #####

runtest() {
MAXNOTIMPL=1 sfecho <<EOF
a
b
c
d
EOF

MAXNOTIMPL=0 sfecho <<EOF
a
b
c
d
EOF
}
run_compare_test tests/smtpfront-maxnotimpl  <<END_OF_TEST_RESULTS
500 Not implemented.^M
503-Too many unimplemented commands.^M
503 Closing connection.^M
500 Not implemented.^M
500 Not implemented.^M
500 Not implemented.^M
500 Not implemented.^M
END_OF_TEST_RESULTS


##### Test tests/rules-both #####

runtest() {
cat >$tmp/rules <<EOF
ka@a:a@b:K1
ka@a:b@b:K2
kb@a:a@b:K3
kb@a:b@b:K4
EOF

MAILRULES=$tmp/rules sfecho <<EOF
MAIL FROM:<a@a>
RCPT TO:<a@b>
RCPT TO:<b@b>
MAIL FROM:<b@a>
RCPT TO:<a@b>
RCPT TO:<b@b>
EOF

rm -f $tmp/rules
}
run_compare_test tests/rules-both  <<END_OF_TEST_RESULTS
250 Sender='a@a'.^M
250 K1^M
250 K2^M
250 Sender='b@a'.^M
250 K3^M
250 K4^M
END_OF_TEST_RESULTS


##### Test tests/rules-list #####

runtest() {
cat >$tmp/rules <<EOF
k[[$tmp/list]]:*:LIST
k[[@$tmp/atlist]]:*:ATLIST
d*:*:DD
EOF

cat >$tmp/list <<EOF
a@a
@c
EOF

cat >$tmp/atlist <<EOF
b
EOF

MAILRULES=$tmp/rules sfecho <<EOF
MAIL FROM:<a@a>
MAIL FROM:<b@a>
MAIL FROM:<a@b>
MAIL FROM:<b@B>
MAIL FROM:<c@c>
MAIL FROM:<c@C>
MAIL FROM:<d@d>
EOF

rm -f $tmp/rules $tmp/list $tmp/atlist
}
run_compare_test tests/rules-list  <<END_OF_TEST_RESULTS
250 LIST^M
553 DD^M
250 ATLIST^M
250 ATLIST^M
250 LIST^M
250 LIST^M
553 DD^M
END_OF_TEST_RESULTS


##### Test tests/rules-noop #####

runtest() {
cat >$tmp/rules <<EOF
n*:*:Do not see this:::RELAYCLIENT=@rc
EOF

MAILRULES=$tmp/rules sfecho <<EOF
MAIL FROM:<a@a>
RCPT TO:<b@b>
EOF

rm -f $tmp/rules
}
run_compare_test tests/rules-noop  <<END_OF_TEST_RESULTS
250 Sender='a@a'.^M
250 Recipient='b@b@rc'.^M
END_OF_TEST_RESULTS


##### Test tests/patterns-message #####

runtest() {
cat >$tmp/patterns <<EOF
=response 1
=response 2
/*
=response 3
EOF

PATTERNS=$tmp/patterns sfecho <<EOF
MAIL FROM:<>
RCPT TO:<nobody>
DATA
before

after
.
EOF

rm -f $tmp/patterns
}
run_compare_test tests/patterns-message  <<END_OF_TEST_RESULTS
250 Sender=''.^M
250 Recipient='nobody'.^M
354 End your message with a period.^M
554 response 2^M
END_OF_TEST_RESULTS


##### Test tests/pop3front-auth #####

runtest() {
$src/pop3front-auth false echo Yes. <<EOF 2>/dev/null
QUIT NO
QUIT
QUIT AGAIN
EOF
}
run_compare_test tests/pop3front-auth  <<END_OF_TEST_RESULTS
+OK ^M
-ERR Syntax error^M
+OK ^M
END_OF_TEST_RESULTS


##### Test tests/patterns-normal #####

runtest() {
cat >$tmp/patterns <<EOF
# comment

/before
# comment
EOF

PATTERNS=$tmp/patterns sfecho <<EOF
MAIL FROM:<>
RCPT TO:<nobody>
DATA
before

after
.
EOF

rm -f $tmp/patterns
}
run_compare_test tests/patterns-normal  <<END_OF_TEST_RESULTS
250 Sender=''.^M
250 Recipient='nobody'.^M
354 End your message with a period.^M
554 This message contains prohibited content^M
END_OF_TEST_RESULTS


##### Test tests/imapfront-auth-login #####

runtest() {
export CVM_SASL_PLAIN=$tmp/cvm

ifauth false <<EOF
1 AUTHENTICATE LOGIN
dGVzdHVzZXI=
dGVzdHBhc3x=
2 AUTHENTICATE LOGIN
dGVzdHVzZXI=
dGVzdHBhc3M=
EOF

ifauth false <<EOF
3 AUTHENTICATE LOGIN dGVzdHVzZXI=
dGVzdHBhc3M=
EOF

ifauth false <<EOF
4 AUTHENTICATE LOGIN
dGVzdHVzZXI=
*
EOF

unset CVM_SASL_PLAIN
}
run_compare_test tests/imapfront-auth-login  <<END_OF_TEST_RESULTS
+ VXNlcm5hbWU6^M
+ UGFzc3dvcmQ6^M
1 NO AUTHENTICATE failed: Authentication failed.^M
+ VXNlcm5hbWU6^M
+ UGFzc3dvcmQ6^M
Yes: 2
+ UGFzc3dvcmQ6^M
Yes: 3
+ VXNlcm5hbWU6^M
+ UGFzc3dvcmQ6^M
4 NO AUTHENTICATE failed: Authentication failed.^M
END_OF_TEST_RESULTS


##### Test tests/imapfront-auth-plain #####

runtest() {
export CVM_SASL_PLAIN=$tmp/cvm

ifauth <<EOF
1 AUTHENTICATE PLAIN dGVzdHVzZXIAdGVzdHVzZXIAdGVzdHBhc3x=
2 AUTHENTICATE PLAIN dGVzdHVzZXIAdGVzdHVzZXIAdGVzdHBhc3M=
EOF

ifauth <<EOF
3 AUTHENTICATE PLAIN
dGVzdHVzZXIAdGVzdHVzZXIAdGVzdHBhc3M=
EOF

ifauth <<EOF
4 AUTHENTICATE PLAIN
*
EOF

unset CVM_SASL_PLAIN
}
run_compare_test tests/imapfront-auth-plain  <<END_OF_TEST_RESULTS
1 NO AUTHENTICATE failed: Authentication failed.^M
Yes: 2
+ ^M
Yes: 3
+ ^M
4 NO AUTHENTICATE failed: Authentication failed.^M
END_OF_TEST_RESULTS


##### Test tests/pop3front-maildir-flags #####

runtest() {
pftest() {
  (
    for line in "$@"; do
      echo $line
    done
  ) | pfmaildir $maildir
  ( cd $maildir && ls -1 */* )
}

# Does it properly parse existing v2 flags
maildir
tstmsg new/1000000000.12345.here:2,FX
pftest 'RETR 1' QUIT

# Does it properly ignore non-v2 flags
maildir
tstmsg new/1000000000.12345.here:1fsd
pftest 'RETR 1' QUIT
}
run_compare_test tests/pop3front-maildir-flags  <<END_OF_TEST_RESULTS
+OK ^M
Header: foo^M
^M
body^M
^M
.^M
+OK ^M
cur/1000000000.12345.here:2,FXS
+OK ^M
Header: foo^M
^M
body^M
^M
.^M
+OK ^M
cur/1000000000.12345.here:1fsd
END_OF_TEST_RESULTS


##### Test tests/smtpfront-looping-received #####

runtest() {
MAXHOPS=1 sfecho <<EOF
MAIL FROM:<somebody>
RCPT TO:<nobody@nowhere>
DATA
Received: foo
.
EOF

echo

MAXHOPS=1 sfecho <<EOF
MAIL FROM:<somebody>
RCPT TO:<nobody@nowhere>
DATA
Received: foo
Received: foo
.
EOF
}
run_compare_test tests/smtpfront-looping-received  <<END_OF_TEST_RESULTS
250 Sender='somebody'.^M
250 Recipient='nobody@nowhere'.^M
354 End your message with a period.^M
250 Received 142 bytes.^M

250 Sender='somebody'.^M
250 Recipient='nobody@nowhere'.^M
354 End your message with a period.^M
554 This message is looping, too many hops.^M
END_OF_TEST_RESULTS


##### Test tests/pop3front-maildir-state #####

runtest() {
local quit="$1"
local command="$2"
maildir
tstmsg new/1
(
  echo $command
  if $quit; then echo QUIT; fi
) | pfmaildir $maildir
( cd $maildir && find new cur -type f )
}
run_compare_test tests/pop3front-maildir-state 'false' 'UIDL' <<END_OF_TEST_RESULTS
+OK ^M
1 1^M
.^M
new/1
END_OF_TEST_RESULTS

run_compare_test tests/pop3front-maildir-state 'false' 'TOP 1 0' <<END_OF_TEST_RESULTS
+OK ^M
Header: foo^M
^M
^M
.^M
new/1
END_OF_TEST_RESULTS

run_compare_test tests/pop3front-maildir-state 'false' 'RETR 1' <<END_OF_TEST_RESULTS
+OK ^M
Header: foo^M
^M
body^M
^M
.^M
new/1
END_OF_TEST_RESULTS

run_compare_test tests/pop3front-maildir-state 'true' 'UIDL' <<END_OF_TEST_RESULTS
+OK ^M
1 1^M
.^M
+OK ^M
cur/1
END_OF_TEST_RESULTS

run_compare_test tests/pop3front-maildir-state 'true' 'TOP 1 0' <<END_OF_TEST_RESULTS
+OK ^M
Header: foo^M
^M
^M
.^M
+OK ^M
cur/1
END_OF_TEST_RESULTS

run_compare_test tests/pop3front-maildir-state 'true' 'RETR 1' <<END_OF_TEST_RESULTS
+OK ^M
Header: foo^M
^M
body^M
^M
.^M
+OK ^M
cur/1:2,S
END_OF_TEST_RESULTS


##### Test tests/smtpfront-databytes #####

runtest() {
DATABYTES=123 sfecho <<EOF
EHLO hostname
MAIL FROM:<somebody>
MAIL FROM:<somebody> SIZE
MAIL FROM:<somebody> SIZE=
MAIL FROM:<somebody> SIZE=100
MAIL FROM:<somebody> SIZE=123
MAIL FROM:<somebody> SIZE=124
RCPT TO:<nobody@nowhere>
EOF
}
run_compare_test tests/smtpfront-databytes  <<END_OF_TEST_RESULTS
250-unknown^M
250-SIZE 123^M
250-8BITMIME^M
250 PIPELINING^M
250 Sender='somebody'.^M
250 Sender='somebody'.^M
250 Sender='somebody'.^M
250 Sender='somebody'.^M
250 Sender='somebody'.^M
552 The message would exceed the maximum message size.^M
503 You must send MAIL FROM: first^M
END_OF_TEST_RESULTS


##### Test tests/smtpgreeting #####

runtest() {
env SMTPGREETING='hostname hello' $src/smtpfront-echo 2>/dev/null </dev/null
}
run_compare_test tests/smtpgreeting  <<END_OF_TEST_RESULTS
220 hostname hello ESMTP^M
END_OF_TEST_RESULTS


##### Test tests/smtpfront-content #####

runtest() {
RELAYCLIENT= sfecho <<EOF
MAIL FROM:<user@remote>
RCPT TO:<user@local>
DATA
Subject: test

foo
..
bar
.
MAIL FROM:<user@remote>
RCPT TO:<user@local>
DATA
Subject: test

foo
..
bar

.
EOF
}
run_compare_test tests/smtpfront-content  <<END_OF_TEST_RESULTS
250 Sender='user@remote'.^M
250 Recipient='user@local'.^M
354 End your message with a period.^M
250 Received 153 bytes.^M
250 Sender='user@remote'.^M
250 Recipient='user@local'.^M
354 End your message with a period.^M
250 Received 154 bytes.^M
END_OF_TEST_RESULTS


##### Test tests/smtpfront-auth-login #####

runtest() {
sfqmail <<EOF
AUTH LOGIN
EOF

export CVM_SASL_LOGIN=$tmp/cvm

sfqmail <<EOF
MAIL FROM: <user@remote>
RCPT TO: <user@remote>
AUTH LOGIN
dGVzdHVzZXI=
dGVzdHBhc3x=
AUTH LOGIN
dGVzdHVzZXI=
dGVzdHBhc3M=
AUTH LOGIN
MAIL FROM: <user@remote>
RCPT TO: <user@remote>
EOF

sfqmail << EOF
AUTH LOGIN dGVzdHVzZXI=
dGVzdHBhc3M=
EOF

sfqmail <<EOF
AUTH LOGIN
dGVzdHVzZXI=
*
MAIL FROM: <user@remote>
RCPT TO: <user@remote>
EOF

unset CVM_SASL_PLAIN
}
run_compare_test tests/smtpfront-auth-login  <<END_OF_TEST_RESULTS
504 Unrecognized authentication mechanism.^M
250 Sender accepted.^M
553 Sorry, that domain isn't in my list of allowed rcpthosts.^M
334 VXNlcm5hbWU6^M
334 UGFzc3dvcmQ6^M
501 Authentication failed.^M
334 VXNlcm5hbWU6^M
334 UGFzc3dvcmQ6^M
235 Authentication succeeded.^M
503 You are already authenticated.^M
250 Sender accepted.^M
250 Recipient accepted.^M
334 UGFzc3dvcmQ6^M
235 Authentication succeeded.^M
334 VXNlcm5hbWU6^M
334 UGFzc3dvcmQ6^M
501 Authentication failed.^M
250 Sender accepted.^M
553 Sorry, that domain isn't in my list of allowed rcpthosts.^M
END_OF_TEST_RESULTS


##### Test tests/rules-selector #####

runtest() {
cat >$tmp/rules <<EOF
# This selector forces what would normally be a recipient rule to be
# applied only to the sender.
:sender
kone@one:two@two:A
# This selector forces what would normally be a sender rule to be
# applied only to recipients.
:recipient
ktwo@two:*:B
EOF

MAILRULES=$tmp/rules sfecho <<EOF
MAIL FROM:<one@one>
RCPT TO:<three@three>
RCPT TO:<four@four>
MAIL FROM:<two@two>
RCPT TO:<three@three>
RCPT TO:<four@four>
EOF

rm -f $tmp/rules
}
run_compare_test tests/rules-selector  <<END_OF_TEST_RESULTS
250 A^M
250 Recipient='three@three'.^M
250 Recipient='four@four'.^M
250 Sender='two@two'.^M
250 B^M
250 B^M
END_OF_TEST_RESULTS


##### Test tests/smtpfront-auth-plain #####

runtest() {
sfqmail <<EOF
AUTH PLAIN dGVzdHVzZXIAdGVzdHVzZXIAdGVzdHBhc3M=
EOF

export CVM_SASL_PLAIN=$tmp/cvm

sfqmail <<EOF
MAIL FROM: <user@remote>
RCPT TO: <user@remote>
AUTH PLAIN dGVzdHVzZXIAdGVzdHVzZXIAdGVzdHBhc3x=
AUTH PLAIN dGVzdHVzZXIAdGVzdHVzZXIAdGVzdHBhc3M=
AUTH PLAIN dGVzdHVzZXIAdGVzdHVzZXIAdGVzdHBhc3M=
MAIL FROM: <user@remote>
RCPT TO: <user@remote>
EOF

sfqmail << EOF
AUTH PLAIN
dGVzdHVzZXIAdGVzdHVzZXIAdGVzdHBhc3M=
EOF

sfqmail <<EOF
AUTH PLAIN
*
MAIL FROM: <user@remote>
RCPT TO: <user@remote>
EOF

sfqmail << EOF
AUTH PLAIN XXXXdHVzZXIAdGVzdHVzZXIAdGVzdHBhc3M=
EOF

unset CVM_SASL_PLAIN
}
run_compare_test tests/smtpfront-auth-plain  <<END_OF_TEST_RESULTS
504 Unrecognized authentication mechanism.^M
250 Sender accepted.^M
553 Sorry, that domain isn't in my list of allowed rcpthosts.^M
501 Authentication failed.^M
235 Authentication succeeded.^M
503 You are already authenticated.^M
250 Sender accepted.^M
250 Recipient accepted.^M
334 ^M
235 Authentication succeeded.^M
334 ^M
501 Authentication failed.^M
250 Sender accepted.^M
553 Sorry, that domain isn't in my list of allowed rcpthosts.^M
235 Authentication succeeded.^M
END_OF_TEST_RESULTS


##### Test tests/rules-defaultmsg #####

runtest() {
cat >$tmp/rules <<EOF
dd:*
zz:*
kk:*
d*:d
z*:z
k*:k
EOF

MAILRULES=$tmp/rules sfecho <<EOF
MAIL FROM:<d>
MAIL FROM:<z>
MAIL FROM:<k>
RCPT TO:<d>
RCPT TO:<z>
RCPT TO:<k>
EOF

rm -f $tmp/rules
}
run_compare_test tests/rules-defaultmsg  <<END_OF_TEST_RESULTS
553 Rejected^M
451 Deferred^M
250 OK^M
553 Rejected^M
451 Deferred^M
250 OK^M
END_OF_TEST_RESULTS


##### Test tests/pop3front-auth-userpass #####

runtest() {
# Should fix this and the others to actually check the stderr output too.
pfauth $tmp/cvm <<EOF 2>/dev/null
USER testuser
PASS testpass
EOF
pfauth $tmp/cvm <<EOF 2>/dev/null
USER testuser
PASS testpasx
EOF
}
run_compare_test tests/pop3front-auth-userpass  <<END_OF_TEST_RESULTS
+OK ^M
Yes.
+OK ^M
-ERR Authentication failed^M
END_OF_TEST_RESULTS


##### Test tests/patterns-after #####

runtest() {
cat >$tmp/patterns <<EOF
\after
EOF

PATTERNS=$tmp/patterns sfecho <<EOF
MAIL FROM:<>
RCPT TO:<nobody>
DATA
before

after
.
EOF

echo

cat >$tmp/patterns <<EOF
\before
EOF

PATTERNS=$tmp/patterns sfecho <<EOF
MAIL FROM:<>
RCPT TO:<nobody>
DATA
before

after
.
EOF

rm -f $tmp/patterns
}
run_compare_test tests/patterns-after  <<END_OF_TEST_RESULTS
250 Sender=''.^M
250 Recipient='nobody'.^M
354 End your message with a period.^M
554 This message contains prohibited content^M

250 Sender=''.^M
250 Recipient='nobody'.^M
354 End your message with a period.^M
250 Received 142 bytes.^M
END_OF_TEST_RESULTS


##### Test tests/rules-maxhops #####

runtest() {
cat >$tmp/rules <<EOF
ka:b:K1:::MAXHOPS=1
EOF

MAILRULES=$tmp/rules sfecho <<EOF
MAIL FROM:<a>
RCPT TO:<b>
DATA
Received: hop1
Received: hop2
.
EOF
MAILRULES=$tmp/rules sfecho <<EOF
MAIL FROM:<a>
RCPT TO:<c>
DATA
Received: hop1
Received: hop1
.
EOF

rm -f $tmp/rules
}
run_compare_test tests/rules-maxhops  <<END_OF_TEST_RESULTS
250 Sender='a'.^M
250 K1^M
354 End your message with a period.^M
554 This message is looping, too many hops.^M
250 Sender='a'.^M
250 Recipient='c'.^M
354 End your message with a period.^M
250 Received 158 bytes.^M
END_OF_TEST_RESULTS


##### Test tests/smtpfront-looping-delivered-to #####

runtest() {
MAXHOPS=1 sfecho <<EOF
MAIL FROM:<somebody>
RCPT TO:<nobody@nowhere>
DATA
Delivered-To: foo
.
EOF

echo

MAXHOPS=1 sfecho <<EOF
MAIL FROM:<somebody>
RCPT TO:<nobody@nowhere>
DATA
Delivered-To: foo
Delivered-To: foo
.
EOF
}
run_compare_test tests/smtpfront-looping-delivered-to  <<END_OF_TEST_RESULTS
250 Sender='somebody'.^M
250 Recipient='nobody@nowhere'.^M
354 End your message with a period.^M
250 Received 146 bytes.^M

250 Sender='somebody'.^M
250 Recipient='nobody@nowhere'.^M
354 End your message with a period.^M
554 This message is looping, too many hops.^M
END_OF_TEST_RESULTS


##### Test tests/pop3front-auth-login #####

runtest() {
pfauth false <<EOF
AUTH LOGIN
EOF

export CVM_SASL_LOGIN=$tmp/cvm

pfauth false <<EOF
AUTH LOGIN
dGVzdHVzZXI=
dGVzdHBhc3x=
AUTH LOGIN
dGVzdHVzZXI=
dGVzdHBhc3M=
EOF

pfauth false <<EOF
AUTH LOGIN dGVzdHVzZXI=
dGVzdHBhc3M=
EOF

pfauth false <<EOF
AUTH LOGIN
dGVzdHVzZXI=
*
EOF

unset CVM_SASL_PLAIN
}
run_compare_test tests/pop3front-auth-login  <<END_OF_TEST_RESULTS
-ERR Unrecognized authentication mechanism.^M
+ VXNlcm5hbWU6^M
+ UGFzc3dvcmQ6^M
-ERR Authentication failed.^M
+ VXNlcm5hbWU6^M
+ UGFzc3dvcmQ6^M
Yes.
+ UGFzc3dvcmQ6^M
Yes.
+ VXNlcm5hbWU6^M
+ UGFzc3dvcmQ6^M
-ERR Authentication failed.^M
END_OF_TEST_RESULTS


##### Test tests/pop3front-auth-plain #####

runtest() {
pfauth false <<EOF
AUTH PLAIN dGVzdHVzZXIAdGVzdHVzZXIAdGVzdHBhc3M=
EOF

export CVM_SASL_PLAIN=$tmp/cvm

pfauth false <<EOF
AUTH PLAIN dGVzdHVzZXIAdGVzdHVzZXIAdGVzdHBhc3x=
AUTH PLAIN dGVzdHVzZXIAdGVzdHVzZXIAdGVzdHBhc3M=
EOF

pfauth false <<EOF
AUTH PLAIN
dGVzdHVzZXIAdGVzdHVzZXIAdGVzdHBhc3M=
EOF

pfauth false <<EOF
AUTH PLAIN
*
EOF

unset CVM_SASL_PLAIN
}
run_compare_test tests/pop3front-auth-plain  <<END_OF_TEST_RESULTS
-ERR Unrecognized authentication mechanism.^M
-ERR Authentication failed.^M
Yes.
+ ^M
Yes.
+ ^M
-ERR Authentication failed.^M
END_OF_TEST_RESULTS


##### Test tests/smtpfront-bad-bounce #####

runtest() {
sfecho <<EOF
MAIL FROM:<not@bounce>
RCPT TO:<addr1@domain>
RCPT TO:<addr2@domain>
DATA
.
EOF

sfecho <<EOF
MAIL FROM:<>
RCPT TO:<addr1@domain>
DATA
.
EOF

sfecho <<EOF
MAIL FROM:<>
RCPT TO:<addr1@domain>
RCPT TO:<addr2@domain>
DATA
EOF
}
run_compare_test tests/smtpfront-bad-bounce  <<END_OF_TEST_RESULTS
250 Sender='not@bounce'.^M
250 Recipient='addr1@domain'.^M
250 Recipient='addr2@domain'.^M
354 End your message with a period.^M
250 Received 128 bytes.^M
250 Sender=''.^M
250 Recipient='addr1@domain'.^M
354 End your message with a period.^M
250 Received 128 bytes.^M
250 Sender=''.^M
250 Recipient='addr1@domain'.^M
550 Bounce messages should have a single recipient.^M
550 Bounce messages should have a single recipient.^M
END_OF_TEST_RESULTS


##### Test tests/received #####

runtest() {
sfecho2() {
$src/smtpfront-echo 2>&1 >/dev/null <<EOF | \
	sed -n -e 's/^.* Received: //p'
MAIL FROM:<>
RCPT TO:<test>
DATA
.
EOF
}

TESTLOCALIP=   TESTLOCALHOST=   TESTREMOTEIP=   TESTREMOTEHOST=   sfecho2
TESTLOCALIP=li TESTLOCALHOST=   TESTREMOTEIP=   TESTREMOTEHOST=   sfecho2
TESTLOCALIP=   TESTLOCALHOST=lh TESTREMOTEIP=   TESTREMOTEHOST=   sfecho2
TESTLOCALIP=li TESTLOCALHOST=lh TESTREMOTEIP=   TESTREMOTEHOST=   sfecho2
TESTLOCALIP=   TESTLOCALHOST=   TESTREMOTEIP=ri TESTREMOTEHOST=   sfecho2
TESTLOCALIP=li TESTLOCALHOST=   TESTREMOTEIP=ri TESTREMOTEHOST=   sfecho2
TESTLOCALIP=   TESTLOCALHOST=lh TESTREMOTEIP=ri TESTREMOTEHOST=   sfecho2
TESTLOCALIP=li TESTLOCALHOST=lh TESTREMOTEIP=ri TESTREMOTEHOST=   sfecho2
TESTLOCALIP=   TESTLOCALHOST=   TESTREMOTEIP=   TESTREMOTEHOST=rh sfecho2
TESTLOCALIP=li TESTLOCALHOST=   TESTREMOTEIP=   TESTREMOTEHOST=rh sfecho2
TESTLOCALIP=   TESTLOCALHOST=lh TESTREMOTEIP=   TESTREMOTEHOST=rh sfecho2
TESTLOCALIP=li TESTLOCALHOST=lh TESTREMOTEIP=   TESTREMOTEHOST=rh sfecho2
TESTLOCALIP=   TESTLOCALHOST=   TESTREMOTEIP=ri TESTREMOTEHOST=rh sfecho2
TESTLOCALIP=li TESTLOCALHOST=   TESTREMOTEIP=ri TESTREMOTEHOST=rh sfecho2
TESTLOCALIP=   TESTLOCALHOST=lh TESTREMOTEIP=ri TESTREMOTEHOST=rh sfecho2
TESTLOCALIP=li TESTLOCALHOST=lh TESTREMOTEIP=ri TESTREMOTEHOST=rh sfecho2

sfecho3() {
$src/smtpfront-echo 2>&1 >/dev/null <<EOF | \
	sed -n -e 's/^.* Received: //p'
EHLO hh
MAIL FROM:<>
RCPT TO:<test>
DATA
.
EOF
}

TESTLOCALIP=   TESTLOCALHOST=   TESTREMOTEIP=   TESTREMOTEHOST=   sfecho3
TESTLOCALIP=li TESTLOCALHOST=   TESTREMOTEIP=   TESTREMOTEHOST=   sfecho3
TESTLOCALIP=   TESTLOCALHOST=lh TESTREMOTEIP=   TESTREMOTEHOST=   sfecho3
TESTLOCALIP=li TESTLOCALHOST=lh TESTREMOTEIP=   TESTREMOTEHOST=   sfecho3
TESTLOCALIP=   TESTLOCALHOST=   TESTREMOTEIP=ri TESTREMOTEHOST=   sfecho3
TESTLOCALIP=li TESTLOCALHOST=   TESTREMOTEIP=ri TESTREMOTEHOST=   sfecho3
TESTLOCALIP=   TESTLOCALHOST=lh TESTREMOTEIP=ri TESTREMOTEHOST=   sfecho3
TESTLOCALIP=li TESTLOCALHOST=lh TESTREMOTEIP=ri TESTREMOTEHOST=   sfecho3
TESTLOCALIP=   TESTLOCALHOST=   TESTREMOTEIP=   TESTREMOTEHOST=rh sfecho3
TESTLOCALIP=li TESTLOCALHOST=   TESTREMOTEIP=   TESTREMOTEHOST=rh sfecho3
TESTLOCALIP=   TESTLOCALHOST=lh TESTREMOTEIP=   TESTREMOTEHOST=rh sfecho3
TESTLOCALIP=li TESTLOCALHOST=lh TESTREMOTEIP=   TESTREMOTEHOST=rh sfecho3
TESTLOCALIP=   TESTLOCALHOST=   TESTREMOTEIP=ri TESTREMOTEHOST=rh sfecho3
TESTLOCALIP=li TESTLOCALHOST=   TESTREMOTEIP=ri TESTREMOTEHOST=rh sfecho3
TESTLOCALIP=   TESTLOCALHOST=lh TESTREMOTEIP=ri TESTREMOTEHOST=rh sfecho3
TESTLOCALIP=li TESTLOCALHOST=lh TESTREMOTEIP=ri TESTREMOTEHOST=rh sfecho3
}
run_compare_test tests/received  <<END_OF_TEST_RESULTS
from unknown  by unknown
from unknown  by li ([li])
from unknown  by lh
from unknown  by lh ([li])
from ri ([ri])  by unknown
from ri ([ri])  by li ([li])
from ri ([ri])  by lh
from ri ([ri])  by lh ([li])
from rh (rh)  by unknown
from rh (rh)  by li ([li])
from rh (rh)  by lh
from rh (rh)  by lh ([li])
from rh (rh [ri])  by unknown
from rh (rh [ri])  by li ([li])
from rh (rh [ri])  by lh
from rh (rh [ri])  by lh ([li])
from hh  by unknown
from hh  by li ([li])
from hh  by lh
from hh  by lh ([li])
from hh ([ri])  by unknown
from hh ([ri])  by li ([li])
from hh ([ri])  by lh
from hh ([ri])  by lh ([li])
from hh (rh)  by unknown
from hh (rh)  by li ([li])
from hh (rh)  by lh
from hh (rh)  by lh ([li])
from hh (rh [ri])  by unknown
from hh (rh [ri])  by li ([li])
from hh (rh [ri])  by lh
from hh (rh [ri])  by lh ([li])
END_OF_TEST_RESULTS


##### Test tests/rules-multiline #####

runtest() {
cat >$tmp/rules <<EOF
ka@a:*:ONE\nTWO
ka@b:*:ONE\:TWO
ka@c:*:ONE\\\\TWO:
EOF

MAILRULES=$tmp/rules sfecho <<EOF
MAIL FROM:<a@a>
MAIL FROM:<a@b>
MAIL FROM:<a@c>
EOF

rm -f $tmp/rules
}
run_compare_test tests/rules-multiline  <<END_OF_TEST_RESULTS
250-ONE^M
250 TWO^M
250 ONE:TWO^M
250 ONE\TWO^M
END_OF_TEST_RESULTS


##### Test tests/rules-asterisk #####

runtest() {
cat >$tmp/rules <<EOF
ka@a:*:K1
k*@a:*:K2
kc@*:*:K3
k*:*:K4
EOF

MAILRULES=$tmp/rules sfecho <<EOF
MAIL FROM:<a@a>
MAIL FROM:<b@a>
MAIL FROM:<c@b>
MAIL FROM:<d@d>
MAIL FROM:<>
EOF

rm -f $tmp/rules
}
run_compare_test tests/rules-asterisk  <<END_OF_TEST_RESULTS
250 K1^M
250 K2^M
250 K3^M
250 K4^M
250 K4^M
END_OF_TEST_RESULTS


rm -rf $tmp
echo $tests_count tests executed, $tests_failed failures
if [ $tests_failed != 0 ]; then exit 1; fi
