// SocketConnection.cc - source file for the mailfilter program
// Copyright (c) 2000 - 2004  Andreas Bauer <baueran@in.tum.de>
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
// USA.


#include <string>
#include <cstring>
#include <csignal>
#include <cerrno>
#include <stdexcept>
#include <iostream>
extern "C" {
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/select.h>
#include <sys/time.h>
#include <netdb.h>
#include <netinet/in.h>
#include <unistd.h>
#include <fcntl.h>
#include <setjmp.h>
}
#include "Connection.hh"
#include "SocketConnection.hh"
#include "mailfilter.hh"

using namespace std;

// Please also read the comments in Connection.hh

namespace conn {

  // Used for alarm signal handling
  static sigjmp_buf myEnv;

  SocketConnection::SocketConnection() {
    mySocket = 0;
    port = 0;
    timeOut = 30; // Default time out in seconds
  }


  SocketConnection::~SocketConnection() {
  }

  
  int SocketConnection::connectHost(const string& hostname, int thePort, int timeout) {
    struct sockaddr_in socketAddress;
    struct hostent *myHost;
    struct sigaction sigact;
      
    // Set some local values
    port = thePort;
    timeOut = timeout;

    // Install alarm signal handler
    sigact.sa_handler = connectAlarm;
    sigemptyset(&sigact.sa_mask);
    sigact.sa_flags = 0;
    
    if (sigaction(SIGALRM, &sigact, NULL) < 0)
      return SIGNAL_FAILURE;

    if (sigsetjmp(myEnv, 1) == 0) {
      alarm(timeOut);

      // Create socket and try to connect
      if ( (mySocket = socket(PF_INET, SOCK_STREAM, 0)) < 0 )
	return SOCKET_CONNECTION_FAILURE;          // Connection failure
      else {
	myHost = gethostbyname(hostname.c_str());
	
	if (!myHost) {
	  return DNS_LOOKUP_FAILURE;               // DNS lookup failure
	}
	else {
	  memset( (char*)&socketAddress, 0, sizeof(struct sockaddr_in) );
	  socketAddress.sin_family = AF_INET;
	  memcpy( &(socketAddress.sin_addr.s_addr), myHost->h_addr, myHost->h_length);
	  socketAddress.sin_port = htons(port);
	  
	  if (connect(mySocket, (struct sockaddr*)&socketAddress, sizeof(struct sockaddr)) < 0)
	    return SOCKET_CONNECTION_FAILURE;      // Connection failure
	}
      }
      
      // Reset alarm signal handler
      alarm(0);
    
      // Connection successfully established
      return 0;
    }
    else
      return ALARM_FAILURE;
  }


  int SocketConnection::disconnectHost(void) {
    return close(mySocket);
  }


  int SocketConnection::sendHost(const string& command) {
    ssize_t noBytes = write(mySocket, command.c_str(), strlen(command.c_str()));

    if (noBytes < 0) {
      string writeError = strerror(errno);
      throw runtime_error( writeError );
    }
    else 
      return noBytes;
  }
  
  
  // According to RFC 1939 a single line response is terminated with \r\n and may be up to 512 bytes long
  // Multi-line messages are ended by \r\n.\r\n and can contain many \r\n in between!
  string SocketConnection::receiveHost(const bool singleLine) {
    char buffer[MAX_BYTES + 20];   // Leave some space for expansion
    int flags = 0, error = 0, counter = 0, bytes = 0, last = 0;
    struct timeval tv;
    fd_set rfds;
    string input;
    
    // Memorize old flags of the open socket
    if ( (flags = fcntl(mySocket, F_GETFL, 0)) == -1 )
      throw IOException();

    // Change the socket communication stream to nonblocking
    if (fcntl(mySocket, F_SETFL, flags | O_NONBLOCK) == -1 )
      throw IOException();
    
    do {
      memset(buffer, 0, sizeof(buffer));
      last = bytes;
      tv.tv_sec = timeOut;
      tv.tv_usec = 0;
      FD_ZERO(&rfds);
      FD_SET(mySocket, &rfds);
      
      // Only read if the socket is ready and sending data, i.e. if error > 0
      error = select(mySocket + 1, &rfds, NULL, NULL, &tv);
      
      if (error < 0) {
	string selectError = strerror(errno);
	throw runtime_error( selectError );
      }
      else if (error > 0 && FD_ISSET(mySocket, &rfds)) {
	bytes = read(mySocket, buffer, MAX_BYTES);
	
	if (bytes > 0) {
	  counter += bytes;
	  
	  if (input.length() > 0)
	    input.append(buffer, bytes);   // Append certain number of bytes to output stream; those are not null-terminated, we do that later!
	  else {
	    buffer[bytes] = '\0';          // Terminate the string, even though we check for it again later...
	    input = (string)buffer;
	  }
	}
	else if (bytes == 0)
	  break;
	else
	  throw IOException();
      }
    } while ( error > 0  &&  
	      ( (singleLine && input.find_last_of('\n', input.length()) == string::npos) || (!singleLine && !isHeaderEnd(input)) ) );
    
    // Put line end behind the buffer, if needed
    if (input.find_last_of('\0', input.length()) == string::npos)
      input[counter] = '\0';

    // Reset socket communication to blocking mode
    if ( (fcntl(mySocket, F_SETFL, flags)) == -1)
      throw IOException();
    
    // Return the server response
    return(input);
  }


  //! Alarm signal call-back function
  void SocketConnection::connectAlarm(int signo) {
    // No need to mask signals in a one-liner, is there?
    siglongjmp(myEnv, signo);
  }
  

}
