#!/bin/bash
#
# syntax         : checkrules.sh [(-M |--mailfilterrc=)<rcfile>] <mailfile>
#                  checkrules.sh <-h|--help>
#                  checkrules.sh <-v|--version>
#
# description    : The given mailfile is checked against the rules defined in
#                  your mailfilterrc. This script prints all matches and if
#                  the mail is deleted from the server and why.
#
#                  IMPORTANT NOTE: Set the variable MAILFILTERRC below to
#                                  point to your mailfilterrc file.
#
# author         : Kai Hildebrandt <kai.hildebrandt@web.de>
#
# date           : 2004-03-21
#
# version        : 0.6.3 (for mailfilter version 0.5.1 to 0.6.x)
#
# license        : Licensed under the same terms as Mailfilter, GPL v2 or later
# -----------------------------------------------------------------------------
# this variable should point to your mailfilterrc
# -----------------------------------------------------------------------------
MAILFILTERRC=${HOME}/.mailfilterrc
# -----------------------------------------------------------------------------
# dont edit anything below this line
# -----------------------------------------------------------------------------
ME=$( basename $0 )
VERSION=0.6.3
# -----------------------------------------------------------------------------
function clean_exit()
{
  # Remove tempfiles
    rm -f ${TMPFILE}
    rm -f ${TMPFILE2}

  # reset LANG
    LANG=${OLDLANG}

    exit 0
}


function get_options()
{
    if [ $# -eq 0 ]; then
	usage
    fi

    ARGC=0
    FLAG_RC=0

    for ARG in $*; do

	ARGC=$(( ${ARGC} + 1 ))

	if [ ${FLAG_RC} -ne 0 ]; then
	    MAILFILTERRC=${ARG}
	    FLAG_RC=0
	    continue
	fi

	case ${ARG} in
	    -M) FLAG_RC=1
		;;

	    --mailfilterrc=*)
		MAILFILTERRC=$( echo ${ARG} | cut -d"=" -f2 )
		;;

	    -h|--help)
		usage
		;;

	    -v|--version)
		echo "${ME} ${VERSION}"
		echo "(for mailfilter 0.5.x - 0.6.x)"
		echo
		echo "Copyright (c) 2003 - 2004  Kai Hildebrandt  <kai.hildebrandt@web.de>"
		echo
		echo "This is free software; see the source for copying conditions.  There is NO"
		echo "warranty; not even for MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE."
		exit 0
		;;

	    *)
		if [ $ARGC -lt $# ]; then
		    echo "unknown option: ${ARG}"
		    usage
		else
		    MAILFILE=${ARG}
		fi
		;;
	esac
    done
}


function get_value()
{
    # syntax: getvalue <pattern> <tmpfile>
    #
    # gets a value for a setting like
    # REG_CASE=no -> no
    #
    # comments are discarded, last match hit (tail -1)
    #
    if [ -r "$2" ]; then
	grep "$1" $2 | tail -1 | sed -e "s/[[:space:]]*=[[:space:]]*/=/" | cut -d= -f2 | sed -e "s/#.*//"
    else
	# error!
	echo "-1"
    fi
}


function usage()
{
    echo "usage: ${ME} [(-M |--mailfilterrc=)<your mailfilterrc>] <file>"
    echo "       ${ME} <-h|--help>"
    echo "       ${ME} <-v|--version>"
    exit 1
}


########
# main #
########

# check command line options if any
get_options $*


# check for files
if [ ! -f ${MAILFILTERRC} ]; then
    echo "error: ${MAILFILTERRC} not found."
    exit 2
fi

if [ ! -f ${MAILFILE} ]; then
    echo "error: ${MAILFILE} not found."
    exit 2
fi


# unset LANG
OLDLANG=${LANG}
unset LANG


# tempfiles
TMPFILE=/var/tmp/${ME}.1.$$.tmp
TMPFILE2=/var/tmp/${ME}.2.$$.tmp


# make tempfiles unreadable by others
umask 077


# Catch EXIT to clean up tempfiles at exit from shell
trap "clean_exit" EXIT


# Create temporary mailfilterrc with all include files
cat ${MAILFILTERRC} $( grep "^INCLUDE" ${MAILFILTERRC} | cut -f2 -d= ) > ${TMPFILE}


# get setting for REG_TYPE, REG_CASE and NORMALIZE
REG_TYPE=$( get_value "^REG_TYPE" ${TMPFILE} )
REG_CASE=$( get_value "^REG_CASE" ${TMPFILE} )
NORMAL=$( get_value "^NORMAL" ${TMPFILE}  )

if [ "${REG_TYPE}" = "extended" ]; then
    EXTENDED="-E"
else
    EXTENDED=""
fi

if [ "${REG_CASE}" != "yes" ]; then
    REG_CASE="no"
fi

if [ "${NORMAL}" != "yes" ]; then
    NORMAL="no"
fi


# get rules from mailfilterrc and included files
grep -E "^(DENY|ALLOW|SCORE|HIGHSCORE)" ${TMPFILE} | sort -r > ${TMPFILE2}


# search for self-defined HIGHSCORE
HIGHSCORE=$( get_value "^HIGHSCORE" ${TMPFILE2} )

# set default HIGHSCORE
if [ -z "${HIGHSCORE}" ]; then
    HIGHSCORE=100
fi

# delete HIGHSCORE lines and substitute \ with \\ (\\\ ?)
sed -e "/^HIGHSCORE/d" -e "s/\\\/\\\\\\\/g" ${TMPFILE2} > ${TMPFILE}


echo "REG_TYPE  is ${REG_TYPE}"
echo "REG_CASE  is ${REG_CASE}"
echo "NORMAL    is ${NORMAL}"
echo "HIGHSCORE is ${HIGHSCORE}"
echo "------------------------------------------------------------------------------"


# reset values
SCORE=0
ALLOWED=0
DENIED=0


# Generate normalized Subject
NORMALIZED_SUBJECT=$( echo $( grep "^Subject:" ${MAILFILE} ) | sed -e "s/^Subject: *//" -e "s/[^[:alnum:][:space:]]//g" )


echo "Normalized subject is \"${NORMALIZED_SUBJECT}\""
echo "------------------------------------------------------------------------------"


# check all rules against file
while read LINE; do

    if [ ! -z "$( echo ${LINE} | grep -E '[[:upper:][:digit:]_ +-]+=' )" ]; then
	# equal rule
	WHAT=$( echo ${LINE} | cut -f1 -d= )
	PATTERN=$( echo ${LINE} | cut -f2- -d= )
	NOT=0
    else
	# not equal rule
	WHAT=$( echo ${LINE} | cut -f1 -d"<" )
	PATTERN=$( echo ${LINE} | cut -f2- -d"<" | cut -b2- )
	NOT=1
    fi


    # are filters case sensitive?
    CASE=""

    # global setting REG_CASE=(yes|no)
    if [ "${WHAT}" = "ALLOW" -o "${REG_CASE}" = "no" ]; then
	CASE="-i"
    fi

    # only for rule e.g. DENY_CASE or SCORE_NOCASE
    T="$( echo "${WHAT}" | cut -f2 -d_ | cut -d' ' -f1 )"

    if [ "${T}" != "${WHAT}" ]; then
	if [ "${T}" = "CASE" ]; then
	    # CASE=CASE
	    CASE=""
	elif [ "${T}" = "NOCASE" ]; then
            # CASE=NOCASE
	    CASE="-i"
	fi
    fi

    MATCHES=$( grep ${EXTENDED} ${CASE} -e "${PATTERN}" ${MAILFILE} | wc -l )

    # Check normalized Subject
    if [ $( echo "${PATTERN}" | grep -i "^Subject:" | wc -l ) -gt 0 ]; then
	if [ ${MATCHES} -eq 0 -a "${NORMAL}" = "yes" ]; then
	    MATCHES=$( echo "${NORMALIZED_SUBJECT}" | grep ${EXTENDED} ${CASE} -e "${PATTERN}" | wc -l )
	fi
    fi


    if [ ${NOT} -eq 1 ]; then
	SHOWLINE="\"<>${PATTERN}\""
    else
	SHOWLINE="\"${PATTERN}\" to \"$( grep ${EXTENDED} ${CASE} -e "${PATTERN}" ${MAILFILE} )\""
    fi


    if [ ${MATCHES} -ne 0 -a ${NOT} -eq 0 -o ${MATCHES} -eq 0 -a ${NOT} -ne 0 ]; then

	case ${WHAT} in

	    ALLOW)
		echo "ALLOW rule match: ${SHOWLINE}"
		ALLOWED=1
		;;

	    DENY)
		echo "DENY rule match: ${SHOWLINE}"
		DENIED=1
		;;

	    SCORE*)
		echo "SCORE ($( echo ${WHAT} | cut -f2 -d" " )) rule match: ${SHOWLINE}"
		SCORE=$(( ${SCORE} $( echo ${WHAT} | cut -f2 -d" " ) ))
		;;
	esac
    fi

done < ${TMPFILE}


echo
echo "Total score: ${SCORE}"
echo


if [ ${ALLOWED} -eq 1 ]; then
    echo "Mail would be downloaded from server because of an ALLOW rule."

elif [ ${DENIED} -eq 1 ]; then
    echo "Mail would be deleted from server because of a DENY rule."

elif [ ${SCORE} -ge ${HIGHSCORE} ]; then
    echo "Mail would be deleted from server because of reached HIGHSCORE."

else
    echo "Mail would be downloaded from server."

fi

clean_exit
