/*
madman - a music manager
Copyright (C) 2003  Andreas Kloeckner <ak@ixion.net>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/




#include <pwd.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>
#include <fstream>
#include <stdexcept>

#include <qtoolbutton.h>
#include <qtextbrowser.h>
#include <qheader.h>
#include <qprogressdialog.h>
#include <qdragobject.h>
#include <qapplication.h>
#include <qstatusbar.h>
#include <qfiledialog.h>
#include <qpushbutton.h>
#include <qlistbox.h>
#include <qmessagebox.h>
#include <qcombobox.h>
#include <qlineedit.h>
#include <qradiobutton.h>
#include <qfile.h>
#include <qstring.h>
#include <qaction.h>
#include <qsplitter.h>

#include "mainwin.h"
#include "utility/player.h"
#include "utility/progress.h"
#include "database/song_set.h"
#include "database/criterion.h"
#include "database/song_list_tools.h"
#include "designer/helpbrowser.h"
#include "ui/set_list_view.h"
#include "ui/multiline.h"
#include "ui/overview.h"
#include "ui/trayicon.h"
#include "ui/passive_popup.h"
#include "ui/progress_impl.h"




// tools ----------------------------------------------------------------------
namespace {
}




// tool functions -------------------------------------------------------------
QString getConfigFile()
{
  passwd * pwd = getpwuid(getuid());
  if (pwd->pw_dir)
    return QString(pwd->pw_dir) + "/.madmandb";
  else
    return ".madmandb";
}




// tMainWindow ----------------------------------------------------------------
tMainWindow::tMainWindow(const QString &filename_to_open)
    : SearchViewManager(*lstAllSongs, *this, "/madman/allsongsview"),
    PlaylistEditor(*lstSetEditor, *this, "/madman/songseteditor"),
    CurrentSearchProgress(NULL),
    HttpDaemon(NULL), SystemTrayIcon(NULL),
    OverviewManager(*lstOverview, this)
{
  // load and realize prefs ---------------------------------------------------
  Preferences.load(Settings);
  SearchViewManager.loadListViewAppearance(Settings);
  PlaylistEditor.loadListViewAppearance(Settings);

  loadSplitterAppearance(splitter1, Settings, "/madman/splitters/splitter1.state");
  loadSplitterAppearance(splitter2, Settings, "/madman/splitters/splitter2.state");
  loadSplitterAppearance(splitter3, Settings, "/madman/splitters/splitter3.state");

  // set up gui ---------------------------------------------------------------
  lstOverview->setSorting(-1);
  lstOverview->header()->hide();
  lstSets->header()->hide();

  lstSets->setAcceptDrops(true);
  lstSets->viewport()->setAcceptDrops(true);
  
  tbMain->setShown(Settings.readBoolEntry("/madman/main_toolbar_shown", 1));
  tbMediaControl->setShown(Settings.readBoolEntry("/madman/media_control_toolbar_shown", 1));
  tbRating->setShown(Settings.readBoolEntry("/madman/rating_toolbar_shown", 0));
  
  if (Preferences.RememberGeometry)
  {
    {
      QString temp = Settings.readEntry("/madman/main_window_position");
      if (!temp.isNull())
      {
	int x,y;
	QTextIStream is(&temp);
	is >> x >> y;
	QPoint p(x,y);
	move(p);
      }
    }

    {
      QString temp = Settings.readEntry("/madman/main_window_size");
      if (!temp.isNull())
      {
	int x,y;
	QTextIStream is(&temp);
	is >> x >> y;
	QSize s(x,y);
	resize(s);
      }
    }
  }

  // various buttons ----------------------------------------------------------
  connect(btnNewSet, SIGNAL(clicked()), this, SLOT(addPlaylist()));
  connect(btnBookmarkSet, SIGNAL(clicked()), this, SLOT(bookmarkCurrentSearch()));
  connect(btnImportSet, SIGNAL(clicked()), this, SLOT(importPlaylist()));
  connect(btnCopySet, SIGNAL(clicked()), this, SLOT(duplicatePlaylist()));
  connect(btnRemoveSet, SIGNAL(clicked()), this, SLOT(removePlaylist()));
  connect(btnEditSetCriterion, SIGNAL(clicked()), this, SLOT(editMultilinePlaylistCriterion()));

  connect(btnHelpAll, SIGNAL(clicked()), this, SLOT(help()));
  connect(btnHelpSet, SIGNAL(clicked()), this, SLOT(help()));
  connect(btnEditSetCriterion, SIGNAL(clicked()),
           this, SLOT(songSetCriterionChanged()));
  connect(btnUpdateSetCriterion, SIGNAL(clicked()),
           this, SLOT(songSetCriterionChanged()));

  // various UI events --------------------------------------------------------
  connect(lstSets, SIGNAL(itemRenamed(QListViewItem *, int, const QString &)),
           this, SLOT(renamePlaylist(QListViewItem *, int, const QString &)));
  connect(lstSets, SIGNAL(selectionChanged()),
           this, SLOT(songSetSelectionChanged()));
  connect(editPlaylistCriterion, SIGNAL(returnPressed()),
           this, SLOT(songSetCriterionChanged()));
  connect(&SearchViewManager, SIGNAL(add (const tSongList &)),
             this, SLOT(add (const tSongList &)));
  connect(editSearch, SIGNAL(textChanged(const QString &)),
           this, SLOT(searchChanged(const QString &)));
  connect(lstOverview, SIGNAL(selectionChanged(QListViewItem *)),
           this, SLOT(overviewSelectionChanged(QListViewItem *)));
  connect(&SearchViewManager, SIGNAL(notifySearchChangeRequested(const QString &)),
      editSearch, SLOT(setText(const QString &)));
  connect(&PlaylistEditor, SIGNAL(notifySearchChangeRequested(const QString &)),
      editSearch, SLOT(setText(const QString &)));
  connect(&PlaylistEditor, SIGNAL(notifySongSetChanged()),
      this, SLOT(noticeSongSetChanged()));

  // menu ---------------------------------------------------------------------
  connect(fileNewAction, SIGNAL(activated()), this, SLOT(fileNew()));
  connect(fileOpenAction, SIGNAL(activated()), this, SLOT(fileOpen()));
  connect(fileSaveAction, SIGNAL(activated()), this, SLOT(fileSave()));
  connect(fileSaveAsAction, SIGNAL(activated()), this, SLOT(fileSaveAs()));
  connect(filePreferencesAction, SIGNAL(activated()), this, SLOT(filePreferences()));
  connect(fileExitAction, SIGNAL(activated()), this, SLOT(close()));
  connect(rescanAction, SIGNAL(activated()), this, SLOT(rescan()));
  connect(rereadTagsAction, SIGNAL(activated()), 
      this, SLOT(rereadTags()));
  connect(updateOverviewAction, SIGNAL(activated()), this, SLOT(buildOverviewTree()));
  connect(helpAboutAction, SIGNAL(activated()), this, SLOT(helpAbout()));
  connect(helpAction, SIGNAL(activated()), this, SLOT(help()));
  connect(minimizeWindowAction, SIGNAL(activated()), this, SLOT(showMinimized()));
  connect(restoreWindowAction, SIGNAL(activated()), this, SLOT(showNormal()));
  connect(hideWindowAction, SIGNAL(activated()), this, SLOT(hide()));

  // player control -----------------------------------------------------------
  connect(playAction, SIGNAL(activated()), this, SLOT(play()));
  connect(pauseAction, SIGNAL(activated()), this, SLOT(pause()));
  connect(stopAction, SIGNAL(activated()), this, SLOT(stop()));
  connect(skipForwardAction, SIGNAL(activated()), this, SLOT(skipForward()));
  connect(skipBackAction, SIGNAL(activated()), this, SLOT(skipBack()));
  connect(sbarSongPosition, SIGNAL(sliderMoved(int)), this, SLOT(skipTo(int)));
  connect(sbarSongPosition, SIGNAL(nextLine()), this, SLOT(skipToWrapper()));
  connect(sbarSongPosition, SIGNAL(prevLine()), this, SLOT(skipToWrapper()));
  connect(sbarSongPosition, SIGNAL(nextPage()), this, SLOT(skipToWrapper()));
  connect(sbarSongPosition, SIGNAL(prevPage()), this, SLOT(skipToWrapper()));

  connect(highlightSongAction, SIGNAL(activated()), this, SLOT(highlightCurrentSong()));

  // rating -------------------------------------------------------------------
  connect(rate0Action, SIGNAL(activated()), this, SLOT(rate0()));
  connect(rate1Action, SIGNAL(activated()), this, SLOT(rate1()));
  connect(rate2Action, SIGNAL(activated()), this, SLOT(rate2()));
  connect(rate3Action, SIGNAL(activated()), this, SLOT(rate3()));
  connect(rate4Action, SIGNAL(activated()), this, SLOT(rate4()));
  connect(rate5Action, SIGNAL(activated()), this, SLOT(rate5()));

  // set up song set tree -----------------------------------------------------
  lstSets->setAcceptDrops(true);
  lstSets->viewport()->setAcceptDrops(true);
  lstSets->setSorting(-1);

  // player state monitoring --------------------------------------------------
  connect(&UpdateTimer, SIGNAL(timeout()),
      this, SLOT(updatePlayerStatus()));
  UpdateTimer.start(500);

  WasPlaying = false;
  AccumulatedPlayTime = 0;

  // load everything ----------------------------------------------------------
  try
  {
    auto_ptr<tDatabase> new_db(new tDatabase);

    CurrentFilename = filename_to_open;
    FilenameValid = !filename_to_open.isNull();

    if (!FilenameValid)
    {
      CurrentFilename = Settings.readEntry("/madman/startup_file", QString::null, &FilenameValid);
    }

    if (!FilenameValid)
      throw runtime_error(QString2string(tr("No startup file found")));

    auto_ptr<tProgressDialog> progress(new tProgressDialog(this, false));
    progress->setWhat(tr("Loading database..."));
    new_db->load(CurrentFilename, progress.get());

    if (Preferences.ScanAtStartup)
      rescan(new_db.get());

    setDatabase(new_db.release());
  }
  catch (runtime_error &ex)
  {
    if (ex.what() != QString2string(tr("No startup file found")))
    {
      QMessageBox::information(this, tr("madman"),
	  tr("Couldn't restore previously opened file:\n%1\nCreating a new file.").
	  arg(QString(ex.what())),
	  QMessageBox::Ok);
    }
    else
    {
      QMessageBox::information(this, tr("Welcome to madman."),
	  tr("Welcome to madman.\n\n"
	  "We sincerely hope that you're about to discover a powerful new way\n"
	  "to enjoy your music. madman is a powerful tool, and we advise you to take\n"
	  "the time to familiarize yourself with its many features--the madman website at\n"
	  "http://madman.sf.net has lots of useful hints.\n\n"
	  "First, you might want to tell madman where you keep your music."), 
	  QMessageBox::Ok);
    }

    startNewDatabase();
  }

  // set up lists -------------------------------------------------------------
  SearchSongSet.setCriterion("" );
  SearchSongSet.reevaluateCriterion();

  SearchViewManager.setSongSet(&SearchSongSet);

  SearchViewManager.setup();
  PlaylistEditor.setup();

  // player control -----------------------------------------------------------
  connect(&Preferences.Player, SIGNAL(currentSongChanged()), 
      this, SLOT(songOrStateChanged()));
  connect(&Preferences.Player, SIGNAL(stateChanged()), 
      this, SLOT(songOrStateChanged()));
  connect(&Preferences.Player, SIGNAL(stateChanged()), 
      this, SLOT(updatePlayerStatus()));

  realizeSystemTrayIconSettings();
  realizeHttpdSettings();
  songOrStateChanged();
}




tMainWindow::~tMainWindow()
{
  QPoint p = pos();
  QSize s = size();

  {
    QString temp;
    {
      QTextOStream os(&temp);
      os << p.x() << ' ' << p.y();
    }
    Settings.writeEntry("/madman/main_window_position", temp);
  }

  {
    QString temp;
    {
      QTextOStream os(&temp);
      os << s.width() << ' ' << s.height();
    }
    Settings.writeEntry("/madman/main_window_size", temp);
  }

  Preferences.save(Settings);

  SearchViewManager.saveListViewAppearance(Settings);
  PlaylistEditor.saveListViewAppearance(Settings);

  saveSplitterAppearance(splitter1, Settings, "/madman/splitters/splitter1.state");
  saveSplitterAppearance(splitter2, Settings, "/madman/splitters/splitter2.state");
  saveSplitterAppearance(splitter3, Settings, "/madman/splitters/splitter3.state");

  Settings.writeEntry("/madman/main_toolbar_shown", tbMain->isShown());
  Settings.writeEntry("/madman/media_control_toolbar_shown", tbMediaControl->isShown());
  Settings.writeEntry("/madman/rating_toolbar_shown", tbRating->isShown());

  if (HttpDaemon)
    delete HttpDaemon;

  if (SystemTrayIcon)
    delete SystemTrayIcon;
}




void tMainWindow::setDatabase(tDatabase *db)
{
  if (Database.get())
    disconnect(&Database->SongCollection, NULL, this, NULL);

  PlaylistEditor.setSongSet(NULL);
  auto_ptr<tDatabase> new_db(db);
  Database = new_db;
  SearchSongSet.setSongCollection(&Database->SongCollection);
  updateAll();

  if (Database.get())
  {
    connect(&Database->SongCollection, SIGNAL(notifySongModified(const tSong*, tSongField)),
	this, SLOT(noticeSongModified(const tSong*, tSongField)));
    connect(Database.get(), SIGNAL(notifyPlaylistTreeChanged()),
	this, SLOT(updatePlaylistTree()));
  }
}




void tMainWindow::closeEvent(QCloseEvent *event)
{
  hide();
  if (fileSaveWithResult())
  {
    Settings.writeEntry("/madman/startup_file", CurrentFilename);
    tMainWindowBase::closeEvent(event);
  }
  else
  {
    if (QMessageBox::warning(this, tr("madman"),
	  tr("Failed to save your database. Do you still want to quit?"),
	  QMessageBox::Yes, QMessageBox::No) == QMessageBox::Yes)
      tMainWindowBase::closeEvent(event);
    else
    {
      event->ignore();
      show();
    }
  }
}




void tMainWindow::updateWindowCaption()
{
  if (FilenameValid)
  {
    QFileInfo info(CurrentFilename);
    setCaption(tr("madman - %1").arg(info.fileName()));
  }
  else
    setCaption(tr("madman - <unnamed>"));
}




void tMainWindow::updateAll()
{
  updatePlaylistTree(false, false);
  songSetSelectionChanged();
  buildOverviewTree();
  updateWindowCaption();
  updateRatingButtons();
  updateTrayIconStatus();
}




void tMainWindow::realizeSystemTrayIconSettings()
{
  if (Preferences.EnableSystemTrayIcon && !SystemTrayIcon)
  {
    SystemTrayIcon = new TrayIcon(this);

    updateTrayIconStatus();

    QPopupMenu *rating_menu = new QPopupMenu(this, "tray_rating_menu");
    rate0Action->addTo(rating_menu);
    rate1Action->addTo(rating_menu);
    rate2Action->addTo(rating_menu);
    rate3Action->addTo(rating_menu);
    rate4Action->addTo(rating_menu);
    rate5Action->addTo(rating_menu);

    QPopupMenu *mainwin_menu = new QPopupMenu(this, "tray_mainwin_menu");
    minimizeWindowAction->addTo(mainwin_menu);
    restoreWindowAction->addTo(mainwin_menu);
    hideWindowAction->addTo(mainwin_menu);

    QPopupMenu *tray_menu = new QPopupMenu(this, "tray_menu");

    playAction->addTo(tray_menu);
    pauseAction->addTo(tray_menu);
    stopAction->addTo(tray_menu);

    tray_menu->insertSeparator();

    tray_menu->insertItem(tr("&Rate currently playing song"), rating_menu);

    tray_menu->insertSeparator();

    skipForwardAction->addTo(tray_menu);
    skipBackAction->addTo(tray_menu);

    tray_menu->insertSeparator();

    tray_menu->insertItem(tr("&Main window"), mainwin_menu);

    tray_menu->insertSeparator();

    fileExitAction->addTo(tray_menu);

    SystemTrayIcon->setPopup(tray_menu);

    SystemTrayIcon->show();

    connect(SystemTrayIcon, SIGNAL(clicked(const QPoint &, int)),
	  this, SLOT(trayIconClicked(const QPoint &, int)));
  }
  if (!Preferences.EnableSystemTrayIcon && SystemTrayIcon)
  {
    delete SystemTrayIcon;
    SystemTrayIcon = NULL;
  }
}




void tMainWindow::realizeHttpdSettings()
{
  if (HttpDaemon)
  {
    delete HttpDaemon;
    HttpDaemon = NULL;
  }

  if (Preferences.HttpDaemonEnabled)
  {
    try
    {
      HttpDaemon = new tHttpDaemon(Preferences.HttpDaemonPort, Preferences.HttpRestrictToLocalhost);

      addResponders(HttpDaemon, this);
    }
    catch (runtime_error &ex)
    {
      if (QMessageBox::information(this, tr("madman"),
	    tr("Could not start web server:\n%1\n"
	      "This could be because another instance of madman is running,\n"
	      "which is not a good idea. Stop now?").
	    arg(QString(ex.what())),
	    QMessageBox::Yes, QMessageBox::No) == QMessageBox::Yes)
	throw;
    }
  }
}




void tMainWindow::quitApplication()
{
  QTimer::singleShot(0, this, SLOT(close()));
}




void tMainWindow::setStatus(const QString &status)
{
  statusBar()->message(status, 10000);
}




// slots ----------------------------------------------------------------------
void tMainWindow::rescan()
{
  rescan(Database.get());
}




void tMainWindow::rescan(tDatabase *db)
{
  try
  {
    auto_ptr<tProgressDialog> progress(new tProgressDialog(this, false));
    db->SongCollection.scan(db->DirectoryList, progress.get());
  }
  catch (exception &ex)
  {
    QMessageBox::warning(this, tr("madman"),
	tr("Error while rescanning:\n%1").arg(ex.what()), QMessageBox::Ok, QMessageBox::NoButton);
  }
}




void tMainWindow::rereadTags()
{
  try
  {
    auto_ptr<tProgressDialog> progress(new tProgressDialog(this, true));
    Database->SongCollection.rereadTags(progress.get());
  }
  catch (exception &ex)
  {
    QMessageBox::warning(this, tr("madman"),
	tr("Error rereading tags:\n%1").arg(ex.what()), QMessageBox::Ok, QMessageBox::NoButton);
  }
}




void tMainWindow::startNewDatabase()
{
  FilenameValid = false;
  auto_ptr<tDatabase> new_db(new tDatabase);
  new_db->startNew();
  showPreferences(new_db.get(), 2);
  setDatabase(new_db.release());
}




void tMainWindow::fileNew()
{
  if (QMessageBox::warning(this, tr("madman"),
	tr("Are you sure you want to start over from scratch?"),
	QMessageBox::Yes, QMessageBox::No) == QMessageBox::Yes)
    startNewDatabase();
}




void tMainWindow::fileOpen()
{
  if (QMessageBox::warning(this, tr("madman"),
	tr("Save database before continuing?"),
	QMessageBox::Yes, QMessageBox::No) == QMessageBox::Yes)
  {
    if (!fileSaveWithResult())
      return;
  }

  try
  {
    QString filename = QFileDialog::getOpenFileName(QString::null,
	tr("madman Database (*.mad)"),
	this);
    if (filename.isNull())
      return;

    PlaylistEditor.setSongSet(NULL);

    auto_ptr<tDatabase> new_db(new tDatabase);

    auto_ptr<tProgressDialog> progress(new tProgressDialog(this, false));
    progress->setWhat(tr("Loading database..."));
    new_db->load(filename, progress.get());

    CurrentFilename = filename;
    FilenameValid = true;
    setDatabase(new_db.release());
  }
  catch (exception &ex)
  {
    QMessageBox::warning(this, tr("madman"),
	tr("Error opening the file.\nReason: %1").arg(string2QString(ex.what())),
	QMessageBox::Ok, 0);
  }
}




void tMainWindow::saveDatabase(const QString &name)
{
  QFileInfo file_info(name);
  QString dir_name = file_info.dirPath();
  QString trunk = file_info.baseName();
  QString ext = file_info.extension();

  if (ext.length())
    ext = "." + ext;

  if (Preferences.BackupCount)
  {
    QDir my_dir(dir_name);
    my_dir.remove((trunk+"-backup-%1").arg(Preferences.BackupCount) + ext);

    for (int dest_i = Preferences.BackupCount; dest_i > 1; dest_i--)
    {
      QString srcname =(trunk+"-backup-%1").arg(dest_i - 1) + ext;
      QString destname =(trunk+"-backup-%1").arg(dest_i) + ext;
      my_dir.rename(srcname, destname);
    }

    QString srcname = trunk + ext;
    QString destname =(trunk+"-backup-%1").arg(1) + ext;
    my_dir.rename(srcname, destname);
  }

  auto_ptr<tProgressDialog> progress(new tProgressDialog(this, false));
  progress->setWhat(tr("Saving database..."));
  Database->save(name, progress.get());
}




void tMainWindow::fileSave()
{
  fileSaveWithResult();
}





bool tMainWindow::fileSaveWithResult()
{
  if (!FilenameValid)
  {
    return fileSaveAsWithResult();
  }

  try
  {
    saveDatabase(CurrentFilename);
    return true;
  }
  catch (exception &ex)
  {
    QMessageBox::warning(this, tr("madman"),
	tr("Error saving file '%1'").arg(string2QString(ex.what())),
	QMessageBox::Ok, 0);
    return false;
  }
}




void tMainWindow::fileSaveAs()
{
  fileSaveAsWithResult();
}




bool tMainWindow::fileSaveAsWithResult()
{
  try
  {
    QString filename;
    while (true)
    {
      filename = QFileDialog::getSaveFileName(QString::null, "madman Database (*.mad)", this);
      if (filename.isNull())
	return false;

      QFileInfo info(filename);
      if (info.exists())
      {
	int result = QMessageBox::warning(this, tr("Warning"), 
	      tr("The file '%1' exists. Overwrite?").arg(filename), 
	      QMessageBox::Yes, QMessageBox::No | QMessageBox::Default,
	      QMessageBox::Cancel);
	if (result == QMessageBox::Yes)
	  break;
	if (result == QMessageBox::Cancel)
	  return false;
      }
      else
	break;
    }
    saveDatabase(filename);
    CurrentFilename = filename;
    FilenameValid = true;
    updateWindowCaption();
    return true;
  }
  catch (exception &ex)
  {
    QMessageBox::warning(this, tr("madman"),
	tr("Error saving file '%1'").arg(string2QString(ex.what())),
	QMessageBox::Ok, 0);
    return false;
  }
}





void tMainWindow::showPreferences(tDatabase *db, int tab)
{
  pair<bool,bool> ok_rescan = editPreferences(this, Preferences, db->DirectoryList, Settings, tab);
  if (ok_rescan.first)
  {
    SearchViewManager.redisplay(false);
    PlaylistEditor.redisplay(false);

    SearchViewManager.setup();
    PlaylistEditor.setup();

    realizeSystemTrayIconSettings();
    realizeHttpdSettings();

    if (ok_rescan.second)
      rescan(db);
  }
}




void tMainWindow::filePreferences()
{
  showPreferences(Database.get(), 0);
}




#include "designer/about.h"
void tMainWindow::helpAbout()
{
  tAboutDialog about;
  about.labelMain->setText(about.labelMain->text().arg(STRINGIFY(MADMAN_VERSION)));
  about.labelAbout->setText(about.labelAbout->text().arg(STRINGIFY(MADMAN_VERSION)));
  about.exec();
}




void tMainWindow::help()
{
  tHelpBrowser *hb = new tHelpBrowser(this);
  hb->browserHelp->setTextFormat(Qt::RichText);
  hb->browserHelp->setText(
      tr(
	"<h2>Table of contents</h2>"
	"1. <a href=\"#search\">Searching in madman</a><br>"
	"2. <a href=\"#faq\">Frequently Asked Questions</a><p>"
	"<a name=\"search\"><h2>Searching in madman</h2>"
	"Criteria can be combined via parentheses, <tt>|</tt> (or), <tt>&</tt> (and), "
	"<tt>!</tt> (not). "
	"Plain text will be interpreted as <tt>~any(...)</tt>. "
	"Unconnected criteria will automatically be connected by \"and\". "
	"Comments can be included if enclosed in <tt>{ ... }</tt>. "
	"<h3>Text criteria</h3>"
	"The following criteria exist:<p>"
	"<ul>"
	"<li><tt>~any(...)</tt>: selects all songs that contain ... in any field."
	"<li><tt>~artist(...)</tt>: selects all songs that contain ... in the artist field."
	"<li><tt>~performer(...)</tt>: selects all songs that contain ... in the performer field."
	"<li><tt>~album(...)</tt>: selects all songs that contain ... in the album field."
	"<li><tt>~title(...)</tt>: selects all songs that contain ... in the title field."
	"<li><tt>~genre(...)</tt>: selects all songs that contain ... in the genre field."
	"<li><tt>~filename(...)</tt>: selects all songs that contain ... in the filename field."
	"</ul>"
	"Texts mentioned above as '...' can take several match modifiers:<p>"
	"<ul>"
	"<li><tt>re:...</tt> for a regular expression"
	"<li><tt>complete:...</tt> for a complete match"
	"<li><tt>substring:...</tt> for a substring match"
	"<li><tt>fuzzy:...</tt> for a fuzzy match"
	"</ul>"
	"Substring matching is assumed if nothing is specified.<p>"

	"<h3>Numeric criteria</h3>"
	"The following criteria exist:<p>"
	"<ul>"
	"<li><tt>~year(&lt;|&gt;|&lt;=|=|&gt;=...)</tt>: selects all songs that have a a year less/greater/.. than ..."
	"<li><tt>~track_number(&lt;|&gt;|&lt;=|=|&gt;=...)</tt>: selects all songs that have a a track number less/greater/.. than ..."
	"<li><tt>~rating(&lt;|&gt;|&lt;=|=|&gt;=...)</tt>: selects all songs that have a a rating less/greater/.. than ..."
	"<li><tt>~unrated</tt>: selects songs that are as yet unrated."
	"<li><tt>~play_count(&lt;|&gt;|&lt;=|=|&gt;=...)</tt>: selects all songs that have a a play count less/greater/.. than ..."
	"<li><tt>~full_play_count(&lt;|&gt;|&lt;=|=|&gt;=...)</tt>: selects all songs that have a full play count "
	"less/greater/.. than ..."
	"<li><tt>~partial_play_count(&lt;|&gt;|&lt;=|=|&gt;=...)</tt>: selects all songs that have a partial play count "
	"less/greater/.. than ..."
	"<li><tt>~existed_for_days(&lt;|&gt;|&lt;=|=|&gt;=...)</tt>: selects all songs that have been in your collection "
	"for less/greater/.. than ... days"
	"<li><tt>~full_play_ratio(&lt;|&gt;|&lt;=|=|&gt;=...)</tt>: selects all songs that have a ratio "
	"of being played in full to the total play count less/greater/.. than ... "
	"(argument should be between 0.0 and 1.0) "
	"<li><tt>~last_played_n_days_ago(&lt;|&gt;|&lt;=|=|&gt;=...)</tt>: selects songs that were last played "
	"within a number of days less/greater/.. than ... (argument may be a non-integer)"
	"<li><tt>~uniqueid(&lt;|&gt;|&lt;=|=|&gt;=...)</tt>: selects songs that have a database key unique ID "
	"less/greater/.. than ..."
	"</ul>"
	"Equal matching is assumed if nothing is specified.<p>"

	"<h3>Criteria examples</h3>"
	"<ul>"
	"<li> <tt>man moon</tt>: Songs that contain both the words \"man\" and \"moon\" "
	  "in one of their fields."
	"<li> <tt>man|moon</tt>: Songs that contain the word \"man\" or the word \"moon\" "
	  "in one of their fields."
	"<li> <tt>\"3 doors down\"</tt>: Just like on Google, quoting ensures that "
	  "you only get songs that contain the phrase \"3 doors down\" in succession."
	"<li> <tt>~artist(complete:creed)</tt>: That's how you say that you want "
	  "Creed, but not Creedence Clearwater Revival."
	"</ul>"
	"And here are some smart ideas for playlists that you could create:"
	"<ul>"
	"<li> <b>Absolute favorites:</b> <tt>~rating(&gt;=4)</tt>"
	"<li> <b>Can't stand to listen to them:</b> <tt>~full_play_ratio(&lt;0.3)&amp;~play_count(&gt;3)</tt>"
	"<li> <b>Always finish listening to them:</b> <tt>~full_play_ratio(&gt;0.5)&amp;~play_count(&gt;3)</tt>"
	"<li> <b>Never listened:</b> <tt>~play_count(=0)</tt>"
	"<li> <b>Love songs:</b> <tt>~title(love)|~album(love)</tt>"
	"</ul>"
	"<a name=\"faq\"><h2>FAQ</h2>"
	"<b>Q. Why can't I drag the column headers like I used to?</b><p>"
	"Just hold Control while dragging."
      ));
  hb->show();
}




void tMainWindow::rateCurrentSong(int rating)
{
  tFilename song_file = Preferences.Player.currentFilename();

  if (Database.get())
  {
    tSong *new_song = Database->SongCollection.getByFilename(song_file);
    if (new_song)
    {
      if (new_song->rating() == rating)
	new_song->setRating(-1);
      else
	new_song->setRating(rating);
    }
    else
      QMessageBox::warning(this, tr("madman"),
	  tr("Currently playing song is not in database. Sorry."), QMessageBox::Ok, QMessageBox::NoButton);
  }
}




void tMainWindow::addPlaylist(tPlaylistNode *node, tPlaylistNode *parent)
{
  if (Database->playlistTree() == NULL)
    Database->setPlaylistTree(node);
  else if (parent == NULL)
    Database->playlistTree()->addChild(node);
  else
    parent->addChild(node);

  updatePlaylistTree(true, true);
}




void tMainWindow::addPlaylist()
{
  try 
  {
    tPlaylistNode *parent = currentNode();

    auto_ptr<tPlaylistNode> node(new tPlaylistNode(Database.get(), new tPlaylist()));
    node->data()->setSongCollection(&Database->SongCollection);
    bool successful = false;
    unsigned index = 1;
    while (!successful)
    {
      try 
      {
	node->setName(tr("New Playlist %1").arg(index));
	addPlaylist(node.get(), parent);
	successful = true;
      }
      catch (...)
      { }
      index++;
    }
    node.release();
  }
  catch (exception &ex)
  {
    QMessageBox::warning(this, tr("madman"),
	tr("Can't create playlist:\n%1").arg(ex.what()), QMessageBox::Ok, QMessageBox::NoButton);
  }
}




void tMainWindow::bookmarkCurrentSearch()
{
  try 
  {
    tPlaylistNode *parent = currentNode();

    auto_ptr<tPlaylistNode> node(new tPlaylistNode(Database.get(), new tPlaylist()));
    node->data()->setCriterion(editSearch->text());
    node->data()->setSongCollection(&Database->SongCollection);

    bool successful = false;
    unsigned index = 1;
    while (!successful)
    {
      try 
      {
	node->setName(tr("Bookmarked Search %1").arg(index));
	addPlaylist(node.get(), parent);
	successful = true;
      }
      catch (...)
      { }
      index++;
    }
    node.release();
  }
  catch (exception &ex)
  {
    QMessageBox::warning(this, tr("madman"),
	tr("Can't create playlist from search:\n%1").arg(ex.what()), QMessageBox::Ok, QMessageBox::NoButton);
  }
}




void tMainWindow::importPlaylist()
{
  try
  {
    tPlaylistNode *parent = currentNode();

    auto_ptr<tPlaylistNode> node(new tPlaylistNode(Database.get(), new tPlaylist()));
    node->data()->setSongCollection(&Database->SongCollection);

    QString fn = QFileDialog::getOpenFileName(QString::null,
	qApp->translate("importM3UIntoPlaylist", "Playlist (*.m3u)"),
	qApp->mainWidget());
    if (fn.isNull())
      return;

    importM3UIntoPlaylist(node->data(), fn);

    QFileInfo info(fn);
    node->setName(info.fileName());
    addPlaylist(node.get(), parent);
    node.release();
  }
  catch (exception &ex)
  {
    QMessageBox::warning(this, tr("madman"),
	tr("Can't import playlist:\n%1").arg(ex.what()), QMessageBox::Ok, QMessageBox::NoButton);
  }
}




void tMainWindow::duplicatePlaylist()
{
  tPlaylistNode *current_node = currentNode();
  if (current_node == NULL)
    return;

  try 
  {
    auto_ptr<tPlaylistNode> new_node(new tPlaylistNode(Database.get(), new tPlaylist));
    new_node->setName(tr("Copy of %1").arg(current_node->name()));
    *(new_node->data()) = *(current_node->data());
    current_node->addChild(new_node.get());
    new_node.release();

    updatePlaylistTree(true, true);
  }
  catch (exception &ex)
  {
    QMessageBox::warning(this, tr("madman"),
	tr("Can't duplicate playlist:\n%1").arg(ex.what()), QMessageBox::Ok, QMessageBox::NoButton);
  }
}




void tMainWindow::removePlaylist()
{
  if (!lstSets->currentItem())
    return ;

  tPlaylistNode *node = currentNode();
  if (node->begin() != node->end())
  {
    if (QMessageBox::warning(this, tr("Warning"), 
	  tr("This playlist has children. Do you still want to delete it?"), 
	  QMessageBox::Yes, QMessageBox::No | QMessageBox::Default) == QMessageBox::No)
      return ;
  }

  // delete from structure
  if (node->parent() == NULL)
    Database->setPlaylistTree(NULL);
  else
    node->parent()->removeChild(node);
  delete node;

  PlaylistEditor.setSongSet(NULL);

  updatePlaylistTree(false, true);
}




void tMainWindow::editMultilinePlaylistCriterion()
{
  if (!lstSets->currentItem())
    return ;
  tPlaylistNode *node = currentNode();
  QString criterion = node->data()->criterion();
  while (multilineEdit(tr("madman"), 
	tr("Criterion for dynamic playlist"), criterion, this))
  {
    try
    {
      if (node->data()->criterion() != criterion)
      {
	node->data()->setCriterion(criterion);
	node->data()->reevaluateCriterion();
	songSetSelectionChanged();
      }
      break;
    }
    catch (exception &ex)
    {
      QMessageBox::warning(this, tr("madman"),
	tr("Error in Criterion: %1").arg(string2QString(ex.what())),
	QMessageBox::Ok, 0);
    }
  }
}




void tMainWindow::slotDropPlaylistNode(const QString &node_name, tPlaylistNode *onto)
{

  tPlaylistNode *node = Database->playlistTree()->resolve(node_name);
  if (node == NULL)
    return;

  // this also catches the case when root is being dragged away
  if (node == onto || onto->hasParent(node))
    return;

  // by now, node can't be root, so it has a parent.
  node->parent()->removeChild(node);
  onto->insertChild(node, onto->begin());

  updatePlaylistTree(false, true);
}




void tMainWindow::renamePlaylist(QListViewItem *item, int col, const QString &text)
{
  try
  {
    nodeFromItem(item)->setName(text);
  }
  catch (exception &ex)
  {
    QMessageBox::warning(this, tr("madman"), string2QString(ex.what()), QMessageBox::Ok, 0);
  }
  updatePlaylistTree(true, true);
}




void tMainWindow::songSetSelectionChanged()
{
  tPlaylistNode *node = currentNode();
  PlaylistEditor.setSongSet(node ? node->data() : NULL);

  if (node && node->data())
  {
    tSongList list;
    node->data()->render(list);
    statusBar()->message(
	stringifySongListSummary(list), 5000);
  }
}




void tMainWindow::songSetCriterionChanged()
{
  tPlaylistNode *node = currentNode();

  if (node)
  {
    tPlaylist *song_set = node->data();
    try {
      QString newcrit = editPlaylistCriterion->text();
      if (song_set->criterion() != newcrit)
      {
	song_set->setCriterion(newcrit);
	song_set->reevaluateCriterion();

	{
	  tSongList list;
	  song_set->render(list);
	  statusBar()->message(
	      stringifySongListSummary(list), 5000);
	}
      }
    }
    catch (exception &ex)
    {
      QMessageBox::warning(this, tr("madman"),
	tr("Error in Criterion: %1").arg(string2QString(ex.what())),
	QMessageBox::Ok, 0);
    }
  }
}




void tMainWindow::noticeSongSetChanged()
{
  tSongSet *song_set = PlaylistEditor.songSet();
  if (song_set)
  {
    editPlaylistCriterion->setEnabled(true);
    editPlaylistCriterion->setText(song_set->criterion());
  }
  else
  {
    editPlaylistCriterion->setEnabled(false);
    editPlaylistCriterion->setText("");
  }
}




// set contents management ----------------------------------------------------
void tMainWindow::add(const tSongList &songlist)
{
  FOREACH_CONST(first, songlist, tSongList)
    PlaylistEditor.playlist()->add(*first);
}




void tMainWindow::searchChanged()
{
  try 
  {
    // Workaround for Qt 3.2
    editSearch->repaint();

    QString new_crit = editSearch->text();
    if (SearchSongSet.criterion() != new_crit)
    {
      if (CurrentSearchProgress)
	CurrentSearchProgress->cancel();

      SearchSongSet.setCriterion(new_crit);

      tCancellableStatusBarProgress progress(statusBar());
      CurrentSearchProgress = &progress;

      SearchSongSet.reevaluateCriterion(&progress);

      if (CurrentSearchProgress == &progress)
	CurrentSearchProgress = NULL;
    }

    tSongList list;
    SearchSongSet.render(list);
    statusBar()->message(
	stringifySongListSummary(list), 5000);
  }
  catch (exception &ex)
  {
    statusBar()->message(tr("Error in expression: %1.").arg(string2QString(ex.what())), 2000);
  }
}




void tMainWindow::buildOverviewTree()
{
  lstOverview->clear();

  new tGenreOverviewItem(*Database, lstOverview, NULL, tr("By Genre"), QString::null);
  new tAlbumOverviewItem(*Database, lstOverview, NULL, tr("By Album"), QString::null);
  new tArtistsOverviewItem(*Database, lstOverview, NULL, tr("By Artist"), QString::null);

  new tOverviewItem(lstOverview, NULL, tr("All"), "");
}




void tMainWindow::overviewSelectionChanged(QListViewItem *item)
{
  tOverviewItem *ov_item = dynamic_cast< tOverviewItem * >(item);
  if (ov_item->criterion() != QString::null)
    editSearch->setText(ov_item->criterion());
}




void tMainWindow::highlightCurrentSong()
{
  if (SearchSongSet.criterion() != "")
    editSearch->setText("");

  lstAllSongs->setFocus();
  SearchViewManager.highlightCurrentSong();
}




void tMainWindow::play()
{
  Preferences.Player.play();
}
void tMainWindow::pause()
{
  if (Preferences.Player.isPaused())
    Preferences.Player.play();
  else
    Preferences.Player.pause();
}
void tMainWindow::stop()
{
  Preferences.Player.stop();
}
void tMainWindow::skipForward()
{
  Preferences.Player.skipForward();
}
void tMainWindow::skipBack()
{
  Preferences.Player.skipBack();
}
void tMainWindow::skipTo(int value)
{
  Preferences.Player.skipTo(float(value) / 100);
}
void tMainWindow::skipToWrapper()
{
  skipTo(sbarSongPosition->value() );
}




void tMainWindow::updatePlayerStatus()
{
  float current_time = Preferences.Player.currentTime();
  float total_time = Preferences.Player.totalTime();
  if (total_time == 0)
    sbarSongPosition->setValue (0);
  else
    sbarSongPosition->setValue (int(10000 * current_time / total_time));

  QString current,total_duration;
  current.sprintf("%d:%02d", int(current_time) / 60, int (current_time) % 60);
  total_duration.sprintf("%d:%02d", int(total_time) / 60, int (total_time) % 60);

  labelSongPosition->setText(QString("%1 / %2").arg(current).arg(total_duration));
}




void tMainWindow::songOrStateChanged()
{
  // detect state changes
  time_t now = time(NULL);

  if (WasPlaying)
    AccumulatedPlayTime += now - PlayStartTime;

  WasPlaying = Preferences.Player.isPlaying() && !Preferences.Player.isPaused();
  PlayStartTime = now;

  pauseAction->setOn(Preferences.Player.isPaused() && Preferences.Player.isPlaying());

  // detect song changes
  tFilename song_file = Preferences.Player.currentFilename();

  if (song_file != CurrentSongFilename)
  {
    if (Database.get())
    {
      if (CurrentSongFilename != "" && Preferences.CollectHistory)
      {
	tSong *old_song = Database->SongCollection.getByFilename(CurrentSongFilename);
	if (old_song)
	{
	  old_song->played(time(NULL), AccumulatedPlayTime > 0.6 * old_song->duration());
	  Database->History.played(old_song->uniqueId(), 
	      time(NULL), AccumulatedPlayTime);
	}
      }

      if (Preferences.EnablePassivePopupSongAnnouncements)
      {
	tSong *new_song = Database->SongCollection.getByFilename(song_file);
	if (new_song)
	{
	  new tPassivePopup(substituteSongFields(Preferences.PassivePopupFormat, new_song, true), 3000);
	}
      }
    }

    updateTrayIconStatus();
    updateRatingButtons();

    CurrentSongFilename = song_file;
    AccumulatedPlayTime = 0;
  }
}




void tMainWindow::trayIconClicked(const QPoint &where, int button)
{
  if (isShown())
    hide();
  else
    showNormal();
}




// helpers --------------------------------------------------------------------
void tMainWindow::updatePlaylistTree()
{
  updatePlaylistTree(false, false);
}




void tMainWindow::updatePlaylistTree(bool keep_selection, bool keep_scroll)
{
  int contents_x = lstSets->contentsX();
  int contents_y = lstSets->contentsY();
  tPlaylistNodeListViewItem *selected_item = NULL;
  if (keep_selection)
    selected_item = dynamic_cast<tPlaylistNodeListViewItem *>(lstSets->currentItem());
  tPlaylistNode *selected_node = NULL;
  if (selected_item)
    selected_node = selected_item->node();

  lstSets->clear();
  if (Database->playlistTree())
    lstSets->insertItem(
	createPlaylistNodeItem(Database->playlistTree(), NULL, 
	  selected_node, selected_item));

  if (keep_scroll)
    lstSets->setContentsPos(contents_x, contents_y);
  if (keep_selection && selected_item)
    lstSets->setSelected(selected_item, true);
}




tPlaylistNode *tMainWindow::nodeFromItem(QListViewItem *item)
{
  tPlaylistNodeListViewItem *cast_item = dynamic_cast<tPlaylistNodeListViewItem *>(item);
  if (cast_item)
    return cast_item->node();
  else
    return NULL;
}





tPlaylistNode *tMainWindow::currentNode()
{
  QListViewItem *current_item = lstSets->currentItem();
  if (current_item == NULL)
    return NULL;
  else
    return nodeFromItem(current_item);
}




tPlaylist *tMainWindow::currentPlaylist()
{
  tPlaylistNode * node = currentNode();
  if (node)
    return node->data();
  else
    return NULL;
}




void tMainWindow::insertAtPosition(QListViewItem *parent, QListViewItem *item, tIndex pos)
{
  parent->insertItem(item);
  if (pos != 0)
  {
    QListViewItem * after = item;
    while (pos--)
      after = after->nextSibling();
    item->moveItem(after);
  }
}




QListViewItem *tMainWindow::createPlaylistNodeItem(tPlaylistNode *node, QListViewItem *parent,
    tPlaylistNode *selected_node, tPlaylistNodeListViewItem *&selected_item)
{
  tPlaylistNodeListViewItem *result;
  if (parent)
    result = new tPlaylistNodeListViewItem(parent, node);
  else
    result = new tPlaylistNodeListViewItem(lstSets, node);

  connect(result, SIGNAL(dropNode(const QString &,tPlaylistNode *)),
      this, SLOT(slotDropPlaylistNode(const QString &,tPlaylistNode *)));

  if (node == selected_node)
    selected_item = result;

  tPlaylistNode::iterator first = node->begin(), last = node->end();
  if (first != last)
    do
    {
      last--;
      createPlaylistNodeItem(*last, result, selected_node, selected_item);
    }
    while (first != last);

  lstSets->setOpen(result, true);
  return result;
}




void tMainWindow::loadSplitterAppearance(QSplitter *splitter, QSettings &settings, QString const &where)
{
  QString representation;
  representation = settings.readEntry(where);
  {
    QTextStream stream(&representation, IO_ReadOnly);
    stream >> *splitter;
  }
}




void tMainWindow::saveSplitterAppearance(QSplitter *splitter, QSettings &settings, QString const &where)
{
  QString representation;
  {
    QTextStream stream(&representation, IO_WriteOnly);
    stream << *splitter;
  }
  settings.writeEntry(where, representation);
}




void tMainWindow::showEvent(QShowEvent *e)
{
  tMainWindowBase::showEvent(e);
}




void tMainWindow::hideEvent(QHideEvent *e)
{
  tMainWindowBase::hideEvent(e);
  if (isMinimized() && Preferences.EnableSystemTrayIcon && Preferences.MinimizeToSystemTray)
    hide();
}




void tMainWindow::updateTrayIconStatus()
{
  if (!SystemTrayIcon)
    return;

  // update tooltip -----------------------------------------------------------
  tSong *new_song;

  if (Database.get() &&
      (new_song = Database->SongCollection.getByFilename(
      Preferences.Player.currentFilename())) && new_song)
  {
    QString rating_string;
    if (new_song->rating() == 0)
    {
      rating_string = tr(" (not rated)");
    }
    else if (new_song->rating() > 0)
    {
      rating_string.fill('*', new_song->rating());
      rating_string = tr(" (rated %1)").arg(rating_string);
    }

    SystemTrayIcon->setToolTip(substituteSongFields(Preferences.TrayTooltipFormat, new_song, true));
  }
  else
    SystemTrayIcon->setToolTip(tr("madman"));

  // update icon --------------------------------------------------------------
  QImage tray_img = icon()->convertToImage();
  tray_img = tray_img.smoothScale(16, 16);

  QPixmap tray_pixmap;
  tray_pixmap.convertFromImage(tray_img);

  SystemTrayIcon->setIcon(tray_pixmap);
}




void tMainWindow::updateRatingButtons()
{
  int current_rating = -1;

  tSong *new_song;
  if (Database.get() &&
      (new_song = Database->SongCollection.getByFilename(
      Preferences.Player.currentFilename())) && new_song)
    current_rating = new_song->rating();

  int const action_count = 6;
  QAction *actions[] = {
    rate0Action,
    rate1Action,
    rate2Action,
    rate3Action,
    rate4Action,
    rate5Action
  };
  
  for (int i = 0; i < action_count; i++)
    actions[ i ]->setOn(false);

  if (current_rating >= 0 && current_rating < action_count)
    actions[ current_rating ]->setOn(true);
}




void tMainWindow::noticeSongModified(const tSong *song, tSongField field)
{
  if (field == FIELD_RATING)
    updateRatingButtons();
}
