#include "../config.h"
#include <sys/types.h>
#include <stdlib.h>
#include <string.h>
#include "libnjb.h"
#include "njb_error.h"
#include "defs.h"
#include "base.h"
#include "byteorder.h"

extern int __sub_depth;

// Routine to unpack EAX block from NJB1 series device
eax_t *eax_unpack(void *data, size_t nbytes)
{
	__dsub= "eax_unpack";
	eax_t *eax; 
	unsigned char *dp= (unsigned char *) data;
	int i= 0;

	__enter;

	eax= malloc(sizeof(eax_t));
	if ( eax == NULL ) {
		NJB_ERROR(EO_NOMEM);
		__leave;
		return NULL;
	}

	memset(eax, 0, sizeof(eax_t));

	eax->volume= (u_int8_t) *dp++;
	eax->volumemin = 0;
	eax->volumemax = 99;
	eax->muted= (u_int8_t) *dp++;
	eax->eq_status= (u_int8_t) *dp++;

	eax->bass= (int8_t) *dp++;
	eax->bassmin= (int8_t) *dp++;
	eax->bassmax= (int8_t) *dp++;

	eax->midrange= (int8_t) *dp++;
	eax->midrangemin= (int8_t) *dp++;
	eax->midrangemax= (int8_t) *dp++;

	eax->treble= (int8_t) *dp++;
	eax->treblemin= (int8_t) *dp++;
	eax->treblemax= (int8_t) *dp++;

	eax->nfreq= (u_int8_t) *dp++;
	eax->freqset= (u_int8_t) *dp++;
	eax->frequencies= malloc(eax->nfreq*2);
	if ( eax->frequencies == NULL ) {
		NJB_ERROR(EO_NOMEM);
		eax_destroy(eax);
		__leave;
		return NULL;
	}

	for (i= 0; i< eax->nfreq; i++) {
		eax->frequencies[i] = njb1_bytes_to_16bit(&dp[0]);
		dp+= 2;
	}

	eax->neffects= (u_int8_t) *dp++;
	eax->acteffect= (u_int8_t) *dp++;
	eax->effects= (char **) malloc(eax->neffects * sizeof(char *));
	if ( eax->effects == NULL ) {
		NJB_ERROR(EO_NOMEM);
		eax_destroy(eax);
		__leave;
		return NULL;
	}
	memset(eax->effects, 0, eax->neffects * sizeof(char *));

	eax->effamts= (u_int8_t *) malloc(eax->neffects);
	if ( eax->effamts == NULL ) {
		NJB_ERROR(EO_NOMEM);
		eax_destroy(eax);
		__leave;
		return NULL;
	}
	memset(eax->effamts, 0, eax->neffects);

	for (i= 0; i<eax->neffects; i++) {
		u_int8_t efflen= (u_int8_t) *dp++;

		eax->effects[i]= (char *) malloc(efflen+1);
		if ( eax->effects[i] == NULL ) {
			NJB_ERROR(EO_NOMEM);
			eax_destroy(eax);
			__leave;
			return NULL;
		}

		memcpy(eax->effects[i], dp, efflen);
		eax->effects[i][efflen]= '\0';
		dp+= efflen;

		eax->effamts[i]= (u_int8_t) *dp++;
	}

	eax->nphone= (u_int8_t) *dp++;
	eax->actphone= (u_int8_t) *dp++;
	eax->phonemodes= (char **) malloc (eax->nphone * sizeof(char *));
	if ( eax->phonemodes == NULL ) {
		NJB_ERROR(EO_NOMEM);
		eax_destroy(eax);
		__leave;
		return NULL;
	}

	memset(eax->phonemodes, 0, eax->nphone * sizeof(char *));
	for (i= 0; i< eax->nphone; i++) {
		u_int8_t phonelen= (u_int8_t) *dp++;
		eax->phonemodes[i]= (char *) malloc(phonelen+1);
		if ( eax->phonemodes[i] == NULL ) {
			NJB_ERROR(EO_NOMEM);
			eax_destroy(eax);
			__leave;
			return NULL;
		}

		memcpy(eax->phonemodes[i], dp, phonelen);
		eax->phonemodes[i][phonelen]= '\0';
		dp+= phonelen;
	}

	eax->nrear= (u_int8_t) *dp++;
	eax->actrear= (u_int8_t) *dp++;
	eax->rears= (char **) malloc(eax->nrear * sizeof(char *));
	if ( eax->rears == NULL ) {
		NJB_ERROR(EO_NOMEM);
		eax_destroy(eax);
		__leave;
		return NULL;
	}

	memset(eax->rears, 0, eax->nrear * sizeof(char *));
	for (i= 0; i< eax->nrear; i++) {
		u_int8_t rearlen= (u_int8_t) *dp++;
		eax->rears[i]= (char *) malloc(rearlen+1);
		if ( eax->rears[i] == NULL ) {
			NJB_ERROR(EO_NOMEM);
			eax_destroy(eax);
			__leave;
			return NULL;
		}

		memcpy(eax->rears[i], dp, rearlen);
		eax->rears[i][rearlen]= '\0';
		dp+= rearlen;
	}

	__leave;
	return eax;
}

void eax_destroy(eax_t * eax)
{
	/* Free the memory used by the EAX structure 
	 * extra care taken for NULL checking since
	 * it is used in case an error occurs in
	 * the parsing in protocol.c */
	u_int8_t i;

	if (eax != NULL) {
		if (eax->rears != NULL) {
			for (i = 0; i < eax->nrear; i++) {
				if (eax->rears[i] != NULL)
					free(eax->rears[i]);
			}
			free(eax->rears);
		}
		if (eax->phonemodes != NULL) {
			for (i = 0; i < eax->nphone; i++) {
				if (eax->phonemodes[i] != NULL)
					free(eax->phonemodes[i]);
			}
			free(eax->phonemodes);
		}
		if (eax->effects != NULL) {
			for (i = 0; i < eax->neffects; i++) {
				if (eax->effects[i] != NULL)
					free(eax->effects[i]);
			}
			free(eax->effects);
		}
		if (eax->effamts != NULL)
			free(eax->effamts);
		if (eax->frequencies != NULL)
			free(eax->frequencies);
		free(eax);
	}
}

// Routine to refresh EAX for an NJB1 series device
void eax_refresh(eax_t *eax, unsigned char *data, size_t numbytes)
{
	/*
	 * The NJB1 has a special "refresh" command that is perhaps better
	 * left unused, but here it is anyway.
	 */
	char *dp= (char *) data;
	int i;

	if (numbytes != 15)
		return;

	dp += 5;
	eax->volume= (u_int8_t) *dp++;
	eax->muted= (u_int8_t) *dp++;
	eax->eq_status= (u_int8_t) *dp++;
	eax->bass= (int8_t) *dp++;
	eax->midrange= (int8_t) *dp++;
	eax->treble= (int8_t) *dp++;
	eax->freqset= (u_int8_t) *dp++;
	eax->acteffect= (u_int8_t) *dp++;
	for (i= 0; i<eax->neffects; i++) {
		eax->effamts[i]= (u_int8_t) *dp;
	}
	dp++;
	eax->actphone= (u_int8_t) *dp++;
	eax->actrear= (u_int8_t) *dp;
}

// EAX unpacking routine for an NJB3 series device
eax_t *eax_unpack3(void *data)
{
  __dsub = "eax_unpack3";
  eax_t *eax; 
  // unsigned char *dp = (unsigned char *) data;
  // int i = 0;
  
  __enter;
  
  eax= malloc(sizeof(eax_t));
  if ( eax == NULL ) {
    NJB_ERROR(EO_NOMEM);
    __leave;
    return NULL;
  }
  
  memset(eax, 0, sizeof(eax_t));
  
  eax->volume = 0;
  eax->volumemin = 0;
  eax->volumemax = 0;

  eax->muted = 0;
  eax->eq_status = 0;
  
  eax->bass = 0;
  eax->bassmin = 0;
  eax->bassmax = 0;
  
  eax->midrange = 0;
  eax->midrangemin = 0;
  eax->midrangemax = 0;
  
  eax->treble = 0;
  eax->treblemin = 0;
  eax->treblemax = 0;
  
  eax->nfreq = 0;
  eax->freqset = 0;
  eax->frequencies = malloc(eax->nfreq*2);
  
  eax->neffects = 0;
  eax->acteffect = 0;
  eax->effects = (char **) malloc(eax->neffects * sizeof(char *));
  memset(eax->effects, 0, eax->neffects * sizeof(char *));
  eax->effamts = (u_int8_t *) malloc(eax->neffects);
  memset(eax->effamts, 0, eax->neffects);
  
  eax->nphone = 0;
  eax->actphone = 0;
  eax->phonemodes = (char **) malloc (eax->nphone * sizeof(char *));
  memset(eax->phonemodes, 0, eax->nphone * sizeof(char *));
  
  eax->nrear = 0;
  eax->actrear = 0;
  eax->rears = (char **) malloc(eax->nrear * sizeof(char *));
  memset(eax->rears, 0, eax->nrear * sizeof(char *));

  __leave;
  return eax;
}
