// -*- C++ -*-
/* This file is part of
 * ======================================================
 *
 *           LyX, The Document Processor
 *           Copyright 1995 Matthias Ettrich
 *
 *           This file is Copyleft 1996
 *           Lars Gullik Bjnnes
 *
 * ====================================================== */

#ifndef BUFFER_H
#define BUFFER_H

#ifdef __GNUG__
#pragma interface
#endif

#include "LString.h"
#include "undo.h"
#include "support/limited_stack.h"

#include "lyxvc.h"
#include "bufferparams.h"
#include "texrow.h"
#include "ParagraphList.h"
#include "paragraph.h"

#include <boost/shared_ptr.hpp>

class BufferView;
class LyXRC;
class TeXErrors;
class LaTeXFeatures;
class Language;
class ParIterator;
class ParConstIterator;


///
struct DEPCLEAN {
	///
	bool clean;
	///
	string master;
	///
	DEPCLEAN * next;
};

/** The buffer object.
  This is the buffer object. It contains all the informations about
  a document loaded into LyX. I am not sure if the class is complete or
  minimal, probably not.
  \author Lars Gullik Bjnnes
  */
class Buffer {
public:
	/// What type of log will \c getLogName() return?
	enum LogType {
		latexlog, ///< LaTeX log
		buildlog  ///< Literate build log
	};

	/** Constructor
	    \param file
	    \param b  optional \c false by default
	*/
	explicit Buffer(string const & file, bool b = false);

	/// Destructor
	~Buffer();

	/** High-level interface to buffer functionality.
	    This function parses a command string and executes it
	*/
	bool dispatch(string const & command, bool * result = 0);

	/// Maybe we know the function already by number...
	bool dispatch(int ac, string const & argument, bool * result = 0);

	///
	void resizeInsets(BufferView *);

	/// Update window titles of all users.
	void updateTitles() const;

	/// Reset autosave timers for all users.
	void resetAutosaveTimers() const;

	/** Adds the BufferView to the users list.
	    Later this func will insert the \c BufferView into a real list,
	    not just setting a pointer.
	*/
	void addUser(BufferView * u);

	/** Removes the #BufferView# from the users list.
	    Since we only can have one at the moment, we just reset it.
	*/
	void delUser(BufferView *);

	///
	void redraw();

	/// Load the autosaved file.
	void loadAutoSaveFile();

	/** Reads a file.
	    \param par if != 0 insert the file.
	    \return \c false if method fails.
	*/
	bool readFile(LyXLex &, string const &, Paragraph * par = 0);

	/** Reads a file without header.
	    \param par if != 0 insert the file.
	    \return \c false if file is not completely read.
	*/
	bool readLyXformat2(LyXLex &, Paragraph * par = 0);

	/// This parses a single LyXformat-Token.
	bool parseSingleLyXformat2Token(LyXLex &, Paragraph *& par,
					Paragraph *& return_par,
					string const & token, int & pos,
					Paragraph::depth_type & depth,
					LyXFont &);
	///
	void insertStringAsLines(Paragraph *&, lyx::pos_type &,
				 LyXFont const &, string const &) const;
	///
	Paragraph * getParFromID(int id) const;
private:
	/// Parse a single inset.
	void readInset(LyXLex &, Paragraph *& par, int & pos, LyXFont &);
public:
	/** Save file.
	    Takes care of auto-save files and backup file if requested.
	    Returns \c true if the save is successful, \c false otherwise.
	*/
	bool save() const;

	/// Write file. Returns \c false if unsuccesful.
	bool writeFile(string const &) const;

	///
	void writeFileAscii(string const & , int);
	///
	void writeFileAscii(std::ostream &, int);
	///
	string const asciiParagraph(Paragraph const &, unsigned int linelen,
				    bool noparbreak = false) const;
	/// Just a wrapper for the method below, first creating the ofstream.
	void makeLaTeXFile(string const & filename,
			   string const & original_path,
			   bool nice,
			   bool only_body = false,
			   bool only_preamble = false);
	///
	void makeLaTeXFile(std::ostream & os,
			   string const & original_path,
			   bool nice,
			   bool only_body = false,
			   bool only_preamble = false);
	/** LaTeX all paragraphs from par to endpar.
	    \param \a endpar if == 0 then to the end
	*/
	void latexParagraphs(std::ostream & os, Paragraph * par,
			     Paragraph * endpar, TexRow & texrow, bool moving_arg = false) const;
	///
	void simpleDocBookOnePar(std::ostream &,
				 Paragraph * par, int & desc_on,
				 Paragraph::depth_type depth) const ;
	///
	void simpleLinuxDocOnePar(std::ostream & os, Paragraph * par,
				  Paragraph::depth_type depth);
	///
	void makeLinuxDocFile(string const & filename,
			      bool nice, bool only_body = false);
	///
	void makeDocBookFile(string const & filename,
			     bool nice, bool only_body = false);
	///
	void sgmlError(Paragraph * par, int pos, string const & message) const;

	/// returns the main language for the buffer (document)
	Language const * getLanguage() const;
	///
	int runChktex();
	/// return true if the main lyx file does not need saving
	bool isClean() const;
	///
	bool isBakClean() const;
	///
	bool isDepClean(string const & name) const;

	/// mark the main lyx file as not needing saving
	void markClean() const;

	///
	void markBakClean();

	///
	void markDepClean(string const & name);

	///
	void setUnnamed(bool flag = true);

	///
	bool isUnnamed();

	/// Mark this buffer as dirty.
	void markDirty();

	/// Returns the buffer's filename. It is always an absolute path.
	string const & fileName() const;

	/// Returns the the path where the buffer lives.
	/// It is always an absolute path.
	string const & filePath() const;

	/** A transformed version of the file name, adequate for LaTeX.
	    \param no_path optional if \c true then the path is stripped.
	*/
	string const getLatexName(bool no_path = true) const;

	/// Get the name and type of the log.
	std::pair<LogType, string> const getLogName() const;

	/// Change name of buffer. Updates "read-only" flag.
	void setFileName(string const & newfile);

	/// Name of the document's parent
	void setParentName(string const &);

	/// Is buffer read-only?
	bool isReadonly() const;

	/// Set buffer read-only flag
	void setReadonly(bool flag = true);

	/// returns \c true if the buffer contains a LaTeX document
	bool isLatex() const;
	/// returns \c true if the buffer contains a LinuxDoc document
	bool isLinuxDoc() const;
	/// returns \c true if the buffer contains a DocBook document
	bool isDocBook() const;
	/** returns \c true if the buffer contains either a LinuxDoc
	    or DocBook document */
	bool isSGML() const;
	/// returns \c true if the buffer contains a Wed document
	bool isLiterate() const;

	/** Validate a buffer for LaTeX.
	    This validates the buffer, and returns a struct for use by
	    #makeLaTeX# and others. Its main use is to figure out what
	    commands and packages need to be included in the LaTeX file.
	    It (should) also check that the needed constructs are there
	    (i.e. that the \refs points to coresponding \labels). It
	    should perhaps inset "error" insets to help the user correct
	    obvious mistakes.
	*/
	void validate(LaTeXFeatures &) const;

	///
	std::vector<std::pair<string, string> > const getBibkeyList() const;
	///
	std::vector<string> const getLabelList() const;

	/** This will clearly have to change later. Later we can have more
	    than one user per buffer. */
	BufferView * getUser() const;

	///
	void changeLanguage(Language const * from, Language const * to);
	///
	bool isMultiLingual();

	/// Does this mean that this is buffer local?
	limited_stack<boost::shared_ptr<Undo> > undostack;

	/// Does this mean that this is buffer local?
	limited_stack<boost::shared_ptr<Undo> > redostack;

	///
	BufferParams params;

	/** The list of paragraphs.
	    This is a linked list of paragraph, this list holds the
	    whole contents of the document.
	 */
	ParagraphList paragraphs;

	/// LyX version control object.
	LyXVC lyxvc;

	/// Where to put temporary files.
	string tmppath;

	/** If we are writing a nice LaTeX file or not.
	    While writing as LaTeX, tells whether we are
	    doing a 'nice' LaTeX file */
	bool niceFile;

	/// Used when typesetting to place errorboxes.
	TexRow texrow;
private:
	/// is save needed
	mutable bool lyx_clean;

	/// is autosave needed
	mutable bool bak_clean;

	/// is this a unnamed file (New...)
	bool unnamed;

	/// is regenerating #.tex# necessary
	DEPCLEAN * dep_clean;

	/// buffer is r/o
	bool read_only;

	/// name of the file the buffer is associated with.
	string filename_;

	/// The path to the document file.
	string filepath_;

	/// Format number of buffer
	int file_format;
	/** A list of views using this buffer.
	    Why not keep a list of the BufferViews that use this buffer?

	    At least then we don't have to do a lot of magic like:
	    #buffer->lyx_gui->bufferview->updateLayoutChoice#. Just ask each
	    of the buffers in the list of users to do a #updateLayoutChoice#.
	*/
	BufferView * users;
public:
	///
	class inset_iterator {
	public:
		typedef std::input_iterator_tag iterator_category;
		typedef Inset value_type;
		typedef ptrdiff_t difference_type;
		typedef Inset * pointer;
		typedef Inset & reference;
		typedef ParagraphList::iterator base_type;

		///
		inset_iterator();
		///
		inset_iterator(base_type p, base_type e);
		///
		inset_iterator(base_type p, lyx::pos_type pos, base_type e);

		/// prefix ++
		inset_iterator & operator++();
		/// postfix ++
		inset_iterator operator++(int);
		///
		reference operator*();
		///
		pointer operator->();

		///
		Paragraph * getPar();
		///
		lyx::pos_type getPos() const;
		///
		friend
		bool operator==(inset_iterator const & iter1,
				inset_iterator const & iter2);
	private:
		///
		void setParagraph();
		///
		ParagraphList::iterator pit;
		///
		ParagraphList::iterator pend;
		///
		InsetList::iterator it;
	};

	///
	inset_iterator inset_iterator_begin() {
		return inset_iterator(paragraphs.begin(), paragraphs.end());
	}

	///
	inset_iterator inset_iterator_end() {
		return inset_iterator();
	}

	///
	inset_iterator inset_const_iterator_begin() const {
		return inset_iterator(paragraphs.begin(), paragraphs.end());
	}

	///
	inset_iterator inset_const_iterator_end() const {
		return inset_iterator();
	}

	///
	ParIterator par_iterator_begin();
	///
	ParConstIterator par_iterator_begin() const;
	///
	ParIterator par_iterator_end();
	///
	ParConstIterator par_iterator_end() const;

	///
	Inset * getInsetFromID(int id_arg) const;
};

bool operator==(Buffer::inset_iterator const & iter1,
		Buffer::inset_iterator const & iter2);

bool operator!=(Buffer::inset_iterator const & iter1,
		Buffer::inset_iterator const & iter2);
#endif
