/*
 *  client/cahce.c
 *		The caching done by the client.
 *
 *
 *  Copyright (C) 1990	Lysator Computer Club,
 *			Linkoping University,  Sweden
 *
 *  Everyone is granted permission to copy, modify and redistribute
 *  this code, provided the people they give it to can.
 *
 *
 *  Author:	Thomas Bellman
 *		Lysator Computer Club
 *		Linkoping University
 *		Sweden
 *
 *  email:	Bellman@Lysator.LiU.SE
 */


#ifndef  CACHE_H_ALREADY_INCLUDED__
#define  CACHE_H_ALREADY_INCLUDED__


#include <kom-types.h>
#include <kom-errno.h>


/*  ==================================================================  */
/*		    Globally accessible variables			*/


extern Pers_no	  cpn;	/* Current Person Number */
extern Conf_no	  ccn;	/* Current Conference Number */



/*  ==================================================================  */
/*	Functions to get various data from the server, possibly		*/
/*	through the cache.  All data is allocated using zmalloc(),	*/
/*	and is zuse():ed for the caller.				*/

/*	All data is valid until explicitly zfree():ed by the caller,	*/
/*	but the caller may not change any data returned by the cache.	*/


/*
 *  Initialize the cache.  Must be called at the start of the
 *  program.
 */
extern  void
init_cache (void);


/* Now it is safe to handle the asynchronous calls 
 * For now, just before the prompt is printed.
 */
extern void
handle_all_asyncs(void);

/*
 *  Tell the cache that we have changed person
 */
extern  void
cache_change_person (Pers_no	pers);

/*
 *  Indicate for the cache that the user has switched the current conference
 */
extern  Conf_no
cache_change_conf (Conf_no	  new_conf,
		   Kom_err	* kom_errno_arg	   );

/*
 *  Get the name of a conference.  Returns EMPTY_STRING if
 *  it can't find the name.
 */
extern  String
conf_name (Conf_no	  conf,
	   Kom_err	* kom_errno_arg);

/*
 *  Get the last local_no of a conference.  Returns -1 if
 *  it can't find the conf.
 */
extern  int
conf_last (Conf_no	  conf,
	   Kom_err	* error);

/*
 *  Return a pointer to a Conference struct for the requested
 *  conference.
 *  A NULL pointer is returned if no data about the conference
 *  can be obtained.
 */
extern  Conference *
conf_stat (Conf_no	  conf,
	   Kom_err	* kom_errno_arg);

/*
 *  Get the list of members for a conference.  Returns NULL if none
 *  can be found.
 */
extern  Member_list *
member_list (Conf_no	  conf,
	     Kom_err	* error);



/*
 *  Return a pointer to a Person struct for the requested
 *  person.
 *  A NULL pointer is returned if no data about the person
 *  can be obtained.
 */
extern  Person *
pers_stat (Pers_no	  pers,
	   Kom_err	* kom_errno_arg);

/*
 *  Return pointer to Text_stat struct for the requested text.
 *  A NULL pointer is returned if no data about the text can be
 *  obtained.
 *  Note: no real caching is done, since the text stats tend to
 *  change rather often.
 */
extern  Text_stat *
text_stat (Text_no	  text);

/*
 *  Return a 'String' object containing the text of the requested
 *  text, or EMPTY_STRING if the text can't be obtained.
 */
extern  String
get_text (Text_no	  text,
	  String_size	  start_char,
	  String_size	  end_char,
	  Kom_err	* kom_errno_arg	);

/*
 *  In the conference with the number CONF, find the global
 *  Text_no of a text with the local number LOCAL_NO, or 0 if it
 *  can't find it, in which case it sets *ERRNO to the error, or
 *  KOM_NO_ERROR if the cause for the 0 was that the text had
 *  been GC:ed, deleted or removed from the conference.
 */
extern  Text_no
map_local_to_global (Conf_no		  conf,
		     Local_text_no	  local_no,
		     Kom_err		* kom_errno_arg	   );


/*
 *  Find the data about person PERS:s membership in CONF and put the
 *  data in *RESULT if RESULT is non-NULL.  Returns the position in
 *  the (zero-originated) list of memberships, or -1 if not found.
 *  The result does not contain a list of read texts.
 *  //////////////////////////////////////////////////
 *  Should perhaps return an error number too...
 */
extern  int
locate_membership (Conf_no	  conf,
		   Pers_no	  pers,
		   Membership	* result);


/*  Find the data about the first membership */
extern	int
first_membership (Pers_no	  pers,
		  Membership	* result);

/*  Find the data about the next membership */
extern	int
next_membership (Membership	* result);



/* Mark a single text as read in the cache and sends the message on the 
 * server. The texts is references by the GLOBAL_TEXT_NO. The conf BUTCONF
 * is skipped.
 */
extern void
mark_global_as_read(Text_no	  global,
		    Conf_no	  butconf);


/* Mark a list of texts as read in the cache but does not send anything to
 * the server or store it anywere.
 */
extern Success
mark_as_read_locally(Conf_no			conf_no,
		     int			len,
		     const Local_text_no      * text_arr);

/* Mark a list of texts as read in the cache and sends the message on to the
 * server.
 * Arguments as the kom_mark_as_read.
 */
extern Success
mark_as_read (Conf_no			conf_no,
	      int			len,
	      const Local_text_no     * text_arr    );

/* Return the number of conferences with unread. */
extern int
get_unread_confs_count(void);


#endif  /* CACHE_H_ALREADY_INCLUDED__ */
