package Lire::ImportJob;

use strict;

use Carp;
use POSIX qw/ strftime /;

use Lire::Error qw/ file_not_readable /;
use Lire::I18N qw/ set_fh_encoding /;
use Lire::Utils qw/ check_param check_object_param /;
use Lire::DlfConverterProcess;

=pod

=head1 NAME

Lire::ImportJob - Object used to represent an periodical source of log data

=head1 SYNOPSIS

  use Lire::ImportJob;

  my $src = new Lire::ImportJob( "name",
                                 'pattern'    => "/var/log/messsages",
                                 'period'     => "unique",
                                 'processor'  => $name,
                                );

  my $file = $src->file,

=head1 DESCRIPTION

The Lire::ImportJob object is used to represent a source of log data.
ImportJobs can also be instantiated from configuration data (The
'import_job' configuration specification.)

=head1 new( $name, ... );

Create a new Lire::ImportJob object. As mandatory parameter, it takes
the name of the log source.

Additional parameters:

=over

=item pattern

The path to the file that contains the log data. This filename can
contain strftime(3) patterns that will be substituted to obtain the
name of the file. (Useful for automatic rotation).

=item period

The periodicity to which this ImportJob should be processed. Defaults
to 'unique' which means this is a one-shot ImportJob. Other
recognized values are 'hourly', 'daily', 'weekly', 'monthly' and 'yearly'.

=item converter

The name of the converter that can be used to process the ImportJob.
This must be known to the Lire::PluginManager.

=item converter_config

Configuration parameter for the DlfConverter.

=item filter

A shell command that will be used to filter the log file.

=item encoding

Encoding for the log file. If omitted, the default system encoding
will be used.

=back

=cut

sub new {
    my ($pkg, $name, %args) = @_;

    check_param( $name, 'name' );

    $args{'period'} ||= "unique";

    my $self = bless { '_name' => $name,
                       '_converter' => undef,
                       '_converer_config' => undef,
                       '_pattern' => undef,
                       '_period' => undef,
                       '_filter' => undef,
                       '_encoding' => undef,
                     }, ref $pkg || $pkg;

    foreach my $p ( qw/pattern period converter converter_config filter encoding/ ) {
        no strict 'refs';
        $self->$p( $args{$p} ) if exists $args{$p};
    }

    return $self;
}

=pod

=head2 name()

Returns the name of this ImportJob.

=cut

sub name { $_[0]{'_name'} };

=pod

=head2 period( [$new_period] )

With no argument, returns the period of this ImportJob. With a
parameter, the ImportJob's period is set to this new value.

=cut

sub period {
    my ( $self, $period ) = @_;

    if ( @_ == 2 ) {
        check_param( $period, 'period', qr/^(unique|hourly|daily|weekly|monthly|yearly)$/,
                     "'period' parameter can only be 'unique', 'daily', 'weekly' or 'monthly'" );
        $self->{'_period'} = $period;
    }

    $self->{'_period'};
}

=pod

=head2 file( [$time] )

Returns the file to be used. The optional $time parameter (which
defaults to now) is used to compute the file name. To find the file to
use for yesterday's data, one could pass time - 86400.

=cut

sub file {
    my ( $self, $time ) = @_;

    $time ||= time;

    return strftime $self->{'_pattern'}, localtime $time;
}

=pod

=head2 pattern( [$new_file] )

Returns the file pattern to use as log data source. This pattern may
contain strftime(3) substitutions. If an argument is used, it will
change the file to use.

=cut

sub pattern {
    my ( $self, $pattern ) = @_;

    if ( @_ == 2 ) {
        check_param( $pattern, 'pattern' );
        $self->{'_pattern'} = $pattern;
    }

    return $self->{'_pattern'};
}

=pod

=head2 filter( [$new_filter] )

Returns the shell command that will be used as filter. Undef or ''
will not use a filter.

=cut

sub filter {
    my ( $self, $filter ) = @_;

    if ( @_ == 2 ) {
        $self->{'_filter'} = $filter ? $filter : undef;
    }

    return $self->{'_filter'};
}

=pod

=head2 encoding( [$new_encoding] )

Returns the encoding of the log file. Undef and '' means to use
the system default encoding.

=cut

sub encoding {
    my ( $self, $encoding ) = @_;

    if ( @_ == 2 ) {
        $self->{'_encoding'} = $encoding ? $encoding : undef;
    }

    return $self->{'_encoding'};
}

=pod

=head2 converter( [$new_name] )

Returns the name of the converter that should be used for this
ImportJob.

=cut

sub converter {
    my ( $self, $converter ) = @_;

    if ( @_ == 2 ) {
        check_param( $converter, 'converter' );
        $self->{'_converter'} = $converter;
    }

    return $self->{'_converter'};
}

=pod

=head2 converter_config( [$new_config] )

Returns a configuration object that should be used by the DlfConverter
(through the set_configuration() method).

=cut

sub converter_config {
    my ( $self, $config ) = @_;

    if ( @_ == 2 ) {
        $self->{'_converter_config'} = $config;
    }

    return $self->{'_converter_config'};
}

=pod

=head2 log_fh( [ $time ] )

Returns a file handle that can be used to read from the log. $time is
the parameter taken by file to determine the final filename. All
filters and encoding will be setup properly.

=cut

sub log_fh {
    my ( $self, $time ) = @_;

    my $filename = $self->file( $time );

    my $fh;
    if ( $self->{'_filter'} ) {
        if ( index( $self->{'_filter'}, '|' ) != -1 ) {
            $filename = "cat '$filename' | $self->{'_filter'} |"
        } else {
            $filename = "$self->{'_filter'} < '$filename' |"
        }
    }
    open $fh, $filename
          or die file_not_readable( $filename );
    set_fh_encoding( $fh, $self->{'_encoding'} )
      if $self->{'_encoding'};

    return $fh;
}

=pod

=head2 run( $store )

Runs this ImportJob and import it into the $store DlfStore.

=cut

sub run {
    my ( $self, $store ) = @_;

    check_object_param( $store, 'store', 'Lire::DlfStore' );

    my $process = new Lire::DlfConverterProcess( $self, $store );
    $process->run_import_job();

    my $converter = Lire::PluginManager->get_plugin( 'dlf_converter',
                                                     $self->{'_converter'} );
    foreach my $name ( $converter->schemas() ) {
        $store->run_analysers( $name, $process->job_id() )
          if $store->has_dlf_stream( $name );
    }
    return;
}

# Used to instantiate from Lire::Config::Object
sub new_from_config {
    my ( $class, $config ) = @_;

    $config = $config->Lire::Config::Dictionary::as_value();
    return new Lire::ImportJob( $config->{'name'},
                                'period' => $config->{'period'},
                                'encoding' => $config->{'log_encoding'},
                                'pattern' => $config->{'log_file'},
                                'converter' => $config->{'service'}{'plugin'},
                                'converter_config' => $config->{'service'}{'properties'},
                                'filter' => $config->{'filter'} );
}

# keep perl happy
1;

__END__

=pod

=head1 SEE ALSO

Lire::DlfStore(3pm) Lire::DlfConverter(3pm)

=head1 AUTHOR

  Francis J. Lacoste <flacoste@logreport.org>

=head1 VERSION

$Id: ImportJob.pm,v 1.11 2004/06/29 18:37:03 flacoste Exp $

=head1 COPYRIGHT

Copyright (C) 2002-2004 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=cut
