# Copyright 2002, Steve Kowalik <stevenk@debian.org>, based on DpkgVersion.py
# This module implements dpkg version numbers and comparison routines
# Copyright 2002 Wichert Akkerman <wichert@deephackmode.org>

import re, string, types

# Regular expression to take a version number appart
VersionRegex = "^(?:(?P<epoch>[0-9]+):)?(?P<version>[a-zA-Z0-9.+-]+?)(?:-(?P<revision>[a-zA-Z0-9.+]+))?$"

# Precompile the regular expressions 
VersionMatcher = re.compile(VersionRegex)

def __isdigit(x):
    "Helper function to check if a character is an ASCII digit"
    return x >= '0' and x <= '9'

def __isalpha(x):
    "Helper function to check if a character is an ASCII letter"
    return (x >= 'a' and x <= 'z') or (x >= 'A' and x <= 'Z')

def __order(x):
    '''Helper function for SimpleCompare. Determines a ranking order
    for a character.'''

    # The order of these if statements is important
    if __isdigit(x):
        return 0
    elif __isalpha(x):
        return ord(x)
    elif ord(x) == 0:
        return 0
    elif x == '~':
        return -1
    else:
        return ord(x)+256

def SimpleCompare(orgval, orgref):
    '''The basic version comparison routine. The algorithm used is taken
    from lib/vercmp.c from the dpkg source tree, slightly modified for
    python'''

    # Add a terminal so val and ref can't suddenly become empty and
    # we start doing illegal list operations
    val = orgval+"\x00"
    ref = orgref+"\x00"

    while ord(val[0]) or ord(ref[0]):
        first_diff = 0

        while ((ord(val[0]) and not __isdigit(val[0])) or
                (ord(ref[0]) and not __isdigit(ref[0]))):
            vc = __order(val[0])
            rc = __order(ref[0])
            if (vc != rc):
                return (vc - rc)
            val = val[1:]
            ref = ref[1:]
        
        while val[0] == '0':
            val = val[1:]
        while ref[0] == '0':
            ref = ref[1:]
        while __isdigit(val[0]) and __isdigit(ref[0]):
            if not first_diff:
                first_diff = ord(val[0]) - ord(ref[0])
            val = val[1:]
            ref = ref[1:]

        if __isdigit(val[0]):
            return 1
        if __isdigit(ref[0]):
            return -1
        if first_diff:
            return first_diff
    
    return 0

class DpkgVersionException(Exception):
    pass

class DpkgVersion:
    '''Class representing a dpkg version number consisting of an epoch,
    an (upstream) version and a package revision.'''

    def __init__(self, version=None):
        self.epoch = 0
        self.version = ""
        self.revision = ""

        if version:
            self.Set(version)

    def __str__(self):
        ver = ""
        if self.epoch:
            ver = "%d:" % self.epoch
        ver += self.version
        if self.revision:
            ver += "-"+self.revision

        return ver

    def __realcmp(self, other):
        if (self.epoch != other.epoch):
            return self.epoch - other.epoch

        res = SimpleCompare(self.version, other.version)
        if res:
            return res

        if self.revision is None:
            self.revision = ''
        if other.revision is None:
            other.revision = ''
        res = SimpleCompare(self.revision, other.revision)
        if res:
            return res

        return 0

    def __cmp__(self, other):
        if type(self) == type(other):
            return self.__realcmp(other)
        elif type(other) == types.StringType:
            return self.__realcmp(DpkgVersion(other))
        else:
            raise TypeError

    def Set(self, version):
        '''Assign a version number. This will split the version in
        the epoch, version and revision.'''

        m = VersionMatcher.match(version)
        if not m:
            raise DpkgVersionException, "Invalid version"

        if m.group("epoch"):
            self.epoch = int(m.group("epoch"))
        else:
            self.epoch = 0
        self.version = m.group("version")
        self.revision = m.group("revision")

    def is_native(self):
        native = False
        if not self.revision:
            native = True
        return native

