/***************************************************************************
                           kxesettings.h
                           -------------
    begin                : Tue Dec 02 2003
    copyright            : (C) 2003 by The KXMLEditor Team
    email                : hartig@users.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef KXESETTINGS_H
#define KXESETTINGS_H

#include <qobject.h>

class KConfig;
class QFrame;

/**
 * This is an abstract base class for classes representing a group, that stores
 * several configuration settings and manages a corresponding page in the
 * configuration dialog.
 *
 * @author Olaf Hartig
 */
class KXESettings : public QObject
{
		Q_OBJECT

	public:

		/**
		 * the constructor
		 *
		 * @param strConfigGroup the name of the corresponding group in the config file,
		 *                       it is copied to the member @ref m_strConfigGroup
		 */
		KXESettings( const QString & strConfigGroup, QObject * pParent = 0, const char * pszName = 0 );

		/**
		 * Stores this settings to the given @ref KConfig object, by doing the
		 * following things:
		 *
		 * - set the config group by calling @ref setConfigGroup
		 * - write the data to the given @ref KConfig object by calling @ref write
		 *   (which is implemented in the child class)
		 */
		void store( KConfig * ) const;
		/**
		 * Restores the settings from the given @ref KConfig object, by doing
		 * following things:
		 *
		 * - set the config group by calling @ref setConfigGroup
		 * - read the data from the given @ref KConfig object by calling @ref read
		 *   (which is implemented in the child class)
		 * - try to update the corresponding config.page by calling @ref updatePage
		 *   (implemented in the child class)
		 * - emit the signal sigChanged
		 */
		void restore( KConfig * );
		/**
		 * If the data in the corresponding configuration dialog page has been
		 * changed (@ref m_bPageChanged), this data gets applied by the
		 * @ref setFromPage member function implemented in child classes,
		 * the flag @ref m_bPageChanged is reset (to false) and the signal
		 * sigChanged is emitted.
		 */
		void apply();

		/**
		 * Override this function and return the name of the corresponding
		 * configuration dialog page. This is the name used in the list of
		 * pages in the configuration dialog.
		 * This name has to be internationalized.
		 */
		virtual QString dialogPageName() const = 0;
		/**
		 * You can override this function and return a header line to
		 * be used for the corresponding configuration dialog page.
		 * If it's not overridden, the pages name is used (see
		 * @ref dialogPageName)
		 * This string has to be internationalized.
		 */
		virtual QString dialogPageHeader() const { return QString::null; }
		/**
		 * Override this function and return the (file-)name of the icon
		 * to be used in the configuration dialog for the corresponding page.
		 */
		virtual QString dialogPageIcon() const = 0;
		/**
		 * Override this function to create the corresponding configuration
		 * dialog page with the given parent and return it.
		 * Update the page's widgets (usually by using @ref updatePage).
		 * Connect the signal @ref sigDialogPageChanged to the page's
		 * data changed signal(s).
		 */
		virtual QWidget * dialogPage( QFrame * pParent ) = 0;


	signals:

		/**
		 * This signal is always emitted when the settings change.
		 * It is emitted by @ref restore and @ref apply.
		 */
		void sigChanged();
		/**
		 * Emitted, when the data in the corresponding dialog page
		 * has been changed.
		 * Connect this signal to the page's changed signal in your
		 * child class in @ref dialogPage.
		 */
		void sigDialogPageChanged();

	protected:

		/**
		 * Overide this function and write your settings to the given KConfig object.
		 * You mustn't change the config group.
		 */
		virtual void write( KConfig * ) const = 0;
		/**
		 * Overide this function and read the settings from the given KConfig object.
		 * You mustn't change the config group.
		 */
		virtual void read( const KConfig * ) = 0;
		/**
		 * Override this function to set this object's data to the data in the
		 * corresponding configuration dialog page, if a page has already been
		 * created (check this!).
		 */
		 virtual void setFromPage() = 0;
		 /**
		  * Override this function to update the widgets in the corresponding
		  * configuration dialog page with the current data (from your child class
		  * object), if a page has already been created (check this!).
		  */
		 virtual void updatePage() const = 0;

		/**
		 * Sets the config group of the given @ref KConfig object to
		 * this setting's config group (@ref m_strConfigGroup).
		 */
		void setConfigGroup( KConfig * ) const;

		/**
		 * This flag is set to true if the data in the corresponding configuration
		 * dialog's page has been changed but not applied yet.
		 */
		bool m_bPageChanged;

	protected slots:

		/**
		 * Sets the flag @ref m_bPageChanged to true.
		 */
		void slotDialogPageChanged();

	private:

		/**
		 * name of the config group for this group of settings
		 */
		const QString m_strConfigGroup;

};

#endif
