/* @(#) uplinkprotocol.c 1.24 @(#) */
/***************************************************************\
*	Copyright (c) 1999-2000 First Step Internet Services, Inc.
*		All Rights Reserved
*	Distributed under the BSD Licenese
*
*	Module: CORE
\***************************************************************/

#define _KOALAMUD_UPLINKPROTOCOL_C "@(#) nitehawk@localhost.1ststep.net|lib/koala/uplinkprotocol.c|20000827025248|47703 @(#)"

#include "autoconf.h"

#include "version.h"
#include "koalatypes.h"
#include "daemon.h"
#include "network.h"
#include "log.h"
#include "uplinkprotocol.h"
#include "llist.h"
#include "memory.h"
#include "buffer.h"

/* uplinknegotiatestage -
 *		Read data from socket and handle current stage of protocol negotiation
 *		
 *		There must be data on the socket or the socket must be marked
 *		non-blocking, otherwise this function *will* block progress
 *
 *		This function does not handle the authentication of uplink daemons.
 */
koalaerror uplinknegotiatestage(pdescriptor desc)
{
	char buf[80] = {'\0'};
	unsigned numread = 80;
	char linkerrmsg[] = "Protocol Negotiation Error - Disconnecting\r\n";
	koalaerror kerr;

	/* Validate descriptor */
	if (!desc || desc->type != DESCRIPTOR_UNKNOWN)
	{
		logmsg(LOGERR, "uplinknegotiatestage caught invalid descriptor");
		return KEBADOPT;
	}

	/* If the descriptor is NOMINAL or further in its lifeline, then we
	 * shouldn't be trying to handle creating an uplink
	 */
	if (desc->status >= STATUS_NOMINAL)
	{
		return KESUCCESS;
	}

	/* Read in data to determine current stage of uplink negotiation */
	numread = MAGICLEN;
	kerr = buffer_readbytes(desc, buf, &numread, FALSE);
	if (kerr == KENOTENOUGH)
	{
		/* No data on socket, just return success now */
		return KESUCCESS;
	}

	/* At this point, we should have all the information we need to determine
	 * what stage we are at and make the appropriate reply */
	if (desc->status == STATUS_CONNECTING)
	{
		/* We havn't received a daemon type from the remote machine yet.  The
		 * data we just read should either be the protocol magic or the magic
		 * of the daemon type.
		 */
		if (strcmp(buf, magic.daemonmagic) == 0)
		{
			/* We are the initiator */
			/* First stage - Reply with the magic string for our daemon type
			 * and fall out */
			switch(koptions.daemontype)
			{
				case DAEMON_CLIENT:
					netwrite(desc, magic.clientmagic,
							strlen(magic.clientmagic));
					break;

				case DAEMON_HUB:
					netwrite(desc, magic.hubmagic,
							strlen(magic.hubmagic));
					break;

				case DAEMON_ZONE:
					netwrite(desc, magic.zonemagic,
							strlen(magic.zonemagic));
					break;
				
				case DAEMON_UNKNOWN:
					/* It is impossible to reach this point, do nothing */
			}
			/* Set the status to say we sent the type */
			desc->status = STATUS_TYPESENT;
			return KESUCCESS;
		}
	}

	/* We just received the type of daemon on the other side of the link.  If
	 * we received this data and the status is 'CONNECTING' then we need to
	 * send our type back.  Otherwise we should be in 'TYPESENT' status and
	 * we have already sent our daemon type.
	 */
	if (strcmp(buf, magic.clientmagic) == 0)
	{
		/* Remote is a client daemon */
		desc->type = DESCRIPTOR_CLIENTSRV;
		desc->data.clientsrv = kmalloc(sizeof(struct TAG_CLIENTSERVER),
				ALLOC_DESCRIPTOR);
		desc->data.clientsrv->curmsg.messageid = 0;
	}
	else if (strcmp(buf, magic.hubmagic) == 0)
	{
		/* Remote is a client daemon */
		desc->type = DESCRIPTOR_HUBSRV;
		desc->data.hubsrv = kmalloc(sizeof(struct TAG_HUBSERVER),
				ALLOC_DESCRIPTOR);
		desc->data.hubsrv->curmsg.messageid = 0;
	}
	else if (strcmp(buf, magic.zonemagic) == 0)
	{
		/* Remote is a client daemon */
		desc->type = DESCRIPTOR_ZONESRV;
		desc->data.zonesrv = kmalloc(sizeof(struct TAG_ZONESERVER),
				ALLOC_DESCRIPTOR);
		desc->data.zonesrv->curmsg.messageid = 0;
	}
	else
	{
		/* Protocol problem - Close the socket */
		logmsg(LOGERR, "Protocol negotiation error!  Closing socket");
		netwrite(desc, linkerrmsg, strlen(linkerrmsg));
		desc->status = STATUS_CLOSE;
	}

	/* If we havn't sent the type yet, send it now */
	if (desc->status == STATUS_CONNECTING)
	{
		switch(koptions.daemontype)
		{
		case DAEMON_CLIENT:
			netwrite(desc, magic.clientmagic,
					strlen(magic.clientmagic));
			break;

		case DAEMON_HUB:
			netwrite(desc, magic.hubmagic,
					strlen(magic.hubmagic));
			break;

		case DAEMON_ZONE:
			netwrite(desc, magic.zonemagic,
					strlen(magic.zonemagic));
			break;
			
		case DAEMON_UNKNOWN:
			/* It is impossible to reach this point, do nothing */
		}
	}
	else
	{
		/* Send first part of ident squak */
		message_data data = {NULL};
		uplinksendmessage(desc, 0, MSGTYPE_IDENT, &data, 0);
	}

	/* Set the status to say we sent the type */
	desc->status = STATUS_NOMINAL;
	return KESUCCESS;
}

/* Send Message - Fill in a message struct and send it to the remote host
 */
koalaerror uplinksendmessage(pdescriptor desc, unsigned int dest, msgtype type,
		message_data *msgstruct, unsigned int msgid)
{
	uplinkmsg_header header;

	/* first verify our pointers */
	if (!desc || !msgstruct)
	{
		logmsg(LOGERR, "Caught NULL pointer");
		return KEMISSINGARG;
	}

	/* Now fill in the message header */
	header.messageid = msgid == 0 ? (unsigned int)random() : msgid;
	header.sourcenodeid = kstate.nodeid;
	header.messagetype = type;
	header.destnodeid = dest;
	header.messagedatalen = 0;

	/* The rest of the handling depends on the type of message being sent */
	switch (type)
	{
		case MSGTYPE_IDENT:
			/* Finish header */
			header.messagedatalen = sizeof(message_ident);
			header.destnodeid = desc->nodeid;

			/* First see if we were passed a struct to use, otherwise
			 * allocate one */
			if (msgstruct->data == NULL)
			{
				msgstruct->data = kmalloc(sizeof(message_ident), ALLOC_GENERIC);
				if (msgstruct->data == NULL)
				{
					logmsg(LOGCRIT, "Unable to allocate memory"
							" for ident struct");
					return KENOMEM;
				}
			}
			/* Fill in the struct */
			msgstruct->msgident->mynodeid = kstate.nodeid;
			/* This will be 0 if we don't yet know their ID */
			/* receiving 0 in 'yournodeid' is a signal to respond with the
			 * same message type */
			msgstruct->msgident->yournodeid = desc->nodeid;

			/* Send the data out */
			netwrite(desc, (char *)&header, sizeof(uplinkmsg_header));
			netwrite(desc, (char *)msgstruct->msgident, sizeof(message_ident));

			/* Free the message data, even if it wasn't us allocating it */
			kmfree(msgstruct->data, ALLOC_GENERIC);
			break;

		case MSGTYPE_GLOBAL_MSG:
			/* This isn't *really* global.  It is sent to a specific nodeid
			 * or the class global ID.  It can also be sent to an id group
			 * (this is a nodeid that sends to a group of nodes)
			 */
			/* If the caller didn't give us message data, we have no way to
			 * construct a message to send */
			if (msgstruct->data == NULL)
			{
				logmsg(LOGWARN, "Global message cannot be sent"
						" without message data");
				return KEMISSINGARG;
			}

			if (dest == NODEID_UNSPEC ||
					dest == NODEID_GLOBAL ||
					dest == NODEID_ALLHUB ||
					dest == NODEID_ALLZONE)
			{
				header.destnodeid = NODEID_ALLCLIENT;
			}

			/* Write packet to the network */
			netwrite(desc, (char *)&header, sizeof(header));
			netwrite(desc, (char *)msgstruct->data, sizeof(message_global_msg));
			break;

		case MSGTYPE_DISCONNECT:
			// Set the destination to the remote nodeid
			header.destnodeid = desc->nodeid;
			
			// Write the packet to the network
			netwrite(desc, (char *)&header, sizeof(uplinkmsg_header));

			// Set the descriptor status to close it
			desc->status = STATUS_CLOSE;
			break;

		case MSGTYPE_SHUTDOWN:
			// Write the packet to the network
			netwrite(desc, (char *)&header, sizeof(uplinkmsg_header));
			break;

		case MSGTYPE_REBOOT:
			// Write the packet to the network
			netwrite(desc, (char *)&header, sizeof(uplinkmsg_header));
			break;

		default:
			logmsg(LOGWARN, "Attempt to send unknown message type");
			return KEUNKNOWNMESSAGE;
	}

	return KESUCCESS;
}

/* Uplinkreadmessage -- Read message from network
 * Note: We will block progress until we get the entire message packet
 */
koalaerror uplinkreadmessage(pdescriptor desc, uplinkmsg_header *header, 
		message_data *data)
{
	int num;
	koalaerror kerr;
	
	/* Verify we got pointers */
	if (!desc || !data)
	{
		logmsg(LOGERR, "Caught null pointers!");
		return KEMISSINGARG;
	}
	
	/* If we aren't currently handling a message */
	if (header->messageid == 0)
	{
		/* Read the header from the network */
		num = sizeof(uplinkmsg_header);
		kerr = buffer_readbytes(desc, (char *)header, &num, TRUE);
		if (kerr == KENOTENOUGH)
		{
			return KENOTENOUGH;
		}
	}

	if (header->messagedatalen > 0)
	{
		data->data = kmalloc(header->messagedatalen, ALLOC_GENERIC);
		if (data->data == NULL)
		{
			logmsg(LOGCRIT, "Unable to allocate memory for message struct");
			return KENOMEM;
		}
		num = header->messagedatalen;
		kerr = buffer_readbytes(desc, (char*)data->data, &num, TRUE);
		if (kerr == KENOTENOUGH)
		{
			return KENOTENOUGH;
		}
	}
	return KESUCCESS;
}

/* uplinkhandlemessage -- Handle a packet that is destined for us
 */
koalaerror uplinkhandlemessage(pdescriptor desc, uplinkmsg_header *header,
		message_data *data)
{
	/* Verify we got pointers */
	if (!desc || !header || !data)
	{
		logmsg(LOGERR, "Caught null pointers!");
		return KEMISSINGARG;
	}

	/* Switch into correct message handling */
	switch (header->messagetype)
	{
		case MSGTYPE_IDENT:
			/* Copy 'mynodeid' into the descriptor struct */
			desc->nodeid = data->msgident->mynodeid;

			/* If the 'yournodeid' field is zero, send a message back */
			if (data->msgident->yournodeid == 0)
			{
				kmfree(data->data, ALLOC_GENERIC);
				data->data = NULL;
				uplinksendmessage(desc, desc->nodeid, MSGTYPE_IDENT, data,
						header->messageid);
			}
			else
			{
				kmfree(data->data, ALLOC_GENERIC);
				data->data = NULL;
			}

			break;

		case MSGTYPE_GLOBAL_MSG:
			/* If this is not a client server, we don't handle this message */
			if (koptions.daemontype != DAEMON_CLIENT)
			{
				kmfree(data->data, ALLOC_GENERIC);
				return KESUCCESS;
			}
			break;

		case MSGTYPE_DISCONNECT:  // No ack needed, just close it
			// Set the descriptor status to close it
			desc->status = STATUS_CLOSE;
			break;

		case MSGTYPE_SHUTDOWN:  // No ack needed, just close it
			kstate.running = DSTATE_SHUTDOWN;
			break;

		case MSGTYPE_REBOOT:  // No ack needed, just close it
			kstate.running = DSTATE_REBOOT;
			break;

		default:
			logmsg(LOGWARN, "Unknown message type");
			return KEUNKNOWNMESSAGE;
	}
	header->messageid = 0;

	return KESUCCESS;
}

/* Uplinkroutemessage - Read message from network and route it on to its
 * destination
 */
koalaerror uplinkroutemessage(pdescriptor desc)
{
	uplinkmsg_header *header;
	message_data data;
	koalaerror kerr = KESUCCESS;
	int transmitcount = 0;
	listnodeptr tmplist;
	pdescriptor tmpdesc;
	int msglen;

	/* Initialize stuff */
	data.data = NULL;

	/* Point local header to descriptors header */
	switch(desc->type)
	{
		case DESCRIPTOR_HUBSRV:
			header = &(desc->data.hubsrv->curmsg);
			break;
		case DESCRIPTOR_ZONESRV:
			header = &(desc->data.zonesrv->curmsg);
			break;
		case DESCRIPTOR_CLIENTSRV:
			header = &(desc->data.clientsrv->curmsg);
			break;
		default:
			return KEBADOPT;
	}

	/* Read the message from the network */
	kerr = uplinkreadmessage(desc, header, &data);
	if (kerr != KESUCCESS)
	{
		/* We can't do anything useful if the read failed */
		return kerr;
	}

	/* Figure out the size of the message we are sending */
	switch(header->messagetype)
	{
		case MSGTYPE_GLOBAL_MSG:
			msglen = sizeof(message_global_msg);
			break;

		case MSGTYPE_XML:
			msglen = data.msgxml->xmllen + sizeof(short);
			break;

		default:
			msglen = 0;
	}

	/* Route the packet to target descriptors */
	/* first: If this is our packet, we can skip the entire routing loop and
	 * just handle it
	 */
	if (header->destnodeid != kstate.nodeid)
	{
		/* If this is an ident squawk, it cannot be routed past us, toss it to
		 * the handler.  Same with disconnects. */
		if (header->messagetype == MSGTYPE_IDENT ||
				header->messagetype == MSGTYPE_DISCONNECT)
		{
			kerr = uplinkhandlemessage(desc, header, &data);
			return kerr;
		}

		/* This packet is not ours (or it is global or unknown),
		 * 	route it on its way */
		if (header->destnodeid == NODEID_GLOBAL)
		{
			/* This is a global message, loop through all descriptors and
			 * forward to all uplinks
			 */
			for (tmplist = getdescriptorlist(); tmplist;
					tmplist = listnextnode(tmplist))
			{
				tmpdesc = tmplist->data.desc;

				if (tmpdesc->type == DESCRIPTOR_HUBSRV ||
						tmpdesc->type == DESCRIPTOR_ZONESRV ||
						tmpdesc->type == DESCRIPTOR_CLIENTSRV)
				{
					transmitcount++;
					netwrite(tmpdesc, (char *)header, sizeof(uplinkmsg_header));

					/* Send out message data */
					switch(header->messagetype)
					{
						case MSGTYPE_SHUTDOWN:
						case MSGTYPE_REBOOT:
							/* No data piece to write */
							break;
						default:
							/* All other possible message types have message
							 * data to send */
							netwrite(tmpdesc, (char *)data.data, msglen);
					}
				}
			}
		} // end Global message loop

		/* This packet is targeted for all servers of a specific class */
		if (header->destnodeid == NODEID_ALLHUB ||
				header->destnodeid == NODEID_ALLZONE ||
				header->destnodeid == NODEID_ALLCLIENT)
		{
			/* This is a global message, loop through all descriptors and
			 * forward to all uplinks
			 */
			for (tmplist = getdescriptorlist(); tmplist;
					tmplist = listnextnode(tmplist))
			{
				tmpdesc = tmplist->data.desc;

				if (tmpdesc->type == header->destnodeid)
				{
					transmitcount++;
					netwrite(tmpdesc, (char *)header, sizeof(uplinkmsg_header));

					/* Send out message data */
					switch(header->messagetype)
					{
						case MSGTYPE_SHUTDOWN:
						case MSGTYPE_REBOOT:
							/* No data piece to write */
							break;

						default:
							/* All other possible message types have message
							 * data to send */
							netwrite(tmpdesc, (char *)data.data, msglen);
					}
				}
			}
		}  // end Server class global message loop

		/* Find entry in routing table and forward packet along */
	}

	/* If this message is global or to us, handle it now
	 * Two reasons for this:
	 * 		1: Global shutdowns need to be routed outward first
	 * 		2: Header and data are freed in the handler
	 */
	if (header->destnodeid == NODEID_GLOBAL ||
			header->destnodeid == NODEID_ALLHUB ||
			header->destnodeid == kstate.nodeid)
	{
		/* Call the handler */
		kerr = uplinkhandlemessage(desc, header, &data);
		/* Data is freed by the message handler */
		header->messageid = 0;
		return kerr;
	}

	/* We didn't handle the packet ourselves, we need to free the memory used
	 * by the message data
	 */
	if (data.data)
	{
		kmfree(data.data, ALLOC_GENERIC);
	}
	header->messageid = 0;

	return KESUCCESS;
}

/* Uplinkreceivemessage - Read a message from network and handle it
 */
koalaerror uplinkreceivemessage(pdescriptor desc)
{
	uplinkmsg_header *header;
	message_data data;
	koalaerror kerr = KESUCCESS;

	/* Initialize stuff */
	data.data = NULL;

	/* Point local header to descriptors header */
	switch(desc->type)
	{
		case DESCRIPTOR_HUBSRV:
			header = &(desc->data.hubsrv->curmsg);
			break;
		case DESCRIPTOR_ZONESRV:
			header = &(desc->data.zonesrv->curmsg);
			break;
		case DESCRIPTOR_CLIENTSRV:
			header = &(desc->data.clientsrv->curmsg);
			break;
		default:
			return KEBADOPT;
	}

	/* Read the message from the network */
	kerr = uplinkreadmessage(desc, header, &data);
	if (kerr != KESUCCESS)
	{
		/* We can't do anything useful if the read failed */
		return kerr;
	}

	/* Verify that this message is either global or for us */
	if (header->destnodeid == NODEID_GLOBAL ||
			header->destnodeid == kstate.nodeid ||
			/* Hub servers don't use this loop so we don't need to check for
			 * them */
			((koptions.daemontype == DAEMON_CLIENT &&
			  header->destnodeid == DESCRIPTOR_CLIENTSRV) ||
			 (koptions.daemontype == DAEMON_ZONE &&
			  header->destnodeid == DESCRIPTOR_ZONESRV)))
	{
		/* Call the handler */
		kerr = uplinkhandlemessage(desc, header, &data);
		/* Data is freed by the message handler */
		header->messageid = 0;
		return kerr;
	}

	/* Message wasn't for us, free the data and silently ignore it */
	if (data.data)
	{
		kmfree(data.data, ALLOC_GENERIC);
	}
	header->messageid = 0;

	return KESUCCESS;
}
