/* @(#) .del-fs.c 1.11 @(#) */
/***************************************************************\
*	Copyright (c) 1999 First Step Internet Services, Inc.
*		All Rights Reserved
*	Distributed under the BSD Licenese
*
*	Module: CORE
\***************************************************************/

#define _KOALAMUD_FS_C "@(#) nitehawk@winghove.1ststep.net|BitKeeper/deleted/.del-fs.c|20000307194803|11687 @(#)"

#include "autoconf.h"
#include <sys/stat.h>
#include <sys/mman.h>

#include "version.h"
#include "koalatypes.h"
#include "log.h"
#include "conf.h"
#include "lib/dbinternal.h"
#include "db.h"

/* See database.format for the specification detailing the workings of the
 * database
 */

/* dbnewfs
 * 	Create a new database file and populate basic structures.
 */
koalaerror dbnewfs(char *dbpath)
{
	int dbfd = -1;	// File handle to the database
	void *baseaddr = NULL;
	void *blkaddr = NULL;
	superblock *sb = NULL;
	struct timeval tod;
	int blocknum = 0;
	int blockdnode = 0;
	int dnodenum = 0;
	dnode *dn = NULL;
	kdbfs_blockid_t	blockid;
	u_int8_t *byte;

	/* First we'll try to open the file for writing.  This will also have the
	 * effect of truncating the file to 0 bytes
	 */
	if ((dbfd = open(dbpath, O_RDWR | O_CREAT | O_TRUNC, 0660)) == -1)
	{
		logerr("Unable to open database file");
		return KEFOPEN;
	}

	/* Zero out block 0 */
	zeroblock(dbfd, 0);

	/* Now we try mapping the database file into memory */
	if ((baseaddr = mmapblockaligned(dbfd, 0, 1)) == NULL)
	{
		close(dbfd);
		logerr("Unable to map new database file");
		return KENOMEM;
	}

	/* fill in the superblock */
	{
		/* We now have the new database file mapped in with enough space for
		 * block 0 and the first superblock */
		sb = (superblock *)baseaddr;

		sb->fs_clean = FS_CLEAN;

		/* Database FS size */
		sb->fs_dsize = FS_DNODESBASE;
		sb->fs_sbdsize = FS_DNODESBASE;
		sb->fs_size = FS_BLOCKSBASE;
		sb->fs_sbsize = FS_BLOCKSBASE;

		/* Find position for block bitmap */
		sb->fs_btno = 1;
		sb->fs_btsize = FS_BITMAPBLOCKS;

		/* Setup dnode table location and reserve */
		sb->fs_dbno = sb->fs_btno + FS_BITMAPBLOCKS; 
		sb->fs_dtsize = sb->fs_sbdsize / FS_DNODESPERBLOCK;

		/* Number of free blocks and dnodes */
		sb->fs_dfree = FS_DNODESBASE;
		sb->fs_bfree = sb->fs_size - sb->fs_dtsize - sb->fs_btsize - 1;

		/* Location of first free objects */
		sb->fs_ffdnode = DNODE_ROOTOWN + 1;
		sb->fs_ffblock = sb->fs_dtsize + sb->fs_btsize + 1;

		/* get the time of day */
		gettimeofday(&tod, NULL);
		sb->fs_ctime = tod.tv_sec;
		sb->fs_ctimesec = tod.tv_usec;
		sb->fs_ltime = tod.tv_sec;
		sb->fs_ltimesec = tod.tv_usec;

		/* Set the last mount to 'unmounted' so that we know this is a new
		 * database
		 */
		strcpy(sb->fs_lastmnt, "unmounted");
	}

	/* Create block bitmap */
	{
		/* First write out enough blocks for the map */
		for (blocknum = sb->fs_btno; blocknum < sb->fs_btno + sb->fs_btsize;
				blocknum++)
		{
			zeroblock(dbfd, blocknum);
		}

		/* Map the entire bitmap */
		if ((blkaddr = mmapblockaligned(dbfd, sb->fs_btno,
						sb->fs_btsize)) == NULL)
		{
			closemapfile(dbfd);
			logerr("Unable to map new blocks");
			return KENOMEM;
		}

		/* Now we just need to turn on the bits for all the blocks used by the
		 * dnode table, super block, and block bitmap */
		for (blockid = 0; blockid < (1 + sb->fs_btsize + sb->fs_dtsize);
				blockid++)
		{
			/* This is kindof a tricky way to turn on the bit representing the
			 * block without touching anything around it.  Read carefully */
			byte = blkaddr + (blockid/8);
			*byte |= ((u_int8_t)0x80>> (blockid%8));
		}

		/* We're done building the block bitmap, unmap the block(s) now */
		mapfree(blkaddr);
	}

	/* Build the dnode table */
	for (blocknum = sb->fs_dbno; blocknum < (sb->fs_dbno + sb->fs_dtsize);
			blocknum++)
	{
		/* Zero the block out for the current portion of the inode table */
		zeroblock(dbfd, blocknum);

		/* Map the block into memory */
		if ((blkaddr = mmapblockaligned(dbfd, blocknum,
						1)) == NULL)
		{
			closemapfile(dbfd);
			logerr("Unable to map new blocks");
			return KENOMEM;
		}

		for (blockdnode = 0; blockdnode < FS_DNODESPERBLOCK; blockdnode++)
		{
			/* Point to the dnode */
			dn = blkaddr + (blockdnode * sizeof(dnode));

			/* Fill in dnode attributes */ 
			dn->d_num = dnodenum;
			dn->d_nodetype = DNTYPE_FREE;

			dnodenum++;
		}

		/* Unmap the block */
		mapfree(blkaddr);
	}

	/* Create root directory entry - Dnode 0 */
	{
		void *fblockaddr;
		kdb_dirent *dir;

		/* Map to the dnode table */
		if ((blkaddr = mmapblockaligned(dbfd, sb->fs_dbno,
						sb->fs_dtsize)) == NULL)
		{
			closemapfile(dbfd);
			logerr("Unable to map new blocks");
			return KENOMEM;
		}
		/* Point to the root dnode */
		dn = blkaddr + DNODE_ROOT * sizeof(dnode);
		dn->d_nodetype = DNTYPE_DIRECTORY;
		dn->d_links = 2;
		dn->d_size = 512;
		dn->d_blocks = 1;

		/* Set owners */
		dn->d_uid = DNODE_ROOTOWN;
		dn->d_gid = DNODE_ROOTOWN;

		/* Allocate blocks for root directory */
		dn->d_db[0] = allocateblock(dbfd, sb);

		/* Timestamp root node */
		gettimeofday(&tod, NULL);
		dn->d_ctime = dn->d_atime = dn->d_mtime = tod.tv_sec;
		dn->d_ctimesec = dn->d_atimesec = dn->d_mtimesec = tod.tv_usec;

		/* Map to the file block */
		fblockaddr = mmapblockaligned(dbfd, dn->d_db[0], 1);

		/* Now we need to write out the base directory structure */
		dir = fblockaddr;
		dir->d_dnode = DNODE_ROOT;
		dir->d_type = DE_DIR;
		strcpy(dir->d_name, ".");
		dir->d_namelen = strlen(dir->d_name);
		dir->d_reclen = sizeof(dir->d_dnode) + sizeof(dir->d_reclen) + 1 +
				sizeof(dir->d_namelen) + dir->d_namelen + sizeof(dir->d_type);

		dir = (void*)dir + dir->d_reclen + 1;
		dir->d_dnode = DNODE_ROOT;
		dir->d_type = DE_DIR;
		strcpy(dir->d_name, "..");
		dir->d_namelen = strlen(dir->d_name);
		dir->d_reclen = sizeof(dir->d_dnode) + sizeof(dir->d_reclen) + 1 +
				sizeof(dir->d_namelen) + dir->d_namelen + sizeof(dir->d_type);

		/* We are done with the file block now */
		mapfree(fblockaddr);

		/* Unmap dnode table */
		mapfree(blkaddr);
	}

	/* We're done building the first superblock.  unmap the new database and
	 * close its file handle */
	mapfree(baseaddr);
	closemapfile(dbfd);

	return KESUCCESS;
}

/* dbinitdb
 * 	Create a new database.  This creates the default root player and other
 * 	important objects
 */
koalaerror dbinitdb(char *dbpath)
{
	koalaerror kerr;
	int dbfd;
	void *baseaddr;
	void *dnodebase;
	void *blkbase;
	dnode *dn;
	superblock *sb;
	kdb_dirent *dir;
	struct timeval tod;

	/* log message */
	logmsg("Creating new root database file system");

	/* Create a new filesytem on the given database file */
	if ((kerr = dbnewfs(dbpath)) != KESUCCESS)
	{
		logerr("Problem writing new filesystem");
		return KEDBFATAL;
	}

	/* Now that we have the new filesystem, we have to reopen the database */
	if ((dbfd = open(dbpath, O_RDWR)) == -1)
	{
		/* Error reopening database file?  How'd that happen */
		logerr("Unable to reopen database file.");
		return KEDBFATAL;
	}

	/* Map the superblock */
	if ((baseaddr = mmapblockaligned(dbfd, 0, 1)) == NULL)
	{
		logerr("Failed to map to superblock");
		closemapfile(dbfd);
		return KEDBFATAL;
	}
	sb = baseaddr;

	/* Map to the dnode table */
	if ((dnodebase = mmapblockaligned(dbfd, sb->fs_dbno, sb->fs_dtsize)) == NULL)
	{
		logerr("Failed to map to dnode table");
		closemapfile(dbfd);
		return KEDBFATAL;
	}

	/* Create /players/ tree */
	{
		/* Point to the /players/ dnode */
		dn = dnodebase + DNODE_PLAYERS * sizeof(dnode);
		dn->d_nodetype = DNTYPE_DIRECTORY;
		dn->d_links = 3;
		dn->d_size = 512;
		dn->d_blocks = 1;

		/* Set owners */
		dn->d_uid = DNODE_ROOTOWN;
		dn->d_gid = DNODE_ROOTOWN;

		/* Allocate blocks for root directory */
		dn->d_db[0] = allocateblock(dbfd, sb);

		/* Timestamp root node */
		gettimeofday(&tod, NULL);
		dn->d_ctime = dn->d_atime = dn->d_mtime = tod.tv_sec;
		dn->d_ctimesec = dn->d_atimesec = dn->d_mtimesec = tod.tv_usec;

		/* Map to the file block */
		blkbase = mmapblockaligned(dbfd, dn->d_db[0], 1);

		/* Now we need to write out the base directory structure */
		dir = blkbase;
		dir->d_dnode = dn->d_num;
		dir->d_type = DE_DIR;
		strcpy(dir->d_name, ".");
		dir->d_namelen = strlen(dir->d_name);
		dir->d_reclen = sizeof(dir->d_dnode) + sizeof(dir->d_reclen) + 1 +
				sizeof(dir->d_namelen) + dir->d_namelen + sizeof(dir->d_type);

		dir = (void*)dir + dir->d_reclen + 1;
		dir->d_dnode = DNODE_ROOT;
		strcpy(dir->d_name, "..");
		dir->d_namelen = strlen(dir->d_name);
		dir->d_type = DE_DIR;
		dir->d_reclen = sizeof(dir->d_dnode) + sizeof(dir->d_reclen) + 1 +
				sizeof(dir->d_namelen) + dir->d_namelen + sizeof(dir->d_type);

		dir = (void*)dir + dir->d_reclen + 1;
		dir->d_dnode = DNODE_ROOTOWN;
		dir->d_type = DE_DIR;
		strcpy(dir->d_name, "root");
		dir->d_namelen = strlen(dir->d_name);
		dir->d_reclen = sizeof(dir->d_dnode) + sizeof(dir->d_reclen) + 1 +
				sizeof(dir->d_namelen) + dir->d_namelen + sizeof(dir->d_type);

		/* We are done with the file block now */
		mapfree(blkbase);
	}

	/* Create /players/root/ tree */
	{
		/* Point to the /players/ dnode */
		dn = dnodebase + DNODE_ROOTOWN * sizeof(dnode);
		dn->d_nodetype = DNTYPE_DIRECTORY;
		dn->d_links = 2;
		dn->d_size = 512;
		dn->d_blocks = 1;

		/* Set owners */
		dn->d_uid = DNODE_ROOTOWN;
		dn->d_gid = DNODE_ROOTOWN;

		/* Allocate blocks for root directory */
		dn->d_db[0] = allocateblock(dbfd, sb);

		/* Timestamp root node */
		gettimeofday(&tod, NULL);
		dn->d_ctime = dn->d_atime = dn->d_mtime = tod.tv_sec;
		dn->d_ctimesec = dn->d_atimesec = dn->d_mtimesec = tod.tv_usec;

		/* Map to the file block */
		blkbase = mmapblockaligned(dbfd, dn->d_db[0], 1);

		/* Now we need to write out the base directory structure */
		dir = blkbase;
		dir->d_dnode = dn->d_num;
		dir->d_type = DE_DIR;
		strcpy(dir->d_name, ".");
		dir->d_namelen = strlen(dir->d_name);
		dir->d_reclen = sizeof(dir->d_dnode) + sizeof(dir->d_reclen) + 1 +
				sizeof(dir->d_namelen) + dir->d_namelen + sizeof(dir->d_type);

		dir = (void*)dir + dir->d_reclen + 1;
		dir->d_dnode = DNODE_PLAYERS;
		strcpy(dir->d_name, "..");
		dir->d_namelen = strlen(dir->d_name);
		dir->d_type = DE_DIR;
		dir->d_reclen = sizeof(dir->d_dnode) + sizeof(dir->d_reclen) + 1 +
				sizeof(dir->d_namelen) + dir->d_namelen + sizeof(dir->d_type);

		/* We are done with the file block now */
		mapfree(blkbase);
	}

	/* Map /players/ into / */
	{
		/* Point to the appropriate dnode */
		dn = dnodebase + DNODE_ROOT * sizeof(dnode);

		/* Bump up the link count since there is a link from /players to us */
		dn->d_links++;

		/* Map to the directory block */
		blkbase = mmapblockaligned(dbfd, dn->d_db[0], 1);

		/* Point to the first directory entry */
		dir = blkbase;

		/* Find the end of the directory listing */
		while (dir->d_reclen > 0)
		{
			dir = (void*)dir + dir->d_reclen + 1;
		}

		/* Now add a link into players */
		dir->d_dnode = DNODE_PLAYERS;
		strcpy(dir->d_name, "players");
		dir->d_namelen = strlen(dir->d_name);
		dir->d_type = DE_DIR;
		dir->d_reclen = sizeof(dir->d_dnode) + sizeof(dir->d_reclen) + 1 +
				sizeof(dir->d_namelen) + dir->d_namelen + sizeof(dir->d_type);

		mapfree(blkbase);
	}

	/* Close the database file again */
	closemapfile(dbfd);

	/* Return success */
	return KESUCCESS;
}

/* zeroblock
 * 	Write out a null block to the file system.  This function takes the file
 * 	descriptor for the database and a block id to zero.  If the file is
 * 	currently too small, the file will be expanded until it has room for our
 * 	block.
 *
 * 	The file descriptors current offset *will* be modified by this function
 *
 * 	Returns error if the write fails
 */
koalaerror zeroblock(int dbfd, kdbfs_blockid_t block)
{
	ssize_t writeret;
	off_t fdoff;
	char blockzero[BLOCKSIZE] = { '\0' };

	/* seek to the location of the block */
	if ((fdoff = lseek(dbfd, block * BLOCKSIZE, SEEK_SET)) != (block*BLOCKSIZE))
	{
		logerr("DBFATAL: Problem seeking to zero new block");
		return KEDBFATAL;
	}

	/* Write out a block of data */
	if ((writeret = write(dbfd, blockzero, BLOCKSIZE)) != BLOCKSIZE)
	{
		logerr("DBFATAL: Problem seeking to zero new block");
		return KEDBFATAL;
	}

	return KESUCCESS;
}

/* allocateblock
 * 	Allocate a free block from the filesystem containing the passed superblock
 * 	and mark it as allocated.  a return of 0 means that no blocks were able to
 * 	be allocated.
 *
 * 	*theoretically* this should only be needed while building the initial
 * 	database.  Normal object storage routines will need a bit more advanced
 * 	form.
 */
kdbfs_blockid_t allocateblock(int dbfd, superblock *sb)
{
	/* For the time being, (primarily to make this thing work in the first
	 * place) this will just assign the block listed as first free by the
	 * superblock and do housekeeping.
	 */
	void *blockbitmap = NULL;
	kdbfs_blockid_t block = sb->fs_ffblock;
	u_int8_t	* byte;

	/* Get a pointer to the block bit map */
	blockbitmap = mmapblockaligned(dbfd, sb->fs_btno, sb->fs_btsize);

	/* Check to see if the block reported as the first free is actually free
	 */
	byte = blockbitmap + block/8;

	/* This check really should not fail... */
	if ((*byte & ((u_int8_t)0x80>>block%8)) != 0)
	{
		/* For now we don't handle the case of ffblock pointing to a bad block
		 */
		return 0;
	}

	/* Mask the bit to 1 */
	*byte |= (u_int8_t)0x80 >> block % 8;

	/* Increment the first free block to the next block */
	sb->fs_ffblock++;
	sb->fs_bfree--;

	/* Zero out the block */
	zeroblock(dbfd, block);

	return block;
}
