                                         /**
 * A class that lets the user draw with the mouse. The
 * window knows how to redraw itself.
 */

import org.kde.qt.*;

public class QTimerTest extends QWidget {
	
	public static final int COLOR_MENU_ID_BLACK = 0;
	public static final int COLOR_MENU_ID_RED = 1;
	public static final int COLOR_MENU_ID_BLUE = 2;
	public static final int COLOR_MENU_ID_GREEN = 3;
	public static final int COLOR_MENU_ID_YELLOW = 4;
	
	public static int eventCount = 0;
	
	private QMenuBar _menubar;
	private QPopupMenu _filemenu;
	private QPopupMenu _colormenu;
	private QPopupMenu _helpmenu;
	private QScrollView _scrollview;
	private ScribbleArea _scribblearea;
	
	public class ScribbleArea extends QWidget {
	    private QPoint _last;
	    private QColor _currentcolor;
	
	    private QPixmap _buffer;
	    private QPopupMenu _popupmenu;
	
		/**
		  * The constructor. Initializes the member variables.
		  */
		ScribbleArea()
		{
			// initialize member variables
			_buffer = new QPixmap();
			_last = new QPoint();
			_currentcolor = Qt.black();
			
			// don't blank the window before repainting
			setBackgroundMode( NoBackground );
			
			// create a pop-up menu
			_popupmenu = new QPopupMenu();
			_popupmenu.insertItem( "&Clear", this, Qt.SLOT( "slotClearArea()" ) );
		}
		
		/**
		  * This slot sets the curren color for the scribble area. It will be
		  * connected with the colorChanged( QColor ) signal from the
		  * ScribbleWindow.
		  */
		public void setColor( QColor new_color )
		{
			_currentcolor = new_color;
		}
		
		/**
		 * This slot clears the drawing area by filling the off-screen buffer with
		 * white and copying it over to the window.
		 */
		public void slotClearArea()
		{
			// fill the off screen buffer with plain white
			_buffer.fill( Qt.white() );
			
			// and copy it over to the window
			QPaintDevice.bitBlt( this, 0, 0, _buffer );
		}
		
		
		/**
		  * This method does the actual loading. It relies on QPixmap (and the
		  * underlying I/O machinery) to determine the filetype.
		  */
		public void slotLoad( String filename )
		{
			if ( !_buffer.load( filename ) )
				QMessageBox.warning( null, "Load error", "Could not load file" );
				
			repaint();  // refresh the window
		}
		
		
		/**
		 * This method does the actual saving. We hard-code the file type as
		 * BMP. Unix users might want to replace this with something like XPM.
		 */
		public void slotSave( String filename )
		{
			if( !_buffer.save( filename, "BMP" ) )
				QMessageBox.warning( null, "Save error", "Could not save file" );
		}
		
		
		/**
		  * This method is called whenever the user presses the
		  * mouse over the window. It just records the position of the mouse
		  * at the time of the click.
		  */
		public void mousePressEvent(QMouseEvent event)
		{
			if ( event.button() == RightButton )
				_popupmenu.exec( QCursor.pos() );
			else
			  {
				_last = event.pos();	// retrieve the coordinates from the event
			  }
		}
		
		
		/**
		  * The method is called whenever the usr moves the mouse
		  * while the mouse button is pressed. If we had called
		  * setMouseTracking(true) before, the method would also be called
		  * when the mouse was moved with any button pressed. We know that
		  * we haven't, and thus don't have to check whether any buttons are
		  * pressed.
		  */
		public void mouseMoveEvent(QMouseEvent event) {
			// create a QPainter object for drawing onto the window
			QPainter windowpainter = new QPainter();
			// and another QPainter object for drawing int an off-screen pixmap
			QPainter bufferpainter = new QPainter();
			
			// start painting
			windowpainter.begin( this ); // this painter paints onto the window
			bufferpainter.begin( _buffer );  // and this one paints in the buffer
		
			// set a standard pen with the currently selected color
			windowpainter.setPen( _currentcolor );
			bufferpainter.setPen( _currentcolor );
		
			// draw a line in both the window and the buffer
			windowpainter.drawLine( _last, event.pos() );
			bufferpainter.drawLine( _last, event.pos() );
		
			// done with painting
			windowpainter.end();
			bufferpainter.end();
		
			// remember the current mouse position
			_last = event.pos();
			eventCount++;						
		}
		/**
		  * This method is called whenever the widget needs
		  * painting, for example when it has been obscured and then revealed again.
		  */
		public void paintEvent(QPaintEvent event) {
			QPaintDevice.bitBlt(this, 0, 0, _buffer);
		}
	
		/**
		  * This method get called whenever the widget needs
		  * painting, for example, when it has been obscured and then revealed again.
		  */
		public void resizeEvent(QResizeEvent event) {
			QPixmap save = new QPixmap( _buffer );
			_buffer.resize( event.size() );
			_buffer.fill( Qt.white() );
			QPaintDevice.bitBlt( _buffer, 0, 0, save );
		}
	}
	
	QTimerTest()
	{
		/* The next lines build the menu bar. We first create the menus
		* one by one, then add them to the menu bar. */
		_filemenu = new QPopupMenu();  // create a file menu
		_filemenu.insertItem( "&Load", this, Qt.SLOT( "slotLoad()" ) );
		_filemenu.insertItem( "&Save", this, Qt.SLOT( "slotSave()" ) );
		_filemenu.insertSeparator();
		_filemenu.insertItem( "&Quit", Qt.qApp(), Qt.SLOT( "quit()" ) );
		
		_colormenu = new QPopupMenu(); // create a color menu
		_colormenu.insertItem( "B&lack", COLOR_MENU_ID_BLACK);
		_colormenu.insertItem( "&Red", COLOR_MENU_ID_RED);
		_colormenu.insertItem( "&Blue", COLOR_MENU_ID_BLUE);
		_colormenu.insertItem( "&Green", COLOR_MENU_ID_GREEN);
		_colormenu.insertItem( "&Yellow", COLOR_MENU_ID_YELLOW);
		QObject.connect( _colormenu, SIGNAL( "activated(int)" ),
						 this, SLOT( "slotColorMenu(int)" ) );
						
		_helpmenu = new QPopupMenu();  // create a help menu
		_helpmenu.insertItem( "&About QtScribble", this, SLOT( "slotAbout()" ) );
		_helpmenu.insertItem( "&About Qt", this, SLOT( "slotAboutQt()" ) );
		
		_menubar = new QMenuBar( this, "" );  // create a menu bar
		_menubar.insertItem( "&File", _filemenu );
		_menubar.insertItem( "&Color", _colormenu );
		_menubar.insertItem( "&Help", _helpmenu );
		
		/* We create a QScrollView and a ScribbleArea. The ScribbleArea will
		 * be managed by the scroll view.*/
		_scrollview = new QScrollView( this );
		_scrollview.setGeometry( 0, _menubar.height(),
		 							width(), height() - _menubar.height() );
		_scribblearea = new ScribbleArea();
		_scribblearea.setGeometry( 0, 0, 1000, 1000 );
		_scrollview.addChild( _scribblearea );
		QObject.connect( this, SIGNAL( "colorChanged(QColor)" ),
						 _scribblearea, SLOT( "setColor(QColor)" ) );
		QObject.connect( this, SIGNAL( "save(java.lang.String)" ),
						 _scribblearea, SLOT( "slotSave(java.lang.String)" ) );
		QObject.connect( this, SIGNAL( "load(java.lang.String)" ),
						 _scribblearea, SLOT( "slotLoad(java.lang.String)" ) );
	}
	
	public void resizeEvent( QResizeEvent event )
	{
		/* When the whole window is resized, we have to rearrange the geometry
		 * in the ScribbleWindow as well. Note that the ScribbleArea does not need
		 * to be changed. */
		 _scrollview.setGeometry( 0, _menubar.height(),
		 							width(), height() - _menubar.height() );
	}

	
	
	public void slotAbout()
	{
		QMessageBox.information( this, "About QtScribble 5",
										"This is the Scribble 5 application\n" +
										"Copyright 1998 by Mathias Kalle Dalheimer\n"
										);
	}
	
	public void slotAboutQt()
	{
		QMessageBox.aboutQt( this, "About Qt" );
	}
	
	public void slotColorMenu( int item )
	{
		switch( item )
		{
			case COLOR_MENU_ID_BLACK:
				emit("colorChanged", Qt.black());
				break;
			case COLOR_MENU_ID_RED:
				emit("colorChanged", Qt.darkRed());
				break;
			case COLOR_MENU_ID_BLUE:
				emit("colorChanged", Qt.darkBlue());
				break;
			case COLOR_MENU_ID_GREEN:
				emit("colorChanged", Qt.darkGreen());
				break;
			case COLOR_MENU_ID_YELLOW:
				emit("colorChanged", Qt.yellow());
				break;
		}
	}
	
	
	/**
	  * This is the slot for the menu item File/Load. It opens a
	  * QFileDialog to ask the user for a filename, then emits a save()
	  * signal with the filename as parameter.
	  */
	public void slotLoad()
	{
		/* Open a file dialog for loading. The default directory is the
		 * current directory, the filter *.bmp.
		 */
		String filename = QFileDialog.getOpenFileName( ".", "*.bmp", this );
		if ( !filename.equals("") )
			emit("load", filename);
	}
	
	/**
	  * This is the slot for the menu item File/Load. It opens a
	  * QFileDialog to ask the user for a filename, then emits a save()
	  * signal with the filename as parameter.
	  */
	public void slotSave()
	{
		/* Open a file dialog for saving. The default directory is the
		 * current directory, the filter *.bmp.
		 */
		String filename = QFileDialog.getSaveFileName( ".", "*.bmp", this );
		if ( !filename.equals("") )
			emit("save", filename);
	}
	
	public void timerDone()
	{
//		System.out.println("in timerDone...");
	}
	
	public static void main(String[] args)
	{
		QApplication myapp = new QApplication(args);
		QTimerTest mywidget = new QTimerTest();
		mywidget.setGeometry(50, 500, 400, 400);
    	
    	QTimer timer = new QTimer( mywidget );
		connect( timer, Qt.SIGNAL("timeout()"),
					mywidget, Qt.SLOT("timerDone()") );
		timer.start( 2000, false );                 // 2 seconds single-shot
		
		myapp.setMainWidget(mywidget);

		AppWizardBase mywizard = new AppWizardBase(mywidget, "", false, 0);
		mywizard.show();
		FilePropsPageBase mypropspage = new FilePropsPageBase(mywidget, "", 0);
		mypropspage.show();
		
		mywidget.show();
		myapp.exec();
		System.out.println("event count: " + eventCount);
		return;
	}
	
	static {
                qtjava.initialize();
	}
}
